/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Jun 24, 2009
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:  vhaiswwerfej
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.ihe.xca.datasource.configuration;

import gov.va.med.OID;
import gov.va.med.RoutingToken;
import gov.va.med.RoutingTokenImpl;
import gov.va.med.exceptions.RoutingTokenFormatException;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSourceImpl;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.ihe.xca.datasource.XCADocumentSetDataSourceService;
import java.io.Serializable;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author vhaiswwerfej
 *
 */
public class XCADataSourceConfiguration
implements Serializable
{
	private static final long serialVersionUID = -7814886810785999850L;
	
	public final static String defaultKeystoreUrl = "file:///c:/VixConfig/xca.keystore";
	public final static String defaultTruststoreUrl = "file:///c:/vixconfig/xca.truststore";
	public final static String defaultTLSProtocol = "xcas";
	public final static int defaultTLSPort = PORT;
	
	public final static int defaultQueryTimeout = 45000;
	public final static int defaultRetrieveTimeout = 120000;
	
	/**
	 * Create a configuration with the minimum default values
	 * @return
	 * @throws MalformedURLException
	 */
	public static XCADataSourceConfiguration create()
	throws MalformedURLException 
	{
		return new XCADataSourceConfiguration(
			null, null, null, null, null, 443, true, defaultQueryTimeout, defaultRetrieveTimeout
		);
	}
	
	public static XCADataSourceConfiguration create(
		String keystorePassword, String truststorePassword)
	throws MalformedURLException 
	{
		return XCADataSourceConfiguration.create(
			defaultKeystoreUrl, 
			keystorePassword, 
			defaultTruststoreUrl, 
			truststorePassword);
	}
	
	public static XCADataSourceConfiguration create(
		String keystoreUrl, 
		String keystorePassword, 
		String truststoreUrl, 
		String truststorePassword)
	throws MalformedURLException 
	{
		return XCADataSourceConfiguration.create(
			keystoreUrl, 
			keystorePassword, 
			truststoreUrl, 
			truststorePassword, 
			defaultTLSProtocol,
			defaultTLSPort,
			XCADataSourceProxyConfiguration.DEFAULT_ALLOWPARTIALSUCCESS.booleanValue());
	}
	
	public static XCADataSourceConfiguration create(
		String keystoreUrl, 
		String keystorePassword, 
		String truststoreUrl, 
		String truststorePassword, 
		String federationSslProtocol,
		int tlsPort,
		boolean allowPartialSuccess)
	throws MalformedURLException 
	{
		if(keystoreUrl == null)
			keystoreUrl = defaultKeystoreUrl;
		if(truststoreUrl == null)
			truststoreUrl = defaultTruststoreUrl;
		if(federationSslProtocol == null)
			federationSslProtocol = defaultTLSProtocol;
		if(tlsPort <= 0)
			tlsPort = defaultTLSPort;
		
		return new XCADataSourceConfiguration(
			keystoreUrl, 
			keystorePassword, 
			truststoreUrl, 
			truststorePassword, 
			federationSslProtocol,
			tlsPort, 
			allowPartialSuccess,
			defaultQueryTimeout, 
			defaultRetrieveTimeout
		);
	}

	// ====================================================================================
	private SortedSet<XCASiteConfiguration> artifactSourceConfigurations = 
		new TreeSet<XCASiteConfiguration>(new XCASiteConfigurationComparator()); 
	private boolean useCertificate = false;
	private TLSConfiguration tlsConfiguration;
	private XCADataSourceProxyConfiguration proxyConfiguration;
	private transient Logger logger = LogManager.getLogger(this.getClass());
	private Integer queryTimeout;
	private Integer retrieveTimeout;
	
	/**
	 * @return the logger
	 */
	protected synchronized Logger getLogger()
	{
		if(logger == null)
			logger = LogManager.getLogger(this.getClass());
		return this.logger;
	}

	/**
	 * @param siteConfigurations the siteConfigurations to set
	 */
	//public void setSiteConfigurations(List<XCASiteConfiguration> siteConfigurations) {
	//	this.siteConfigurations = siteConfigurations;
	//}
	
	/**
	 * 
	 * @param truststoreUrl
	 * @param keystoreUrl
	 * @param keystorePassword
	 * @param truststorePassword
	 * @param federationSslProtocol
	 * @throws MalformedURLException 
	 */
	private XCADataSourceConfiguration(
		String keystoreUrl,
		String keystorePassword, 
		String truststoreUrl, 
		String truststorePassword, 
		String tlsProtocol,
		int tlsPort,
		boolean allowPartialSuccess,
		int queryTimeout,
		int retrieveTimeout) 
	throws MalformedURLException 
	{
		super();
		this.tlsConfiguration = TLSConfiguration.create(
			keystoreUrl, keystorePassword, 
			truststoreUrl, truststorePassword,
			tlsProtocol, tlsPort);
		this.proxyConfiguration = new XCADataSourceProxyConfiguration( new Boolean(allowPartialSuccess) );
		this.queryTimeout = queryTimeout;
		this.retrieveTimeout = retrieveTimeout;
	}
	
	/**
	 * @return the tlsConfiguration
	 */
	public TLSConfiguration getTlsConfiguration()
	{
		return this.tlsConfiguration;
	}

	/**
	 * @param tlsConfiguration the tlsConfiguration to set
	 */
	public void setTlsConfiguration(TLSConfiguration tlsConfiguration)
	{
		this.tlsConfiguration = tlsConfiguration;
	}

	/**
	 * @return the truststoreUrl
	 */
	public String getTruststoreUrl() 
	{
		return getTlsConfiguration().getTruststoreUrl().toString();
	}

	/**
	 * @param truststoreUrl the truststoreUrl to set
	 * @throws MalformedURLException 
	 */
	public void setTruststoreUrl(String truststoreUrl) 
	throws MalformedURLException 
	{
		getTlsConfiguration().setTruststoreUrl(new URL(truststoreUrl));
	}

	/**
	 * @return the keystoreUrl
	 */
	public String getKeystoreUrl() 
	{
		return getTlsConfiguration().getKeystoreUrl().toString();
	}

	/**
	 * @param keystoreUrl the keystoreUrl to set
	 * @throws MalformedURLException 
	 */
	public void setKeystoreUrl(String keystoreUrl) 
	throws MalformedURLException 
	{
		getTlsConfiguration().setKeystoreUrl(new URL(keystoreUrl));
	}

	/**
	 * @return the keystorePassword
	 */
	public String getKeystorePassword() 
	{
		return getTlsConfiguration().getKeystorePassword();
	}

	/**
	 * @param keystorePassword the keystorePassword to set
	 */
	public void setKeystorePassword(String keystorePassword) 
	{
		getTlsConfiguration().setKeystorePassword(keystorePassword);
	}

	/**
	 * The alias of the certificate to use as the credentials.
	 * 
	 * @param string
	 */
	public String getKeystoreAlias()
	{
		return getTlsConfiguration().getAlias();
	}
	
	public void setKeystoreAlias(String alias)
	{
		getTlsConfiguration().setAlias(alias);
	}
	
	/**
	 * @return the truststorePassword
	 */
	public String getTruststorePassword() 
	{
		return getTlsConfiguration().getTruststorePassword();
	}

	/**
	 * @param truststorePassword the truststorePassword to set
	 */
	public void setTruststorePassword(String truststorePassword) 
	{
		getTlsConfiguration().setTruststorePassword(truststorePassword);
	}

	/**
	 * @return the useCertificate
	 */
	public boolean isUseCertificate() 
	{
		return useCertificate;
	}

	/**
	 * @param useCertificate the useCertificate to set
	 */
	public void setUseCertificate(boolean useCertificate) 
	{
		this.useCertificate = useCertificate;
	}
	
	public Integer getQueryTimeout()
	{
		return queryTimeout;
	}

	public void setQueryTimeout(Integer queryTimeout)
	{
		this.queryTimeout = queryTimeout;
	}

	public Integer getRetrieveTimeout()
	{
		return retrieveTimeout;
	}

	public void setRetrieveTimeout(Integer retrieveTimeout)
	{
		this.retrieveTimeout = retrieveTimeout;
	}

	/**
	 * @return the tlsProtocol
	 */
	public String getTLSProtocol()
	{
		return getTlsConfiguration().getProtocol();
	}

	/**
	 * @return the tlsPort
	 */
	public int getTLSPort()
	{
		return getTlsConfiguration().getPort();
	}

	/**
	 * @param tlsProtocol the tlsProtocol to set
	 */
	public void setTLSProtocol(String protocol)
	{
		getTlsConfiguration().setProtocol(protocol);
	}

	/**
	 * @param tlsPort the tlsPort to set
	 */
	public void setTLSPort(int port)
	{
		getTlsConfiguration().setPort(port);
	}

	/**
	 * @return the proxyConfiguration
	 */
	public XCADataSourceProxyConfiguration getProxyConfiguration()
	{
		return this.proxyConfiguration;
	}

	/**
	 * @param proxyConfiguration the proxyConfiguration to set
	 */
	public void setProxyConfiguration(XCADataSourceProxyConfiguration proxyConfiguration)
	{
		this.proxyConfiguration = proxyConfiguration;
	}

	/**
	 * 
	 */
	public void clear()
	{
		getTlsConfiguration().clear();
		artifactSourceConfigurations.clear();
		getProxyConfiguration().clear();
	}
	
	// =============================================================================
	/**
	 * 
	 * @param configuration
	 * @param site
	 * @return
	 * @throws MethodException
	 */
	public XCASiteConfiguration findSiteConfiguration(ResolvedArtifactSource resolvedArtifactSource)
	throws MethodException
	{
		if(resolvedArtifactSource == null || resolvedArtifactSource.getArtifactSource() == null)
		{
			String msg = "Either resolvedArtifactSource or resolvedArtifactSource.getArtifactSource() are null.";
			getLogger().error(msg);
			return null;
		}
		
		OID homeCommunityId = resolvedArtifactSource.getArtifactSource().getHomeCommunityId();
		String repositoryId = resolvedArtifactSource.getArtifactSource().getRepositoryId();
		
		XCASiteConfiguration siteConfiguration;
		try
		{
			siteConfiguration = get(homeCommunityId, repositoryId);
			if(siteConfiguration == null)
			{
				String msg = "Could not find XCA gateway site configuration for artifact source '" + resolvedArtifactSource.toString() + "'.";
				getLogger().info(msg);
				return null;
			}
			
			return siteConfiguration;
		}
		catch (RoutingTokenFormatException x)
		{
			throw new MethodException("Exception thrown while finding site configuration for " + resolvedArtifactSource.toString() + ".", x);
		}
	}
	
	/**
	 * Replaces the URL with the specified protocol with a URL formed from the
	 * path.  The resulting ResolvedArtifactSource will have only one query and
	 * one retrieve URL, which will be the ones to use for XCA.  Any component
	 * of the URL may be changed by the URLComponentMerger.
	 * 
	 * @param resolvedArtifactSource
	 * @param siteConfiguration
	 * @return
	 * @throws MalformedURLException 
	 */
	public ResolvedArtifactSource fixupURLPaths(
		ResolvedArtifactSource resolvedArtifactSource,
		XCASiteConfiguration siteConfiguration) 
	throws MalformedURLException
	{
		if(siteConfiguration == null)
			return resolvedArtifactSource;
		
		List<URL> fixedUpQueryUrls = new ArrayList<URL>();
		for( URL metadataUrl : resolvedArtifactSource.getMetadataUrls() )
			if( XCADocumentSetDataSourceService.SUPPORTED_PROTOCOL.equals(metadataUrl.getProtocol()) )
			{
				URL fixedUpUrl = siteConfiguration.getQueryComponentMerger().merge(metadataUrl);
				fixedUpQueryUrls.add(fixedUpUrl); 
			}
		
		List<URL> fixedUpRetrieveUrls = new ArrayList<URL>();
		for( URL retrieveUrl : resolvedArtifactSource.getArtifactUrls() )
			if( XCADocumentSetDataSourceService.SUPPORTED_PROTOCOL.equals(retrieveUrl.getProtocol()) )
			{
				URL fixedUpUrl = siteConfiguration.getRetrieveComponentMerger().merge(retrieveUrl);
				fixedUpRetrieveUrls.add(fixedUpUrl); 
			}
		
		ResolvedArtifactSourceImpl result = ResolvedArtifactSourceImpl.create(
			resolvedArtifactSource.getArtifactSource(), fixedUpQueryUrls, fixedUpRetrieveUrls
		);
		
		return result;
	}
	
	/**
	 * @param siteConfiguration
	 */
	public void add(XCASiteConfiguration siteConfiguration)
	{
		synchronized(artifactSourceConfigurations)
		{
			artifactSourceConfigurations.add(siteConfiguration);
		}
	}
	
	/**
	 * 
	 * @param siteNumber
	 * @return
	 * @throws RoutingTokenFormatException 
	 */
	public XCASiteConfiguration get(OID homeCommunityId, String repositoryId) 
	throws RoutingTokenFormatException
	{
		if(homeCommunityId == null || repositoryId == null)
			return null;
		
		return get(RoutingTokenImpl.create(homeCommunityId, repositoryId));
	}
	
	public XCASiteConfiguration get(RoutingToken target) 
	throws RoutingTokenFormatException
	{
		synchronized(artifactSourceConfigurations)
		{
			for(XCASiteConfiguration siteConfig : artifactSourceConfigurations)
				if( target.equals(siteConfig.getRoutingToken()) )
					return siteConfig;
		}
		return null;
	}
	
	/**
	 * 
	 * @param siteNumber
	 * @throws RoutingTokenFormatException 
	 */
	public void remove(OID homeCommunityId, String repositoryId) 
	throws RoutingTokenFormatException
	{
		if(homeCommunityId == null || repositoryId == null)
			return;
		remove(RoutingTokenImpl.create(homeCommunityId, repositoryId));
	}
	
	public void remove(RoutingToken target) 
	throws RoutingTokenFormatException
	{	
		synchronized(artifactSourceConfigurations)
		{
			XCASiteConfiguration siteToRemove = get(target);
			
			if(siteToRemove != null)
				artifactSourceConfigurations.remove(siteToRemove);
		}
	}
	
	/**
	 * 
	 */
	@Override
	public String toString()
	{
		StringBuilder sb = new StringBuilder();
		
		sb.append("BEGIN " + this.getClass().getName() + " \n");
		for(XCASiteConfiguration site : artifactSourceConfigurations)
		{
			sb.append('\t');
			sb.append(site.toString());
			sb.append('\n');
		}
		sb.append("END " + this.getClass().getName() + " \n");
		
		return sb.toString();
	}
}
