/**
 * 
 * Date Created: Apr 26, 2017
 * Developer: vhaisltjahjb
 */
package gov.va.med.imaging.viewer.datasource;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.RoutingToken;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.interfaces.exceptions.SecurityCredentialsExpiredException;
import gov.va.med.imaging.datasource.exceptions.InvalidCredentialsException;
import gov.va.med.imaging.datasource.exceptions.UnsupportedProtocolException;
import gov.va.med.imaging.exchange.business.ResolvedSite;
import gov.va.med.imaging.exchange.business.Site;
import gov.va.med.imaging.exchange.enums.ImagingSecurityContextType;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.url.vista.StringUtils;
import gov.va.med.imaging.url.vista.VistaQuery;
import gov.va.med.imaging.url.vista.exceptions.InvalidVistaCredentialsException;
import gov.va.med.imaging.url.vista.exceptions.VistaMethodException;
import gov.va.med.imaging.viewer.business.DeleteImageUrn;
import gov.va.med.imaging.viewer.business.DeleteImageUrnResult;
import gov.va.med.imaging.viewer.business.FlagSensitiveImageUrn;
import gov.va.med.imaging.viewer.business.FlagSensitiveImageUrnResult;
import gov.va.med.imaging.viewer.business.LogAccessImageUrn;
import gov.va.med.imaging.viewer.business.LogAccessImageUrnResult;
import gov.va.med.imaging.vistadatasource.common.VistaCommonUtilities;
import gov.va.med.imaging.vistadatasource.session.VistaSession;
import gov.va.med.imaging.vistaimagingdatasource.AbstractVistaImagingDataSourceService;
import gov.va.med.imaging.vistaimagingdatasource.common.VistaImagingCommonUtilities;

/**
 * @author Administrator
 *s
 */
public class ViewerImagingDataSourceService
extends AbstractVistaImagingDataSourceService
implements ViewerImagingDataSourceSpi
{
	
	public final static String SUPPORTED_PROTOCOL = "vistaimaging";

	private final static Logger logger = LogManager.getLogger(ViewerImagingDataSourceService.class);
	// The required version of VistA Imaging needed to execute the RPC calls for this operation
	public final static String MAG_REQUIRED_VERSION ="3.0P59"; 
		
	//private SmbStorageUtility storageUtility = new SmbStorageUtility();
	
	/**
	 * @param resolvedArtifactSource
	 * @param protocol
	 */
	public ViewerImagingDataSourceService(ResolvedArtifactSource resolvedArtifactSource, 
			String protocol)
	{
		super(resolvedArtifactSource, protocol);
		if(! (resolvedArtifactSource instanceof ResolvedSite) )
			throw new UnsupportedOperationException("The artifact source must be an instance of ResolvedSite and it is a '" + resolvedArtifactSource.getClass().getSimpleName() + "'.");
	}

	public static ViewerImagingDataSourceService create(ResolvedArtifactSource resolvedArtifactSource, String protocol)
	throws ConnectionException, UnsupportedProtocolException
	{
		return new ViewerImagingDataSourceService(resolvedArtifactSource, protocol);
	}
	
	/**
	 * The artifact source must be checked in the constructor to assure that it is an instance
	 * of ResolvedSite.
	 * 
	 * @return
	 */
	protected ResolvedSite getResolvedSite()
	{
		return (ResolvedSite)getResolvedArtifactSource();
	}
	
	protected Site getSite()
	{
		return getResolvedSite().getSite();
	}
	
	private VistaSession getVistaSession() 
    throws IOException, ConnectionException, MethodException
    {
	    return VistaSession.getOrCreate(getMetadataUrl(), getSite(), ImagingSecurityContextType.MAG_WINDOWS);
    }
	
	private Logger getLogger()
	{
		return logger;
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.datasource.ImagingPatientDataSource#isVersionCompatible()
	 */
	@Override
	public boolean isVersionCompatible() 
	throws SecurityCredentialsExpiredException
	{
		String version = VistaImagingCommonUtilities.getVistaDataSourceImagingVersion(
				getVistaImagingConfiguration(), this.getClass(), 
				MAG_REQUIRED_VERSION);
		
		logger.info("isVersionCompatible searching for version [" + version + "], TransactionContext (" + TransactionContextFactory.get().getDisplayIdentity() + ").");
		VistaSession localVistaSession = null;
		try
		{
			localVistaSession = getVistaSession();	
			return VistaImagingCommonUtilities.isVersionCompatible(version, localVistaSession);	
		}
		catch(SecurityCredentialsExpiredException sceX)
		{
			// caught here to be sure it gets thrown as SecurityCredentialsExpiredException, not ConnectionException
			throw sceX;
		}
		catch(MethodException mX)
		{
			logger.error("There was an error finding the installed Imaging version from VistA", mX);
			TransactionContextFactory.get().addDebugInformation("isVersionCompatible() failed, " + (mX == null ? "<null error>" : mX.getMessage()));
		}
		catch(ConnectionException cX)
		{
			logger.error("There was an error finding the installed Imaging version from VistA", cX);
			TransactionContextFactory.get().addDebugInformation("isVersionCompatible() failed, " + (cX == null ? "<null error>" : cX.getMessage()));
		}		
		catch(IOException ioX)
		{
			logger.error("There was an error finding the installed Imaging version from VistA", ioX);
			TransactionContextFactory.get().addDebugInformation("isVersionCompatible() failed, " + (ioX == null ? "<null error>" : ioX.getMessage()));
		}
		finally
		{
			try{localVistaSession.close();}
			catch(Throwable t){}
		}		
		return false;
	}
	
	protected String getDataSourceVersion()
	{
		return "1";
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.viewer.datasource.ViewerImagingDataSourceSpi#deleteImage(gov.va.med.RoutingToken, String)
	 */
	@Override
	public List<DeleteImageUrnResult> deleteImages(
			RoutingToken globalRoutingToken,
			List<DeleteImageUrn> imageUrns)
	throws MethodException,ConnectionException 
	{
		VistaCommonUtilities.setDataSourceMethodAndVersion("deleteImage", getDataSourceVersion());
		VistaSession vistaSession = null;
		getLogger().info(
				"deleteImage imageUrns(" + imageUrns + ") " +
				"TransactionContext (" + TransactionContextFactory.get().getTransactionId() + ").");
		try
		{
			List<String> p34ImageUrns = getP34ImageFromDeleteImageUrnList(imageUrns);
			List<DeleteImageUrnResult> deletedList = null;
			
			if ((imageUrns != null) && (imageUrns.size() > 0))
			{
				vistaSession = getVistaSession();
				
				Map<String, String> imageUrnMap = MapImageUrnByIen(imageUrns);
				VistaQuery query = ViewerImagingQueryFactory.createDeleteImagesQuery(imageUrns);
				
				String ret = vistaSession.call(query);
				
				deletedList = ViewerImagingTranslator.translateDeleteImagesResult(ret, imageUrnMap);
			}
			
			if ((p34ImageUrns == null) || (p34ImageUrns.size() == 0))
			{
				return deletedList;
			}
			else
			{
				return ViewerImagingTranslator.mergeDeleteImagesErrorResult(
						deletedList, p34ImageUrns);
			}
		}
		catch(VistaMethodException vmX)
		{
			throw new MethodException(vmX);
		}
		catch(InvalidVistaCredentialsException icX)
		{
			throw new InvalidCredentialsException(icX);
		}
		catch(IOException ioX)
		{
			throw new ConnectionException(ioX);
		}
		finally
		{
			try{vistaSession.close();}
			catch(Throwable t){}
		}
	}

	private List<String> getP34ImageFromDeleteImageUrnList(List<DeleteImageUrn> imageUrns) 
	{
		List<String> errList = new ArrayList<String>();
		Iterator<DeleteImageUrn> iter = imageUrns.iterator();
		
		while (iter.hasNext()) {
			DeleteImageUrn urn = iter.next();

			if (urn.getValue().toLowerCase().startsWith("urn:vap34image:"))
			{
				errList.add(urn.getValue());
				iter.remove();
			}
		}
		
		return errList;
	}

	private Map<String, String> MapImageUrnByIen(List<DeleteImageUrn> imageUrns) {
		Map<String, String> result = new HashMap<String, String>();
		for(int i = 0; i < imageUrns.size(); i++)
		{
			result.put(	
					ViewerImagingTranslator.getImageIen(imageUrns.get(i).getValue()), 
					imageUrns.get(i).getValue());
		}

		return result;
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.viewer.datasource.ViewerImagingDataSourceSpi#flagImagesAsSensitive(gov.va.med.RoutingToken, List)
	 */
	@Override
	public List<FlagSensitiveImageUrnResult> flagImagesAsSensitive(
			RoutingToken globalRoutingToken,
			List<FlagSensitiveImageUrn> imageUrns)
	throws MethodException,ConnectionException 
	{
		VistaCommonUtilities.setDataSourceMethodAndVersion("flagImagesAsSensitive", getDataSourceVersion());
		VistaSession vistaSession = null;
		getLogger().info(
				"flagImagesAsSensitive imageUrns(" + imageUrns + ") " +
				"TransactionContext (" + TransactionContextFactory.get().getTransactionId() + ").");
		try
		{
			List<String> p34ImageUrns = getP34ImageFromFlagSensitiveImageUrnList(imageUrns);
			List<FlagSensitiveImageUrnResult> flagSensList = null;

			if ((imageUrns != null) && (imageUrns.size() > 0))
			{
				vistaSession = getVistaSession();
				
				Map<String, String> imageUrnMap = MapFlagSensitiveImageUrnByIen(imageUrns);
				VistaQuery query = ViewerImagingQueryFactory.createFlagSensitiveImagesQuery(imageUrns);
				
				String ret = vistaSession.call(query);

				flagSensList = ViewerImagingTranslator.translateFlagSensitiveImagesResult(
						ret, imageUrnMap);  
			}
			
			if ((p34ImageUrns == null) || (p34ImageUrns.size() == 0))
			{
				return flagSensList;
			}
			else
			{
				return ViewerImagingTranslator.mergeFlagSensitiveErrorResult(flagSensList, p34ImageUrns);
			}
		}
		catch(VistaMethodException vmX)
		{
			throw new MethodException(vmX);
		}
		catch(InvalidVistaCredentialsException icX)
		{
			throw new InvalidCredentialsException(icX);
		}
		catch(IOException ioX)
		{
			throw new ConnectionException(ioX);
		}
		finally
		{
			try{vistaSession.close();}
			catch(Throwable t){}
		}
	}

	private List<String> getP34ImageFromFlagSensitiveImageUrnList(List<FlagSensitiveImageUrn> imageUrns) 
	{
		List<String> errList = new ArrayList<String>();
		Iterator<FlagSensitiveImageUrn> iter = imageUrns.iterator();
		
		while (iter.hasNext()) {
			FlagSensitiveImageUrn urn = iter.next();

			if (urn.getImageUrn().toLowerCase().startsWith("urn:vap34image:"))
			{
				errList.add(urn.getImageUrn());
				iter.remove();
			}
		}
		
		return errList;
	}


	private Map<String, String> MapFlagSensitiveImageUrnByIen(List<FlagSensitiveImageUrn> imageUrns) {
		Map<String, String> result = new HashMap<String, String>();
		for(int i = 0; i < imageUrns.size(); i++)
		{
			result.put(	
					ViewerImagingTranslator.getImageIen(imageUrns.get(i).getImageUrn()), 
					imageUrns.get(i).getImageUrn());
		}

		return result;
	}

	@Override
	public String getUserInformationByUserId(RoutingToken globalRoutingToken, String userId)
	throws MethodException, ConnectionException 
	{
		VistaCommonUtilities.setDataSourceMethodAndVersion("getUserInformationByUserId", getDataSourceVersion());
		VistaSession vistaSession = null;
		logger.info("getUserInformationByUserId TransactionContext (" + TransactionContextFactory.get().getTransactionId() + ").");
		try
		{
			vistaSession = getVistaSession();
			VistaQuery query = ViewerImagingQueryFactory.createGetUserInformationByUserIdQuery(userId);
			logger.info("Executing query '" + query.getRpcName() + "'.");
			String rtn = vistaSession.call(query);
			String[] userInfo = StringUtils.Split(rtn,StringUtils.CARET);
			return userInfo[0] + StringUtils.CARET + userInfo[1];
		}
		catch(VistaMethodException vmX)
		{
			throw new MethodException(vmX);
		}
		catch(InvalidVistaCredentialsException icX)
		{
			throw new InvalidCredentialsException(icX);
		}
		catch(IOException ioX)
		{
			throw new ConnectionException(ioX);
		}
		finally
		{
			try{vistaSession.close();}
			catch(Throwable t){}
		}

	}

	@Override
	public List<LogAccessImageUrnResult> logImageAccessByUrns(
			RoutingToken globalRoutingToken, 
			String patientIcn,
			String patientDfn,
			List<LogAccessImageUrn> imageUrns)
	throws MethodException, ConnectionException 
	{
		VistaCommonUtilities.setDataSourceMethodAndVersion("logImageAccessByUrns", getDataSourceVersion());
		VistaSession vistaSession = null;
		getLogger().info(
				"logImageAccessByUrns imageUrns(" + imageUrns + ") " +
				"TransactionContext (" + TransactionContextFactory.get().getTransactionId() + ").");
		try
		{
			List<String> p34ImageUrns = getP34ImageFromLogAccessImageUrnList(imageUrns);
			List<LogAccessImageUrnResult> logAccessList = null;

			if ((imageUrns != null) && (imageUrns.size() > 0))
			{
				vistaSession = getVistaSession();
				
				if ((patientDfn == null) || (patientDfn.isEmpty()))
				{
					patientDfn = VistaCommonUtilities.getPatientDFN(vistaSession, patientIcn);
					if ((patientDfn == null) || (patientDfn.isEmpty()))
					{
						return null;
					}
				}

				Map<String, String> imageUrnMap = MapLogAccessImageUrnByIen(imageUrns);
				VistaQuery query = ViewerImagingQueryFactory.createLogAccessImagesQuery(patientDfn, imageUrns);
				
				String ret = vistaSession.call(query);

				logAccessList = ViewerImagingTranslator.translateLogAccessImagesResult(ret, imageUrnMap);
			}
			
			if ((p34ImageUrns == null) || (p34ImageUrns.size() == 0))
			{
				return logAccessList;
			}
			else
			{
				return ViewerImagingTranslator.mergeLogAccessErrorResult(logAccessList, p34ImageUrns);
			}
		}
		catch(VistaMethodException vmX)
		{
			throw new MethodException(vmX);
		}
		catch(InvalidVistaCredentialsException icX)
		{
			throw new InvalidCredentialsException(icX);
		}
		catch(IOException ioX)
		{
			throw new ConnectionException(ioX);
		}
		finally
		{
			try{vistaSession.close();}
			catch(Throwable t){}
		}
	}

	private List<String> getP34ImageFromLogAccessImageUrnList(List<LogAccessImageUrn> imageUrns) {
		List<String> errList = new ArrayList<String>();
		Iterator<LogAccessImageUrn> iter = imageUrns.iterator();
		
		while (iter.hasNext()) {
			LogAccessImageUrn urn = iter.next();

			if (urn.getImageUrn().toLowerCase().startsWith("urn:vap34image:"))
			{
				errList.add(urn.getImageUrn());
				iter.remove();
			}
		}

		return errList;
	}

	private Map<String, String> MapLogAccessImageUrnByIen(List<LogAccessImageUrn> imageUrns) {
		Map<String, String> result = new HashMap<String, String>();
		for(int i = 0; i < imageUrns.size(); i++)
		{
			result.put(	
					ViewerImagingTranslator.getImageIen(imageUrns.get(i).getImageUrn()), 
					imageUrns.get(i).getImageUrn());
		}

		return result;
	}
	
}
