package p34.database;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.junit.Test;

import p34.test.BaseTest;
import p34.Utils;
import p34.DB.DBMethodsDataSource;
import p34.DB.InstanceFileN;
import p34.DB.PatientRef;
import p34.DB.ProcedureRef;
import p34.DB.Study;
import p34.DB.Series;
import p34.DB.SOPInstance;
import p34.DB.InstanceFile;
import gov.va.med.imaging.rpc.impl.IConnection;

import java.util.Random;

/**
 * Created on Oct 09, 2009
 * 
 * @author vhaiswtittoc
 *
 */

public class DBMethodTests extends BaseTest {
	
	private static String patRefPK="-1";
	private static String procRefPK="-1";
	private static String studyPK="-1";
	private static String seriesPK="-1";
	private static String sopiPK="-1";
	private static String iFilePK="-1";

	private static String curDFN="123456";
	private static String curAccNum="1235678-125";
	private static String curStudyUID="1.2.840.113754.1.4.671.6939684.8882.1.31506.2179";
	private static String curSeriesUID="1.2.840.113754.1.4.671.6939684.8882.2.31506.2179";
	private static String curSOPIUID="1.2.840.113754.1.4.671.6939684.8882.3.31506.2179";
//	private static String curFName="FileName.DCM";
	private static String curToken="token123456";
//	private static String curWriteNetLoc="C:\\stuff\\";
	private static String lastSeriesFrOfRefUID="1.2.3";
	// DB FileNames:  "IMAGING PATIENT REFERENCE", "IMAGING PROCEDURE REFERENCE",
	//				  "IMAGE STUDY", "IMAGE SERIES", "IMAGE SOP INSTANCE" or "IMAGE INSTANCE FILE"
	
	
	private static Logger logger = LogManager.getLogger(DBMethodTests.class);
		
	private String mockDFN()
	{
		Random adHoc = new Random();
//		Integer randomNum = 1 + adHoc.nextInt(99999);
//		String dfn = "9" + randomNum.toString();
		Integer randomNum = 1201 + adHoc.nextInt(99); // use range of 1201..1299 
		String dfn = randomNum.toString();
		logger.info("DFN picked = " + dfn);

		return dfn;
	}
	private String mockAccessionNumber()
	{
		Random adHoc = new Random();
//		Integer randomNum1 = 1 + adHoc.nextInt(99999)
		Integer randomNum2 = 1 + adHoc.nextInt(99); // 20
		if (randomNum2 < 10)
			randomNum2+=10;
		randomNum2+=6;
//		String accN = "9" + randomNum1.toString() + "-9" + randomNum2.toString(); // 
		String accN = "112403-2" + randomNum2.toString(); // 112403-nnn
		logger.info("ACN picked = " + accN);
		return accN;
	}
	private String mockUID()
	{
		String uid="1.2.840.113754.99."; 
		Random adHoc = new Random();
		Integer randomNum1 = 1 + adHoc.nextInt(999999);
		Integer randomNum2 = 1 + adHoc.nextInt(999999);
		uid += ((Long)System.currentTimeMillis()).toString();
		uid += "." + randomNum1.toString() + "." + randomNum2.toString();
		return uid;
	}
//	private String mockFileName()
//	{
//		Random adHoc = new Random();
//		Integer randomNum1 = 1 + adHoc.nextInt(999999);
//		Integer randomNum2 = 1 + adHoc.nextInt(99);
//		String fName = randomNum1.toString() + randomNum2.toString() + ".dcm";
//		return fName;
//	}
	private String mockToken()
	{
		Random adHoc = new Random();
		Integer randomNum1 = 1 + adHoc.nextInt(999999);
		Integer randomNum2 = 1 + adHoc.nextInt(999999);
		String fName = "token" + randomNum1.toString() + "-" + randomNum2.toString();
		return fName;
	}
	
	// -------------------------------------------------------------
	private String findPatRefTest(String CurrentDFN, String creatorID) throws Exception
		{
		String paRefPK="-1";
		//	test one patientRef at a time
		//							  enterpID	assigningAuthority creatingEntity IDTtype
		PatientRef patRef = new PatientRef(CurrentDFN, "V", creatorID, "D"); // VA DFN
		String[] resList = DBMethodsDataSource.findPatientRef(connection, patRef);
		dumpResultLines("Find Patient Ref", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			paRefPK=resList[2]; // line 3 is IEN
		} else
			paRefPK="-1";
		return paRefPK;
	}

	public String createPatRefTest(String CurrentDFN, String creatorID) throws Exception
	{
		String paRefPK="-1";
		//							  enterpID	assigningAuthority creatingEntity IDTtype
		PatientRef patRef = new PatientRef(CurrentDFN, "V", creatorID, "D"); // VA DFN
		String[] resList = DBMethodsDataSource.createPatientRef(connection, patRef);
		dumpResultLines("Create Patient Ref", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			paRefPK=resList[2]; // line 3 is IEN
		}
		return paRefPK;
	}
		
	public void updatePatRefTest(String CurrentDFN, String creatorID, String paRefPK) throws Exception
	{
		//	test one patientRef at a time
		//							  enterpID	assigningAuthority creatingEntity IDTtype
		PatientRef patRef = new PatientRef(CurrentDFN, "V", creatorID, "D"); // VA DFN
			String[] resList = DBMethodsDataSource.updatePatientRef(connection, paRefPK, patRef);
		dumpResultLines("Update Patient Ref", resList);
	}
	
	public Boolean getPatAttributesTest(String paRefPK) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getPatientAttributes(connection, paRefPK);
		dumpResultLines("Get Patient Attributes", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean getPatInfoTest(String paRefPK) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getPatientInfo(connection, paRefPK);
		dumpResultLines("Get Patient Info", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean inactivatePatientRefTest(String paRefPK) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivatePatient(connection, paRefPK);
		dumpResultLines("Inactivate Patient Ref", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	// -----------------------------------------------
	private String patientRefTests(String CurrentDFN,  String creatorID) throws Exception
	{
		String paRefPK=findPatRefTest(CurrentDFN, creatorID);
		if (paRefPK.startsWith("-1")) //	make a new patRef
		{
			paRefPK=createPatRefTest(CurrentDFN, creatorID);
		}
		if (!paRefPK.startsWith("-1")) // get attributes and update new patRef
		{
			if (inactivatePatientRefTest(paRefPK)) {
				String paRefPK2=createPatRefTest(CurrentDFN, creatorID);
				if (!paRefPK.equals(paRefPK2)) {
					logger.info("ERROR: After inactivation, Create patient Ref picked PK=" + paRefPK2 + " instead of " + paRefPK);
					paRefPK=paRefPK2;
				}
			}
		}
		if (!paRefPK.startsWith("-1")) {
			if (getPatAttributesTest(paRefPK)) {
				updatePatRefTest(CurrentDFN, creatorID, paRefPK);
			}
		}

		if (!paRefPK.startsWith("-1")) {
			getPatAttributesTest(paRefPK);
			getPatInfoTest(paRefPK);
		}
		return paRefPK; 
	}
	// =====================================================
	private String findProcRefTest(String CurAccN, String creatorID, String paRefPK) throws Exception
	{
		String prRefPK="-1";
//		curAccNum = mockAccessionNumber();
		//	test one procedureRef at a time
		//							            procID AssAuth creEntity procIDTtype pocCreDaTime  pkgIx  // classIx  procEvIx,   SpecSubSpecIx  
		ProcedureRef procRef = new ProcedureRef(CurAccN, "V", creatorID, "RAD", "20091009.144832", "RAD"); //, "CLIN", "2005.85,777", "2005.84,88"); //
		String[] resList = DBMethodsDataSource.findProcRef(connection, paRefPK, procRef);
		dumpResultLines("Find Procedure Ref under Patient Ref", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			prRefPK=resList[2]; // line 3 is IEN
		} else
			prRefPK="-1";
		return prRefPK;
	}

	private String attachProcRefToPatTest(String curAccN, String creatorID, String paRefPK) throws Exception
	{
			String prRefPK="-1";
			//	test one procedureRef at a time
			//							            procID AssAuth creEntity procIDTtype pocCreDaTime  pkgIx  // classIx  procEvIx,   SpecSubSpecIx  
			ProcedureRef procRef = new ProcedureRef(curAccN, "V", creatorID, "RAD", "20091009.144832", "RAD"); // "CLIN", "2005.85,777", "2005.84,88");
			String[] resList = DBMethodsDataSource.attachPatProcRef(connection, paRefPK, procRef);
			dumpResultLines("Attach Procedure Ref to Patient Ref", resList);
			if ((resList != null) && (resList[0].startsWith("0"))) {
				prRefPK=resList[2]; // line 3 is IEN
			} 
			return prRefPK; 
	}

	private Boolean updateProcRefTest(String curAccN, String creatorID, String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one procedureRef at a time
		//							            procID AssAuth creEntity procIDTtype pocCreDaTime  pkgIx  // classIx  procEvIx,   SpecSubSpecIx  
		ProcedureRef procRef = new ProcedureRef(curAccN, "V", creatorID, "RAD", "20091009.144832", "RAD"); //, "CLIN", "2005.85,666", "2005.84,99");
		String[] resList = DBMethodsDataSource.updatePatProcRef(connection, paRefPK, prRefPK, ignoreParentKey, procRef);
		dumpResultLines("Update Procedure Ref", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean getProcRefAttributesTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getProcedureRefAttributes(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Get Procedure Ref Attributes", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean getAccessionNumberTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getAccessionNumber(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Get Accession Number", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getProcedureInfoTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getProcedureInfo(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Get Procedure Info", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getReportTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getReport(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Get Report", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getLabInfoTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getLabInfo(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Get LAB Info", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean inactivateProcRefTest(String paRefPK, String prRefPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivateProcRef(connection, paRefPK, prRefPK, ignoreParentKey);
		dumpResultLines("Inactivate Procedure Ref", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	private Boolean traverseProcRefTest(String patRefIEN, String direction, String ProcRefIEN) throws Exception
	{
		// Direction is one of "FIRST" (default), "LAST", "NEXT" or "PREV"
		String theDirection="FIRST";
		if ((direction.startsWith("FIRST")) || (direction.startsWith("LAST")) || (direction.startsWith("NEXT")) || (direction.startsWith("PREV")))
			theDirection=direction;
		String[] resList = DBMethodsDataSource.traverseProcRef(connection, patRefIEN, theDirection, ProcRefIEN);
		dumpResultLines("Traverse Proc Ref (" + theDirection + ")" , resList);
		return ((resList != null) && (resList[0].startsWith("0")));	
	}
	// -----------------------------------------------
	private String procedureRefTests(String curAccN, String creatorID, String paRefPK) throws Exception
	{
		String prRefPK=findProcRefTest(curAccN, creatorID, paRefPK);
		if (prRefPK.startsWith("-1")) //	make a new procRef
		{
			prRefPK=attachProcRefToPatTest(curAccN, creatorID, paRefPK);
			if (!prRefPK.startsWith("-1")) // update new patRef
			{
				if (inactivateProcRefTest(paRefPK, prRefPK, false)) {
					traverseProcRefTest(paRefPK, "FIRST", "0");
					String prRefPK2=attachProcRefToPatTest(curAccN, creatorID, paRefPK);
					if (!prRefPK.equals(prRefPK2)) {
						logger.info("ERROR: After inactivation, Create Procedure Ref picked PK=" + prRefPK2 + " instead of " + prRefPK);
						prRefPK=prRefPK2;
					}
				}
				updateProcRefTest(curAccN, creatorID, paRefPK, prRefPK, false);
			}
		}
		if (!prRefPK.startsWith("-1")) {
			getProcRefAttributesTest(paRefPK, prRefPK, false);
			getAccessionNumberTest(paRefPK, prRefPK, false);
			getProcedureInfoTest(paRefPK, prRefPK, false);
			getReportTest(paRefPK, prRefPK, false);
//			getLabInfoTest(paRefPK, prRefPK, false);
			
			traverseProcRefTest(paRefPK, "FIRST", "0");
			traverseProcRefTest(paRefPK, "LAST", "0");
			traverseProcRefTest(paRefPK, "NEXT", prRefPK);
			traverseProcRefTest(paRefPK, "PREV", prRefPK);
		}
		return prRefPK; 
	}
	// ====================================================
	private String findStudyByUIDTest(String curStdUID) throws Exception
	{
		String stdPK="-1";
		//	test one study at a time
		//					    studyIUID origIUID  studyID       Desc.       ModalitiesInStd   DateTime              reason  isACQdone, originIx Pri AccN   
		Study study = new Study(curStdUID, "", "Study 123", "Lots of bla-bla", "CR\\RF", "20091011.080645", "Test M (Mike's) code", "P", "V", "R", "");
		String[] resList = DBMethodsDataSource.findStudyByUID(connection, study);
		dumpResultLines("Find Study", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			stdPK=resList[2]; // line 3 is IEN
		} else
			stdPK="-1";
		return stdPK;
	}

	private String attachStudyTest(String paRefPK, String prRefPK, String curStdUID, String curAccN) throws Exception
	{
		String stdPK="-1";
		//	test one study at a time
		//					    studyIUID origIUID  studyID       Desc.       ModalitiesInStd   DateTime              reason   isACQdone, originIx Pri AccN   
		Study study = new Study(curStdUID, "", "Study 123", "Lots of bla-bla", "CR\\RF", "20091011.080645", "Test M (Mike's) code", "A", "V", "R", curAccN);
		String[] resList = DBMethodsDataSource.attachStudy(connection, paRefPK, prRefPK, study);
		dumpResultLines("Attach Study", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			stdPK=resList[2]; // line 3 is IEN
		}
		return stdPK;
	}
	private Boolean updateStudyTest(String prRefPK, String stdIEN, Boolean ignoreParentKey, String curStdUID, String curAccN) throws Exception
	{
		String stdPK="-1";
		//	test one study at a time
		//					    studyIUID origIUID  studyID       Desc.             ModalitiesInStd   DateTime              reason   isACQdone, originIx Pri AccN   
		Study study = new Study(curStdUID, "", "Study 321", "Lots of bla-bla updated", "CR\\RF", "20091011.080645", "Test M (Mike's) code", "C", "V", "H", curAccN);
		String[] resList = DBMethodsDataSource.updateStudy(connection, prRefPK, stdIEN, ignoreParentKey, study);
		dumpResultLines("Update Study", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getStudyTest(String prRefPK, String studyPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getStudy(connection, prRefPK, studyPK, ignoreParentKey);
		dumpResultLines("Get Study", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean inactivateStudyTest(String prRefPK, String studyPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivateStudy(connection, prRefPK, studyPK, ignoreParentKey);
		dumpResultLines("Inactivate Study", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	private Boolean traverseStudyTest(String procRefIEN, String direction, String studyIEN) throws Exception
	{
		// Direction is one of "FIRST" (default), "LAST", "NEXT" or "PREV"
		String theDirection="FIRST";
		if ((direction.startsWith("FIRST")) || (direction.startsWith("LAST")) || (direction.startsWith("NEXT")) || (direction.startsWith("PREV")))
			theDirection=direction;
		String[] resList = DBMethodsDataSource.traverseStudy(connection, procRefIEN, theDirection, studyIEN);
		dumpResultLines("Traverse Study (" + theDirection + ")" , resList);
		return ((resList != null) && (resList[0].startsWith("0")));	
	}
// ------------------------------------------------------
	private String studyTests(String paRefPK, String prRefPK, String curStdUID, String curAccN) throws Exception
	{
		String stdPK=findStudyByUIDTest(curStdUID);
		if (stdPK.startsWith("-1")) //	test one study at a time
		{
			stdPK=attachStudyTest(paRefPK, prRefPK, curStdUID, curAccN);
			if (inactivateStudyTest(prRefPK, stdPK, false)) {
				traverseStudyTest(prRefPK, "FIRST", "0");
				String stdPK2=attachStudyTest(paRefPK, prRefPK, curStdUID, curAccN);
				if (!stdPK.equals(stdPK2)) {
					logger.info("ERROR: After inactivation, Attach Study picked PK=" + stdPK2 + " instead of " + stdPK);
					stdPK=stdPK2;
				} else {
					updateStudyTest(prRefPK, stdPK, false, curStdUID, curAccN);
				}
			}
		}
		if (!stdPK.startsWith("-1")) {
			getStudyTest(prRefPK, stdPK, false);

			traverseStudyTest(prRefPK, "FIRST", "0");
			traverseStudyTest(prRefPK, "LAST", "0");
			traverseStudyTest(prRefPK, "NEXT", stdPK);
			traverseStudyTest(prRefPK, "PREV", stdPK);
		}
//		String junk=findStudyByUIDTest("1.2.840.113754.99.1270748120437.913355.250198");
		return stdPK;
	}
	// ======================================================
	public String findSeriesByUIDTest(String stdPK, String curSerUID) throws Exception
	{
		String serPK="-1";
		//	test one series at a time
		//						  seriesIUID origIUID Ser#  Desc.  Modality  BPart   ASite       Ser-DTime     Ser-Creator DevModel FrOfRefUID Lat. SpatPos  srcAE       retrAE   VIACQEntryPoint    
		Series series = new Series(curSerUID, "", "1", "bla-bla", "CR", "HEAD", "St Louis", "20091121.230645", "Fuji", "ACR5000", mockUID(), "R", "HFP", "CRSimu1", "DGW1-SCP", "DICOM Storage", "3", "CLIN", "2005.85,777", "2005.84,88");
		String[] resList = DBMethodsDataSource.findSeriesByUID(connection, series);
		dumpResultLines("Find Series", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			serPK=resList[2]; // line 3 is IEN
		} else
			serPK="-1";
		return serPK;
	}

	private String attachSeriesTest(String stdPK, String curSerUID, String frOfRefUID) throws Exception
	{
		String serPK="-1";
		//	test one series at a time
		//						 studyIUID origIUID Ser#  Desc.  Modality  BPart    ASite       Ser-DTime     Ser-Creator DevModel FrOfRefUID Lat. SpatPos   srcAE       retrAE   VIACQEntryPoint    
		Series series = new Series(curSerUID, "", "1", "bla-bla", "CR", "HEAD", "St Louis", "20091121.230645", "Fuji", "FCR-5000", frOfRefUID, "R", "HFP", "CRSimu1", "DGW1-SCP", "DICOM Storage", "3", "CLIN", "2005.85,777", "2005.84,88");
		String[] resList = DBMethodsDataSource.attachSeries(connection, stdPK, series);
		dumpResultLines("Attach Series", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			serPK=resList[2]; // line 3 is IEN
		}
		return serPK;
	}
	
	private Boolean updateSeriesTest(String stdPK, String serPK, Boolean ignoreParentKey, String curSerUID, String frOfRefUID) throws Exception
	{
		//						 studyIUID origIUID Ser#        Desc.  Modality  BPart    ASite       Ser-DTime     Ser-Creator DevModel FrOfRefUID Lat. SpatPos   srcAE       retrAE   VIACQEntryPoint  status classIx  procEvIx,   SpecSubSpecIx   
		Series series = new Series(curSerUID, "", "2", "bla-bla updated", "CR", "HEAD", "St Louis", "20091121.230645", "Fuji", "FCR-5000", frOfRefUID, "R", "HFP", "CRSimu1", "DGW1-SCP", "DICOM Storage", "0", "CLIN", "2005.85,666", "2005.84,77");
		String[] resList = DBMethodsDataSource.updateSeries(connection, stdPK, serPK, ignoreParentKey, series);
		dumpResultLines("Update Series", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getSeriesTest(String studyPK, String seriesPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getSeries(connection, studyPK, seriesPK, ignoreParentKey);
		dumpResultLines("Get Series", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean inactivateSeriesTest(String studyPK, String seriesPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivateSeries(connection, studyPK, seriesPK, ignoreParentKey);
		dumpResultLines("Inactivate Series", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean findSeriesByRefUIDTest(String refUID) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.findSeriesByRefUID(connection, refUID);
		dumpResultLines("Find Series By Ref UID", resList);
		return ((resList != null) && (resList.length>0) && (resList[0].startsWith("0")));
	}

	private Boolean traverseSeriesTest(String studyIEN, String direction, String seriesIEN) throws Exception
	{
		// Direction is one of "FIRST" (default), "LAST", "NEXT" or "PREV"
		String theDirection="FIRST";
		if ((direction.startsWith("FIRST")) || (direction.startsWith("LAST")) || (direction.startsWith("NEXT")) || (direction.startsWith("PREV")))
			theDirection=direction;
		String[] resList = DBMethodsDataSource.traverseSeries(connection, studyIEN, theDirection, seriesIEN);
		dumpResultLines("Traverse Series (" + theDirection + ")" , resList);
		return ((resList != null) && (resList[0].startsWith("0")));	
	}
	// --------------------------------------------------	
	private String seriesTests(String stdPK, String curSerUID, String frOfRefUID) throws Exception
	{
		String serPK=findSeriesByUIDTest(stdPK, curSerUID);
		if (serPK.startsWith("-1")) //	test one study at a time
		{
			serPK=attachSeriesTest(stdPK, curSerUID, frOfRefUID);
			if (inactivateSeriesTest(stdPK, serPK, false)) {
				traverseSeriesTest(stdPK, "FIRST", "0");
				String serPK2=attachSeriesTest(stdPK, curSerUID, frOfRefUID);
				if (!serPK.equals(serPK2)) {
					logger.info("ERROR: After inactivation, Attach Series picked PK=" + serPK2 + " instead of " + serPK);
					serPK=serPK2;
				} else {
					updateSeriesTest(stdPK, serPK, false, curSerUID, frOfRefUID);
				}
			}
		}
		if (!serPK.startsWith("-1")) {
			getSeriesTest(stdPK, serPK, false);

			traverseSeriesTest(stdPK, "FIRST", "0");
			traverseSeriesTest(stdPK, "LAST", "0");
			traverseSeriesTest(stdPK, "NEXT", serPK);
			traverseSeriesTest(stdPK, "PREV", serPK);
		}
		findSeriesByRefUIDTest(frOfRefUID);
		
		return serPK;
	}
	// =====================================================
	private String attachSOPTest(String serPK, String curSopUID) throws Exception
	{
		String sopPK="-1";
		//	test one series at a time
		//						              sopIUID                           origIUID                         sopClass         TypeIX  Desc.      Acq-DTime         Acq# Inst# KeyImg ImgNOK         CBAgent                       ImgPos            ImgOrient                            PatOrient 
		SOPInstance sopInst = new SOPInstance(curSopUID, "1.2.840.113754.1.7.660.69.20090324.123657.0", "1.2.840.10008.5.1.4.1.1.1", "3", "bla-bla", "20091121.230645", "1", "4", "0",     "1", "Contrast Bolus Agent #1", "-6.36558\\-150.822\\105.245", "0\\1\\0\\-0.0168794\\0\\-0.999858", "ARH", 
		//      PhotMI    #frs   Rows  Cols      pxSpacg SampPPx BitsA BitsS HiBit PxRep RescInt RescSlope WinC WinW PlateID SliceThness ReconDiam ScangSeq ScangVar MRAcqType AcqCDesc                            RepTime EchoTime                RefSopIns                         SrcOfRefInst   MicroscObj  LSID HS  ILat
			"MONOCHROME2", "", "1760", "2140", "0.20\\0.20", "1", "16", "12", "11", "0",   "1.0",    "1.5", "436", "873", "",    "1",        "",     "SE",   "MTC",     "3D",  "ad-hoc params for DB test only",     "2",     "5",     "1.2.840.113754.1.7.660.69.20090324.123657.0", "0008,1199",  "no MicObj", "", "", "B");
		String[] resList = DBMethodsDataSource.attachSOPInstance(connection, serPK, sopInst);
		dumpResultLines("Attach SOP Instance", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			sopPK=resList[2]; // line 3 is IEN
		} else
			sopPK="-1";
		return sopPK;
	}
	
	private Boolean updateSOPTest(String serPK, String sopIEN, Boolean ignoreParentKey, String curSopUID) throws Exception
	{
		//						              sopIUID                           origIUID                         sopClass         TypeIX          Desc.      Acq-DTime         Acq# Inst# KeyImg ImgNOK         CBAgent                       ImgPos            ImgOrient                            PatOrient 
		SOPInstance sopInst = new SOPInstance(curSopUID, "1.2.840.113754.1.7.660.69.20090324.123657.0", "1.2.840.10008.5.1.4.1.1.1", "3", "bla-bla updated", "20091121.230645", "1", "4", "1",     "0", "Contrast Bolus Agent #1", "-6.36558\\-150.822\\105.245", "0\\1\\0\\-0.0168794\\0\\-0.999858", "ARH", 
		//      PhotMI    #frs   Rows  Cols      pxSpacg SampPPx BitsA BitsS HiBit PxRep RescInt RescSlope WinC WinW PlateID SliceThness ReconDiam ScangSeq ScangVar MRAcqType AcqCDesc                            RepTime EchoTime                RefSopIns                         SrcOfRefInst   MicroscObj  LSID HS ILat 
			"MONOCHROME2", "", "1760", "2140", "0.20\\0.20", "1", "16", "12", "11", "0",   "1.0",    "1.5", "436", "873", "",    "1",        "",     "SE",   "MTC",     "3D",  "ad-hoc params for DB test only",     "2",     "5",     "1.2.840.113754.1.7.660.69.20090324.123657.0", "0008,1199",  "no MicObj",  "", "", "R");
		String[] resList = DBMethodsDataSource.updateSOPInstance(connection, serPK,  sopIEN, ignoreParentKey, sopInst);
		dumpResultLines("Update SOP Instance", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	
	public Boolean getSOPTest(String seriesPK, String sopiPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getSOPInstance(connection, seriesPK, sopiPK, ignoreParentKey);
		dumpResultLines("Get SOP Instance", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean inactivateSOPTest(String seriesPK, String sopiPK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivateSOPInstance(connection, seriesPK, sopiPK, ignoreParentKey);
		dumpResultLines("Inactivate SOP Instance", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean findOrigInstanceFileTest(String sopPK) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.findOriginalInstanceFile(connection, sopPK);
		dumpResultLines("Is Original Instance File", resList);
		return ((resList != null) && /*(resList.length>0) && */ (resList[0].startsWith("0")));
	}

	private Boolean traverseSOPInstanceTest(String seriesIEN, String direction, String sopIIEN) throws Exception
	{
		// Direction is one of "FIRST" (default), "LAST", "NEXT" or "PREV"
		String theDirection="FIRST";
		if ((direction.startsWith("FIRST")) || (direction.startsWith("LAST")) || (direction.startsWith("NEXT")) || (direction.startsWith("PREV")))
			theDirection=direction;
		String[] resList = DBMethodsDataSource.traverseSOPInstance(connection, seriesIEN, theDirection, sopIIEN);
		dumpResultLines("Traverse SOP Instance (" + theDirection + ")" , resList);
		return ((resList != null) && (resList[0].startsWith("0")));	
	}

	// ----------------------------------------------------------
	private String sopiTests(String serPK, String curSopUID) throws Exception
	{
		String sopPK=findSeriesByUIDTest(serPK, curSopUID);
		if (sopPK.startsWith("-1")) //	test one study at a time
		{
			sopPK=attachSOPTest(serPK, curSopUID);
			if (inactivateSOPTest(serPK, sopPK, false)) {
				traverseSOPInstanceTest(serPK, "FIRST", "0");
				String sopPK2=attachSOPTest(serPK, curSopUID);
				if (!sopPK.equals(sopPK2)) {
					logger.info("ERROR: After inactivation, Attach SOP Instance picked PK=" + sopPK2 + " instead of " + sopPK);
					sopPK=sopPK2;
				} else {
					updateSOPTest(serPK, sopPK, false, curSopUID);
				}
			}
		}
		if (!serPK.startsWith("-1")) {
			getSOPTest(serPK, sopPK, false);

			traverseSOPInstanceTest(serPK, "FIRST", "0");
			traverseSOPInstanceTest(serPK, "LAST", "0");
			traverseSOPInstanceTest(serPK, "NEXT", sopPK);
			traverseSOPInstanceTest(serPK, "PREV", sopPK);
		}
	
		return sopPK;
	}
	// =====================================================
	private String attachInstanceFileTest(String sopPK, String aToken) throws Exception
	{
		String iFPK="-1";
		//	test one series at a time
		//										  FName    NetLocPtr  FType IsOrig   Save-DateTime Saved-by   Del-DT / By / Reason    Fsize     FileCRC          ImgType    DerDesc  CompRat CompMethod  
//		InstanceFile instFile = new InstanceFile(curFname, "2005.2.7", "DCM", "Y", "20091122.013356", "cpt", "n/a1", "n/a2", "n/a3", "526243", "1864732", "ORIGINAL/PRIMARY", "orig", "1.0", "none");
		//										  token IsOrig IsConf          Del-DT / By / Reason  ImgType            DerDesc            CompRat CompMethod Artifact IEN 
		InstanceFileN instFile = new InstanceFileN(aToken, "1", "0", "20091122.013356", "cpt", "not available1", "ORIGINAL/PRIMARY", "orig", "1.0", "none", 1);
		String[] resList = DBMethodsDataSource.attachInstanceFile(connection, sopPK, instFile);
		dumpResultLines("Attach Instance File", resList);
		if ((resList != null) && (resList[0].startsWith("0"))) {
			iFPK=resList[2]; // line 3 is IEN
		} else
			iFPK="-1";
		return iFPK;
	}

	private Boolean updateInstanceFileTest(String sopPK, String ifiPK, Boolean ignoreParentKey, String aToken) throws Exception
	{
		//										  FName    NetLocPtr  FType IsOrig   Save-DateTime Saved-by   Del-DT / By / Reason    Fsize     FileCRC          ImgType           DerDesc  CompRat CompMethod  
//		InstanceFile instFile = new InstanceFile(curFname, "2005.2.7", "DCM", "Y", "20091122.013356", "cpt", "n/a1", "n/a2", "n/a3", "526243", "1864732", "ORIGINAL/PRIMARY", "orig updated", "1.0", "none");
		//										  token IsOrig IsConf          Del-DT / By / Reason              ImgType            DerDesc            CompRat CompMethod
		InstanceFileN instFile = new InstanceFileN(aToken, "1", "1", "20100331.013859", "cpt", "not available1-updated", "ORIGINAL/PRIMARY", "orig-upd", "1.0", "none", 1 );
		String[] resList = DBMethodsDataSource.updateInstanceFile(connection, sopPK, ifiPK, ignoreParentKey, instFile);
		dumpResultLines("Update Instance File", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	public Boolean getInstanceFileTest(String sopiPK, String iFilePK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.getInstanceFile(connection, sopiPK, iFilePK, ignoreParentKey);
		dumpResultLines("Get Instance File", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}
	public Boolean inactivateInstanceFileTest(String sopiPK, String iFilePK, Boolean ignoreParentKey) throws Exception
	{
		//	test one patientRef at a time
		String[] resList = DBMethodsDataSource.inactivateInstanceFile(connection, sopiPK, iFilePK, ignoreParentKey);
		dumpResultLines("Inactivate Instance File", resList);
		return ((resList != null) && (resList[0].startsWith("0")));
	}

	private Boolean traverseInstanceFileTest(String sopIIEN, String direction, String iFileIEN) throws Exception
	{
		// Direction is one of "FIRST" (default), "LAST", "NEXT" or "PREV"
		String theDirection="FIRST";
		if ((direction.startsWith("FIRST")) || (direction.startsWith("LAST")) || (direction.startsWith("NEXT")) || (direction.startsWith("PREV")))
			theDirection=direction;
		String[] resList = DBMethodsDataSource.traverseInstanceFile(connection, sopIIEN, theDirection, iFileIEN);
		dumpResultLines("Traverse Instance File (" + theDirection + ")" , resList);
		return ((resList != null) && (resList[0].startsWith("0")));	
	}
	// ----------------------------------------------------------
	private String iFileTests(String sopPK, String curToken) throws Exception
	{
		String iFiPK=attachInstanceFileTest(sopPK, curToken);
		if (!iFiPK.startsWith("-1")) // Instance File exists
		if (inactivateInstanceFileTest(sopPK, iFiPK, false)) {
			traverseInstanceFileTest(sopPK, "FIRST", "0");
			String iFiPK2=attachInstanceFileTest(sopPK, curToken);
			if (!iFiPK.equals(iFiPK2)) {
				logger.info("ERROR: After inactivation, Attach Instance File picked PK=" + iFiPK2 + " instead of " + iFiPK);
				iFiPK=iFiPK2;
			} else {
				updateInstanceFileTest(sopiPK, iFiPK, false, curToken);
			}
		}
		getInstanceFileTest(sopPK, iFiPK, false);
		
		traverseInstanceFileTest(sopPK, "FIRST", "0");
		traverseInstanceFileTest(sopPK, "LAST", "0");
		traverseInstanceFileTest(sopPK, "NEXT", iFiPK);
		traverseInstanceFileTest(sopPK, "PREV", iFiPK);

		return iFiPK;
	}
	// ===============================================================
	private void getTests(String parPK, String prrPK, String stdPK, String serPK, String sopPK, String ifiPK) throws Exception
	{
		getPatAttributesTest(parPK);
		// getPatInfoTest(parPK);
		getProcRefAttributesTest(parPK, prrPK, false);
		getStudyTest(prrPK, stdPK, false);
		getSeriesTest(stdPK, serPK, false);
		getSOPTest(serPK, sopPK, false);
		getInstanceFileTest(sopPK, ifiPK, false);
	}

	// ===============================================================
	private void CleanDBTest() throws Exception
	{
		String[] resList = DBMethodsDataSource.VaporizeDB(connection);
		dumpResultLines("Clean P34 DB", resList);
	}
 
	// ===============================================================

	private void testNewDB(String theDFN, String LocationID, String theACC, String stdUID, String SerUID, String sopUID, String token,
						   Boolean inactivate, Boolean eraseDB) throws Exception
	// test happy path with read backs of data written + empty DB?
	{
		curDFN = theDFN;
		patRefPK=patientRefTests(curDFN, LocationID);
		if (!patRefPK.startsWith("-1")) // patRef exists
		{
			curAccNum = theACC;
			procRefPK=procedureRefTests(curAccNum, LocationID, patRefPK);
			if (!procRefPK.startsWith("-1")) // procRef exists
			{
				curStudyUID = stdUID;
				studyPK=studyTests(patRefPK, procRefPK, curStudyUID, curAccNum);
				if (!studyPK.startsWith("-1")) // study exists
				{
					curSeriesUID = SerUID;
					lastSeriesFrOfRefUID=mockUID();
					seriesPK=seriesTests(studyPK, curSeriesUID, lastSeriesFrOfRefUID);
					if (!seriesPK.startsWith("-1")) // series exists
					{
						curSOPIUID = sopUID;
						sopiPK=sopiTests(seriesPK, curSOPIUID);
						if (!sopiPK.startsWith("-1")) // SOP Instance exists
						{
							curToken = token;
							iFilePK=iFileTests(sopiPK, curToken);
							if (!iFilePK.startsWith("-1")) // Instance File exists
							{
								findOrigInstanceFileTest(sopiPK);
							}
						}
					}
				}
			}
		}
		if (!iFilePK.startsWith("-1"))
			getTests(patRefPK, procRefPK, studyPK, seriesPK, sopiPK, iFilePK);
		logger.info("curDFN=" + curDFN + "; curAccNum=" +  curAccNum + "; curStudyUID=" + curStudyUID + "; curSeriesUID="
				+ curSeriesUID + "; curSOPIUID=" + curSOPIUID + "; curFName=" + curToken); // curFName);
		logger.info("patRefPK=" + patRefPK + "; procRefPK=" +  procRefPK + "; studyPK=" + studyPK + "; seriesPK="
				+ seriesPK + "; sopiPK=" + sopiPK + "; iFilePK=" + iFilePK);

		if (eraseDB) {
			CleanDBTest();
		}
	}
	private void testNewDBx(String theDFN, String LocationID, String prPK, String stdPK, String SerUID, String sopUID, String token,
						    Boolean inactivate, Boolean eraseDB) throws Exception
	// test happy path with read backs of data written + empty DB? - stub ProcRef and Study processing!
	{
		curDFN = theDFN;
		patRefPK=patientRefTests(curDFN, LocationID);
		if (!patRefPK.startsWith("-1")) // patRef exists
		{
			// curAccNum = theACC;
			procRefPK=prPK; // procedureRefTests(curAccNum, patRefPK); // find ProcRef does not work yet
			if (!procRefPK.startsWith("-1")) // procRef exists
			{
			//	curStudyUID = stdUID;
				studyPK=stdPK; // studyTests(patRefPK, procRefPK, curStudyUID); // find ProcRef does not work yet
				if (!studyPK.startsWith("-1")) // study exists
				{
					curSeriesUID = SerUID;
					lastSeriesFrOfRefUID=mockUID();
					seriesPK=seriesTests(studyPK, curSeriesUID, lastSeriesFrOfRefUID);
					if (!seriesPK.startsWith("-1")) // series exists
					{
						curSOPIUID = sopUID;
						sopiPK=attachSOPTest(seriesPK, curSOPIUID);
						if (!sopiPK.startsWith("-1")) // SOP Instance exists
						{
							getSOPTest(seriesPK, sopiPK, false);
														
							curToken = token;
							iFilePK=attachInstanceFileTest(sopiPK, curToken);
							if (!iFilePK.startsWith("-1")) // Instance File exists
							{
								getInstanceFileTest(sopiPK, iFilePK, false);
//								isOrigInstanceFileTest(sopiPK);
							}
						}
					}
				}
			}
		}
		logger.info("curDFN=" + curDFN + "; curAccNum=" +  curAccNum + "; curStudyUID=" + curStudyUID + "; curSeriesUID="
				+ curSeriesUID + "; curSOPIUID=" + curSOPIUID + "; curFName=" + curToken); // curFName);
		logger.info("patRefPK=" + patRefPK + "; procRefPK=" +  procRefPK + "; studyPK=" + studyPK + "; seriesPK="
				+ seriesPK + "; sopiPK=" + sopiPK + "; iFilePK=" + iFilePK);

		if (eraseDB) {
			CleanDBTest();
		}
	}
	
	@Test
	public void testNewDB_Take1() throws Exception
	// test happy path with read backs of data written + empty DB
	{
		testNewDB(mockDFN(), "660", mockAccessionNumber(), mockUID(), mockUID(), mockUID(), mockToken(), false, false);
		// create UID test patients on empty DB
		// 					 p2 - o3 - G - I - M/N		p2="1313"	o3="020210-331"
		//					 p1 - o1 - A - F - U/X      p1="1312"	o1="020410-332"
		//					 p1 - o2 - S - R - O/P					o2="020410-333"
		//
		//							   G="1.2.840.113754.1.4.660.6899797.9154.1.20210.331"
		//								   I="1.2.840.113754.1.4.660.6899797.9154.1.20210.331.1"
		//									   M="1.2.840.113754.1.7.660.3.20100225.154603.0"
		//									   N="1.2.840.113754.1.7.660.3.20100225.154605.0"
		//							   A="1.2.840.113754.1.4.660.6899795.8891.1.20410.332"
		//								   F="1.2.840.113754.1.4.660.6899795.8891.1.20410.332.1"
		//									   U="1.2.840.113754.1.7.660.3.20100222.140112.0"
		//									   X="1.2.840.113754.1.7.660.3.20100222.140115.0"
		//							   S="1.2.840.113754.1.4.660.6899795.8891.1.20410.333"
		//								   R="1.2.840.113754.1.4.660.6899795.8891.1.20410.333.1"
		//									   O="1.2.840.113754.1.7.660.3.20100222.140116.0"
		//									   P="1.2.840.113754.1.7.660.3.20100222.140117.0"
		//
//		testNewDB("1313", "660", "020210-331", "1.2.840.113754.1.4.660.6899797.9154.1.20210.331",
//				  "1.2.840.113754.1.4.660.6899797.9154.1.20210.331.1", "1.2.840.113754.1.7.660.3.20100225.154603.0",
//				  mockToken(), false, false); // patRefPK=1; procRefPK=1; studyPK=1; seriesPK=1; sopiPK=1; iFilePK=1
//		testNewDBx("1313", "660", procRefPK, studyPK,
//				  "1.2.840.113754.1.4.660.6899797.9154.1.20210.331.1", "1.2.840.113754.1.7.660.3.20100225.154605.0",
//				  mockToken(), false, false); // patRefPK=1; procRefPK=1; studyPK=1; seriesPK=1; sopiPK=2; iFilePK=2
//		testNewDB("1312", "660", "020410-332", "1.2.840.113754.1.4.660.6899795.8891.1.20410.332",
//				  "1.2.840.113754.1.4.660.6899795.8891.1.20410.332.1", "1.2.840.113754.1.7.660.3.20100222.140112.0",
//				  mockToken(), false, false); // patRefPK=2; procRefPK=2; studyPK=2; seriesPK=2; sopiPK=3; iFilePK=3
//		testNewDBx("1312", "660", procRefPK, studyPK,
//				  "1.2.840.113754.1.4.660.6899795.8891.1.20410.332.1", "1.2.840.113754.1.7.660.3.20100222.140115.0",
//				  mockToken(), false, false); // patRefPK=2; procRefPK=2; studyPK=2; seriesPK=2; sopiPK=4; iFilePK=4
//		testNewDB("1312", "660", "020410-333", "1.2.840.113754.1.4.660.6899795.8891.1.20410.333",
//		  		  "1.2.840.113754.1.4.660.6899795.8891.1.20410.333.1", "1.2.840.113754.1.7.660.3.20100222.140116.0",
//		  		  mockToken(), false, false); // patRefPK=2; procRefPK=3; studyPK=3; seriesPK=3; sopiPK=5; iFilePK=5
//		testNewDBx("1312", "660", procRefPK, studyPK,
//		  		  "1.2.840.113754.1.4.660.6899795.8891.1.20410.333.1", "1.2.840.113754.1.7.660.3.20100222.140117.0",
//		  		  mockToken(), false, false); // patRefPK=2; procRefPK=3; studyPK=3; seriesPK=3; sopiPK=6; iFilePK=6
	}
	
//	@Test
//	public void testConsult() throws Exception
//	{
//		logger.info(" ");
//		logger.info(" Consult test STARTED...");
//		String paRefPK="19";
//		String prRefPK="90";
//		
//		getPatAttributesTest(paRefPK);
//		getPatInfoTest(paRefPK);
//		
//		getProcRefAttributesTest(paRefPK, prRefPK, false);
//		getAccessionNumberTest(paRefPK, prRefPK, false);
//		getProcedureInfoTest(paRefPK, prRefPK, false);
//		getReportTest(paRefPK, prRefPK, false);
//
//		logger.info(" ... consult test FINISHED.");
//	}
//		
//	@Test
//	public void testRadReprot() throws Exception
//	{
//		logger.info(" ");
//		logger.info(" RAD Report test STARTED...");
//		String paRefPK="30";
//		String prRefPK="92";
////		updatePatRefTest("1265", paRefPK);
//
//		getPatAttributesTest(paRefPK);
//		getPatInfoTest(paRefPK);
//		
//		getProcRefAttributesTest(paRefPK, prRefPK, false);
//		getAccessionNumberTest(paRefPK, prRefPK, false);
//		getProcedureInfoTest(paRefPK, prRefPK, false);
//		getReportTest(paRefPK, prRefPK, false);
//
//		logger.info(" ... RAD Report test FINISHED.");
//	}
		
	//------------------------- dump result utility --------------------------------------
	private void dumpResultLines(String resType, String[] resList)
	{
		// Write out the items
		String infoText = resType + " RPC returned:";
		logger.info(infoText);
		int i=1;
		for (String rL : resList)
		{
			logger.info(i++ + ": " + rL);
		}
		if (i==1)
			logger.info("               *** N O T H I N G *** ");
	}
}
