/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Nov 8, 2017
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:  vhaisltjahjb
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.url.mssqlcache.configuration;

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;


/**
 * This contains a copy of code from AbstractBaseFacadeConfiguration. It needed to be copied to not
 * create a circular dependency.
 * 
 * @author vhaisltjahjb
 *
 */
public class SqlCacheConnectionConfiguration 
{
	private Logger logger = LogManager.getLogger(SqlCacheConnectionConfiguration.class);
	
	private Long disconnectReadTimeout = null;
	private Long callReadTimeout = null;
	private Long connectReadTimeout = null;
	private Long readPollingInterval = null;
	private String sqlUserPassword = null;
	private String sqlUserLogin = null;
	private String sqlPort = null;
	private String sqlHost = null;
	
	private final static long defaultCallReadTimeout = 240000L;
	private final static long defaultConnectionReadTimeout = 60000L;
	private final static long defaultDisconnectReadTimeout = 60000L;
	private final static long defaultReadPollingInterval = 10L;
	private static final String defaultSqlHost = "localhost";
	private static final String defaultSqlPort = "1433";
	private static final String defaultUserPassword = "R00tbeer";
	private static final String defaultUserLogin = "alexdelarge";
	
	public SqlCacheConnectionConfiguration()
	{
		super();
	}
	
	
	public SqlCacheConnectionConfiguration(
			String host,
			String port,
			String login,
			String pwd,
			long callReadTimeout, long connectionReadTimeout, 
			long disconnectReadTimeout, long readPollingInterval)
	{
		super();
		this.connectReadTimeout = callReadTimeout;
		this.connectReadTimeout = connectionReadTimeout;
		this.disconnectReadTimeout = disconnectReadTimeout;
		this.readPollingInterval = readPollingInterval;
		this.sqlHost = host;
		this.sqlPort = port;
		this.sqlUserLogin = login;
		this.sqlUserPassword = pwd;
	}

	private static SqlCacheConnectionConfiguration configuration = null;
	public synchronized static SqlCacheConnectionConfiguration getSqlCacheConnectionConfiguration()
	{
		if(configuration == null)
		{
			SqlCacheConnectionConfiguration config = new SqlCacheConnectionConfiguration();
			configuration = config.loadConfiguration();			
		}
		return configuration;
	}
	

	/**
	 * @return the disconnectReadTimeout
	 */
	public Long getDisconnectReadTimeout()
	{
		return disconnectReadTimeout;
	}

	/**
	 * @param disconnectReadTimeout the disconnectReadTimeout to set
	 */
	public void setDisconnectReadTimeout(Long disconnectReadTimeout)
	{
		this.disconnectReadTimeout = disconnectReadTimeout;
	}

	/**
	 * @return the callReadTimeout
	 */
	public Long getCallReadTimeout()
	{
		return callReadTimeout;
	}

	/**
	 * @param callReadTimeout the callReadTimeout to set
	 */
	public void setCallReadTimeout(Long callReadTimeout)
	{
		this.callReadTimeout = callReadTimeout;
	}

	/**
	 * @return the connectReadTimeout
	 */
	public Long getConnectReadTimeout()
	{
		return connectReadTimeout;
	}

	/**
	 * @param connectReadTimeout the connectReadTimeout to set
	 */
	public void setConnectReadTimeout(Long connectReadTimeout)
	{
		this.connectReadTimeout = connectReadTimeout;
	}

	/**
	 * @return the readPollingInterval
	 */
	public Long getReadPollingInterval()
	{
		return readPollingInterval;
	}

	/**
	 * @param readPollingInterval the readPollingInterval to set
	 */
	public void setReadPollingInterval(Long readPollingInterval)
	{
		this.readPollingInterval = readPollingInterval;
	}

	public SqlCacheConnectionConfiguration loadConfiguration()
	{
		SqlCacheConnectionConfiguration config = loadConfigurationFromFile();
		if(config == null)
		{
			config = loadDefaultConfiguration();
			config.storeConfiguration();
		}
		return config;
	}		
	
	public SqlCacheConnectionConfiguration loadDefaultConfiguration()
	{
		this.sqlHost = defaultSqlHost; 
		this.sqlPort = defaultSqlPort;
		this.sqlUserLogin = defaultUserLogin;
		this.sqlUserPassword = defaultUserPassword;
		this.callReadTimeout = defaultCallReadTimeout;
		this.disconnectReadTimeout = defaultDisconnectReadTimeout;
		this.connectReadTimeout = defaultConnectionReadTimeout;
		this.readPollingInterval = defaultReadPollingInterval;
		return this;
	}
	
	private SqlCacheConnectionConfiguration loadConfigurationFromFile()
	{
		XMLDecoder decoder = null;
		try
		{
			File file = new File(getConfigurationFileName());
			if(file.exists())
			{			
				decoder = new XMLDecoder(new FileInputStream(file.getAbsolutePath()));
				SqlCacheConnectionConfiguration configuration = (SqlCacheConnectionConfiguration)decoder.readObject();
				logger.info("Loaded configuration file [" + file.getAbsolutePath() + "]");
				return configuration;
			}
			else
			{
				logger.info("File [" + file.getAbsolutePath() + "] does not exist");
				return null;
			}
		}
		catch(FileNotFoundException fnfX)
		{
			logger.error("Error reading configuration, " + fnfX.getMessage(), fnfX);
			return null;
		}
		finally
		{
			if(decoder != null)
				decoder.close();
		}
	}

	public synchronized void storeConfiguration()
	{
		XMLEncoder encoder = null;
		try
		{
			String filename = getConfigurationFileName();
			encoder = new XMLEncoder(new FileOutputStream(filename));
			encoder.writeObject(this);
			logger.info("Stored configuration file [" + filename + "]");
		}
		catch(IOException ioX)
		{
			logger.error("Error storing configuration, " + ioX.getMessage(), ioX);
		}
		finally
		{
			if(encoder != null)
				encoder.close();
		}		
	}
	
	/**
	 * Get the configuration directory.
	 * Usually, derived classes do not need to access the directory
	 * and just rely on the storeConfiguration)( and loadConfiguration()
	 * methods.  This method is provided for exceptional cases.
	 */
	private File getConfigurationDirectory()
	{
		String configurationDirectoryName = System.getenv("vixconfig");
		if(configurationDirectoryName == null)
			configurationDirectoryName = System.getProperty("user.home");
		if(configurationDirectoryName == null)
			configurationDirectoryName = "/";
		
		File configurationDirectory = new File(configurationDirectoryName);
		if(! configurationDirectory.exists())
			configurationDirectory.mkdirs();		// make the directories if they don't exist
		
		return configurationDirectory;
	}
	
	/**
	 * Build a filename in the standardized format from the
	 * provider name and version.  A Provider that does not
	 * have any persistent configuration must override this 
	 * method to return null.
	 * This method will assure that the parent directory exists
	 * before returning.  It will NOT create the configuration
	 * file if it does not exist.
	 * 
	 * The preferred store locations are (in order):
	 * 1.) The directory of the VIX configuration
	 * 2.) The user home directory
	 * 3.) The root directory
	 * 
	 * @return
	 */
	private String getConfigurationFileName()
	{
		File configurationDirectory = getConfigurationDirectory();	
		return configurationDirectory.getAbsolutePath() + "/" + this.getClass().getSimpleName() + ".config"; 
	}
	
	public static void main(String [] args)
	{
		SqlCacheConnectionConfiguration configuration = new SqlCacheConnectionConfiguration();
		configuration.loadDefaultConfiguration();
		
		if((args != null) && (args.length > 0))
		{
			String host = defaultSqlHost;
			String port = defaultSqlPort;
			String login = defaultUserLogin;
			String pwd = defaultUserPassword;
			long callReadTimeout = defaultCallReadTimeout;
			long connectionReadTimeout = defaultConnectionReadTimeout;
			long disconnectReadTimeout = defaultDisconnectReadTimeout;
			long readPollingInterval = defaultReadPollingInterval;
			
			for(int i = 0; i < args.length; i++)
			{
				if("-host".equals(args[i]))
				{
					host = args[++i];
				}
				else if("-port".equals(args[i]))
				{
					port = args[++i];
				}
				else if("-login".equals(args[i]))
				{
					login = args[++i];
				}
				else if("-pwd".equals(args[i]))
				{
					pwd = args[++i];
				}
				else if("-call".equals(args[i]))
				{
					callReadTimeout = Long.parseLong(args[++i]);
				}
				else if("-connect".equals(args[i]))
				{
					connectionReadTimeout = Long.parseLong(args[++i]);
				}
				else if("-disconnect".equals(args[i]))
				{
					disconnectReadTimeout = Long.parseLong(args[++i]);
				}
				else if("-read".equals(args[i]))
				{
					readPollingInterval = Long.parseLong(args[++i]);
				}
			}
			
			SqlCacheConnectionConfiguration sqlCacheConfiguration = 
					new SqlCacheConnectionConfiguration(
							host, port, login, pwd,
							callReadTimeout, connectionReadTimeout, 
							disconnectReadTimeout, readPollingInterval);
					
			sqlCacheConfiguration.storeConfiguration();		
		}
	}
	

	public void setSqlUserPassword(String pwd) {
		sqlUserPassword = pwd;
	}
	public String getSqlUserPassword()
	{
		return sqlUserPassword;
	}

	public void setSqlUserLogin(String login) 
	{
		sqlUserLogin = login;
	}
	public String getSqlUserLogin() 
	{
		return sqlUserLogin;
	}

	public void setSqlPort(String port) 
	{
		sqlPort = port;
	}
	public String getSqlPort() 
	{
		return sqlPort;
	}

	public void setSqlHost(String host) 
	{
		sqlHost = host;
	}
	public String getSqlHost() 
	{
		return sqlHost;
	}
	
}

