package gov.va.med.imaging.mix.translator;

import java.util.List;
import java.util.ArrayList;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.json.*;

import gov.va.med.imaging.StringUtil;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.mix.DODImageURN;
import gov.va.med.imaging.mix.webservices.rest.exceptions.MIXImagingStudyException;
import gov.va.med.imaging.mix.webservices.rest.types.v1.ErrorResultType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.InstanceType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.ModalitiesType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.ReportStudyListResponseType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.SeriesType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.SeriesTypeComponentInstances;
import gov.va.med.imaging.mix.webservices.rest.types.v1.StudyType;
import gov.va.med.imaging.mix.webservices.rest.types.v1.StudyTypeComponentSeries;


public class MixJSONConverter
{
	private static Logger logger = LogManager.getLogger(MixJSONConverter.class);	

	public static ReportStudyListResponseType ConvertDiagnosticReportToJava(String jsonData)
	{
		ReportStudyListResponseType rslr = null;
		
		try {
			 logger.debug("Convert DiagnosticReport(s) JSON response to JSONObject...");
		     JSONObject obj = new JSONObject(jsonData);
			 logger.debug("Get DRs JSONArray...");
		     JSONArray dRs = obj.getJSONArray("DRs"); // {"DRs":[...]} assumed 
		     if ((dRs==null) || (dRs.length()==0)) {
		    	 return rslr;
		     }
			 logger.debug("Convert " + dRs.length() + " DRs JSONArray items ...");

		     List<StudyType> studies = new ArrayList<StudyType>();
		     List<ErrorResultType> errors = new ArrayList<ErrorResultType>();

		     for (int i = 0; i < dRs.length(); i++) {
				 logger.debug("Convert DRs JSONArray item " + (i+1) + "...");
		    	 JSONObject curDR = dRs.getJSONObject(i);
				 logger.debug("Convert DRs JSONArray item " + (i+1) + " got object:");
			     String status = curDR.getString("status"); // ?check for "final"
				 logger.debug("Convert DRs JSONArray item " + (i+1) + " status=" + status);
			     String discipline = curDR.getString("category"); // ?check for "final"
				 logger.debug("Convert DRs JSONArray item " + (i+1) + " category=" + discipline);
			     String subject = curDR.getJSONObject("subject").getString("reference");
				 logger.debug("Convert DRs JSONArray item " + (i+1) + " subject/reference=" + subject);
			     String patId = subject.replace("Patient/", ""); // take off FHIR Patient resource reference tag, keep ICN 
			     String report = curDR.getJSONObject("text").getString("div");
				 logger.debug("Convert DRs JSONArray item " + (i+1) + " report=" + report);

			     JSONArray shallowStudies = curDR.getJSONArray("shallowStudy");
			     for (int j = 0; j < shallowStudies.length(); j++) {
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + "...");
			    	 StudyType study = new StudyType();

			    	 JSONObject curSStudy = shallowStudies.getJSONObject(j);
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " got object:");
				     String procDate = curSStudy.getString("started"); // *** "yyyy-MM-ddTHH:MI:SS+HHH:MI"
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " started=" + procDate);
				     String proc = curSStudy.getString("procedure"); // description
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " procedure=" + proc);
				     int numSeries = curSStudy.getInt("numberOfSeries");
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " numberOfSeries=" + numSeries);
				     int numInstances = curSStudy.getInt("numberOfInstances");
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " numberOfInstances=" + numInstances);
				     String modList = curSStudy.getString("modalityList"); // *** might need array treatment getting "code" from each member
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " modalityList=" + modList);
				     String[] modalities = StringUtil.split(modList, StringUtil.COMMA);
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " modalities=" + modalities.toString());
				     ModalitiesType mt = new ModalitiesType(modalities);
				     String uid = curSStudy.getString("uid");
					 logger.debug("Convert DRs JSONArray item " + (i+1) + " - study " + (j+1) + " uid=" + uid);
				     study.setProcedureDate(procDate);
				     study.setProcedureDescription(proc);
				     study.setSeriesCount(numSeries);
				     study.setImageCount(numInstances);
				     study.setModalities(mt);
				     study.setDicomUid(uid);
				     study.setReportContent(report);
				     study.setPatientId(patId);
				     if (discipline!=null && !discipline.isEmpty())
				    	 study.setSpecialtyDescription(discipline);
				    	 
				     studies.add(study);
					 logger.debug("Convert DRs JSONArray item " + (i+1) + "; added study, #studies=" + studies.size());
			     }
		     }
		     rslr = new ReportStudyListResponseType();
		     if (studies.size() > 0) {
				 logger.debug("Convert DRs JSONArray item: copy studies...");
				 StudyType[] studyTypeArray = new StudyType[studies.size()];
			     for (int i = 0; i < studies.size(); i++) {
			    	 studyTypeArray[i]=studies.get(i);
			     }
		    	 rslr.setStudies(studyTypeArray);
		     } else
				 logger.debug("Convert DRs JSONArray item: no studies to copy!");
		    	 
		}
		catch (JSONException je) {
			logger.info("FHIR DiagnosticReport response JSON data object error: " + je.getMessage());
			return null;
		}
		// ***
		return rslr;
	}
	
	public static StudyType ConvertImagingStudyToJava(String jsonData)
	throws MIXImagingStudyException {
		StudyType st = null;
		try {
		     JSONObject iS = new JSONObject(jsonData);
		     if (iS.length()==0) {
		    	 return st;
		     }
		     st = new StudyType();
		     List<SeriesType> series = new ArrayList<SeriesType>();
		     DODImageURN dodURN = new DODImageURN();
		     
		     String started = iS.getString("started"); // *** "yyyy-MM-ddTHH:MI:SS+HHH:MI"
		     String patRef = iS.getJSONObject("patient").getString("reference");
		     String patId = patRef.replace("Patient/", ""); // take off FHIR Patient resource reference prefix 
		     String stdUid = iS.getString("uid");
		     String accessionNum = iS.getJSONObject("accession").getString("value"); // no Study ID from HAIMS, use accession#
		     // compose a fake study id as a studyUrn using urn:vastudy:200-<accvalue>-<patId> to pass MixTranslator
		     String stdId = "urn:vastudy:200-" + accessionNum + "-" + patId;
		     JSONArray procList = iS.getJSONArray("procedure"); // array, getting first element's "reference" value
		     String proc = procList.getJSONObject(0).getString("reference");
		     String desc = iS.getString("description");
		     int numSeries = iS.getInt("numberOfSeries");
		     int numInstances = iS.getInt("numberOfInstances");
		     String mtyList = "";
		     JSONArray modalityList = iS.getJSONArray("modalityList"); // array, getting "code" from each member
		     for (int k = 0; k < modalityList.length(); k++) {
		    	 JSONObject curModality = modalityList.getJSONObject(k);
			     String aMty = curModality.getString("code");
			     if (mtyList.isEmpty()) {
			    	 mtyList = aMty;
			     } else {
			    	 mtyList += "," + aMty;
			     }
		     }		     
		     String[] modalities = StringUtil.split(mtyList, StringUtil.COMMA);
		     ModalitiesType mt = new ModalitiesType(modalities);
		     // convert JSON date to DicomDate
		     String procDate="";
		     if ((started!=null) && (started.length()>=20)) {
		    	 procDate = started.replace("-", "").replace("T", "").replace(":", "").substring(0, 14); // UTC is ignored !!!
		     }
		     
		     st.setProcedureDate(procDate);
		     st.setPatientId(patId);
		     st.setStudyId(stdId);
		     st.setDicomUid(stdUid);
		     st.setModalities(mt);
		     st.setSeriesCount(numSeries);
		     st.setImageCount(numInstances);
		     st.setProcedureDescription(proc);
		     st.setDescription(desc);
		     //st.setReportContent(report); // cannot be filled here!!! -  make sure it's not lost from diagnostic report query!

		     JSONArray serieses = iS.getJSONArray("series");
		     for (int i = 0; i < serieses.length(); i++) {
		    	 SeriesType serT = new SeriesType();
			     List<InstanceType> instances = new ArrayList<InstanceType>();
			     
		    	 JSONObject curSeries = serieses.getJSONObject(i);
				 int serNum = curSeries.getInt("number");
			     String modality = curSeries.getJSONObject("modality").getString("code");
			     String serUid = curSeries.getString("uid");
			     // String serDesc = curSeries.getString("description");
			     int numSerInstances = curSeries.getInt("numberOfInstances");
			     serT.setDicomSeriesNumber(serNum);
			     serT.setModality(modality);
			     serT.setDicomUid(serUid);
			     // serT.setDescription(serDesc);
			     serT.setImageCount(numSerInstances);
			     
			     JSONArray insts = curSeries.getJSONArray("instance");
			     for (int j = 0; j < insts.length(); j++) {
			    	 InstanceType insT = new InstanceType();
			    	 JSONObject curIns = insts.getJSONObject(j);
					 int insNum = curIns.getInt("number");
			    	 String sopUid = curIns.getString("uid");
			    	 // String sopClass = curIns.getString("sopClass"); // where to put it? - not in InstanceType, not conveyed anyhow...
			    	 insT.setDicomInstanceNumber(insNum);
				     insT.setDicomUid(sopUid);
				     
				     try {
				    	 insT.setImageUrn((String)dodURN.create(stdUid, serUid, sopUid).toString());
				     } catch(URNFormatException ufe) {
				    	 throw new MIXImagingStudyException("Error creating DOD imageURN: " + ufe.getMessage());
				     }
				    	 
				     instances.add(insT);
			     }
			     
			     // hook instances!
			     InstanceType[] seriesInstances = instances.toArray(new InstanceType[instances.size()]);
			     SeriesTypeComponentInstances instancesWrapper = new SeriesTypeComponentInstances();
			     instancesWrapper.setInstance(seriesInstances);
			     serT.setComponentInstances(instancesWrapper);

			     series.add(serT);
		     }
		     
		     // hook series!
		     SeriesType[] studySeries = series.toArray(new SeriesType[series.size()]);
		     StudyTypeComponentSeries instancesWrapper = new StudyTypeComponentSeries();
		     instancesWrapper.setSeries(studySeries);
		     st.setComponentSeries(instancesWrapper);
			 logger.debug("Converted IS procedure date to DICOM date: " + st.getProcedureDate());
		}
		catch (JSONException je) {
			logger.info("FHIR ImagingStudy response JSON data object error: " + je.getMessage());
			return null;
		}
		// *** log if inconsistencies found!
		return st;
	}
}

