/*
 * Created on Aug 16, 2005
// Per VHA Directive 2004-038, this routine should not be modified.
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 *
 */
package gov.va.med.imaging.dicom.dcftoolkit.listen;

import gov.va.med.imaging.dicom.common.utils.Publisher;
import gov.va.med.imaging.dicom.dcftoolkit.common.observer.SubOperationsStatus;
import gov.va.med.imaging.dicom.dcftoolkit.scp.SubOperationsListener;
import gov.va.med.imaging.exchange.business.dicom.MoveCommandObserver;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.Observable;
import java.util.Observer;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.lbs.DCF.DCFException;
import com.lbs.DCS.AssociationAcceptor;
import com.lbs.DCS.AttributeTag;
import com.lbs.DCS.DCSException;
import com.lbs.DCS.DicomDataSet;
import com.lbs.DCS.DicomUIElement;
import com.lbs.DCS.DimseMessage;
import com.lbs.DCS.DimseStatus;
import com.lbs.DCS.QRDimseStatus;
import com.lbs.DDS.DDSException;
import com.lbs.DSS.QRSCP;

/**
 *
 * However, 
 * the Move SCP Service is responsible for sending update Dimse messages about the C-Store
 * activity.  This is referred to as the SubOperationsStatus message.  This is accomplished
 * through the Listener Pattern.  The Implementation of this Interface also becomes a 
 * Subscriber and receives any updates from the C-Store Publisher.  When the Subscriber 
 * receives an update, the Subcriber creates a SubOperationsStatus Dimse message and sends
 * the message to the C-Move SCU.
 *
 * @author William Peterson
 *
 */
public class SpecializedQRSCP extends QRSCP implements SubOperationsListener, Observer {

    
    private boolean processingMoveRq = false;
        
    private Publisher cancelMovePublisher = null;
    
    private AssociationAcceptor association = null;

    private String affectedSOPClassUID = "";
    
    private int dimseMessageID = 0;
    
    private int context_id = 0;
    
    private MoveCommandObserver moveCommandObserver = null;

    private static Logger logger = LogManager.getLogger (SpecializedQRSCP.class);
            
    /**
     * @param arg0
     * @throws com.lbs.DCF.DCFException
     */
    public SpecializedQRSCP(AssociationAcceptor arg0) throws DCFException {
        super(arg0);
        this.association = arg0;        
    }
    
    public void cMoveRq(DimseMessage msg) throws DCSException{
        
        try{
            //Set switch to flag processing of cMoveRq message.  This is used to notify the cCancel
            // method what action to take.
            logger.debug(this.getClass().getName()+": Performing C-Move-Rq");
            this.processingMoveRq = true;
            
            //Pull attributes out of cMoveRq message to be inserted later into the cMoveRsp
            // message.
            this.context_id = msg.context_id();
            this.affectedSOPClassUID = msg.affectedSopclassUid();
            this.dimseMessageID = msg.messageId();
            
            //invoke DicomDataService_a.moveObjects method
            this.moveCommandObserver = DicomDataService_a.getInstance().moveObjects(this.association, msg, 
                    (Observer)this);
            
            logger.debug(this.getClass().getName()+": Dicom Toolkit Layer: "+
                    "...returned from starting the C-Move-Rq process.");
            
            this.cancelMovePublisher = new Publisher();
            //Add Subscriber to Publisher.
            this.cancelMovePublisher.addObserver(this.moveCommandObserver);

        }
        catch (DDSException ddse){
            logger.error(ddse.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown during processing " +
                    "C-Move Request.");
            SubOperationsStatus subop = new SubOperationsStatus();
            subop.setCompleteStatus(SubOperationsStatus.FAILURE);
            subop.setErrorComment("Exception thrown while processing C-Move Request.");
            this.moveObjectsComplete(subop);
        }        
    }
    
    public void cCancel(DimseMessage msg) throws DCSException{
        
        //Need if Statement to determine what the cCancel is cancelling.  If cancelling a 
        // cFindRq, perform super().  This allows the base method to perform its actions.
        // If cMoveRq, create code to process properly for this specialized case.
        logger.debug("Processing Move Rq State: "+this.processingMoveRq);
        if(this.processingMoveRq){
            //Trigger a instance variable.  The MoveObjectResult method will check this instance 
            // variable for change.  This is how this object will trip the MoveSCP object to 
            // notify the Business Layer.
            logger.debug("Processing cCancel as for C-Move.");
            this.cancelMovePublisher.publish(null);
            this.processingMoveRq = false;
        }
        else{
            //msg.messageIdResp(msg.messageId());
            logger.debug("Processing cCancel as for C-Find.");
            msg.messageId(msg.messageIdResp());
            super.cCancel(msg);
        }
    }
    
    public void moveObjectResult(SubOperationsStatus status){
        try{
                //Build cMoveRsp DimseMessage
                DimseMessage cMoveRsp = new DimseMessage(DimseMessage.C_MOVE_RSP);
                if(this.context_id == 0){
                    throw new DCSException("No Context ID");
                }
                
                cMoveRsp.context_id(this.context_id);
                cMoveRsp.affectedSopclassUid(this.affectedSOPClassUID);
                cMoveRsp.messageIdResp(this.dimseMessageID);
                cMoveRsp.dataSetType(0x0101);
                cMoveRsp.status(DimseStatus.PENDING);
                cMoveRsp.numberOfRemainingSubops(status.getRemainingSubOperations());
                cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                
                //Send built cMoveRsp DimseMessage to SCU.
                association.sendDimseMessage(cMoveRsp, 0);
                logger.debug("Sent Move Suboperations Result.");
        }
        catch(DCSException dcse){
            LOG.info("cMoveRsp Dimse message was not sent to Move SCU.");
            
            logger.error(dcse.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while returning Move" +
                    "suboperations result.");
            logger.error("Trace:", dcse);
        }
    }
    
    //IMPROVE Sending SubOperationsStatus to Move SCU.  Currently, I cannot pass Error Comments
    //	because I do not have the correct signature in the CloseAssociation method calls.
    //	Also, place all status codes in a separate class in one of the common projects.
    public void moveObjectsComplete (SubOperationsStatus status){

        try{
            //Build cMoveRsp DimseMessage
            DimseMessage cMoveRsp = new DimseMessage(DimseMessage.C_MOVE_RSP);
            if(this.context_id == 0){
                throw new DCSException("No Context ID");
            }
            cMoveRsp.context_id(this.context_id);

            switch(status.getCompleteStatus()){
                case(SubOperationsStatus.SUCCESS):
                    cMoveRsp.dataSetType(0x0101);
                    cMoveRsp.status(DimseStatus.DIMSE_SUCCESS);
                    cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                    cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                    cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                    break;
                case(SubOperationsStatus.CANCEL):
                    cMoveRsp.dataSetType(0x0101);
                    cMoveRsp.status(DimseStatus.CANCEL);
                    //Build the SOP Instance list correctly in DimseMessage.
                    if(status.getSOPInstanceUIDs() != null){
                    	if(!status.getSOPInstanceUIDs().isEmpty()){
                    		cMoveRsp.dataSetType(0x0100);
                    		cMoveRsp.data(this.createInstanceListInDDS(status));
                    	}
                    }
                    cMoveRsp.numberOfRemainingSubops(status.getRemainingSubOperations());
                    cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                    cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                    cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                    break;
                case(SubOperationsStatus.FAILURE):
                    cMoveRsp.dataSetType(0x0101);
                    cMoveRsp.status(QRDimseStatus.OUT_OF_RESOURCES_CANT_PERFORM_SUBOPS);
                    //Build the SOP Instance list correctly in DimseMessage.
                    if(status.getSOPInstanceUIDs() != null){
                    	if(!status.getSOPInstanceUIDs().isEmpty()){
                    		cMoveRsp.dataSetType(0x0100);
                    		cMoveRsp.data(this.createInstanceListInDDS(status));
                    	}
                    }
                    cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                    cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                    cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                    break;
                case(SubOperationsStatus.WARNING):
                    cMoveRsp.dataSetType(0x0101);
                    cMoveRsp.status(QRDimseStatus.SUB_OPS_COMPLETED_WITH_FAILURES);
                    //Build the SOP Instance list correctly in DimseMessage.
                    if(status.getSOPInstanceUIDs() != null){
                    	if(!status.getSOPInstanceUIDs().isEmpty()){
                    		cMoveRsp.dataSetType(0x0100);
                    		cMoveRsp.data(this.createInstanceListInDDS(status));
                    	}
                    }
                    cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                    cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                    cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                    break;
                case(SubOperationsStatus.FAILURE_COMMENT):
                    cMoveRsp.dataSetType(0x0101);
                    cMoveRsp.status(QRDimseStatus.OUT_OF_RESOURCES_CANT_PERFORM_SUBOPS);
                    cMoveRsp.errorComment(status.getErrorComment());
                    if(status.getSOPInstanceUIDs() != null){
                    	if(!status.getSOPInstanceUIDs().isEmpty()){
                    		cMoveRsp.dataSetType(0x0100);
                    		cMoveRsp.data(this.createInstanceListInDDS(status));
                    	}
                    }
                    cMoveRsp.numberOfCompletedSubops(status.getSuccessfulSubOperations());
                    cMoveRsp.numberOfFailedSubops(status.getFailedSubOperations());
                    cMoveRsp.numberOfWarningSubops(status.getWarningSubOperations());
                    break;
            }

            //Add common elements to any resultant DimseMessage.
            cMoveRsp.affectedSopclassUid(this.affectedSOPClassUID);
            cMoveRsp.messageIdResp(this.dimseMessageID);

            //Send built cMoveRsp DimseMessage to SCU.
            association.sendDimseMessage(cMoveRsp, 0);     
            logger.debug("Sent Move Suboperations Complete Result.");
        }
        catch(DCSException dcse){
            logger.info("cMoveRsp Dimse message was not sent to Move SCU.");            
            logger.error(dcse.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while returning Move" +
                    "suboperations complete result.");
            logger.error("Trace:", dcse);
        }
        finally{
        	this.moveCommandObserver = null;
        }
    }

    
    /* (non-Javadoc)
     * @see java.util.Observer#update(java.util.Observable, java.lang.Object)
     */
    public void update(Observable o, Object arg){
        
        logger.info(this.getClass().getName()+": Dicom Toolkit Layer: " +
                "...updating SubOperations Status.");
        //Cast the object to a SubOperationsStatus object.
        SubOperationsStatus status = (SubOperationsStatus) arg;
        
        // if connection no longer exist, do not call either method below.
        if(this.association.getConnected()){
        	logger.debug("C-Move Association is still active.");
	        //If this is last SubOperationsStatus, invoke moveObjectComplete 
	        if (status.isSubOperationsComplete()){
	            logger.info(this.getClass().getName()+": Dicom Toolkit Layer: " +
	                    "...SubOperations are Complete.");
	            this.moveObjectsComplete(status);
	        }
	        else{
	            if(status.isSubOperationsActive()){
	                //Invoke Listener method to notify SCU.
	                this.moveObjectResult(status);
	            }
	        }
        }
        else{
        	logger.debug("C-Move Association is not active.");
        }
    }

    
    private DicomDataSet createInstanceListInDDS(SubOperationsStatus status){

        DicomDataSet dds = new DicomDataSet();
        try{
        	ArrayList<String> list = status.getSOPInstanceUIDs();
        	while(list.remove("")){
        	}
        	list.trimToSize();
        	
        	String[] temp = new String[list.size()];
        	Iterator<String> iterator = list.iterator();
        	for(int i=0;iterator.hasNext(); i++){
        		temp[i] = iterator.next();
        	}
            DicomUIElement uidList = new DicomUIElement(new AttributeTag("0008,0058"),
                    temp);
            dds.insert(uidList);
        }
        catch(DCSException dcse){
            logger.error(dcse.getMessage());
            logger.error(this.getClass().getName()+": Could not create the Instance List" +
                    " for the C-Move Response.");
        }
        return dds;
    }
}
