/*
 * Created on Apr 2, 2005
 *
// Per VHA Directive 2004-038, this routine should not be modified.
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 */
package gov.va.med.imaging.dicom.dcftoolkit.listen;

import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.dicom.DicomContext;
import gov.va.med.imaging.dicom.common.SpringContext;
import gov.va.med.imaging.dicom.common.stats.DicomServiceStats;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomListenException;
import gov.va.med.imaging.dicom.dcftoolkit.movescp.interfaces.IDicomMoveSCP;
import gov.va.med.imaging.dicom.dcftoolkit.storagescp.interfaces.IDicomStorageSCP;
import gov.va.med.imaging.dicom.dcftoolkit.storagecommitscp.interfaces.IDicomStorageCommitSCP;
import gov.va.med.imaging.exchange.business.dicom.DicomAE;
import gov.va.med.imaging.exchange.business.dicom.DicomServerConfiguration;
import gov.va.med.imaging.exchange.business.dicom.InstrumentConfig;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.transactioncontext.TransactionContextMemento;
import gov.va.med.server.ServerAgnosticEngineAdapter;

import java.util.Hashtable;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.lbs.CDS.CDSException;
import com.lbs.DCS.AssociationAbortedException;
import com.lbs.DCS.AssociationAcceptor;
import com.lbs.DCS.AssociationConfigPolicyManager;
import com.lbs.DCS.AssociationInfo;
import com.lbs.DCS.AssociationListener;
import com.lbs.DCS.AssociationManager;
import com.lbs.DCS.AssociationRejectedException;
import com.lbs.DCS.DCSException;
import com.lbs.DCS.DicomSessionSettings;
import com.lbs.DCS.PDU;
import com.lbs.DCS.PDUAssociateReject;
import com.lbs.DCS.VerificationServer;
import com.lbs.DSS.StoreServer;
import com.lbs.DSS.StoreCommitServer;

/**
 * Implements the AssociationConfigPolicyManager and AssociationListener Interfaces
 *  from the DCF API.</p>
 * 
 * This class is a Singleton.
 * Class is the SCP Server. Class creates a DICOM listening socket for all DICOM Services.
 * This starts DCF Association Manager and a DCF specific Server object for each available
 * DICOM Service.  This class handles the Association Establishment workflow.  Basically, a SCU
 * sends an Associate-Request PDU message.  This SCP Server object determines if to accept the
 * Association.  If the SCP Server accepts the message, the SCP Server will respond with an
 * Associate-Accept PDU message.  This will establish the Association.  If the SCP Server does
 * not accept the message, the SCP Server will respond with an Associate-Reject message with
 * a reason for the rejection.  There will be no establishment of the Association.</p>
 * 
 * Selecting specific SOP Classes and DICOM Services to accept can be configured through a 
 * Configuration file in this package directory.</p>
 *
 *
 * @author William Peterson
 *
 */
public class Listen implements AssociationConfigPolicyManager, AssociationListener, Runnable {

    /*
     * DCF Toolkit class. There is only one Association Manager per socket.  However,
     * there will be multiple Associations within the Manager.  
     */
    private AssociationManager associationMgr = null;
    
    /*
     * DCF Toolkit class.  Runs inside of Association Manager.  Handles C-Store
     * Services.  However, does not invoke any VA specific classes.
     */
    //private StoreServer storeserver = null;
    
    /*
     * DCF Toolkit class.  Runs inside of Association Manager.  Handles C-Store
     * Services.  However, does not invoke any VA specific classes.
     */
    private StoreCommitServer storecommitserver = null;
    
    /*
     * DCF Toolkit class.  Runs inside of Association Manager.  Handles C-Find
     * Services.
     */
    private SpecializedQRServer qrserver = null;
    
    /*
     * DCF Toolkit class.  Runs inside of Association Manager.  Handles Verification
     * Services.
     */
    private VerificationServer verificationserver = null;
    
    /*
     * VA Class.  Instantiate and saved in hashmap at time of Association beginning.
     * Used later when images come across after the Association is established.
     */
    private IDicomStorageSCP store = null;
        
    /*
     * VA Class.  Instantiate and saved in hashmap at time of Association beginning.
     * Used later when the Storage Commit request come across after the Association is established.
     */
    private IDicomStorageCommitSCP sCommit = null;
        
    //However, I do need declaration of IDicomMoveSCP.
    private IDicomMoveSCP move = null;
    /*
     * Hashmap to save multiple Store objects.  There is a separate Store object
     * for each Association.
     */
    private static Hashtable<AssociationAcceptor, IDicomStorageSCP> store_map = null;
        
    /*
     * Hashmap to save multiple Storage Commit requests.  There is a separate StoreCommit object
     * for each Association.
     */
    private static Hashtable<AssociationAcceptor, IDicomStorageCommitSCP> scommit_map = null;

    //However, I need this for DicomMoveSCP objects.
    private static Hashtable<AssociationAcceptor, IDicomMoveSCP> move_map = null;
    
    //Maintain the DicomAE object with the correct Association Acceptor.
    private static Hashtable<AssociationAcceptor, DicomAE> ae_map = null;
    
    private static Hashtable<AssociationAcceptor, ServerAgnosticEngineAdapter> engineAdapter_map = null;
    
    /*
     * Switch to determine of the AETitles should be authenticated.
     */
    private boolean authenticate_aetitles = false;
    
    private String implementationClassUID = "";
    private String implementationVersionName = "";
    private ServerAgnosticEngineAdapter engineAdapter;
    private InstrumentConfig instrument = null;
    private static Logger logger = LogManager.getLogger (Listen.class);
    private static Logger summaryLogger = LogManager.getLogger("Summary");

    
    public Listen(InstrumentConfig instrument, ServerAgnosticEngineAdapter engineAdapter) throws DCSException, CDSException{
        LOG.info( "Creating new DICOM Listen Object.");

        this.engineAdapter = engineAdapter;
        this.instrument = instrument;
        
        //Setup Association Manager.
        this.AssociationManagerSetup();
        
        // Set tcp port
        this.setAssociationManagerPort(this.instrument.getPort());
        
        
    }
    
	@Override
	public void run()
	{
		try
		{
			// Authenticate against the realm to get our principal object configured
	       	DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
	       	
			String accessCode = config.getAccessCodeString();
			String verifyCode = config.getVerifyCodeString();
			engineAdapter.authenticate(accessCode, verifyCode.getBytes());

			this.listenforSCURequests();
		}
		catch (DicomListenException e)
		{
			logger.error("Error listening for SCU Requests on port " + this.getAssociationManagerPort(), e);
			summaryLogger.error("Port "+ this.getAssociationManagerPort()+" failed to start listening.\n"
					+" Refer to other logs for more detail.");
		}
	}

   
    /**
     * Initializes the Association Manager Object.  The Listen Constructors call this
     * method to initialize the Association Manager.  This method is code that represents a
     * common denominator between the Listen Constructors.
     *  
     * @throws DCSException represents DCS Exceptions from DCF API.  Refer to DCF API.
     * @throws CDSException represents CDS Exceptions from DCF API.  Refer to DCF API.
     */
    private void AssociationManagerSetup() throws DCSException, CDSException{
        
        // Create an AssociationManager
        associationMgr = new AssociationManager();
        //  It belongs somewhere around here.
        
        //set AssociationConfigPolicyManager
        associationMgr.setAssociationConfigPolicyMgr(this);
        //Add this object as another AssociationListener.  This is because this object
        //  also implements an AssociationListener Interface.
        associationMgr.addAssociationListener(this);

        // Create a StoreServer object
        //LOG.info( "Creating new StoreServer Object" );
        //storeserver = new StoreServer( associationMgr );
        
        // Create a StoreCommitServer object
        LOG.info( "Creating new StoreCommitServer Object" );
        storecommitserver = new StoreCommitServer( associationMgr );
        
        // Create a Specialized QRServer object
        LOG.info( "Creating new QRServer Object" );
        qrserver = new SpecializedQRServer( associationMgr );

        // Create a VerificationServer object
        LOG.info( "Creating new VerificationServer Object" );
        verificationserver = new VerificationServer( associationMgr );
    }
    
    /**
     * Starts the actual listening of a selected port for the Listen Object.  The
     * selected port is set in the Association Manager.
     * 
     * @throws DicomListenException
     */
    public void listenforSCURequests()throws DicomListenException{
        
        try{
 
            LOG.debug(CINFO.df_SHOW_GENERAL_FLOW, "Setting up DCF Data Service Adapter." );
            //Setup the DicomDataService adapter.  This is where DCF Toolkit invokes
            //  VA implemented code, via an DCF Interface, for handling of DICOM
            //  Services.
            DicomDataService_a.setup(new String[0]);
            logger.debug("Starting to Listen on socket "+ String.valueOf(associationMgr.getServerTcpPort())+" in new thread.");
            //Call local method to start listening.
            DicomServiceStats.getInstance().setCurrentPortStatus(this.getAssociationManagerPort(), DicomServiceStats.UP);
            this.runServer();
        }
        catch( Exception e ){
            logger.error(e.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while starting the listener.");
            DicomServiceStats.getInstance().setCurrentPortStatus(this.getAssociationManagerPort(), DicomServiceStats.FAILED);
            throw new DicomListenException("Failure to start listener.", e);
        }
    }
    
    /**
     * Stop listening for SCU Requests.  This does not perform a shutdown.  The start method
     * can be called again to restart listening if desired.
     * 
     * @throws DicomListenException
     */
    public void stopListeningforSCURequests() throws DicomListenException{
        //Stop listening on socket.
        this.associationMgr.stop();
        DicomServiceStats.getInstance().setCurrentPortStatus(this.getAssociationManagerPort(), DicomServiceStats.DOWN);
    }

    /**
     * Stop listening for SCU Requests.  This does not perform a shutdown.  The start method
     * can be called again to restart listening if desired.
     * 
     * @throws DicomListenException
     */
    public void shutdownServer() throws DicomListenException{
        //Stop listening on socket.
        this.associationMgr.stop();
        this.associationMgr.shutdown();
        DicomServiceStats.getInstance().setCurrentPortStatus(this.getAssociationManagerPort(), DicomServiceStats.DOWN);        
    }
    
    
    /**
     * Activates the running of the SCP Servers in the Association Manager.
     * 
     * @throws DCSException represents the DCS Exception from the DCF API.  Refer to DCF API.
     */
    public void runServer(){
        try{
            //Activate listening on socket predetermined by the Association Manager.
            //  If this is not in a separate thread, it will halt all action on main
            //  thread.  This run() method is not thread based.
            this.associationMgr.run();
        }
        catch (DCSException dcsError){
            logger.info("\nDICOM Listening failed.\n");
            
            logger.error("Fatal Error: "+dcsError.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while starting the Server.");
            logger.error("Trace:", dcsError);
        }
    }

    /* (non-Javadoc)
     * @see com.lbs.DCS.AssociationConfigPolicyManager#getSessionSettings(com.lbs.DCS.AssociationAcceptor)
     * 
     */
    public DicomSessionSettings getSessionSettings(AssociationAcceptor assoc)
            throws AssociationRejectedException, AssociationAbortedException {


       	DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
       	
    	authenticate_aetitles = DicomServerConfiguration.getConfiguration().isAuthenticateAETitles();
        implementationClassUID = DicomServerConfiguration.getConfiguration().getImplementationClassUID();
        implementationVersionName = DicomServerConfiguration.getConfiguration().getImplementationVersionName();            
        
        //Set VA Implementation fields.
        AssociationInfo ainfo;
        String srcIPAddress = "";
        ainfo = assoc.getAssociationInfo();
        ainfo.calledImplementationClassUid(implementationClassUID);
        ainfo.calledImplementationVersionName(implementationVersionName);
        srcIPAddress = ainfo.callingPresentationAddress();

        try
        {
        	// Establish a security context on this thread
    		String accessCode = config.getAccessCodeString();
    		String verifyCode = config.getVerifyCodeString();
    		engineAdapter.authenticate(accessCode, verifyCode.getBytes());

    		//get both AETitles
            String calledaet = assoc.getAssociationInfo().calledTitle();
            String callingaet = assoc.getAssociationInfo().callingTitle();
                        
            DicomAE dicomAE = DicomContext.getRouter().getRemoteAE(DicomAE.searchMode.REMOTE_AE, callingaet, DicomServerConfiguration.getConfiguration().getSiteId());

            if(authenticate_aetitles){
                if(!dicomAE.isRemoteAEValid()){
                    logger.error("...throwing due to rejected Remote AETitle, "
                            +callingaet+", for "+srcIPAddress+".");
        			summaryLogger.error("The Remote AETitle, "+callingaet
        					+", does not have permission to access VistA Imaging.\n"
        					+"This permission is configurable using DICOM AE Security Matrix.");
                    //This thrown exception is picked up by DCF and automatically 
                    //  converted into an Association Reject PDU. 
                	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
                	DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);                                
                    throw new AssociationRejectedException (
                            PDUAssociateReject.RESULT_PERMANENT,
                            PDUAssociateReject.SOURCE_SERVICE_USER,
                            PDUAssociateReject.REASON_BAD_CALLING);
            	}
                //Is the called AETitle authenticated?
                if(!dicomAE.getLocalAETitle().equalsIgnoreCase(calledaet)){
                    logger.error("...throwing due to rejected Local AETitle, "
                            +calledaet+", for Remote AETitle "+callingaet+".");
        			summaryLogger.error("The Local AETitle, "+calledaet
        					+", is invalid with Remote AETitle, "+callingaet+", to access VistA Imaging.\n"
        					+"This permission is configurable using DICOM AE Security Matrix.");
                    //This thrown exception is picked up by DCF and automatically 
                    //  converted into an Association Reject PDU.
                	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
                	DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);
                    throw new AssociationRejectedException (
                            PDUAssociateReject.RESULT_PERMANENT,
                            PDUAssociateReject.SOURCE_SERVICE_USER,
                            PDUAssociateReject.REASON_BAD_CALLED);              
                }
                if(ae_map == null){
                	ae_map = new Hashtable<AssociationAcceptor, DicomAE>();
                }
                ae_map.put(assoc, dicomAE);
                
                if(engineAdapter_map == null){
                	engineAdapter_map = new Hashtable<AssociationAcceptor, ServerAgnosticEngineAdapter>();
                }
                engineAdapter_map.put(assoc, engineAdapter);
            }
        }
        catch(MethodException me){
                
            logger.error(me.getMessage());
            logger.error(this.getClass().getName()+": AETitle Authentication Exception thrown" +
                    " during Association Request. Throwing Association Rejection.");
            summaryLogger.error("AETitle Authentication internal exception thrown.  Refer to other logs for more detail.");
            //This thrown exception is picked up by DCF and automatically 
            //  converted into an Association Reject PDU.
        	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
            DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);
            throw new AssociationRejectedException (
                    PDUAssociateReject.RESULT_PERMANENT,
                    PDUAssociateReject.SOURCE_SERVICE_PROVIDER_ACSE,
                    PDUAssociateReject.REASON_NO_REASON);
        }
        catch(ConnectionException ce){
            
            logger.error(ce.getMessage());
            logger.error(this.getClass().getName()+": AETitle Authentication Exception thrown" +
                    " during Association Request. Throwing Association Rejection.");
            summaryLogger.error("AETitle Authentication internal exception thrown.  Refer to other logs for more detail.");

            //This thrown exception is picked up by DCF and automatically 
            //  converted into an Association Reject PDU. 
        	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
            DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);
            throw new AssociationRejectedException (
                    PDUAssociateReject.RESULT_PERMANENT,
                    PDUAssociateReject.SOURCE_SERVICE_PROVIDER_ACSE,
                    PDUAssociateReject.REASON_NO_REASON);
       }

       try{
           logger.info("AETitles are valid for "+srcIPAddress+". Returning DicomSessionSettings.");
           return new DicomSessionSettings();
       }
       catch(DCSException dcs){
           LOG.info("...throwing due to DCF Exception.");
            
           logger.error(dcs.getMessage());
           logger.error(this.getClass().getName()+": Exception thrown while DCF was creating "+
                   "DICOM Session Settings. Throwing Association Rejection.");
           summaryLogger.error("DICOM Toolkit internal exception thrown.  Refer to other logs for more detail.");
           //This thrown exception is picked up by DCF and automatically 
           //  converted into an Association Reject PDU. 
       	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
        DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);
           throw new AssociationRejectedException (
                   PDUAssociateReject.RESULT_PERMANENT,
                   PDUAssociateReject.SOURCE_SERVICE_PROVIDER_ACSE,
                   PDUAssociateReject.REASON_NO_REASON);
       }
    }

    /**
     * Temporary method.  Set/Change the listening network socket.  Issue this method call
     * before the listenforSCURequest method.
     * 
     * @param port  represents the network port or socket number for listening.
     */
    public void setAssociationManagerPort(int port){
        associationMgr.setServerTcpPort(port);
        //REENG Need to add code to protect from assigning a port if the 
        //  Association Manager is already running.  Only allow the ability
        //  to assign a port if the Association Manager is not in a "run" state.
    }
    
    public int getAssociationManagerPort(){
        //Return socket from Association Manager.
        return (int)this.associationMgr.getServerTcpPort();
    }
    
    /**
	 * Implementation of AssociationListener interface.
	 * 
	 * Indicates that a new association has been created. The
	 * AssociationAcceptor has selected configuration settings for the
	 * association, and has processed the A-Associate-Request PDU. The
	 * association has not yet been accepted.
	 * </p>
	 * 
	 * The sole purpose of implementing the AssociationListener Interface for
	 * this class was to keep the StoreSCPControl object alive between
	 * C-Store-Rq Dimse messages. This allows the StoreSCPControl object to
	 * track Study and Series Instance UIDs. Thus, giving the usability of
	 * convenience methods to store the instance to Persistence.
	 * 
	 * @param assoc
	 *            the object handling the association.
	 * @throws AssociationRejectedException
	 *             Indicates that the association should be rejected
	 *             immediately.
	 * @throws AssociationAbortedException
	 *             Indicates that the association should be aborted immediately.
	 */
    public void beginAssociation( AssociationAcceptor assoc )
    throws AssociationRejectedException, AssociationAbortedException {
        
        String threadID = " [" + Long.toString(Thread.currentThread().getId()) + "]";
    	
        AssociationInfo ainfo = assoc.getAssociationInfo();

		boolean test = false;
        logger.info("Facade has received an association request on port " + assoc.getAssociationManager().getServerTcpPort());
        // Instantiate Store Object via Spring Factory.
        store = (IDicomStorageSCP)SpringContext.getContext().getBean("DicomStorageSCP");
        if(store == null){
            test = true;
        }
        sCommit = (IDicomStorageCommitSCP)SpringContext.getContext().getBean("DicomStorageCommitSCP");
        if(sCommit == null){
            test = true;
        }

        //Instantiate Move Object via Spring Factory.
        move = (IDicomMoveSCP)SpringContext.getContext().getBean("DicomMoveSCP");
        if(move == null){
            test = true;
        }
        if(test){
            //This thrown exception is picked up by DCF and automatically 
            //  converted into an Association Reject PDU. 
        	String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
        	DicomServiceStats.getInstance().incrementInboundAssociationRejectCount(ainfo.callingTitle(), ipAddr);
        	throw new AssociationRejectedException (
                    PDUAssociateReject.RESULT_PERMANENT,
                    PDUAssociateReject.SOURCE_SERVICE_PROVIDER_ACSE,
                    PDUAssociateReject.REASON_TEMP_CONGESTION);
        }
         
        LOG.debug( CINFO.df_SHOW_GENERAL_FLOW, "Listen: beginAssociation" );
        //Create hashmap object to save Store object(s).
        if ( store_map == null ){
            store_map = new Hashtable<AssociationAcceptor, IDicomStorageSCP>();
        }
        //Create hashmap object to save StorageCommit object.
        if ( scommit_map == null ){
        	scommit_map = new Hashtable<AssociationAcceptor, IDicomStorageCommitSCP>();
        }
        
        //I need this for MoveSCP Objects
        if(move_map == null){
            move_map = new Hashtable<AssociationAcceptor, IDicomMoveSCP>();
        }
        
        //Place instantiated Store object into the hashmap for later use.  The model
        //  required the Store object(s) be created now, but not used until the
        //  association receives c-store-rq DIMSE messages.
        store_map.put( assoc, store );
        
        //Place instantiated Store Commit object into the hashmap for later use.  The model
        //  required the SCommit object(s) be created now, but not used until the
        //  association receives N-ACTION DIMSE messages.
        scommit_map.put( assoc, sCommit );

        //Place instantiated Move object into the hashmap for later use.  The model
        //  required the Move object(s) be created now, but not used until the
        //  association receives c-move-rq DIMSE messages.
        move_map.put(assoc, move);
        LOG.info("Association has started: " + assoc.getAssociationInfo());
        LOG.info("...completed beginAssociation in Listen Object.\n");

        String ipAddr = getSourceIPAddress(ainfo.callingPresentationAddress());
        DicomServiceStats.getInstance().incrementInboundAssociationAcceptCount(ainfo.callingTitle(), ipAddr);
    }
 
    
    /**
    * Implementation of AssociationListener interface.
    *
    * Indicates that an association has ended.
    * @param assoc the object handling the association.
    */
    public void endAssociation( AssociationAcceptor assoc ){
      
        logger.info("Facade is ending an association on port " + assoc.getAssociationManager().getServerTcpPort());
        //Since the association is ending, remove the Store object related to this
        //  association.
        store_map.remove(assoc);
        
        //Since the association is ending, remove the SCommit object related to this
        //  association.
        scommit_map.remove(assoc);
        
        //Since the association is ending, remove the Move object related to this
        //  association.
        move_map.remove(assoc);
        LOG.info("...completed endAssociation in Listen Object.\n");
        
        ae_map.remove(assoc);
        
        engineAdapter_map.remove(assoc);
        
        // Clear the transaction context on this thread
        TransactionContextFactory.get().clear();
        
    }
    @Override
    public void acsePduIn(String connection_id, AssociationAcceptor acceptor, PDU pdu)
    {
        LOG.info("acsePduIn: connection id ="
        + connection_id
        + System.getProperty( "line.separator" )
        + pdu); 
    }

    @Override
    public void acsePduOut(String connection_id, AssociationAcceptor acceptor, PDU pdu)
    {
        LOG.info("acsePduOut: connection id ="
        + connection_id
        + System.getProperty( "line.separator" )
        + pdu); 
    }

    @Override
    public void associationSocketConnected(String connection_id, String local_address, String remote_address)
    {
        LOG.info("associationSocketConnected: connection id ="
        + connection_id
        + System.getProperty( "line.separator" )
        + "local address = " + local_address
        + System.getProperty( "line.separator" )
        + "remote address = " + remote_address );   
    }

    @Override
    public void associationSocketDisconnected(String connection_id, String local_address, String remote_address)
    {
        LOG.info("associationSocketDisconnected: connection id ="
        + connection_id
        + System.getProperty( "line.separator" )
        + "local address = " + local_address
        + System.getProperty( "line.separator" )
        + "remote address = " + remote_address);    
    }
    /**
     * Maps DicomStorageSCPImpl Objects to their respective AssociationAcceptor Object.
     *  
     * @param a represents the AssociationAcceptor object.
     * @return
     */
    public static IDicomStorageSCP getStoreObject (AssociationAcceptor a){
        //Return casted Store object from hashmap.
        //FUTURE Need to correct this.  It should be casted using an implementation class.
        return store_map.get(a);
    }
        
    /**
     * Maps DicomStorageCommitSCPImpl Objects to their respective AssociationAcceptor Object.
     *  
     * @param a represents the AssociationAcceptor object.
     * @return
     */
    public static IDicomStorageCommitSCP getStoreCommitRequestObject (AssociationAcceptor a){
        //Return casted StorageCommit object from hashmap.
        //FUTURE Need to correct this.  It should be casted using an implementation class.
        return scommit_map.get(a);
    }
    /**
     * Maps DicomMoveSCPImpl Objects to their respective AssociationAcceptor Object.
     *  
     * @param a represents the AssociationAcceptor object.
     * @return
     */
    public static IDicomMoveSCP getMoveObject (AssociationAcceptor a){
        //Return casted Store object from hashmap.
        return move_map.get(a);
    }
    
    public static DicomAE getDicomAEObject (AssociationAcceptor a){
    	return ae_map.get(a);
    }
    
    public static ServerAgnosticEngineAdapter getEngineAdapterObject(AssociationAcceptor a){
    	return engineAdapter_map.get(a);
    }
    
    public boolean isServerRunning(){
        return associationMgr.isRunning();
    }

    private String getSourceIPAddress(String presentationAddress){
    	
    	String values[] =  presentationAddress.split(":");
    	
    	return values[0];
    }

}
