/*
 * Created on Apr 2, 2005
// Per VHA Directive 2004-038, this routine should not be modified.
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 *
 */
package gov.va.med.imaging.dicom.dcftoolkit.listen;

import gov.va.med.imaging.StringUtil;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.InvalidUserCredentialsException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.router.facade.InternalContext;
import gov.va.med.imaging.dicom.DicomContext;
import gov.va.med.imaging.dicom.DicomRouter;
import gov.va.med.imaging.dicom.common.DicomFileMetaInfo;
import gov.va.med.imaging.dicom.common.SpringContext;
import gov.va.med.imaging.dicom.common.stats.DicomServiceStats;
import gov.va.med.imaging.dicom.dcftoolkit.common.DataSetByteReader;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomFindSCPException;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomMoveSCPException;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomStorageCommitSCPException;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomStorageSCPException;
import gov.va.med.imaging.dicom.dcftoolkit.findscp.interfaces.IDicomFindSCP;
import gov.va.med.imaging.dicom.dcftoolkit.movescp.interfaces.IDicomMoveSCP;
import gov.va.med.imaging.dicom.dcftoolkit.startup.DicomEngineAdapter;
import gov.va.med.imaging.dicom.dcftoolkit.storagecommitscp.interfaces.IDicomStorageCommitSCP;
import gov.va.med.imaging.dicom.dcftoolkit.storagescp.interfaces.IDicomStorageSCP;
import gov.va.med.imaging.dicom.email.DicomEmailInfo;
import gov.va.med.imaging.dicom.email.DicomEmailType;
import gov.va.med.imaging.dicom.email.DicomLevel;
import gov.va.med.imaging.exchange.business.AuditEvent;
import gov.va.med.imaging.exchange.business.EmailMessage;
import gov.va.med.imaging.exchange.business.dicom.DicomAE;
import gov.va.med.imaging.exchange.business.dicom.DicomServerConfiguration;
import gov.va.med.imaging.exchange.business.dicom.InstrumentConfig;
import gov.va.med.imaging.exchange.business.dicom.MoveCommandObserver;
import gov.va.med.imaging.exchange.business.dicom.exceptions.IODViolationException;
import gov.va.med.imaging.notifications.NotificationFacade;
import gov.va.med.imaging.notifications.NotificationTypes;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.transactioncontext.TransactionContextMemento;
import gov.va.med.server.ServerAgnosticEngineAdapter;

import java.io.FileOutputStream;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Observer;
import java.util.Vector;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.lbs.CDS.CFGGroup;
import com.lbs.DCF.DCFException;
import com.lbs.DCS.AcceptedPresentationContext;
import com.lbs.DCS.AssociationAcceptor;
import com.lbs.DCS.AssociationInfo;
import com.lbs.DCS.AssociationRequester;
import com.lbs.DCS.DCM;
import com.lbs.DCS.DCMUID;
import com.lbs.DCS.DCSException;
import com.lbs.DCS.DicomDataDictionary;
import com.lbs.DCS.DicomDataSet;
import com.lbs.DCS.DicomFileOutput;
import com.lbs.DCS.DicomPersistentObjectDescriptor;
import com.lbs.DCS.DicomSessionSettings;
import com.lbs.DCS.DimseMessage;
import com.lbs.DCS.DimseServiceUser;
import com.lbs.DCS.QRDimseStatus;
import com.lbs.DCS.UID;
import com.lbs.DDS.DDSException;
import com.lbs.DDS.DicomDataService;
import com.lbs.DDS.DicomDataServiceListener;

/**
 *
 * This is a Singleton.  This inherits the DCF DicomDataService adapter.  This basically contains the implementation
 * code to handle received DICOM objects from an established Association.  This is where the
 * DCF toolkit has DICOM Service hooks to interface to a non-DCF environment.
 *
 *
 * @author William Peterson
 *
 */

//orginal: public class DicomDataService_a extends com.lbs.DDS_a.DicomDataService_a {
public class DicomDataService_a extends DicomDataService{

    /*
    * The singleton instance.
    */
    private static DicomDataService_a instance_ = null;

    /*
    * The directory for storing image files locally.
    */
    private String image_directory_ = null;

    /*
    * If true, we create new uid's for objects that are stored.
    */
    private boolean f_make_new_uids_ = false;

    /*
    * Transfer syntax to use when writing dicom files to disk.
    */
    private String ts_uid_ = null;

    /*
     * Affects which demo code in storeObject is executed.
     * If true, uses the code that demonstrates the DataSetByteReader
     * class - i.e. pulling stream data from the network data set.
     * If false, use DicomFileOutput that pushes data to a file.
     * This must be set true to stream to Persistence layer.
     */
    private boolean f_use_byte_reader_ = true;
    
                
    private Vector<String> studyUIDList;
    
    private static Logger logger = LogManager.getLogger (DicomDataService_a.class);
    private static Logger summaryLogger = LogManager.getLogger("Summary");
    
    private String[] args;

    
    /**
     * Constructor.
     * 
     * @param arg0 represents command line arguments.
     * @throws com.lbs.DDS.DDSException
     */
    public DicomDataService_a(String[] arg0) throws DDSException {
        this.args = arg0;
        
        studyUIDList = new Vector<String>();
        try{
            //Get the configuration info from the config file.
            CFGGroup cfg = CINFO.getConfig();
            
            //assign config attributes to object variables.
            image_directory_ = cfg.getAttributeValue( "image_directory" );
            ts_uid_ = cfg.getAttributeValue( "transfer_syntax_uid", "" );
            f_make_new_uids_ = cfg.getAttributeBoolValue( "make_new_uids" );
            f_use_byte_reader_ = cfg.getAttributeBoolValue( "use_byte_reader" );

            LOG.debug( CINFO.df_SHOW_CONSTRUCTORS|CINFO.df_SHOW_GENERAL_FLOW,
                    "Image directory = " + image_directory_ );
        }
        catch( DCFException e ){
            LOG.error( -1, "DicomDataService_a.init: error" , e );
            
            logger.error(e.getMessage());
            logger.error(this.getClass().getName()+ ": Exception thrown while reading Attributes"+
                    " from DCF configuration file.");
            summaryLogger.error("Failed to read information from the DCF Configuration file.\n  The HDIG will not" +
            		"function properly with this failure.");
            throw new DDSException( "failed initialization" );
        }
    }
    
    /**
    * Install this adapter as the implementation of the DicomDataService interface.
    * 
    * @param args array of argument strings
    * @throws DDSException if the object could not be opened
    */
    public static synchronized void setup( String args[] )
        throws DDSException {
        LOG.debug( CINFO.df_SHOW_CONSTRUCTORS|CINFO.df_SHOW_GENERAL_FLOW,
            "DicomDataService_a setup" );
        if ( instance_ != null )
        {
            instance_ = null;
        }
        instance_ = new DicomDataService_a( args );
        setInstance( instance_ );
    }
    
    public static synchronized DicomDataService_a getInstance(){
        
        return instance_;
    }


    /**
    * Store a dicom object, and return the DicomPersistentObjectDescriptor
    * which references that object.  This signature writes the image object to a 
    * local file.  The initial reference count for the object will be one.
    * The VA does not use this method unless for some simple testing.  It is not
    * part of the primary implementation for storing DICOM objects.  The next 
    * storeObject() method is used by the VA.
    * 
    * @param obj DicomDataSet whose data will be stored
    * @return persistent object descriptor.  The object may be retrieved
    * later using this descriptor.
    * @throws DDSException if any other error occurs.
    */
    public synchronized DicomPersistentObjectDescriptor storeObject( DicomDataSet obj )
        throws DDSException
    {
        try
        {
            LOG.info("Entering facade processing");
            if (CINFO.testDebugFlags( CINFO.df_SHOW_GENERAL_FLOW ))
            {
                // split the tests-flags/prepare-message, so we
                // don't bother to format the entire dimse message
                // if we aren't logging.
                LOG.debug( "DicomDataService_a.storeObject: " + obj );
            }

            // generate a file name.
            String sop_class_uid = obj.getElementStringValue( DCM.E_SOPCLASS_UID, "" ).trim();
            String sop_instance_uid = ( f_make_new_uids_ 
                                        ? DicomDataDictionary.makeUID()
                                        : obj.getElementStringValue( DCM.E_SOPINSTANCE_UID ) );

            String filename = new String( image_directory_ + "/"
                + sop_instance_uid + ".dcm" );

            //
            if (f_use_byte_reader_ )
            {
                // Create default session settings.  This is needed for the
                //  DataSetByteReader object.
                DicomSessionSettings session_settings = new DicomSessionSettings();

                // Create DataSetByteReader - this produces output
                // that can be written directly to a DICOM file.
                // we use our configured transfer-syntax-uid,
                // and request non-chapter-10 style encoding.
                // See DataSetByteReader for other options.
                DataSetByteReader dbr = new DataSetByteReader(
                        obj,
                        ts_uid_,
                        false,
                        false,
                        null,
                        null,
                        session_settings );
 
                //get the inputstream from the DatasetByteReader object.
                InputStream in = dbr.getInputStream();
                //Call helper method to push inputstream to file.
                transferInputStreamToFile( filename, in );
                LOG.info("\nSaved message to disk using InputStream from DataByteReader.\n");
                dbr.closeInputStream();
            }
            else
            {
                //This is offered to give an alternative method to write the file
                //  to disk.  This does not use streaming.
                DicomFileOutput dfo = new DicomFileOutput(
                        filename, ts_uid_, true, true );
                dfo.open();
                dfo.writeDataSet( obj );
                dfo.close();
                LOG.info("\nSaved message to disk using DicomFileOutput. \n");
            }

            // Return the descriptor that might be used to
            // retrieve this data later, using loadObject.
            DicomPersistentObjectDescriptor dpod_ret =
                new DicomPersistentObjectDescriptor(
                            sop_class_uid,
                            sop_instance_uid,
                            filename,
                            "C-Store-Rq" );

            return dpod_ret;
        }
        catch( Exception e )
        {
            LOG.error(-1, "error during storeObject:", e );
            
            logger.error(e.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while storing object.");
            throw new DDSException( "storeObject failed:\n" + e );
        }
    }


    /**
    * Store a dicom object, and return the DicomPersistentObjectDescriptor
    * which references that object.  This method is normally called by StoreSCP
    * when a C-Store_Request is received.  The image object would normally pass to 
    * the Generic DICOM Layer.  It may be optioned to write to a local file.
    * 
    * @param acceptor the current AssociationAcceptor. This contains
    * information about the client AE, negotiated contexts, etc.
    * @param c_store_rq the dimse message containing
    * the data set to be stored.
    * @param c_store_rsp the dimse message containing the C-Store Response.
    * @return persistent object descriptor.  The object may be retrieved
    * later using this descriptor.
    * @throws DDSException if any other error occurs. A dimse error
    * status should be returned to the SCU if this method throws
    * an exception.
    */
    public DicomPersistentObjectDescriptor storeObject(AssociationAcceptor associationAcceptor, DimseMessage c_store_rq, DimseMessage c_store_rsp )
        throws DDSException{
    	
        String threadID = " [" + Long.toString(Thread.currentThread().getId()) + "]";
        DicomPersistentObjectDescriptor dpod_ret = null;
        String sop_instance_uid = null;
        String sopClassName = null;
        
   		try {
			int pcID = c_store_rq.context_id();
			int messageID = c_store_rq.messageId();
	        logger.debug(this.getClass().getName()+": Store Object: dimse-message = " + messageID + "and \nPresentation Context ID = "+pcID+" -- ThreadID=" + threadID);
		} catch (DCSException e) {
			
		}

        DicomAE dicomAE = Listen.getDicomAEObject(associationAcceptor);
    	boolean authenticateService = DicomServerConfiguration.getConfiguration().isAuthenticateAETitles();
    	checkAuthenticatedServiceForStoreObject(associationAcceptor, dicomAE, authenticateService);
        
        //assign AssociationInfo to local variable.
        AssociationInfo ainfo = associationAcceptor.getAssociationInfo();
        
        logger.info("Received Storage request from SCU AE = " + ainfo.callingTitle() + " (" + ainfo.callingPresentationAddress() +
                ") to SCP AE = " + ainfo.calledTitle() + "(" + ainfo.calledPresentationAddress() + ") -- ThreadID=" + threadID);

        String emailRecepient = DicomServerConfiguration.getConfiguration().getDgwEmailInfo().getEMailAddress();
        String[] eMailTOs=new String[1];
        eMailTOs[0] = emailRecepient;

        
        //This is how to get the proper Store object for this specific association.
        //  The Store object is created during the beginning of the Association and 
        //  saved in a hashmap.  The Store object is where specific Storage SCP
        //  functionality happens.
        logger.debug("Pulling correct Store Object from Hashmap - " + dicomAE.getRemoteAETitle() + "/" + threadID);
        IDicomStorageSCP store = Listen.getStoreObject(associationAcceptor);  
        
        logger.debug("Extracting dataset out of the DimseMessage Object - " + dicomAE.getRemoteAETitle() + "/" +threadID);
        DicomDataSet dataset = c_store_rq.data();
        
        //Extract Manufacturer, Model, and Modality out of the dataset.  This is strictly for use with tracking statistics.
        String manufacturer = dataset.getElementStringValue(DCM.E_MANUFACTURER, "");
        String model = dataset.getElementStringValue(DCM.E_MANUFACTURERS_MODEL_NAME, "");
        String sop_class_uid = dataset.getElementStringValue( DCM.E_SOPCLASS_UID, "" );
        sop_instance_uid = dataset.getElementStringValue( DCM.E_SOPINSTANCE_UID, "" );
        if(sop_class_uid.length() > 0){
        	sop_class_uid = sop_class_uid.trim();
        }
        if(sop_instance_uid.length() > 0){
        	sop_instance_uid = sop_instance_uid.trim();
        }
        
        try{			
	        sopClassName = getSOPClassName(sop_class_uid);

       		int presentationContextID = c_store_rq.context_id(); // this can throw a DCSException! 
       		// which if happens, something is fatally wrong before calling StoreObject
	        
	        //Build an object to contain the File Meta Information from the Association.
	        DicomFileMetaInfo fileMeta = new DicomFileMetaInfo();
	        fileMeta.setImplementationClassUID(ainfo.callingImplementationClassUid());
	        fileMeta.setImplementationVersionName(ainfo.calledImplementationVersionName());
	        fileMeta.setSourceAET(ainfo.callingTitle());
	        String transferSyntax = getAcceptedTransferSyntax(ainfo, presentationContextID);
	        fileMeta.setTransfersyntaxUID(transferSyntax);
	        
	        logger.debug("Building DicomPersistentObjectDescriptor - " + dicomAE.getRemoteAETitle() + "/" + threadID);
	        String persistent_id = new String(sop_instance_uid + ".dcm" );
	        
	        dpod_ret = new DicomPersistentObjectDescriptor(
	                        sop_class_uid,
	                        sop_instance_uid,
	                        persistent_id,
	                        "C-Store Rq");
	
	        // Create and pass the session settings to the Store object.
			DicomSessionSettings session_settings = new DicomSessionSettings();
			store.setSessionSettings(session_settings);

			logger.debug("Calling receiveIOD method in Store Object - " + dicomAE.getRemoteAETitle() + "/" + threadID);
			// This method passes the DICOM Dataset to the Store object.
			// This is basically where to wait for success or an exception
			// to be thrown. The result here is passed through a DIMSE C-Store
			// message notifying the SCU.
			int port = associationAcceptor.getAssociationManager().getServerTcpPort();
			InstrumentConfig instrument = DicomServerConfiguration.getConfiguration().getInstrumentByPort(port);

			store.receiveIOD(dataset, dicomAE, instrument, fileMeta, f_use_byte_reader_);
			DicomServiceStats.getInstance().incrementInboundAcceptCount(ainfo.callingTitle(), manufacturer, model, sopClassName);
        }
		catch(IODViolationException iodvX){	
			//I placed this here only because there are at least two different places that can call this 
			//	exception.  I am trying my best to keep statistical tracking for devices to this class.  This is the
			//	last VA implemented class before re-entering the DCF toolkit.  Also, I do not care if the response message
			//	returns a warning or a rejection.  I want to increment the IODViolation statistical counter in both cases.
			String msg = iodvX.getMessage();
			String [] msgPieces = StringUtil.split(msg, StringUtil.EQUALS); 
			DicomEmailInfo info=null;

			if (msgPieces[0].startsWith("Error")) { 	// FAILURE cases 
				
				// filter and post e-mail to queue instead of sending for each instance
				
				if (msgPieces[1].startsWith("IOD")) { 
					// for IOD error, msgPieces[3] is Series UID!
					info = new DicomEmailInfo(DicomEmailType.IOD, DicomLevel.SERIES, msgPieces[3]);
				} 
				else {	
					// for RESERR,  msgPieces[2] is SOPClass UID!
					info = new DicomEmailInfo(DicomEmailType.RESERR, DicomLevel.SERIES, msgPieces[2]);	
				}
				trackAndPostEmail(
						info,
			        	eMailTOs,	// recepients
			        	"DICOM Storage SCP " + (msgPieces[1].startsWith("IOD")? "IOD Violation":"Resource") + " Error",  // subject
			        	dicomAE.getRemoteAETitle() + " -> Exception thrown.\n" + msg+
			        	"\nNotice: The solution may need to be corrected by the device vendor.",	// body
		        		dicomAE.getRemoteAETitle() + "/" +threadID,
		        		dicomAE, threadID);	// context)

				logger.error(this.getClass().getName()+": AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM object "+sop_instance_uid
						+" in VistA Imaging due to IOD Validation.\n"+iodvX.getMessage());
				summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM object in VistA Imaging with SOP Instance UID "+sop_instance_uid+
						".\nIOD Violations.  Refer to other logs for more detail.");					
				DicomServiceStats.getInstance().incrementInboundRejectCount(ainfo.callingTitle(), manufacturer, model, sopClassName);

			    if (dicomAE.isRejectMessage()) { // send reject message only if configuration says so
			    	sendStoreRejectMessage(c_store_rsp, threadID,
							dicomAE, ainfo, manufacturer, model, sopClassName,
							msg, msgPieces);	// note: only up to 64 Chars (LO) can be sent to sender in the msgPieces.
				}
				else{
					throw new DDSException(); // do we need to throw Exception for all cases or only in e-mail case?
				}
			} 
			else{ 									// WARNING case				
				
				// filter and post e-mail to queue instead of sending for each instance
				
				// for Warning, msgPieces[2] is original SOP Instance UID!
				info = new DicomEmailInfo(DicomEmailType.UID, 
										  (msgPieces[1].contains("Study Instance UID")? DicomLevel.STUDY : DicomLevel.SERIES),
										  msgPieces[2]);
				trackAndPostEmail(
						info,
			        		eMailTOs,	// recepients
			        		"DICOM Storage SCP Replaced DUP/Illegal UID",  // subject
			        		dicomAE.getRemoteAETitle() + " -> Original SOP Instance UID of object, " + msgPieces[2] + ",\n" + msg,	// body
			        		dicomAE.getRemoteAETitle() + "/" +threadID,  // context
			        		dicomAE, threadID);	// context)		        

				logger.warn(this.getClass().getName()+": AETitle "+dicomAE.getRemoteAETitle()+": Storing DICOM object in VistA Imaging, but Warning due to Duplicate/Illegal " +
						"Instance UID:\n"+msg);
				DicomServiceStats.getInstance().incrementInboundAcceptCount(ainfo.callingTitle(), manufacturer, model, sopClassName);
			    if (dicomAE.isWarningMessage()) { // send warning message only if configuration says so
			    	sendStoreWarningMessage(c_store_rsp, threadID,
							dicomAE, ainfo, manufacturer, model, sopClassName,
							msgPieces);
				}
			}
		}
        catch (InvalidUserCredentialsException credentialsException) 
        {
        	handleInvalidCredentialsAndThrowDDSException(credentialsException);
		} 
        catch (DicomStorageSCPException storageException) {
			// If the exception is thrown, the DCF Toolkit automatically sends a
			// DIMSE C-Store message to the SCU stating the image failed
			// to store.
			logger.error(this.getClass().getName()+ ": DICOM Storage SCP Exception thrown while storing object received from "+ dicomAE.getRemoteAETitle() + "/" +threadID);
			logger.error(storageException.getMessage());
			summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM object in VistA Imaging with SOP Instance UID "+sop_instance_uid+
					".\nRefer to other logs for more detail.");					

			DicomEmailInfo info = new DicomEmailInfo(DicomEmailType.RESERR, DicomLevel.SERIES, dataset.getElementStringValue( DCM.E_SOPCLASS_UID, "" ).trim());	
			trackAndPostEmail(
					info,
					eMailTOs,											// recepients
	        		"DICOM Storage SCP Processing/DB Connection Error", // subject
	        		"Failed to store DICOM object with SOP Instance UID "+sop_instance_uid+":\n"+storageException.getMessage(),	// body
	        		dicomAE.getRemoteAETitle() + "/" +threadID,			// context)
	        		dicomAE, threadID);
			
			DicomServiceStats.getInstance().incrementInboundRejectCount(ainfo.callingTitle(), manufacturer, model, sopClassName);
			throw new DDSException();
		} 
        catch (DCSException dcs){
			logger.error(this.getClass().getName()
					+ ": DCS Exception - Could not read Presentation Context ID from Association/Dimse message. - " + dicomAE.getRemoteAETitle() + "/" +threadID);
			logger.error(dcs.getMessage());
			summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM object in VistA Imaging with SOP Instance UID "+sop_instance_uid
					+".\nRefer to other logs for more detail.");					

			DicomEmailInfo info = new DicomEmailInfo(DicomEmailType.RESERR, DicomLevel.SERIES, dataset.getElementStringValue( DCM.E_SOPCLASS_UID, "" ).trim());
			trackAndPostEmail(
					info,
					eMailTOs,									// recepients
	        		"DICOM Storage SCP Toolkit Level Error",	// subject
	        		dcs.getMessage() ,							// body
	        		dicomAE.getRemoteAETitle() + "/" +threadID,	// context)
	        		dicomAE, threadID);
						
			DicomServiceStats.getInstance().incrementInboundRejectCount(ainfo.callingTitle(), manufacturer, model, sopClassName);
			throw new DDSException();
		}
        
        logger.debug("Returning DicomPersistentObjectDescriptor - "+ dicomAE.getRemoteAETitle() + "/" +threadID);
        logger.debug(this.getClass().getName()+": DPOD:\n"+dpod_ret);
        return dpod_ret;
    }
  
    
    /**
     * Place email message into the VISA queue mechanism for later processing and sending.
     * @param info represents DicomEmailInfo.
     * @param eMailTOs represents the receivers of this email message.
     * @param msg represents the email subject line.
     * @param bodyLine represents the email body line.
     * @param context represents the context in which the email was created.
     * @param dicomAE represents the DICOM AE.  This contains the AETitles.
     * @param threadID represents the thread ID which is used to track activity log file if needed.
     */
    private void trackAndPostEmail (DicomEmailInfo info, 
    								 String[] eMailTOs, String subjectLine, String msg, String context,
    								 DicomAE dicomAE, String threadID){
        EmailMessage email = new EmailMessage(eMailTOs, subjectLine, msg);
    	DicomRouter rtr = DicomContext.getRouter();
	    //Normally, I remove commented code.  But this is an exception for now.
    	//	This has been here from the beginning.  But now the analyst are stating that every email generated should
    	//	should be sent, regardless if an similiar email was already sent.  If there are 5 failures, then there
    	//	should be 5 email messages sent.  Thus, I am commenting out this code based on the analyst.  If someone changes
    	//	their mind later, the code is still here.
    	//if (!DicomEmailTrackingCache.hasEmailBeenSent(info)){
	    	// post email to queue
	    	try { 
	    		rtr.postToEmailQueue(email, context);
	    	} 
	    	catch (MethodException me) {
				logger.error("Process error while queueing Email for \n'" + msg + "' - " + dicomAE.getRemoteAETitle() + "/" +threadID);	
	        } 
	    	catch (ConnectionException ce) {
				logger.error("DB Connection error while queueing Email for \n'" + msg + "' - " + dicomAE.getRemoteAETitle() + "/" +threadID);	
	        }
	    //}
    }
    
    /**
     * Process and save a dicom Storage commit request object and return request response to the sender.
     * This method is normally called by StoreSCP
     * when an N-ACTION_Storage Commit Request is received.  The received object would normally passed to 
     * the Generic DICOM Layer.
     * 
     * @param acceptor the current AssociationAcceptor. This contains
     * information about the client AE, negotiated contexts, etc.
     * @param nActionRequest the DIMSE message containing the N-Action request to be checked & stored.
     * @param nActionResponse the DIMSE message containing the N-Action response.
     * @throws DDSException if any other error occurs. A DIMSE error
     * status should be returned to the SCU if this method throws an exception.
     */
     public void storeCommitObject(AssociationAcceptor acceptor, DimseMessage nActionRequest, DimseMessage nActionResponse )
         throws DDSException{
         String threadID = " [" + Long.toString(Thread.currentThread().getId()) + "]";

         logger.debug("DicomDataService_a.storeCommitObject: dimse-message = " + nActionRequest + " -- ThreadID=" + threadID);

         DicomAE dicomAE = Listen.getDicomAEObject(acceptor);
     	 int status=0;
         boolean authenticateService = DicomServerConfiguration.getConfiguration().isAuthenticateAETitles();
     	 checkAuthenticatedServiceForStoreObject(acceptor, dicomAE, authenticateService);
         
         //assign AssociationInfo to local variable.
         AssociationInfo ainfo = acceptor.getAssociationInfo();
         
         logger.info("Received Storage Commit Request from SCU AE = " + ainfo.callingTitle() + " (" + ainfo.callingPresentationAddress() +
                 ") to SCP AE = " + ainfo.calledTitle() + "(" + ainfo.calledPresentationAddress() + ") -- ThreadID=" + threadID);

         String emailRecepient = DicomServerConfiguration.getConfiguration().getDgwEmailInfo().getEMailAddress();
         String[] eMailTOs=new String[1];
         eMailTOs[0] = emailRecepient;
         DicomDataSet dataset=null;
         
         try{
	        // This is how to get the proper Store Commit object for this specific association.
	        //  The Store Commit object is created during the beginning of the Association and 
	        //  saved in a hashmap.  The Store object is where specific Storage SCP
	        //  functionality happens.
	        logger.debug("Pulling correct Storage Commit Object from Hashmap - " + dicomAE.getRemoteAETitle() + "/" + threadID);
	        IDicomStorageCommitSCP sCReq = Listen.getStoreCommitRequestObject(acceptor);  

	        if (nActionRequest.actionTypeId()!=1)
	        	status=0x0123;
	        else {
		        logger.debug("Extracting dataset out of the DimseMessage Object - " + dicomAE.getRemoteAETitle() + "/" +threadID);
		        dataset = nActionRequest.data();       
	
	  	        // Create and pass the session settings to the Store object.
	 			DicomSessionSettings session_settings = new DicomSessionSettings();
	 			sCReq.setSessionSettings(session_settings);
	
	 			logger.debug("Calling receiveSCRequest method in storeCommitObject - " + dicomAE.getRemoteAETitle() + "/" + threadID);
	 			// This method passes the DICOM SC request to the DICOM StorageCommit SCP object.
	 			// This is basically where to wait for success or an exception
	 			// to be thrown. The result here is passed through a DIMSE N-ACTION
	 			// message notifying the SCU.
	 			int port = acceptor.getAssociationManager().getServerTcpPort();
	 			InstrumentConfig instrument = DicomServerConfiguration.getConfiguration().getInstrumentByPort(port);
	
	 			status = sCReq.receiveSCRequest(nActionRequest, dicomAE, instrument);
	        }
         }
 		catch(IODViolationException iodvX){	
     		
 			String msg = iodvX.getMessage();
 			
 			// parse message content and decide what to do
 			String [] msgPieces = StringUtil.split(msg, StringUtil.EQUALS); // note: only up to 64 Chars (LO) can be sent to sender
 			DicomEmailInfo info=null;
 			String dumpFile = null;
 			
//             if (DicomServerConfiguration.getConfiguration().isDicomDebugDumpEnabled() || (msgPieces[0].startsWith("Error"))){
//             	dumpFile = dumpObjectToDisk(threadID, dataset);
//             	logger.warn(this.getClass().getName()+": Dumped DICOM SC Request to disk due to IOD Violation. Filename is "+dumpFile);
//             }
 			if (msgPieces.length<=2) { 	// validation error
 				status=0x120; // missing attribute
 			} else {
				status=0x106; // invalid attribute value
				// insert TA UID into front of piece 2
				if (dataset!=null)
					try {
						String value="";
						value = dataset.getElementStringValue(DCM.E_TRANSACTION_UID);
						if ((value!=null) && !value.isEmpty()) {
							value = "=" + value.trim(); // take out leading and trailing whitespaces
							msgPieces[2]= value + msgPieces[2];
						}
					} catch (Exception e) {}
 			}

 			// filter and post e-mail to queue instead of sending for each instance
			info = new DicomEmailInfo(DicomEmailType.IOD, DicomLevel.NA, (msgPieces.length>2)?msgPieces[2]:"???"); // taUID or ???

			trackAndPostEmail(
						info,
			        	eMailTOs,	// recepients
			        	"DICOM Storage Commit Request Validation Error",  // subject
			        	dicomAE.getRemoteAETitle() + " -> \n" + msg + "\nStored File -> \n" + dumpFile,	// body
		        		dicomAE.getRemoteAETitle() + "/" +threadID,
		        		dicomAE, threadID);	// context)

				logger.error(this.getClass().getName()+": AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM Storage Commit Request in VistA Imaging due to IOD Violation: \n"+msg);
				summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM Storage Commit Request in VistA Imaging" + // with SOP Instance UID "+transactionUID+
				".  IOD Violation(s).  Refer to other logs for more detail.");					
 		 }
         catch (DicomStorageCommitSCPException dsce) {
 			// If the exception is thrown, the DCF Toolkit automatically sends a
 			// DIMSE C-Store message to the SCU stating the image failed
 			// to store.
 			logger.error(this.getClass().getName()
 					+ ": DICOM Storage Commit SCP Exception thrown while storing SC Request received from "+ dicomAE.getRemoteAETitle() + "/" +threadID);
 			logger.error(dsce.getMessage());
 			summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM Storage Commit Request in VistA Imaging" + // with Transaction UID "+transactionUID+
 					".  Refer to other logs for more detail.");					

 			DicomEmailInfo info = new DicomEmailInfo(DicomEmailType.RESERR, DicomLevel.NA, dataset.getElementStringValue( DCM.E_SOPCLASS_UID, "" ).trim());	
 			trackAndPostEmail(
 					info,
 					eMailTOs,											// recepients
 	        		"DICOM Storage Commit SCP Processing/DB Connection Error", // subject
 	        		dsce.getMessage() ,						// body
 	        		dicomAE.getRemoteAETitle() + "/" +threadID,			// context)
 	        		dicomAE, threadID);

 			if (dsce.getMessage().startsWith("T"))
 				status=0x0131; // The Transaction UID of the Storage Commitment Request is already in use
 			else									 // W, B, N: W - WI Tag(s), B - Bad List Item Count or N - No Data Errors
				status=0x106; // invalid attribute value
         } 
         catch (DCSException dcs){
 			logger.error(this.getClass().getName()
         	        + ": DCS Exception - Could not establish session settings. - " + dicomAE.getRemoteAETitle() + "/" +threadID);
 			logger.error(dcs.getMessage());
 			summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM Storage Commit Request in VistA Imaging" + // with SOP Instance UID "+sop_instance_uid
 					".  Refer to other logs for more detail.");					

 			DicomEmailInfo info = new DicomEmailInfo(DicomEmailType.RESERR, DicomLevel.SERIES, dataset.getElementStringValue( DCM.E_SOPCLASS_UID, "" ).trim());
 			trackAndPostEmail(
 					info,
 					eMailTOs,									// recepients
 	        		"DICOM Storage Commit SCP Toolkit Level Error",	// subject
 	        		dcs.getMessage() ,							// body
 	        		dicomAE.getRemoteAETitle() + "/" +threadID,	// context)
 	        		dicomAE, threadID);
 						
 			status=0x213; // resources limitation
 		}
		finally {
			if (status!=0) {
				try {
					nActionResponse.status(status);
				} catch (DCSException dcs) {
		 			logger.error(this.getClass().getName()
		         	        + ": DCS Exception - Could not set N-ACTION response status. - " + dicomAE.getRemoteAETitle() + "/" +threadID);
		 			logger.error(dcs.getMessage());
		 			summaryLogger.error("AETitle "+dicomAE.getRemoteAETitle()+": Failed to store DICOM Storage Commit Request in VistA Imaging" + // with SOP Instance UID "+sop_instance_uid
		 					".  Refer to other logs for more detail.");					
					
				}
//	 			throw new DDSException();
			}
		}
        logger.debug("Returning from "+ dicomAE.getRemoteAETitle() + "/" +threadID);    
     }
    
    public void findObjects(AssociationAcceptor assoc_acceptor, DimseMessage c_find_rq, 
            DicomDataServiceListener ddsListener) throws DDSException{

		
    	TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
        
        try
        {
        	// Establish a security context on this thread
        	DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
        	
    		String accessCode = config.getAccessCodeString();
    		String verifyCode = config.getVerifyCodeString();

    		ServerAgnosticEngineAdapter engineAdapter = Listen.getEngineAdapterObject(assoc_acceptor);
    		engineAdapter.authenticate(accessCode, verifyCode.getBytes());

        	//Instantiate Find object via Spring Factory.
             IDicomFindSCP find = (IDicomFindSCP)SpringContext.getContext().getBean("DicomFindSCP");
            //call method to pass parameters and execute the query command to the
            //Find object.  The Find boundary object will handle the sending of
            //responses to the SCU. 
            logger.info(this.getClass().getName()+": Dicom Toolkit Layer: " +
                    "...calling C-Find Service.");
            
            DicomAE dicomAE = Listen.getDicomAEObject(assoc_acceptor);
        	boolean authenticateService = DicomServerConfiguration.getConfiguration().isAuthenticateAETitles();
        	if(authenticateService){
	            if(!dicomAE.isServiceAndRoleValid("C-FIND", "SCU")){            
		            try {
		            	HashMap<String, String> eventElements = new HashMap<String, String>();
		            	eventElements.put("AETitle", dicomAE.getRemoteAETitle());
		            	String message = dicomAE.getRemoteAETitle()+" Authentication failure while attempting a CFind.";
		    			InternalContext.getRouter().postAuditEvent(
		    					null,
		    					new AuditEvent(AuditEvent.AUTHENTICATION_FAILURE,
		    							DicomServerConfiguration.getConfiguration().getHostName(),
		    							DicomServerConfiguration.getConfiguration().getApplicationName(),
		    							message,
		    							eventElements));
		    		}   
		            catch (ConnectionException cX) {
		                logger.error(cX.getMessage());
		                logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");			
		    		} 
		    	    catch (MethodException mX) {
		                logger.error(mX.getMessage());
		                logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");			
		    		}
	    			DicomServiceStats.getInstance().incrementInboundDimseMessageRejectCount(
	    					assoc_acceptor.getAssociationInfo().callingTitle(), 
	    					DicomServiceStats.CFIND);
		            logger.error(this.getClass().getName()+": "+dicomAE.getRemoteAETitle()+" is invalid to perform C-Find Requests.");
	    			summaryLogger.error("The Remote AETitle, "+dicomAE.getRemoteAETitle()
	    					+", does not have permission to perform a C-Find Dimse Service.\n"
	    					+"This permission is configurable using DICOM AE Security Matrix.");
		            ddsListener.findObjectsComplete(QRDimseStatus.PROCESSING_FAILURE, null);
		            throw new DDSException(dicomAE.getRemoteAETitle()+" is invalid to perform C-Find Requests.");
		        }
        	}
        	
            find.receiveQuery(assoc_acceptor, c_find_rq, ddsListener);

            DicomServiceStats.getInstance().incrementInboundDimseMessageAcceptCount(
    				assoc_acceptor.getAssociationInfo().callingTitle(), 
    				DicomServiceStats.CFIND);
    		
            summaryLogger.info("The C-Find Dimse message was accepted.");
        }
        catch (InvalidUserCredentialsException credentialsException) 
        {
        	handleInvalidCredentialsAndThrowDDSException(credentialsException);
		} 
        catch (DicomFindSCPException dfsException){
            //I determined the following exception needs to be thrown.  I'm not exactly 
            //  sure what it does with the association.  It might throw a C-Find-Rsp
            //  back to the SCU?
			DicomServiceStats.getInstance().incrementInboundDimseMessageRejectCount(
					assoc_acceptor.getAssociationInfo().callingTitle(), 
					DicomServiceStats.CFIND);
			summaryLogger.error("The C-Find Dimse message was rejected.  Refer to other logs for more detail.");
            throw new DDSException(dfsException.getMessage());
        }
        finally
        {
        	TransactionContextFactory.restoreTransactionContext(memento);
        }
    }
    
      
   public MoveCommandObserver moveObjects(AssociationAcceptor assocAcceptor, DimseMessage cMoveRq, 
           Observer subOperationsSubscriber) throws DDSException{
       
      	MoveCommandObserver cancelMoveHandle = null;

    	TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
    	try
    	{
           	//create the Move object
           	IDicomMoveSCP move = Listen.getMoveObject(assocAcceptor);

           
           	// Establish a security context on this thread
           	DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
           	
    		String accessCode = config.getAccessCodeString();
    		String verifyCode = config.getVerifyCodeString();
    		ServerAgnosticEngineAdapter engineAdapter = Listen.getEngineAdapterObject(assocAcceptor);
    		engineAdapter.authenticate(accessCode, verifyCode.getBytes());

           //call method to pass parameters and execute the query command to the
           //Move object.  The Move boundary object will handle the sending of
           //responses to the SCU. 
           logger.info(this.getClass().getName()+": Dicom Toolkit Layer: " +
                "...calling C-Move Service.");
           
           DicomAE dicomAE = Listen.getDicomAEObject(assocAcceptor);
       	   boolean authenticateService = DicomServerConfiguration.getConfiguration().isAuthenticateAETitles();
       	   if(authenticateService){
	           if(!dicomAE.isServiceAndRoleValid("C-MOVE", "SCU")){            
	        	   try {
			           	HashMap<String, String> eventElements = new HashMap<String, String>();
			           	eventElements.put("AETitle", dicomAE.getRemoteAETitle());
			           	String message = dicomAE.getRemoteAETitle()+" Authentication failure while attempting a CMove.";
			   			InternalContext.getRouter().postAuditEvent(
			   					null,
			   					new AuditEvent(AuditEvent.AUTHENTICATION_FAILURE,
			   								DicomServerConfiguration.getConfiguration().getHostName(),
			   								DicomServerConfiguration.getConfiguration().getApplicationName(),
			   								message,
			   								eventElements));
		   			}   
		           	catch (ConnectionException cX) {
		               logger.error(cX.getMessage());
		               logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");			
		           	} 
		           	catch (MethodException mX) {
		               logger.error(mX.getMessage());
		               logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");			
		           	}
		    		DicomServiceStats.getInstance().incrementInboundDimseMessageRejectCount(
		    				assocAcceptor.getAssociationInfo().callingTitle(), 
		    				DicomServiceStats.CMOVE);
		           	logger.error(this.getClass().getName()+": "+dicomAE.getRemoteAETitle()+" is invalid to perform C-Move Requests.");
	    			summaryLogger.error("The Remote AETitle, "+dicomAE.getRemoteAETitle()
	    					+", does not have permission to perform a C-Move Dimse Service.\n"
	    					+"This permission is configurable using DICOM AE Security Matrix.");
		           	throw new DDSException(dicomAE.getRemoteAETitle()+" is invalid to perform C-Find Requests.");
	           }
       	   }
           cancelMoveHandle = move.receiveMove(assocAcceptor, cMoveRq, 
        		   subOperationsSubscriber);           

           DicomServiceStats.getInstance().incrementInboundDimseMessageAcceptCount(
   				assocAcceptor.getAssociationInfo().callingTitle(), 
   				DicomServiceStats.CMOVE);
           summaryLogger.info("The C-Move Dimse message was accepted.");

       }
       catch (InvalidUserCredentialsException credentialsException) 
       {
    	   handleInvalidCredentialsAndThrowDDSException(credentialsException);
       } 
       catch (DicomMoveSCPException dfsException){
           logger.error(dfsException.getMessage());
           logger.error(this.getClass().getName()+": Exception thrown while moving objects.");
   			DicomServiceStats.getInstance().incrementInboundDimseMessageRejectCount(
				assocAcceptor.getAssociationInfo().callingTitle(), 
				DicomServiceStats.CMOVE);
			summaryLogger.error("The C-Move Dimse message was rejected.  Refer to other logs for more detail.");
           throw new DDSException(dfsException.getMessage());
       }
       finally
       {
       		TransactionContextFactory.restoreTransactionContext(memento);
       }

       return cancelMoveHandle;
   }
   
   public void updateObject(AssociationAcceptor association, DimseMessage n_set_request)
           throws DDSException{
       
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not Implemented.");
   }
   
   //NOTE Compiler warning about Collection not parameterized.  This is cause by the 
   //	DCF Toolkit.  Cannot be changed.
   public Vector findObjectsForTransfer(AssociationAcceptor association, DimseMessage query)
           throws DDSException{
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not Implemented.");
   }
   
   public void deleteObject(DicomPersistentObjectDescriptor dpod, boolean f_read_pixel_data)
           throws DDSException{
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
   public DicomPersistentObjectDescriptor duplicateObject(
           DicomPersistentObjectDescriptor target_dpod) throws DDSException{
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
   public void referenceObject(DicomPersistentObjectDescriptor dpod)
           throws DDSException{
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
   public void dereferenceObject(DicomPersistentObjectDescriptor dpod)
           throws DDSException{
       //FUTURE Method not currently used or implmented.
       throw new DDSException("Method not implemented.");
   }
   
   public DicomDataSet loadObject(DicomPersistentObjectDescriptor dpod, 
           boolean f_read_pixel_data) throws DDSException{
       
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
    /*
     * Utility method to write the inputstream to disk.
     */
   private void transferInputStreamToFile( String filename, InputStream in )
    throws Exception{
        int block_size = 16384;
        byte buffer[] = new byte[ block_size ];
    
        FileOutputStream out = new FileOutputStream( filename );
        int count;
    
        while ( ( count = in.read( buffer )) > 0 ){
            out.write( buffer, 0, count );
        }
        out.close();
    }
   
   public void commitCompleted(DimseServiceUser user, DimseMessage message)
           throws DDSException{
       
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
   public void commitRequestReceived(AssociationAcceptor assoc_accept, 
           DimseMessage message)throws DDSException{
       logger.info("DicomDataService_a.commitRequestReceived: dimse-msg = [" + message + "]");

       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }
   
   public void commitRequestSent(AssociationRequester requester, DimseMessage msg1, 
           DimseMessage msg2) throws DDSException{
       
       //FUTURE Method not currently used or implemented.
       throw new DDSException("Method not implemented.");
   }

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.lbs.DDS.DicomDataService#commitCompleted(com.lbs.DCS.DimseServiceUser,
	 *      com.lbs.DCS.DimseMessage, com.lbs.DCS.DimseMessage)
	 */
	@Override
	public void commitCompleted(DimseServiceUser arg0, DimseMessage arg1, DimseMessage arg2) throws DDSException
	{
       logger.info("DicomDataService_a.commitCompleted (Override): dimse-msg1 = [" + arg1 + "];  dimse-msg2 = [" + arg2 + "]");
		// TODO Auto-generated method stub
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.lbs.DDS.DicomDataService#commitRequestReceived(com.lbs.DCS.AssociationAcceptor,
	 *      com.lbs.DCS.DimseMessage, com.lbs.DCS.DimseMessage)
	 */
	@Override
	public void commitRequestReceived(AssociationAcceptor associationAcceptor, DimseMessage nActionRequest, DimseMessage nActionResponse) throws DDSException
	{
//	    logger.info("DicomDataService_a.commitRequestReceived (Override): dimse-N-ACTION-request = [" + nActionRequest + "];   dimse-N-ACTION-response = [" + nActionResponse + "]");
	    this.storeCommitObject(associationAcceptor, nActionRequest, nActionResponse);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.lbs.DDS.DicomDataService#loadObject(com.lbs.DCS.DicomPersistentObjectDescriptor,
	 *      boolean, com.lbs.DCS.DicomSessionSettings)
	 */
	@Override
	public DicomDataSet loadObject(DicomPersistentObjectDescriptor arg0, boolean arg1, DicomSessionSettings arg2) throws DDSException
	{
		// TODO Auto-generated method stub
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.lbs.DDS.DicomDataService#storeObject(com.lbs.DCS.AssociationAcceptor,
	 *      com.lbs.DCS.DimseMessage, com.lbs.DCS.DimseMessage,
	 *      com.lbs.DCS.DicomPersistentObjectDescriptor)
	 */
	@Override
	public void storeObject(AssociationAcceptor associationAcceptor, DimseMessage cStoreRequest, DimseMessage cStoreResponse, DicomPersistentObjectDescriptor dicomPersistentObjectDescriptor) throws DDSException
	{
		// forward the call on to our implementation method....
		dicomPersistentObjectDescriptor = this.storeObject(associationAcceptor, cStoreRequest, cStoreResponse);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see com.lbs.DDS.DicomDataService#storeObject(com.lbs.DCS.DicomDataSet,
	 *      com.lbs.DCS.DicomPersistentObjectDescriptor,
	 *      com.lbs.DCS.DicomSessionSettings)
	 */
	@Override
	public void storeObject(DicomDataSet arg0, DicomPersistentObjectDescriptor arg1, DicomSessionSettings arg2) throws DDSException
	{
		// TODO Auto-generated method stub

	}
	
	
	
	/**
	 * @param associationAcceptor
	 * @param dicomAE
	 * @param authenticateService
	 * @throws DDSException
	 */
	private void checkAuthenticatedServiceForStoreObject(
			AssociationAcceptor associationAcceptor, DicomAE dicomAE,
			boolean authenticateService) throws DDSException {
		if(authenticateService){
    		if(!dicomAE.isServiceAndRoleValid("C-STORE", "SCU")){
    			logger.error(this.getClass().getName()+": "+dicomAE.getRemoteAETitle()+" is invalid to perform C-Store Requests.");
    			summaryLogger.error("The Remote AETitle, "+dicomAE.getRemoteAETitle()
    					+", does not have permission to perform a C-Store Dimse Service.\n"
    					+"This permission is configurable using DICOM AE Security Matrix.");
    			try{
    			DicomServiceStats.getInstance().incrementInboundDimseMessageRejectCount(
    					associationAcceptor.getAssociationInfo().callingTitle(), 
    					DicomServiceStats.CSTORE);
    			}
    			catch(Exception X){
    				logger.warn(this.getClass().getName()+"Failed to update Statistics.");
    			}
    			throw new DDSException(dicomAE.getRemoteAETitle()+" is invalid to perform C-Store Requests.");
    		}
    	}
	}
	
	
	/**
	 * @param ainfo
	 * @param presentationContextID
	 * @param transferSyntax
	 * @return
	 */
	private String getAcceptedTransferSyntax(AssociationInfo ainfo, int presentationContextID) {
		// Look for TS in accepted Presentation Context
        String transferSyntax = null;
		Vector<AcceptedPresentationContext> apcCollection = ainfo.acceptedPresentationContextList();
		Iterator<AcceptedPresentationContext> iter2 = apcCollection.iterator();
		while(iter2.hasNext()){
			AcceptedPresentationContext apc = (AcceptedPresentationContext)iter2.next();
			if(apc.id() == presentationContextID){
				transferSyntax = apc.transferSyntax();
			}
		}
		if(transferSyntax == null){
			transferSyntax = UID.TRANSFERLITTLEENDIANEXPLICIT;
		}
		
		//This IF is here to allow the config file to override network transfer syntax if necessary.
		if(this.ts_uid_ != null){
			if(!(this.ts_uid_.equals(""))){
		    	transferSyntax = this.ts_uid_;        		
			}
		}
		return transferSyntax;
	}

	
	/**
	 * Get the human readable name of the SOP Class.
	 * 
	 * @param sopClassUID
	 * @return
	 */
	private String getSOPClassName(String sopClassUID){
        String sopClassName = "UNKNOWN";
		sopClassName = DCMUID.name(sopClassUID);
		return sopClassName;
	}

	
	
	/**
	 * @param c_store_rsp
	 * @param threadID
	 * @param dicomAE
	 * @param ainfo
	 * @param manufacturer
	 * @param model
	 * @param sopClassName
	 * @param msgPieces
	 */
	private void sendStoreWarningMessage(DimseMessage c_store_rsp,
			String threadID, DicomAE dicomAE, AssociationInfo ainfo,
			String manufacturer, String model, String sopClassName,
			String[] msgPieces) {
		int status;
		try {
			status=0xB000; // WARNING / Warning: Coercion of Data Elements -- set (0000, 0901)? and (0000,0902)
			c_store_rsp.status(status); 
			c_store_rsp.errorComment("Duplicate/Illegal UIDs for "+msgPieces[2]); // (0000,0902)
		} catch (DCSException dcse) { // while creating a DicomUSElement  for the value
			logger.error("Failed to set warning status and/or message for '" + msgPieces[1]  +
						 "' in c_store_rsp response" + dicomAE.getRemoteAETitle() + "/" +threadID);	
		}
	}

	
	/**
	 * @param c_store_rsp
	 * @param threadID
	 * @param dicomAE
	 * @param ainfo
	 * @param manufacturer
	 * @param model
	 * @param sopClassName
	 * @param msg
	 * @param msgPieces
	 */
	private void sendStoreRejectMessage(DimseMessage c_store_rsp,
			String threadID, DicomAE dicomAE, AssociationInfo ainfo,
			String manufacturer, String model, String sopClassName, String msg,
			String[] msgPieces) {
		int status;
		DicomServiceStats.getInstance().incrementInboundRejectCount(ainfo.callingTitle(), manufacturer, model, sopClassName);
		try {
			if (!msgPieces[1].startsWith("IOD")) { 
				status = 0xA701; // RESERR -- Refused: Out of resource or Unsupported SOP Class -- set (0000,0902)
			} 
			else {
				status = 0xA901; // REJECT -- Reject: Validation error; set (0000, 0901)? and (0000,0902)
			}
			c_store_rsp.status(status);
			c_store_rsp.errorComment("Resource Error for "+msgPieces[2]);	// (0000,0902)
		} catch (DCSException dcse) { // while creating a DicomUSElement  for the value
			logger.error("Failed to set error status and/or message for '" + msg +
						 "' in c_store_rsp response - " + dicomAE.getRemoteAETitle() + "/" +threadID);	
		}
	}
	
    protected void handleInvalidCredentialsAndThrowDDSException(InvalidUserCredentialsException e) 
    throws DDSException
    {
    	// log the error
		logger.error("The HDIG currently has invalid service account credentials.", e);
		summaryLogger.error("The HDIG currently has invalid service account credentials.");					
    	
		// Send notification
		String subject = "DICOM listeners are shutting down: invalid service account credentials";
		String message = "The system is shutting down all DICOM listeners because the service account credentials" +
						 " are invalid.";
		NotificationFacade.sendNotification(NotificationTypes.InvalidServiceAccountCredentials, subject, message);

    	// Shut down the listers
		DicomEngineAdapter.stopListeners();
		
		throw new DDSException();
    }

}
