/*
 * Created on Aug 1, 2005
// Per VHA Directive 2004-038, this routine should not be modified.
//+---------------------------------------------------------------+
//| Property of the US Government.                                |
//| No permission to copy or redistribute this software is given. |
//| Use of unreleased versions of this software requires the user |
//| to execute a written test agreement with the VistA Imaging    |
//| Development Office of the Department of Veterans Affairs,     |
//| telephone (301) 734-0100.                                     |
//|                                                               |
//| The Food and Drug Administration classifies this software as  |
//| a medical device.  As such, it may not be changed in any way. |
//| Modifications to this software may result in an adulterated   |
//| medical device under 21CFR820, the use of which is considered |
//| to be a violation of US Federal Statutes.                     |
//+---------------------------------------------------------------+
 *
*/
package gov.va.med.imaging.dicom.dcftoolkit.findscp.impl;

import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.InvalidUserCredentialsException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.router.facade.InternalContext;
import gov.va.med.imaging.dicom.common.SpringContext;
import gov.va.med.imaging.dicom.common.interfaces.IDicomDataSet;
import gov.va.med.imaging.dicom.common.interfaces.IFindSCPResponseCallback;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.DicomFindSCPException;
import gov.va.med.imaging.dicom.dcftoolkit.common.exceptions.ParseIODException;
import gov.va.med.imaging.dicom.dcftoolkit.findscp.interfaces.IDicomFindSCP;
import gov.va.med.imaging.dicom.dcftoolkit.listen.Listen;
import gov.va.med.imaging.dicom.dcftoolkit.scp.ServiceSCP;
import gov.va.med.imaging.dicom.scp.exceptions.GenericDicomSCPException;
import gov.va.med.imaging.dicom.scp.findscp.interfaces.IFindSCPControl;
import gov.va.med.imaging.exchange.business.AuditEvent;
import gov.va.med.imaging.exchange.business.dicom.CFindResults;
import gov.va.med.imaging.exchange.business.dicom.DicomAE;
import gov.va.med.imaging.exchange.business.dicom.DicomServerConfiguration;

import java.util.HashMap;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.lbs.DCS.AssociationAcceptor;
import com.lbs.DCS.DCSException;
import com.lbs.DCS.DicomDataSet;
import com.lbs.DCS.DimseMessage;
import com.lbs.DCS.QRDimseStatus;
import com.lbs.DCS.ValidationErrorList;
import com.lbs.DDS.DicomDataServiceListener;

/**
 *
 * This class is an implementation of the DicomFindSCP Interface. This implementation
 * receives the toolkit specific DicomDataset.  The DICOM Dataset is converted to a 
 * generic DicomDataSetThe and then passed to the Generic DICOM Layer.
 * 
 * The Find SCP Service needs to return multiple dicom messages to the Find SCU. The
 * first step is to convert the format of the data back into a toolkit specific
 * DicomDataSet.  The Implementation shall perform this format conversion.  The 
 * second step is to return the DicomDataSet(s) individually to the Find SCU.  The 
 * implementation shall perform returning the Dicom responses to the original 
 * C-Find Query.
 * 
 * The DICOM Dataset may go through a message validation process.
 *
 * @author William Peterson
 *
 */
public class DicomFindSCPImpl extends ServiceSCP implements IDicomFindSCP {

    
    /*
     * VA class. StoreControl works at the Generic DICOM layer.  This is for use with Spring.
     */
    private IFindSCPControl findControl = null;        
    private static final Logger logger = LogManager.getLogger(DicomFindSCPImpl.class);
     
    /**
     * 
     * Constructor
     * 
     */
    public DicomFindSCPImpl() {

    }
       
    /* (non-Javadoc)
     * @see gov.va.med.imaging.dicom.dcftoolkit.findscp.interfaces.IDicomFindSCP#receiveQuery(com.lbs.DDS.DicomDataServiceListener)
     */
    public void receiveQuery(AssociationAcceptor assoc_acceptor, DimseMessage c_find_rq, DicomDataServiceListener ddsListener) 
    throws DicomFindSCPException, InvalidUserCredentialsException
    {
        
        findControl = (IFindSCPControl)SpringContext.getContext().getBean("FindSCPControl");
        DicomDataSet dds = null;
        CFindResults nonDicomResultSet = null;
        String AETitle = assoc_acceptor.getAssociationInfo().callingTitle();
        logger.debug("Calling AETitle for C-FIND message: "+AETitle);

        try {
        	HashMap<String, String> eventElements = new HashMap<String, String>();
        	eventElements.put("AETitle", AETitle);
        	String message = AETitle+", CFind Request.";
			InternalContext.getRouter().postAuditEvent(
					null,
					new AuditEvent(AuditEvent.DICOM_QUERY, 
						DicomServerConfiguration.getConfiguration().getHostName(),
						DicomServerConfiguration.getConfiguration().getApplicationName(),
						message,
						eventElements
					));
		} 
        catch (InvalidUserCredentialsException e)
        {
        	// Rethrow here so it can be handled at the top level.
        	throw(e);
        }
        catch (ConnectionException cX) {
            logger.error(cX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");
			
		} 
        catch (MethodException mX) {
            logger.error(mX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown posting to Audit Log.");			
		}
        
        try{
            DicomAE dicomAE = Listen.getDicomAEObject(assoc_acceptor);
            //VR-Validate the incoming identifier.
            ValidationErrorList violations = this.validateVRs(dicomAE, c_find_rq);
    			if(violations.hasErrors()){
				this.logVRViolationsForDimseMessage(dicomAE, c_find_rq, violations);
    	            ddsListener.findObjectsComplete(QRDimseStatus.INVALID_ATTR, null);
    	            throw new DicomFindSCPException("VR Violation Error(s) in C-Find-RQ message");
    			}
                            
            //Parse the dataset from the Dimse Message.
            logger.info(this.getClass().getName()+": Dicom Toolkit layer: " +
                    "...parsing incoming IOD.");

            dds = this.parseIOD(c_find_rq);
            logger.info(this.getClass().getName()+": Dicom Toolkit layer: " +
                    "Received C-Find DataSet: ");
            logger.info("\n"+dds.toString());
        

            //make sure the UID exist before proceeding.
            String queryRootUID = c_find_rq.affectedSopclassUid();            
            if((queryRootUID == null) || (queryRootUID.equals(""))){
                logger.error(this.getClass().getName()+": Dicom Toolkit layer: " +
                        "...throwing exception because of empty Query Root UID.");
                throw new DicomFindSCPException("No Query SOP Class UID.");
            }
            //Pass parameters to the FindSCPControl object.
            //Creating a Generic DICOM Dataset to exist at the Generic DICOM layer.
            //  At this point, the DICOM Dataset becomes toolkit independent by way
            //  of the Interface.
            IDicomDataSet dataset = (IDicomDataSet)SpringContext.getContext().getBean("DicomDataSet");
            dataset.setDicomDataSet(dds);
            dataset.setAffectedSOPClass(queryRootUID);
            
            nonDicomResultSet = findControl.requestMatchingItems(dataset);
            
        }
        catch (DCSException dcsX){
            logger.error(dcsX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while processing Query.");
            ddsListener.findObjectsComplete(QRDimseStatus.OUT_OF_RESOURCES_REFUSED, null);
            throw new DicomFindSCPException("Failure from the Dicom Toolkit.", dcsX);
        }
        catch (ParseIODException pX){
            logger.error(pX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while parsing DIMSE message.");
            ddsListener.findObjectsComplete(QRDimseStatus.OUT_OF_RESOURCES_REFUSED, null);
            throw new DicomFindSCPException("Failure to parse DicomDataSet from DIMSE message.", pX);
        }
        catch (GenericDicomSCPException gdscpX){
            logger.error(gdscpX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while processing Query.");
            ddsListener.findObjectsComplete(QRDimseStatus.OUT_OF_RESOURCES_REFUSED, null);
            throw new DicomFindSCPException(gdscpX);
        }
        catch (NullPointerException npX){
            logger.error(npX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown processing Query Request.");
            logger.error("Trace:", npX);
            ddsListener.findObjectsComplete(QRDimseStatus.OUT_OF_RESOURCES_REFUSED, null);
            throw new DicomFindSCPException("Null Pointer Exception.", npX);
        }
        
        try{
            IFindSCPResponseCallback responseCallback = new FindSCPResponseCallbackImpl(ddsListener);

            logger.info(this.getClass().getName()+": Dicom Toolkit layer: " +
            "...converting RowSets to Dicom Datasets.");

            DicomAE dicomAE = Listen.getDicomAEObject(assoc_acceptor);
            findControl.sendCFindResults(nonDicomResultSet, dicomAE, responseCallback);
        }
        catch(GenericDicomSCPException gdscpX){
            logger.error(gdscpX.getMessage());
            logger.error(this.getClass().getName()+": Exception thrown while returning Responses.");
        }
    }
}
