/**
 * 
 */
package gov.va.med.imaging.hi5.client;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gwt.canvas.dom.client.CanvasPixelArray;
import com.google.gwt.touch.client.Point;

/**
 * @author       BECKEC
 *
 */
public class LinearWindowLevel
{
    private Logger logger = LogManager.getLogger("LinearWindowLevel");
    
    private final ImageStatistics imageStats;
    private final AffineTransform affineTransform; 
    private final PixelTransform pixelTransform;
    private double level;
    private double window;
    
    public LinearWindowLevel(ImageStatistics imageStats, AffineTransform affineTransform, PixelTransform pixelTransform)
    {
        this.imageStats = imageStats;
        this.affineTransform = affineTransform;
        this.pixelTransform = pixelTransform;
        
        this.setLevel(imageStats.getLevel());
        this.setWindow(imageStats.getWindow());
    }
    
    public ImageStatistics getImageStats()
    {
        return imageStats;
    }

    public AffineTransform getAffineTransform()
    {
        return affineTransform;
    }

    public PixelTransform getPixelTransform()
    {
        return pixelTransform;
    }

    public double getLevel()
    {
        return level;
    }

    public void setLevel(double level)
    {
        if(level < 0.0)
            this.level = 0.0;
        else if(level > getImageStats().getMaximum())
            this.level = getImageStats().getMaximum();
        else
            this.level = level;
    }

    public double getWindow()
    {
        return window;
    }

    public void setWindow(double window)
    {
        if(window < 1.0)
            this.window = 1.0;
        else if(window > getImageStats().getMaximumWindow())
            this.window = getImageStats().getMaximumWindow();
        else
            this.window = window;
    }
    
    /**
     * 
     * @param context
     * @param width
     * @param height
     * @param pixelData
     */
    public void loadCanvasPixelArray( int[] pixelData, double width, double height, CanvasPixelArray cpa)
    { 
        double bucketSize = getWindow() / getImageStats().getDisplayGrayScaleLevels();
        double minWindowValue = getLevel() - (getWindow() / 2.0);
        double maxWindowValue = minWindowValue + getWindow();
        logger.info("start loadContext [" + getLevel() + "-" + getWindow() + "] " + bucketSize);

        for (int y = 0; y < height; ++y)
        {
            int rowOffset = y * (int)width;
            //StringBuilder rowDataMsg = new StringBuilder("Row: " + y + " ");
            
            for (int x = 0; x < width; ++x)
            {
                double pixelValue = pixelData[rowOffset + x];
                
                int value = 0;
                if(pixelValue > maxWindowValue)
                    value = 255;
                else if(pixelValue < minWindowValue)
                    value = 0;
                else
                    value = (int)((pixelValue - minWindowValue) / bucketSize); 
                //rowDataMsg.append(Integer.toHexString(pixelValue) + " ->" + Integer.toHexString(value) + ", ");
                
                // Do any pixel transformation required (e.g. invert)
                // Pixel transformations change pixel values but not position in the image.
                RGBPixel originPixel = new RGBPixel(value, value, value);
                RGBPixel transformedPixel = getPixelTransform().transformPixel(originPixel);

                // Do the affine transformation requested.
                // This maps the origin point to the destination image point and could include
                // rotation and flipping.
                Point originPoint = new Point(x - (width /2 ), y - (height / 2));        // the origin is a double array, one double per pixel
                Point destPoint = getAffineTransform().transformPoint(originPoint);
                int destOffset = getAffineTransform().destinationCPAPixelOffset(destPoint);

                transformedPixel.setValueIn(cpa, destOffset);
            }
            //logger.info(rowDataMsg.toString());
        }
        logger.info("stop loadContext [" + getLevel() + "-" + getWindow() + "] " + bucketSize);
    }

    /**
     * 
     * @param pixelData
     */
    public static ImageStatistics analyze(int width, int height, int grayScaleLevels, int displayGrayScaleLevels, int[] pixelData)
    {
        int histogramBucketCount = grayScaleLevels / displayGrayScaleLevels;
        int[] histogram = new int[histogramBucketCount];
        for(int index=0; index<histogramBucketCount; ++index)
            histogram[index] = 0;
        
        double minimum = Double.MAX_VALUE;
        double maximum = Double.MIN_VALUE;
        double window = 0.0;
        double level = 0.0;
            
        for(int pixel : pixelData)
        {
            if(pixel > maximum) maximum = pixel;
            if(pixel < minimum) minimum = pixel;
            ++histogram[pixel / displayGrayScaleLevels];
        }

        StringBuilder sb = new StringBuilder("Histogram data [");
        for(int index=0; index<histogramBucketCount; ++index)
            sb.append( histogram[index] + ",");
        
        sb.append("]");
        
        window = (maximum - minimum) / 4;
        
        level = ((maximum - minimum) / 2) + minimum;
        
        ImageStatistics imageStats = new ImageStatistics(
            width, height, 
            grayScaleLevels, displayGrayScaleLevels, 
            window, level,
            maximum, minimum,
            histogram);
        
        return imageStats;
    }
    
}
