package gov.va.med.imaging.hi5.client;

import gov.va.med.imaging.hi5.shared.ArrayConversion;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gwt.canvas.client.Canvas;
import com.google.gwt.canvas.dom.client.CanvasGradient;
import com.google.gwt.canvas.dom.client.CanvasPixelArray;
import com.google.gwt.canvas.dom.client.Context2d;
import com.google.gwt.canvas.dom.client.ImageData;
import com.google.gwt.core.client.EntryPoint;
import com.google.gwt.core.client.GWT;
import com.google.gwt.dom.client.Element;
import com.google.gwt.dom.client.NativeEvent;
import com.google.gwt.event.dom.client.MouseDownEvent;
import com.google.gwt.event.dom.client.MouseDownHandler;
import com.google.gwt.event.dom.client.MouseMoveEvent;
import com.google.gwt.event.dom.client.MouseMoveHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseUpEvent;
import com.google.gwt.event.dom.client.MouseUpHandler;
import com.google.gwt.event.shared.UmbrellaException;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.DockPanel;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.TextArea;
import com.smartgwt.client.types.Side;
import com.smartgwt.client.widgets.tab.Tab;
import com.smartgwt.client.widgets.tab.TabSet;
import com.smartgwt.client.widgets.toolbar.ToolStrip;

/**
 * Entry point classes define <code>onModuleLoad()</code>.
 */
public class Hi5 
implements EntryPoint, DataSourceEventListener
{
    private static final String IMAGE_CANVAS_ELEMENT_ID = "imageCanvas";

    private static final String DICOM_IMAGE_PIXELS_PATH = "/Hi5/dicomImagePixels/";
    
    private static final int MESSAGE_BUFFER_SIZE = 2048;
    
    private final static String ICON_PACKAGE_24 = "buttons/24X24/";
    //private final static String ICON_PACKAGE_72 = "buttons/72X72/";
    private String ICON_PACKAGE = ICON_PACKAGE_24;
   
    private final String CONTAINER_ID = "container";

    private DockPanel rootDockPanel;
    private TabSet elementsOrPixelsTabSet;
    private Element containerElement;
    private ToolStrip genericTools;
    
    private DicomDataSetDataSource dicomDataSource;
    private DicomDataSetGrid dataSetGrid;
    private ToolStrip elementTools;
    
    private Canvas imageCanvas;
    private ToolStrip imageTools;
    private Element imageCanvasContainingElement;    
    private Context2d context;
    private Canvas backBufferCanvas;
    private Context2d backBufferContext;
    
    private TextArea messageLabel;
    private DisplayResolution displayResolution;
    
    private Logger logger = LogManager.getLogger("Hi5");

    boolean isDevelopmentMode() 
    {
        return !GWT.isScript() && GWT.isClient();
    }
    
    /**
     * This is the entry point method.
     */
    public void onModuleLoad()
    {
        GWT.setUncaughtExceptionHandler(new GWT.UncaughtExceptionHandler()
        {
            @Override
            public void onUncaughtException(Throwable e)
            {
                logException(e);
            }
        });
        this.dicomDataSource = new DicomDataSetDataSource();
        this.dicomDataSource.addDataSourceEventListener(this);

        this.rootDockPanel = new DockPanel();
        
        this.elementsOrPixelsTabSet = new TabSet();  
        this.elementsOrPixelsTabSet.setTabBarPosition(Side.LEFT);  
        this.elementsOrPixelsTabSet.setTabBarAlign(Side.TOP);  
        this.containerElement = RootPanel.get(CONTAINER_ID).getElement();

        Tab pixelTab = new Tab("Image", getICON_PACKAGE() + "drawing_board.png");
        pixelTab.setID("pixelTab");
        DockPanel pixelDockPanel = new DockPanel();
        try
        {
            this.imageCanvas = createImageCanvas();
            this.backBufferCanvas = Canvas.createIfSupported();
        } 
        catch (BrowserInsufficientException e1)
        {
            Window.alert(e1.getMessage());
            return;
        }
        this.context = imageCanvas.getContext2d();
        this.imageCanvasContainingElement = imageCanvas.getElement();
        this.backBufferContext = backBufferCanvas.getContext2d();
        //this.imageCanvas.setWidth( "512px" );
        //this.imageCanvas.setHeight( "512px" );
        
        this.imageTools = new ImageToolStrip(this);
        pixelDockPanel.add(this.imageTools, DockPanel.NORTH);
        pixelDockPanel.add(this.imageCanvas, DockPanel.CENTER);
        
        com.smartgwt.client.widgets.Canvas pixelPane = new com.smartgwt.client.widgets.Canvas();
        pixelPane.addChild(pixelDockPanel);
        pixelTab.setPane(pixelPane);
        
        Tab elementTab = new Tab("Header", getICON_PACKAGE() + "document.png");
        DockPanel elementDockPanel = new DockPanel();
        this.dataSetGrid = new DicomDataSetGrid(this.dicomDataSource);
        ScrollPanel dataSetGridScroll = new ScrollPanel(this.dataSetGrid);
        this.elementTools = new ElementToolStrip(this);
        elementDockPanel.add(this.elementTools, DockPanel.NORTH);
        elementDockPanel.add(dataSetGridScroll, DockPanel.CENTER);
        com.smartgwt.client.widgets.Canvas elementPane = new com.smartgwt.client.widgets.Canvas();
        elementPane.addChild(elementDockPanel);
        elementTab.setPane(elementPane);

        
        this.elementsOrPixelsTabSet.addTab(pixelTab);
        this.elementsOrPixelsTabSet.addTab(elementTab);
        
        // fill up the available space
        elementsOrPixelsTabSet.setWidth100();
        elementsOrPixelsTabSet.setHeight100();
        
        this.genericTools = new GenericToolStrip(this);
        this.genericTools.getElement().setId("genericTools");
        this.messageLabel = new TextArea();
        this.messageLabel.getElement().setId("messageLabel");
        
        this.rootDockPanel.add(this.genericTools, DockPanel.NORTH);
        this.rootDockPanel.add(this.elementsOrPixelsTabSet, DockPanel.CENTER);
        this.rootDockPanel.add(this.messageLabel, DockPanel.SOUTH);
        
        RootPanel.get(CONTAINER_ID).add(this.rootDockPanel);
    }
    
    /**
     * 
     * @return
     */
    public Element getContainerElement()
    {
        return containerElement;
    }

    /**
     * 
     * @param e
     */
    private void logException(Throwable e)
    {
        if(e instanceof UmbrellaException)
        {
            logger.error(e.getMessage());
            for( StackTraceElement element : e.getStackTrace() )
                logger.error("        " + element.getClassName() + "." + element.getMethodName() + element.getLineNumber());
            for( Throwable cause : ((UmbrellaException)e).getCauses() )
            {
                logger.error("    " + cause.getMessage());
                for( StackTraceElement element : cause.getStackTrace() )
                    logger.error("        " + element.getClassName() + "." + element.getMethodName() + element.getLineNumber());
            }
        }
        else
        {
            logger.error(e.getMessage());
            for( StackTraceElement element : e.getStackTrace() )
                logger.error("        " + element.getClassName() + "." + element.getMethodName() + element.getLineNumber());
        }
    }

    /**
     * 
     * @return
     * @throws BrowserInsufficientException
     */
    private Canvas createImageCanvas() 
    throws BrowserInsufficientException
    {
        Canvas canvas = Canvas.createIfSupported();
        if(canvas == null)
            throw new BrowserInsufficientException();
        
        canvas.getElement().setId(IMAGE_CANVAS_ELEMENT_ID);
        canvas.addMouseDownHandler( new MouseDownHandler()
        {
            public void onMouseDown(MouseDownEvent event)
            {
                if(NativeEvent.BUTTON_RIGHT == event.getNativeButton())
                    windowLevelingBegin(event);
            }
        });
        canvas.addMouseMoveHandler( new MouseMoveHandler()
        {
            public void onMouseMove(MouseMoveEvent event)
            {
                if(isInLevelingEvent())
                    windowLevel(event);
            }
        });
        canvas.addMouseUpHandler(new MouseUpHandler()
        {
            public void onMouseUp(MouseUpEvent event)
            {
                if(isInLevelingEvent() && NativeEvent.BUTTON_RIGHT == event.getNativeButton())
                {
                    windowLevelingEnd();
                    event.stopPropagation();
                }
            }
        });
        canvas.addMouseOutHandler(new MouseOutHandler()
        {
            @Override
            public void onMouseOut(MouseOutEvent event)
            {
                if(isInLevelingEvent() && NativeEvent.BUTTON_RIGHT == event.getNativeButton())
                {
                    windowLevelingEnd();
                }
            }
        });
        
        return canvas;
    }

    public String getICON_PACKAGE()
    {
        return ICON_PACKAGE;
    }

    public static void showCrosshairCursor() {
        DOM.setStyleAttribute(RootPanel.getBodyElement(), "cursor", "crosshair");
    }
     
    public static void showDefaultCursor() {
        DOM.setStyleAttribute(RootPanel.getBodyElement(), "cursor", "default");
    }
    
    /**
     * Set the size of the image canvas to maintain the aspect ratio.
     * This may have to get more sophisticated someday.
     * @param width
     * @param height
     */
    private void setImageCanvasSize(double imageWidth, double imageHeight)
    {
        double definedWidth = (double)imageCanvasContainingElement.getClientWidth();
        double definedHeight = (double)imageCanvasContainingElement.getClientHeight();
        
        double aspectRatio = imageWidth / imageHeight;
        double proposedHeight = definedWidth / aspectRatio;

        // if the height comes out as less than available, 
        if(proposedHeight <= definedHeight)
        {
            imageCanvas.setHeight( proposedHeight + "px" );
            backBufferCanvas.setHeight( proposedHeight + "px" );
        }
        else
        {
            double proposedWidth = definedHeight / aspectRatio;
            imageCanvas.setWidth(proposedWidth + "px");
            backBufferCanvas.setWidth(proposedWidth + "px");
        }
        
    }

    
    // =============================================================================================
    // Image Retrieval and Display
    // ==============================================================================================
    private int[] rawPixelData = new int[0];
    private double rawPixelDataWidth = 0.0;
    private double rawPixelDataHeight = 0.0;
    private LinearWindowLevel windowLevel = null;
    private AffineTransform affineTransform = null;
    private PixelTransform pixelTransform = null;
    private boolean inhibitRedraw = false;

    private boolean inLevelingEvent = false;
    private int levelingPreviousX = 0;
    private int levelingPreviousY = 0;
    
    private int levelSensitivity = 100;
    private int windowSensitivity = 100;
    
    private boolean imageLoaded = false;
    
    public boolean isImageLoaded()
    {
        return imageLoaded;
    }

    private boolean isInLevelingEvent(){return this.inLevelingEvent;}
    
    private void windowLevelingBegin(MouseDownEvent event)
    {
        showCrosshairCursor();
        inLevelingEvent = true;
        levelingPreviousX = event.getX();
        levelingPreviousY = event.getY();
    }
    
    private void windowLevel(MouseMoveEvent event)
    {
        int levelDelta = levelingPreviousX - event.getX();        // level is controlled by the X axis
        int windowDelta = levelingPreviousY - event.getY();        // window is controlled by the Y axis

        levelDelta = levelSensitivity * levelDelta;
        windowDelta = windowSensitivity * windowDelta;
        
        levelingPreviousX = event.getX();
        levelingPreviousY = event.getY();
        
        windowLevel(levelDelta, windowDelta);
    }
    
    /**
     * To match VistARAD default behavior,
     * 1.) right-click and drag to window-level
     * 2.) window control is up/down, up increases window size
     * 3.) level is right/left, right increases level value
     * 
     * @param screenX
     * @param screenY
     */
    private void windowLevel(int levelDelta, int windowDelta)
    {
        double currentLevel = windowLevel.getLevel();
        double currentWindow = windowLevel.getWindow();
        double newLevel = currentLevel;
        double newWindow = currentWindow;
        
        newLevel += levelDelta;
        newWindow += windowDelta;

        windowLevel.setLevel(newLevel);
        windowLevel.setWindow(newWindow);
        
        addMessage( "Level " + windowLevel.getLevel() + ", Window " + windowLevel.getWindow());
        
        redraw();
    }
    
    private void windowLevelingEnd()
    {
        showDefaultCursor();
        inLevelingEvent = false;
    }
    
    public boolean isInhibitRedraw()
    {
        return inhibitRedraw;
    }

    public void setInhibitRedraw(boolean inhibitRedraw)
    {
        this.inhibitRedraw = inhibitRedraw;
    }

    public int[] getRawPixelData()
    {
        return rawPixelData;
    }

    public void setRawPixelData(int[] rawPixelData, double width, double height)
    {
        this.rawPixelData = rawPixelData;
        this.rawPixelDataWidth = width;
        this.rawPixelDataHeight = height;
        
        if(!isInhibitRedraw())
            redraw();
        
        this.imageLoaded = true;
    }
    
    public void clearRawPixelData()
    {
        this.rawPixelData = new int[0];
        this.rawPixelDataWidth = 0;
        this.rawPixelDataHeight = 0;
        
        if(!isInhibitRedraw())
            redraw();
        
        this.imageLoaded = false;
    }

    public double getRawPixelDataWidth()
    {
        return rawPixelDataWidth;
    }

    public double getRawPixelDataHeight()
    {
        return rawPixelDataHeight;
    }

    public LinearWindowLevel getWindowLevel()
    {
        return windowLevel;
    }

    public void setWindowLevel(LinearWindowLevel transform)
    {
        this.windowLevel = transform;
        
        if(!isInhibitRedraw())
            redraw();
    }

    private void redraw()
    {
        drawImage(
            getWindowLevel(), 
            getRawPixelData(), getRawPixelDataWidth(), getRawPixelDataHeight()
        );
    }

    /**
     * 
     * @param transform
     * @param pixelData
     * @param width
     * @param height
     * @param fillStyle
     */
    private void drawImage(
        LinearWindowLevel transform,
        int[] pixelData, double width, double height) 
    {
        //imageCanvas.setWidth( (int)width + "px" );
        //imageCanvas.setHeight( (int)height + "px" );
        setImageCanvasSize(width, height);
        this.imageCanvas.setCoordinateSpaceWidth( (int)width );
        this.imageCanvas.setCoordinateSpaceHeight( (int)height );
        this.backBufferCanvas.setCoordinateSpaceWidth( (int)width );
        this.backBufferCanvas.setCoordinateSpaceHeight( (int)height );

        ImageData imageData = backBufferContext.createImageData( (int)width, (int)height );
        CanvasPixelArray cpa = imageData.getData();
        transform.loadCanvasPixelArray(pixelData, width, height, cpa);
//        if(affineTransform != null)
//            affineTransform.execute(cpa, height, width, 8);
//        if(pixelTransform != null)
//            pixelTransform.execute(cpa, height, width, 8);
        this.backBufferContext.putImageData(imageData, 0, 0);
        this.context.drawImage(this.backBufferCanvas.getCanvasElement(), 0.0, 0.0);
        
//        ImageData imageData = context.createImageData( (int)width, (int)height );
//        CanvasPixelArray cpa = imageData.getData();
//        transform.loadCanvasPixelArray(pixelData, width, height, cpa);
//        this.context.putImageData(imageData, 0, 0);
    }

    private void revertImage()
    {
        getWindowLevel().getAffineTransform().revert();
        getWindowLevel().getPixelTransform().revert();
        redraw();
    }
    
    private void scaleImage(double magnification)
    {
        getWindowLevel().getAffineTransform().scale(magnification);
        redraw();
    }
    
    private void translateImage(double x, double y)
    {
        getWindowLevel().getAffineTransform().translate(x, y);
        redraw();
    }
    
    private void rotateImage(double theta)
    {
        getWindowLevel().getAffineTransform().rotate(theta);
        redraw();
    }
    
    private void flipImage()
    {
        getWindowLevel().getAffineTransform().flip();
        redraw();
    }
    
    private void mirrorImage()
    {
        getWindowLevel().getAffineTransform().mirror();
        redraw();
    }
    
    private void invertImage()
    {
        getWindowLevel().getPixelTransform().setInvert(! getWindowLevel().getPixelTransform().isInvert());
        redraw();
    }
    
    private void drawTestPattern()
    {
        int width = this.imageCanvas.getCoordinateSpaceWidth();
        int height = this.imageCanvas.getCoordinateSpaceHeight();
        
        CanvasGradient gradient = this.context.createLinearGradient(0, 0, width-1, 1);
        gradient.addColorStop(0.0, "black");
        gradient.addColorStop(1.0, "red");
        this.context.setFillStyle(gradient);
        this.context.fillRect(0, 0, width, height/3);
        
        gradient = this.context.createLinearGradient(0, 0, width-1, 1);
        gradient.addColorStop(0.0, "black");
        gradient.addColorStop(1.0, "green");
        this.context.setFillStyle(gradient);
        this.context.fillRect(0, height/3, width, height/3);
        
        gradient = this.context.createLinearGradient(0, 0, width-1, 1);
        gradient.addColorStop(0.0, "black");
        gradient.addColorStop(1.0, "blue");
        this.context.setFillStyle(gradient);
        this.context.fillRect(0, 2*(height/3), width, height/3);
        
    }
    
    // =============================================================================================
    // DICOM Data Set Retrieval
    // =============================================================================================

    /**
     * 
     */
    protected void loadDicomDataSet(String imageUrn)
    {
        logger.info("calling service.getDicomDataSet(" + imageUrn + ")");
        addMessage("Loading " + imageUrn);
        this.dicomDataSource.fetch(imageUrn);
        //service.getDataSet(imageName, getDicomImageCallback);
        logger.info("after calling service.getImage(" + imageUrn + ")");
    }

 
    private void addMessage(String messageText)
    {
        String currentMsg = this.messageLabel.getText();
        currentMsg = messageText + "\r\n" + currentMsg;
        if(currentMsg.length() > MESSAGE_BUFFER_SIZE)
            currentMsg = currentMsg.substring(0, MESSAGE_BUFFER_SIZE-1);
        this.messageLabel.setText(currentMsg);   
    }

    // =============================================================================================
    // Data Retrieval Notification
    // =============================================================================================
    
    @Override
    public void dataEvent(DataSourceEvent event)
    {
        if(DataSourceEvent.DataEventType.PIXELS_NEW_EVENT == event.getDataEventType())
        {
            int depth = dicomDataSource.getCurrentPixelDataDepth();
            int grayScale = (2 << depth) - 1;
            int height = dicomDataSource.getCurrentPixelDataHeight();
            int width = dicomDataSource.getCurrentPixelDataWidth();

            addMessage(width + "X" + height + "X" + depth + "(" + grayScale + ") image retrieved, decoding ...");

            setInhibitRedraw(true);
            setRawPixelData( ArrayConversion.byteArrayToIntegerArray(dicomDataSource.getCurrentPixelData()), width, height );
            ImageStatistics imageStats = LinearWindowLevel.analyze(width, height, grayScale, 255, getRawPixelData());
            PixelTransform pixelTransform = new PixelTransform(imageStats);
            AffineTransform affineTransform = new AffineTransform(imageStats);
            setWindowLevel( new LinearWindowLevel(imageStats, affineTransform, pixelTransform) );
            redraw();
            setInhibitRedraw(false);
            
            addMessage(width + "X" + height + "X" + depth + "(" + grayScale + ") image processed ...");
        }
    }
    
    // ==========================================================================================================
    // 
    // ==========================================================================================================
    class ShowBrowserInformationCommand implements Command
    {
        @Override
        public void execute()
        {
            // does not work in dev mode
            if(! isDevelopmentMode())
            {
                displayResolution = new DisplayResolution();
                String msg = 
                    "Width :"
                    + displayResolution.getScreenWidth() 
                    + "("
                    + displayResolution.getAvailableScreenWidth()
                    + ")\r\n"
                    + "Height: "
                    + displayResolution.getScreenHeight() 
                    + "("
                    + displayResolution.getAvailableScreenHeight()
                    + ")\r\n"
                    + "Pixel Depth :"
                    + displayResolution.getPixelDepth()
                    + "("
                    + displayResolution.getColorDepth()
                    + ")]";
                Window.alert(msg);
            }
        }
    }
    
    class LoadCommand implements Command
    {
        @Override
        public void execute()
        {
            String imageUrn = Window.prompt("DICOM Image ID", "brain_001.dcm");
            if(imageUrn != null)
            {
                // String imagePixelsUrl = "http://" + getDicomPixelsSourceHostAndContext() + imageUrn;
                // String imageDataSetUrl = "http://" + getDicomDataSetSourceHostAndContext() + imageId;
                // loadDicomPixels(imagePixelsUrl);
                dicomDataSource.fetch(imageUrn);
            }
        }
    }
    
    class RevertCommand implements Command
    {
        @Override
        public void execute()
        {
            revertImage();
        }
    }
    
    class ZoomInCommand implements Command
    {
        @Override
        public void execute()
        {
            scaleImage(2.0);
        }
    }
    
    class ZoomOutCommand implements Command
    {
        @Override
        public void execute()
        {
            scaleImage(.5);
        }
    }
    
    class PanCommand implements Command
    {
        @Override
        public void execute()
        {
            
        }
    }
    
    class InvertCommand implements Command
    {
        @Override
        public void execute()
        {
            invertImage();
        }
    }
    
    class FlipHorizontalCommand implements Command
    {
        @Override
        public void execute()
        {
            mirrorImage();
        }
    }
   
    class FlipVerticalCommand implements Command
    {
        @Override
        public void execute()
        {
            flipImage();
        }
    }
    
    private double NINETY_DEGREES_CW = (Math.PI / 2.0);
    private double NINETY_DEGREES_CCW = -(Math.PI / 2.0);
    
    class RotateClockwiseCommand implements Command
    {
        @Override
        public void execute()
        {
            rotateImage(NINETY_DEGREES_CW);
        }
    }
    
    class RotateCounterClockwiseCommand implements Command
    {
        @Override
        public void execute()
        {
            rotateImage(NINETY_DEGREES_CCW);
        }
    }
    
    class DrawTestPatternCommand implements Command
    {
        @Override
        public void execute()
        {
            drawTestPattern();
        }
    }
}
