package gov.va.med.imaging.dx.translator;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
//import java.time.format.DateTimeFormatter;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.GlobalArtifactIdentifier;
import gov.va.med.GlobalArtifactIdentifierFactory;
import gov.va.med.MediaType;
import gov.va.med.OID;
import gov.va.med.PatientArtifactIdentifierImpl;
import gov.va.med.WellKnownOID;
import gov.va.med.exceptions.GlobalArtifactIdentifierFormatException;
import gov.va.med.imaging.DocumentURN;
import gov.va.med.imaging.ImageURN;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.dx.DesPollerData;
import gov.va.med.imaging.dx.DesPollerResult;
import gov.va.med.imaging.exceptions.OIDFormatException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.ArtifactResultError;
import gov.va.med.imaging.exchange.business.Image;
import gov.va.med.imaging.exchange.business.Series;
import gov.va.med.imaging.exchange.business.Study;
import gov.va.med.imaging.exchange.business.documents.Document;
import gov.va.med.imaging.exchange.business.documents.DocumentSet;
import gov.va.med.imaging.exchange.business.documents.DocumentSetResult;
import gov.va.med.imaging.exchange.enums.ArtifactResultStatus;
import gov.va.med.imaging.exchange.enums.VistaImageType;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;

public class DxDesTranslator {
	public static Logger logger = LogManager.getLogger(DxDesTranslator.class);

	public static DocumentSetResult translate(DesPollerResult pollerResult) {
		TransactionContext transactionContext = TransactionContextFactory.get();
		
		SortedSet<DocumentSet> documentSets = new TreeSet<DocumentSet>();
		if (pollerResult == null)
		{
			return DocumentSetResult.createFullResult(documentSets);
		}		

		DocumentSet documentSet = null;
		String patientIcn = transactionContext.getPatientID();
		logger.debug("patientIcn: " + patientIcn);
		
		for (DesPollerData data : pollerResult.getDesPollerDataList())
		{
			String homeCommunityId = data.getHomeCommunityId();
			logger.debug("homeCommunityId: " + homeCommunityId);
			
			String repositoryId = data.getRepositoryId();
			logger.debug("repositoryId: " + repositoryId);

			String documentId = data.getDocumentId();
			logger.debug("documentId: " + documentId);

			//Document document = translate(eot, patientIcn, resolvedArtifactSource);
			Document document = Document.create(
					data.getRecordId(), 
					getGlobalArtifactIdentifier(patientIcn, homeCommunityId, repositoryId, documentId)
				); 
			logger.debug("document created");

			Date creationDate = null;
			if (!data.getEnteredDate().equals(""))
			{
				logger.debug("converting creationDate: " + data.getEnteredDate());
				creationDate = getCreationDate(data.getEnteredDate());
			}
			document.setCreationDate(creationDate);
			logger.debug("creationDate: " + creationDate);
			
			document.setClinicalType(data.getClinicalType());
			logger.debug("clinicalType: " + data.getClinicalType());
			
			document.setMediaType(getMediaType(data.getContentType()));
			logger.debug("MediaType: " + data.getContentType());

			document.setContentLength(data.getSize());
			logger.debug("Content size: " + data.getSize());

			//document.setLanguageCode(languageCode);
			//document.setConfidentialityCode(confidentialityCode);
			document.setName(data.getTitle());
			logger.debug("name/title: " + data.getTitle());

			document.setDescription(data.getDescription() + " [DX: " + data.getContent() + "]");
			logger.debug("descr: " + data.getDescription());
			
			OID oid;
			try {
				oid = OID.create(data.getOriginId());
				logger.debug("origin: " + oid);
			} catch (OIDFormatException e) {
				oid = WellKnownOID.LOINC.getCanonicalValue();
			}
			
			if (documentSet == null)
			{
				documentSet = new DocumentSet(
						oid,
						repositoryId, 
						homeCommunityId);
				logger.debug("documentSet created");

				
				//documentSet = new DocumentSet(
				//	resolvedArtifactSource.getArtifactSource().getHomeCommunityId(),
				//	resolvedArtifactSource.getArtifactSource().getRepositoryId(),
				//	document.getDocumentSetIen());
			}
			
			documentSet.setPatientIcn(patientIcn);
			documentSet.add(document);
		
				
		}
		
		if (documentSet != null)
		{
			documentSets.add( documentSet );
		}

		return DocumentSetResult.createFullResult(documentSets);
	}

	private static Date getCreationDate(String enteredDate) {
		
		//String[] eDate = enteredDate.split("T");
		String dateTimeFormat = "yyyy-MM-dd'T'HH:mm:ss";

		try
		{
			SimpleDateFormat df = new SimpleDateFormat(dateTimeFormat);
			Date creationDate = df.parse(enteredDate);
			return creationDate;
		}
		catch (ParseException e)
		{
			logger.error("Date Parse Exception Error:" + e.getMessage());
			return null;
		}
	}

	private static GlobalArtifactIdentifier getGlobalArtifactIdentifier(
			String patientIcn,
			String homeCommunityId,
			String repositoryId, 
			String documentReferenceIdentifier)
	{
		GlobalArtifactIdentifier documentGlobalIdentifier = null;
		try
		{
			documentGlobalIdentifier = GlobalArtifactIdentifierFactory.create(
					homeCommunityId, 
					repositoryId, 
					documentReferenceIdentifier);
			// a PatientArtifactIdentifier includes the patient ID as an extra identifier
			documentGlobalIdentifier = PatientArtifactIdentifierImpl.create(documentGlobalIdentifier, patientIcn);
		}
		catch (GlobalArtifactIdentifierFormatException x)
		{
			logger.error(x);
			//throw new MethodException("Fatal error in translation, unable to build a global artifact ID", x);
		}
		catch (URNFormatException urnfX)
		{
			logger.error(urnfX);
			//throw new TranslationException("Fatal error in translation, unable to build a patient artifact ID", urnfX);
		}
		return documentGlobalIdentifier;
	}

	private static MediaType getMediaType(String contentType)
	{
		
		if (contentType.equals("text/plain"))
		{
			return MediaType.TEXT_PLAIN;
		}
		else if (contentType.equals("text/html"))
		{
			return MediaType.TEXT_HTML;
		}
		else if (contentType.equals("text/xml"))
		{
			return MediaType.TEXT_XML;
		}
		else if (contentType.equals("text/rtf"))
		{
			return MediaType.TEXT_RTF;
		}

		else if (contentType.equals("text/css"))
		{
			return MediaType.TEXT_CSS;
		}
		else if (contentType.equals("text/csv"))
		{
			return MediaType.TEXT_CSV;
		}
		else if (contentType.equals("text/rtf"))
		{
			return MediaType.TEXT_RTF;
		}
		else if (contentType.equals("text/enriched"))
		{
			return MediaType.TEXT_ENRICHED;
		}
		else if (contentType.equals("text/tab-separated-values"))
		{
			return MediaType.TEXT_TSV;
		}
		else if (contentType.equals("text/uri-list"))
		{
			return MediaType.TEXT_URI_LIST;
		}
		else if (contentType.equals("text/xml-external-parsed-entity"))
		{
			return MediaType.TEXT_XML_EXTERNAL_PARSED_ENTITY;
		}
		else if (contentType.equals("application/rtf"))
		{
			return MediaType.APPLICATION_RTF;
		}
		else if (contentType.equals("application/pdf"))
		{
			return MediaType.APPLICATION_PDF;
		}
		else if (contentType.equals("application/msword"))
		{
			return MediaType.APPLICATION_DOC;
		}
		else if (contentType.equals("application/dicom"))
		{
			return MediaType.APPLICATION_DICOM;
		}
		else if (contentType.equals("application/vnd.ms-excel"))
		{
			return MediaType.APPLICATION_EXCEL;
		}
		else if (contentType.equals("application/vnd.ms-powerpoint"))
		{
			return MediaType.APPLICATION_PPT;
		}
		else if (contentType.equals("application/vnd.openxmlformats-officedocument.wordprocessingml.document"))
		{
			return MediaType.APPLICATION_DOCX;
		}
		else if (contentType.equals("image/tiff"))
		{
			return MediaType.IMAGE_TIFF;
		}
		else if (contentType.equals("image/jpeg"))
		{
			return MediaType.IMAGE_JPEG;
		}
		else if (contentType.equals("image/png"))
		{
			return MediaType.IMAGE_PNG;
		}
		else if (contentType.equals("image/bmp"))
		{
			return MediaType.IMAGE_BMP;
		}
		else if (contentType.equals("image/x-bmp"))
		{
			return MediaType.IMAGE_XBMP;
		}
		else if (contentType.equals("image/j2k"))
		{
			return MediaType.IMAGE_J2K;
		}
		else if (contentType.equals("image/jp2"))
		{
			return MediaType.IMAGE_JP2;
		}
		else if (contentType.equals("image/x-targa"))
		{
			return MediaType.IMAGE_TGA;
		}
		else if (contentType.equals("audio/x-wav"))
		{
			return MediaType.AUDIO_WAV;
		}
		else if (contentType.equals("audio/mpeg"))
		{
			return MediaType.AUDIO_MPEG;
		}
		else if (contentType.equals("audio/mp4"))
		{
			return MediaType.AUDIO_MP4;
		}
		else if (contentType.equals("video/BMPEG"))
		{
			return MediaType.VIDEO_BMPEG;
		}
		else if (contentType.equals("video/BMPEG"))
		{
			return MediaType.VIDEO_BMPEG;
		}
		else if (contentType.equals("video/jpeg2000"))
		{
			return MediaType.VIDEO_JPEG2000;
		}
		else if (contentType.equals("video/JPEG"))
		{
			return MediaType.VIDEO_JPEG;
		}
		else if (contentType.equals("video/mp4"))
		{
			return MediaType.VIDEO_MP4;
		}
		else if (contentType.equals("video/mpeg4-generic"))
		{
			return MediaType.VIDEO_MPEG4_GENERIC;
		}
		else if (contentType.equals("video/MPEG"))
		{
			return MediaType.VIDEO_MPEG;
		}
		else if (contentType.equals("video/ogg"))
		{
			return MediaType.VIDEO_OGG;
		}
		else if (contentType.equals("video/quicktime"))
		{
			return MediaType.VIDEO_QUICKTIME;
		}
		else if (contentType.equals("video/x-msvideo"))
		{
			return MediaType.VIDEO_AVI;
		}
		else if (contentType.equals("multipart/form-data"))
		{
			return MediaType.MULTIPART_FORM_DATA;
		}
		else if (contentType.equals("multipart/mixed"))
		{
			return MediaType.MULTIPART_MIXED;
		}
		else
		{
			return MediaType.APPLICATION_OCTETSTREAM;
		}
	}

}
