package gov.va.med.imaging.dicom.parser.io;

import gov.va.med.imaging.dicom.dataset.DataSet;
import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.parser.impl.DataElementFactory;
import gov.va.med.imaging.dicom.parser.impl.DataElementFactoryImpl;

import java.io.IOException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * 
 * @author       BECKEC
 *
 */
public class Part10DataSetReader
{
	// The list of element tags that may occur in a Part 10 header
	// The occurrence of any other element tag will end the reading of the
	// Part 10 header
	public final static DataElementTag[] permissiblePart10HeaderElementTags = new DataElementTag[]
    {
		new DataElementTag(0x0002, 0x0000),		// Group Length
		new DataElementTag(0x0002, 0x0001),		// File Meta Information Version
		new DataElementTag(0x0002, 0x0002),		// Media Storage SOP Class UID
		new DataElementTag(0x0002, 0x0003),		// Media Storage SOP Instance UID
		new DataElementTag(0x0002, 0x0010),		// Transfer Syntax UID
		new DataElementTag(0x0002, 0x0012),		// Implementation Class UID
		new DataElementTag(0x0002, 0x0013),		// Implementation Class UID
		new DataElementTag(0x0002, 0x0016),		// Source Application Entity Title
		new DataElementTag(0x0002, 0x0100),		// Private Information Creator UID
		new DataElementTag(0x0002, 0x0102)		// Private Information
    };
	
	private static boolean isPart10HeaderElementTag(DataElementTag elementTag)
	{
		for(DataElementTag headerTag : permissiblePart10HeaderElementTags)
			if(headerTag.equals(elementTag))
				return true;
		return false;
	}
	
	private final DataElementReader reader;
	private Logger logger = LogManager.getLogger(this.getClass().getName());
	
	/**
	 * 
	 * @param inputStream
	 * @param dataElementFactory
	 */
	public Part10DataSetReader(DataElementLimitedInputStream inputStream)
	{
		logger.trace("Part10DataSetReader instantiating... ");
		DataElementFactory headerElementFactory = DataElementFactoryImpl.getDefaultHeaderElementFactory();
		this.reader = new DataElementReader(inputStream, headerElementFactory);
		logger.trace("Part10DataSetReader instantiated. ");
	}

	/**
	 * Read a DataSet, starting from the current position and stopping when an
	 * element tag not in the Part10 list is encountered.  The stream is left at the
	 * start of the first non-Part10 element.
	 * @return
	 * @throws IOException 
	 * @throws UnsupportedOperationException 
	 * @throws InvalidVRException 
	 * @throws InvalidVRModeException 
	 * @throws DicomFormatException 
	 */
	public DataSet readPart10DataSet() 
	throws 
		UnsupportedOperationException, IOException, 
		InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.traceEntry("readPart10DataSet");
		DataSet dataSet = new DataSet(reader.getDataElementFactory().getTransferSyntaxUid());
		
		for( DataElementTag dataElementTag = reader.getInputStream().peekNextDataElementTag();
			dataElementTag != null && isPart10HeaderElementTag(dataElementTag);
			dataElementTag = reader.getInputStream().peekNextDataElementTag() )
		{
			DataElement<?> dataElement = reader.readNextDataElement();
            //RawValueParser<?,?> rvp = rawValueParserFactory.createRawValueParser(dataElement);
            //rvp.parseRawValue();
            
			dataSet.add(dataElement);
			
			logger.info("Added data element " + dataElement.getDataElementTag().toString() + ", " +
					dataElement.getValueRepresentation()[0].name() + " to data set.");
		}
		
		logger.traceExit("readPart10DataSet");
		return dataSet;
	}
}
