/**
 * 
 */
package gov.va.med.imaging.dicom.parser.impl;

import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationValueLengthExceededException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationValueLengthInsufficientException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author       BECKEC
 *
 */
public abstract class RawValueParser<T, E extends DataElement<T>>
{
    private Logger logger = LogManager.getLogger(RawValueParser.class.getName());
    private final DataElementFactory factory;
    private final E dataElement;
    
    public RawValueParser(DataElementFactory factory, E dataElement)
    {
        this.factory = factory;
        this.dataElement = dataElement;
    }
    
    public DataElementFactory getFactory()
    {
        return factory;
    }

    public E getDataElement()
    {
        return dataElement;
    }

    protected Logger getLogger()
    {
        return logger;
    }
    
    /**
     * Parse the raw value (byte array) and setValue() with the result.
     * 
     * @param element
     * @throws ValueRepresentationInterpretationException
     */
    public abstract void parseRawValue() 
    throws DicomFormatException;


    /**
     * Do basic validation of the length field and the length of the value against the
     * specified values.
     *  
     * @throws ValueRepresentationInterpretationException
     */
    protected void validateLengthValues(DataElement<?> dataElement) 
    throws ValueRepresentationInterpretationException
    {
        if(dataElement.getValueLength() < dataElement.getValueRepresentation()[0].getMinLengthOfValue() &&
                !dataElement.getValueRepresentation()[0].isAllowUndefinedLength())
            throw new ValueRepresentationValueLengthInsufficientException(
                "Element '" + dataElement.getDataElementTag() + "' " + dataElement.getValueRepresentation()[0].toString() + 
                " length of " + dataElement.getValueLength() + " is outside range of [" + 
                dataElement.getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
                dataElement.getValueRepresentation()[0].getMaxLengthOfValue() + "].");
        
        if(dataElement.getValueLength() > dataElement.getValueRepresentation()[0].getMaxLengthOfValue() &&
                !dataElement.getValueRepresentation()[0].isAllowUndefinedLength())
            throw new ValueRepresentationValueLengthExceededException(
                "Element '" + dataElement.getDataElementTag() + "' " + dataElement.getValueRepresentation()[0].toString() + 
                " length of " + dataElement.getValueLength() + " is outside range of [" + 
                dataElement.getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
                dataElement.getValueRepresentation()[0].getMaxLengthOfValue() + "].");
        
        if(dataElement.getRawValue() == null)
            return;
        
        if(dataElement.getRawValue().length < dataElement.getValueRepresentation()[0].getMinLengthOfValue() &&
                !dataElement.getValueRepresentation()[0].isAllowUndefinedLength())
            throw new ValueRepresentationValueLengthInsufficientException(
                "Element '" + dataElement.getDataElementTag() + "' " + dataElement.getValueRepresentation()[0].toString() + 
                " raw value length of " + dataElement.getRawValue().length + " is outside range of [" + 
                dataElement.getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
                dataElement.getValueRepresentation()[0].getMaxLengthOfValue() + "].");
        
        if(dataElement.getValueLength() > dataElement.getRawValue().length &&
                !dataElement.getValueRepresentation()[0].isAllowUndefinedLength())
            throw new ValueRepresentationValueLengthExceededException(
                "Element '" + dataElement.getDataElementTag() + "' " + dataElement.getValueRepresentation()[0].toString() + 
                " raw value length of " + dataElement.getRawValue().length + " is outside range of [" + 
                dataElement.getValueRepresentation()[0].getMinLengthOfValue() + "-" + 
                dataElement.getValueRepresentation()[0].getMaxLengthOfValue() + "].");
    }
    
    // ========================================================================================
    // Helper Methods to do various, usually bit or byte level, manipulations
    // ========================================================================================
    protected static String transformValueToStringRemoveNulls(byte[] rawValue, long valueLength)
    {
        for (int j = 0; j < valueLength; j++)
            if (rawValue[j] == 0) rawValue[j] = 20;
        return new String(rawValue);
    }
    
    protected static String getRawValueCharacterRepresentation(DataElement<?> dataElement)
    {
        StringBuffer sb = new StringBuffer();
        
        byte[] raw = dataElement.getRawValue();
        if(raw != null)
            for(byte c : dataElement.getRawValue())
                sb.append("0x" + Integer.toHexString(c) + " ");
        return sb.toString();
    }
   
}
