/**
 * 
 */
package gov.va.med.imaging.dicom.dataset;


import gov.va.med.imaging.dicom.dataset.elements.DataElement;
import gov.va.med.imaging.dicom.dataset.elements.DataElementTag;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;

import java.io.Serializable;
import java.util.Collection;
import java.util.Comparator;
import java.util.Date;
import java.util.Iterator;
import java.util.SortedMap;
import java.util.SortedSet;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.gwt.user.client.rpc.IsSerializable;

/**
 * This class is a Set of all of the DataElement instances in a single DICOM Part 10 file.
 * The DataElement instances are stored in a Map, using the DataElementTag (group and element numbers) as
 * the key to allow rapid access by that attribute.  Logically (without regard to access times) this
 * is a Set of DataElement instances.
 * 
 * From the spec:
 * A Data Element is uniquely identified by a Data Element Tag. 
 * The Data Elements in a Data Set shall be ordered by increasing Data Element Tag Number and shall 
 * occur at most once in a Data Set.
 * 
 * @author       BECKEC
 *
 */
public class DataSet
implements Serializable, SortedSet<DataElement<?>>, IsSerializable
{
	private static final long serialVersionUID = 1L;
	protected SortedMap<DataElementTag, DataElement<?>> wrappedMap = new TreeMap<DataElementTag, DataElement<?>>();
	private TransferSyntaxUid transferSyntax;
	private final transient Logger logger = LogManager.getLogger(this.getClass().getName());
	
	/**
	 * Required no-arg constructor
	 */
	public DataSet()
    {
        super();
    }

    public DataSet(TransferSyntaxUid transferSyntax)
	{
		this.transferSyntax = transferSyntax;
	}
	
	/**
     * @return the transferSyntax
     */
    public TransferSyntaxUid getTransferSyntax()
    {
    	return transferSyntax;
    }

    // =======================================================================================================
    //
    // =======================================================================================================
	/**
	 * 
	 * @param groupNumber
	 * @param elementNumber
	 * @return
	 */
	public DataElement<?> get(int groupNumber, int elementNumber)
	{
		return wrappedMap.get(new DataElementTag(groupNumber, elementNumber));
	}
	
	public DataElement<?> get(DataElementTag dataElementTag)
	{
		return wrappedMap.get(dataElementTag);
	}
	
	// ==================================================================================
	// Set implementation
	// ==================================================================================
	/**
     * @see java.util.Set#add(java.lang.Object)
     */
    @Override
    public boolean add(DataElement<?> dataElement)
    {
    	if( wrappedMap.get(dataElement.getDataElementTag()) != null )
    		return false;
    	
    	wrappedMap.put(dataElement.getDataElementTag(), dataElement);
    	return true;
    }

	/**
     * @see java.util.Set#addAll(java.util.Collection)
     */
    @Override
    public boolean addAll(Collection<? extends DataElement<?>> dataElements)
    {
    	boolean thisSetChanged = false;
    	for(DataElement<?> dataElement : dataElements)
    		thisSetChanged |= add(dataElement);		// if any element was added then the set was changed
    	
	    return thisSetChanged;
    }

	/**
     * @see java.util.Set#contains(java.lang.Object)
     */
    @Override
    public boolean contains(Object o)
    {
    	if(o instanceof DataElement<?>)
    		return wrappedMap.get( ((DataElement<?>)o).getDataElementTag() ) != null;
    	
	    return false;
    }

	/**
     * @see java.util.Set#containsAll(java.util.Collection)
     */
    @Override
    public boolean containsAll(Collection<?> c)
    {
    	for(Object o : c)
    		if(! contains(o))
    			return false;
    	return true;
    }

	/**
     * @see java.util.Set#iterator()
     */
    @Override
    public Iterator<DataElement<?>> iterator()
    {
	    return wrappedMap.values().iterator();
    }
    
    public Iterator<DataElementTag> keyIterator()
    {
        return wrappedMap.keySet().iterator();
    }

	/**
     * @see java.util.Set#removeAll(java.util.Collection)
     */
    @Override
    public boolean removeAll(Collection<?> c)
    {
    	boolean thisSetChanged = false;
    	
    	for(Object o : c)
    		if(o instanceof DataElement)
    			if( wrappedMap.values().contains(o) )
    				thisSetChanged |= (wrappedMap.remove( ((DataElement<?>)o).getDataElementTag() ) != null);
    			
	    return thisSetChanged;
    }

	/**
     * @see java.util.Set#retainAll(java.util.Collection)
     */
    @Override
    public boolean retainAll(Collection<?> c)
    {
    	boolean thisSetChanged = false;
    	
    	for(DataElement<?> dataElement : this)
    		if( ! c.contains(dataElement) )
    		{
    			remove(dataElement);
    			thisSetChanged = true;
    		}
    	
    	return thisSetChanged;
    }

	/**
     * @see java.util.Set#toArray()
     */
    @Override
    public Object[] toArray()
    {
    	return wrappedMap.values().toArray(new DataElement<?>[wrappedMap.size()]);
    }

	/**
     * @see java.util.Set#toArray(T[])
     */
    @Override
    public <T> T[] toArray(T[] a)
    {
    	return wrappedMap.values().toArray(a);
    }
	
	/**
     * @see java.util.Set#clear()
     */
    @Override
    public void clear()
    {
    	wrappedMap.clear();
    }

	/**
     * @see java.util.Set#isEmpty()
     */
    @Override
    public boolean isEmpty()
    {
	    return wrappedMap.isEmpty();
    }

	/**
     * @see java.util.Set#remove(java.lang.Object)
     */
    @Override
    public boolean remove(Object o)
    {
    	if(o instanceof DataElement<?>)
    		return wrappedMap.remove(((DataElement<?>)o).getDataElementTag()) != null;
    	
    	return false;
    }

	/**
     * @see java.util.Set#size()
     */
    @Override
    public int size()
    {
	    return wrappedMap.values().size();
    }
    
	/**
     * @see java.util.SortedSet#comparator()
     * @return null, because this always uses the natural ordering of DataElement 
     */
    @Override
    public Comparator<? super DataElement<?>> comparator()
    {
	    return null;
    }

	/**
     * @see java.util.SortedSet#first()
     */
    @Override
    public DataElement<?> first()
    {
    	return wrappedMap.get(wrappedMap.firstKey());
    }

	/**
     * @see java.util.SortedSet#headSet(java.lang.Object)
     */
    @Override
    public SortedSet<DataElement<?>> headSet(DataElement<?> toElement)
    {
    	return new TreeSet<DataElement<?>>( wrappedMap.headMap(toElement.getDataElementTag()).values() );
    }

	/**
     * @see java.util.SortedSet#last()
     */
    @Override
    public DataElement<?> last()
    {
    	return wrappedMap.get(wrappedMap.lastKey());
    }

	/**
     * @see java.util.SortedSet#subSet(java.lang.Object, java.lang.Object)
     */
    @Override
    public SortedSet<DataElement<?>> subSet(DataElement<?> fromElement, DataElement<?> toElement)
    {
    	return new TreeSet<DataElement<?>>( wrappedMap.subMap(fromElement.getDataElementTag(), toElement.getDataElementTag()).values() );
    }

	/**
     * @see java.util.SortedSet#tailSet(java.lang.Object)
     */
    @Override
    public SortedSet<DataElement<?>> tailSet(DataElement<?> fromElement)
    {
    	return new TreeSet<DataElement<?>>( wrappedMap.tailMap(fromElement.getDataElementTag()).values() );
    }

    /**
     * Get an iterator over all of the elements in this with the specified group number
     * @param groupNumber
     */
	public Iterator<DataElement<?>> iterator(int groupNumber)
    {
		return new GroupIterator(groupNumber);
    }
	
	class GroupIterator
	implements Iterator<DataElement<?>>
	{
		final private int groupNumber;
		private DataElement<?> nextDataElement = null;
		private final Iterator<DataElement<?>> wrappedIterator;
		
		GroupIterator(int groupNumber)
		{
			this.groupNumber = groupNumber;
			wrappedIterator = iterator();
		}
		
		@Override
        public boolean hasNext()
        {
			position();
			return nextDataElement != null;
        }

		@Override
        public DataElement<?> next()
        {
			position();
			if(nextDataElement != null)
			{
				DataElement<?> tmp = nextDataElement;
				nextDataElement = null;
				return tmp;
			}
            return null;
        }

		private synchronized void position()
        {
        	while(wrappedIterator != null && wrappedIterator.hasNext() && nextDataElement == null)
        	{
        		DataElement<?> nextTmp = wrappedIterator.next();
        		if(nextTmp.getDataElementTag().getGroup() == groupNumber)
        			nextDataElement = nextTmp;
        	}
        }

		@Override
        public void remove()
        {
        }
	};

	public Collection<DataElement<?>> subset(int groupNumber)
	{
		return 
			this.wrappedMap.subMap(new DataElementTag(groupNumber, 0x0000), new DataElementTag(groupNumber, 0xFFFF)).values();
	}
	
	// ==================================================================================
	// Convenience Methods, to get some well-known DICOM tag values
	// ==================================================================================
	public DataElement<?> getCommandGroupLength()
	{
		return get( 0x0000, 0x0000 );
	}
	
	// ================================================================================================
	// DICOM Header Elements
	// ================================================================================================
	/**
	 * Number of bytes following this File Meta Element (end of the Value field) up to and including the last 
	 * File Meta Element of the Group 2 File Meta Information
	 * 
	 * @return
	 */
	public DataElement<?> getGroupLength()
	{
		return get(0x0002,0x0000);
	}
	

	/**
	 * 	This is a two byte field where each bit identifies a version of this File Meta Information header. 
	 * In version 1 the first byte value is 00H and the second value byte value is 01H.
	 * Implementations reading Files with Meta Information where this attribute has bit 0 (lsb) of the second byte 
	 * set to 1 may interpret the File Meta Information as specified in this version of PS 3.10. All other bits 
	 * shall not be checked.
	 * Note: A bit field where each bit identifies a version, allows explicit indication of the support of 
	 * multiple previous versions. Future versions of the File Meta Information that can be read by verson 1 
	 * readers will have bit 0 of the second byte set to 1.
	 * 
	 * @return
	 */
	public DataElement<?> getFileMetaInformationVersion()
	{
		return get(0x0002,0x0001);
	}
	
	/**
	 * Uniquely identifies the SOP Class associated with the Data Set. 
	 * SOP Class UIDs allowed for media storage are specified in PS 3.4 of the DICOM Standard - Media Storage 
	 * Application Profiles.
	 * 
	 * @return
	 */
	public DataElement<?> getMediaStorageSOPClassUID()
	{
		return get(0x0002,0x0002);
	}
	
	/**
	 * Uniquely identifies the SOP Instance associated with the Data Set placed in the file and following the 
	 * File Meta Information.
	 * 
	 * @return
	 */
	public DataElement<?> getMediaStorageSOPInstanceUID()
	{
		return get(0x0002,0x0003);
	}
	
	/**
	 * Uniquely identifies the Transfer Syntax used to encode the following Data Set. This Transfer Syntax does not 
	 * apply to the File Meta Information.
	 * Note: It is recommended to use one of the DICOM Transfer Syntaxes supporting explicit Value Representation 
	 * encoding to facilitate interpretation of File Meta Element Values. JPIP Referenced Pixel Data Transfer 
	 * Syntaxes are not used. (See PS 3.5 of the DICOM Standard).
	 * 
	 * @return
	 */
	public DataElement<?> getTransferSyntaxUID()
	{
		return get(DataElementTag.TRANSFER_SYNTAX_UID_TAG);
	}
	
	/**
	 * Uniquely identifies the implementation which wrote this file and its content. It provides an unambiguous 
	 * identification of the type of implementation which last wrote the file in the event of interchange problems. 
	 * It follows the same policies as defined by PS 3.7 of the DICOM Standard (association negotiation).
	 * 
	 * @return
	 */
	public DataElement<?> getImplementationClassUID()
	{
		return get(0x0002,0x0012);
	}
	
	/**
	 * Identifies a version for an Implementation Class UID (0002,0012) using up to 16 characters of the repertoire 
	 * identified in Section 8.5. It follows the same policies as defined by PS 3.7 of the DICOM Standard (association
	 * negotiation).
	 * 
	 * @return
	 */
	public DataElement<?> getImplementationVersionName()
	{
		return get(0x0002,0x0013);
	}

	/**
	 * The DICOM Application Entity (AE) Title of the AE which wrote this file's content (or last updated it). 
	 * If used, it allows the tracing of the source of errors in the event of media interchange problems. 
	 * The policies associated with AE Titles are the same as those defined in PS 3.8 of the DICOM Standard.
	 * 
	 * @return
	 */
	public DataElement<?> getSourceApplicationEntityTitle()
	{
		return get(0x0002,0x0016);
	}
	
	/**
	 * The UID of the creator of the private information (0x0002,0x0102)
	 * 
	 * @return
	 */
	public DataElement<?> getPrivateInformationCreatorUID()
	{
		return get(0x0002,0x0100);
	}
	
	/**
	 * Contains Private Information placed in the File Meta Information. 
	 * The creator shall be identified in (0002,0100). 
	 * Required if Private Information Creator UID (0002,0100) is present.
	 * 
	 * @return
	 */
	public DataElement<?> getPrivateInformation()
	{
		return get(0x0002,0x0102);
	}
	
	// ================================================================================================
	// 
	// ================================================================================================
	/**
	 * DICOM Application Entities (AEs) that extend or replace the default repertoire 
	 * convey this information in the Specific Character Set (0008,0005) Attribute.
	 * 
	 * @return
	 */
	public DataElement<?> getSpecificCharacterSet()
	{
		return get( 0x0008, 0x0005 );
	}
	
	// ================================================================================================
	// Patient Information Group (0x0010)
	// ================================================================================================
	public String getPatientId() 
	throws ValueRepresentationInterpretationException
	{
		return get( 0x0010, 0x0010 ).getValue().toString();
	}
	
	public String getPatientName()
	throws ValueRepresentationInterpretationException
	{
		return get( 0x0010, 0x0020 ).getValue().toString();
	}
	
	public Date getPatientBirthDate()
	throws ValueRepresentationInterpretationException
	{
		return (Date)( get( 0x0010, 0x0030 ).getValue());
	}
	
	public String getPatientSex()
	throws ValueRepresentationInterpretationException
	{
		return get( 0x0010, 0x0040 ).getValue().toString();
	}
	
	public String getPatientAge()
	throws ValueRepresentationInterpretationException
	{
		return get( 0x0010, 0x1010 ).getValue().toString();
	}
	
	// ================================================================================================
	// Image Information Group (0x0028)
	// ================================================================================================
	public DataElement<?> getSamplesPerPixel()
	{
		return get( DataElementTag.SAMPLES_PER_PIXEL );
	}

	public DataElement<?> getPhotometricInterpretation()
	{
		return get( DataElementTag.PHOTOMETRIC_INTERPRETATION );
	}
	
	public int getRows()
	throws ValueRepresentationInterpretationException
	{
		return getIntegerDataElement(DataElementTag.ROWS);
	}
	
	public int getColumns()
	throws ValueRepresentationInterpretationException
	{
		return getIntegerDataElement(DataElementTag.COLUMNS);
	}

	public int getBitsAllocated()
	throws ValueRepresentationInterpretationException
	{
		return getIntegerDataElement(DataElementTag.BITS_ALLOCATED);
	}
	
	public int getBitsStored()
	throws ValueRepresentationInterpretationException
	{
		return getIntegerDataElement(DataElementTag.BITS_STORED);
	}
	
	public int getHighBit()
	throws ValueRepresentationInterpretationException
	{
		return getIntegerDataElement(DataElementTag.HIGH_BIT);
	}

	private int getIntegerDataElement(DataElementTag dataElementTag)
	{
		try
		{
			Integer[] value = (Integer[])( get( dataElementTag ).getValue() );
			return value[0].intValue();
		}
		catch(ClassCastException ccX)
		{
			logger.error("Value returned from getting element '" + dataElementTag + "' is not of the expected type" );
			return 0;
		} 
		catch (ValueRepresentationInterpretationException e)
        {
			logger.error("Getting element '" + dataElementTag + "' resulted in '" + e.getMessage() + "'." );
			return 0;
        }
		
	}
	// ================================================================================================
	// 
	// ================================================================================================
	public boolean isEncapsulated()
    {
		return getTransferSyntax().isEncapsulated();
    }
	
	// ===========================================================================================
	// 
	// ===========================================================================================
	public DataElement<?> getItem()
	{
		return get( DataElementTag.SEQUENCE_ITEM_TAG );
	}
	
	public DataElement<?> getItemDelimitationItem()
	{
		return get( DataElementTag.ITEM_DELIMITATION_TAG );
	}
	
	public DataElement<?> getSequenceDelimitationItem()
	{
		return get( DataElementTag.SEQUENCE_DELIMITATION_TAG );
	}
}
