/**
 * 
 */
package gov.va.med.imaging.storage.cache.impl.memory;

import gov.va.med.imaging.storage.cache.*;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleEvent;
import gov.va.med.imaging.storage.cache.events.GroupLifecycleListener;
import gov.va.med.imaging.storage.cache.events.InstanceLifecycleEvent;
import gov.va.med.imaging.storage.cache.events.InstanceLifecycleListener;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheStateException;
import gov.va.med.imaging.storage.cache.exceptions.RegionDoesNotExistException;
import gov.va.med.imaging.storage.cache.exceptions.RegionInitializationException;
import gov.va.med.imaging.storage.cache.impl.memory.data.MemoryInstanceDataFactory;
import gov.va.med.imaging.storage.cache.impl.memory.memento.MemoryCacheMemento;
import gov.va.med.imaging.storage.cache.memento.EvictionStrategyMemento;
import gov.va.med.imaging.storage.cache.memento.RegionMemento;

import java.net.URI;
import java.util.*;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author       BECKEC
 *
 */
public class MemoryCache 
implements Cache
{
	public final static String protocol = "transient";
	private final URI locationUri;
	private final String name;
	private final EvictionTimer evictionTimer;
	private final Logger logger = LogManager.getLogger(this.getClass());
	
	private Boolean enabled = Boolean.TRUE;
	private Boolean initialized = Boolean.TRUE;
	private Collection<EvictionStrategy> evictionStrategies = new HashSet<EvictionStrategy>();
	private Collection<Region> regions = new HashSet<Region>();
	private Collection<CacheStructureChangeListener> cacheStructureChangeListeners = new ArrayList<CacheStructureChangeListener>();
	private final MemoryInstanceDataFactory instanceDataFactory;
	private final MemoryInstanceByteChannelFactory instanceByteChannelFactory;
	private final MemoryRegionFactory regionFactory;	
	private final GroupFactory groupFactory;
	private final InstanceFactory instanceFactory;
	
	public static MemoryCache create(String name, URI locationUri, EvictionTimer evictionTimer)
	{
		return new MemoryCache(name, locationUri, evictionTimer);
	}
	
	/**
	 * 
	 * @param name
	 * @param locationUri - a URI in the format 
	 *   memory://ram&maxSize=100000?maxInstanceCount=100
	 * @param evictionTimer
	 */
	private MemoryCache(String name, URI locationUri, EvictionTimer evictionTimer)
	{
		super();
		this.locationUri = locationUri;
		parseLocationUri();
		this.evictionTimer = evictionTimer;
		this.name = name;
		instanceDataFactory = new MemoryInstanceDataFactory(getMaxCacheSize());
		instanceByteChannelFactory = new MemoryInstanceByteChannelFactory(instanceDataFactory);
		
		LifecycleLogger lifecycleLogger = new LifecycleLogger();
		
		instanceFactory = new MemoryInstanceFactory(instanceByteChannelFactory, lifecycleLogger);
		groupFactory = new MemoryGroupFactory(lifecycleLogger, instanceFactory);
		regionFactory = new MemoryRegionFactory(getGroupFactory(), getInstanceFactory()); 
	}

	// ==============================================================================================
	// Immutable Properties
	// ==============================================================================================
	@Override
	public String getName()
	{
		return this.name;
	}

	/**
	 * @return the regionFactory
	 */
	public MemoryRegionFactory getRegionFactory()
	{
		return this.regionFactory;
	}

	/**
	 * @return the groupFactory
	 */
	public GroupFactory getGroupFactory()
	{
		return this.groupFactory;
	}

	/**
	 * @return the instanceDataFactory
	 */
	public MemoryInstanceDataFactory getInstanceDataFactory()
	{
		return this.instanceDataFactory;
	}

	/**
	 * @return the instanceFactory
	 */
	public InstanceFactory getInstanceFactory()
	{
		return this.instanceFactory;
	}
	
	/**
	 * The URI may be interpreted as follows:
	 * transient://<name>[? + [maxSize=<cache-size>] + [maxInstances=<instance-count>]]
	 * where: 
	 * <name> is the cache identifier
	 * <cache-size> is a positive integer specifying the maximum size of all cache members (in bytes)
	 * <instance-count> is a positive integer specifying the maximum number of instances in the cache
	 */
	private String locationPath;
	private String locationProtocol;
	private int maxInstanceCount = Integer.MAX_VALUE;
	private long maxCacheSize = Long.MAX_VALUE;
	
	private void parseLocationUri()
	{
		locationPath = getLocationUri().getPath();
		locationProtocol = getLocationUri().getAuthority();
		String query = getLocationUri().getQuery();
		if(query != null)
		{
			String[] queryParameters = query.split("&");
			for(String queryParameter : queryParameters)
			{
				String[] parsedQueryParameter = queryParameter.split("=");
				if(parsedQueryParameter.length == 2)
				{
					try
					{
						if("maxSize".equals(parsedQueryParameter[0]))
							maxCacheSize = Long.parseLong( parsedQueryParameter[1] );
						else if("maxInstanceCount".equals(parsedQueryParameter[0]))
							maxInstanceCount = Integer.parseInt( parsedQueryParameter[1] );
					} 
					catch (NumberFormatException x)
					{
						logger.warn("Error initializing parameter '" + parsedQueryParameter[0] + "', value must be parsable as a positive integer.", x);
					}
				}
			}
		}
	}
	@Override
	public URI getLocationUri()
	{
		return locationUri;
	}
	@Override
	public String getLocationPath()
	{
		return locationPath;
	}
	@Override
	public String getLocationProtocol()
	{
		return locationProtocol;
	}
	public long getMaxCacheSize()
	{
		return maxCacheSize;
	}
	public long getMaxInstanceCount()
	{
		return maxInstanceCount;
	}

	public long getFreeSpace()
	{
		return instanceDataFactory.getFreeSpace();
	}
	
	@Override
	public EvictionTimer getEvictionTimer()
	{
		return evictionTimer;
	}

	@Override
	public InstanceByteChannelFactory<?> getInstanceByteChannelFactory()
	{
		return instanceByteChannelFactory;
	}
	
	// ==============================================================================================
	// Mutable Properties
	// ==============================================================================================
	@Override
	public Boolean isEnabled()
	{
		return enabled;
	}
	
	@Override
	public void setEnabled(Boolean enabled) 
	throws CacheException
	{
		this.enabled = enabled;
	}


	@Override
	public Boolean isInitialized()
	{
		return initialized;
	}
	
	@Override
	public void setInitialized(Boolean initialized) 
	throws CacheException
	{
		this.initialized = initialized;
	}

	/**
	 * 
	 */
	@Override
	public MemoryCacheMemento createMemento()
	{
		MemoryCacheMemento memento = new MemoryCacheMemento();
		memento.setName(getName());
		memento.setByteChannelFactoryMemento( getInstanceByteChannelFactory().createMemento() );
		memento.setEnabled(isEnabled().booleanValue());
		memento.setInitialized(isInitialized().booleanValue());
		
		List<EvictionStrategyMemento> esMementoes = new ArrayList<EvictionStrategyMemento>();
		for(EvictionStrategy evictionStrategy : getEvictionStrategies())
			esMementoes.add( evictionStrategy.createMemento() );
		memento.setEvictionStrategyMementos(esMementoes);
		
		memento.setLocationUri(getLocationUri().toString());
		
		
		List<RegionMemento> regionMementoes = new ArrayList<RegionMemento>();
		for(Region region : getRegions())
			regionMementoes.add( region.createMemento() );
		memento.setRegionMementos(regionMementoes);
		
		return memento;
	}

	// =================================================================================================================================
	// Eviction Strategies
	// =================================================================================================================================
	@Override
	public void addEvictionStrategies(Collection<? extends EvictionStrategy> evictionStrategies) 
	throws CacheStateException
	{
		this.evictionStrategies.addAll(evictionStrategies);
	}

	@Override
	public void addEvictionStrategy(EvictionStrategy evictionStrategy) 
	throws CacheStateException
	{
		this.evictionStrategies.add(evictionStrategy);
	}

	@Override
	public Collection<? extends EvictionStrategy> getEvictionStrategies()
	{
		return evictionStrategies;
	}

	/**
	 * @see gov.va.med.imaging.storage.cache.Cache#getEvictionStrategy(java.lang.String)
	 */
	@Override
	public EvictionStrategy getEvictionStrategy(String evictionStrategyName)
	{
		for(EvictionStrategy evictionStrategy: getEvictionStrategies())
			if( evictionStrategyName.equals(evictionStrategy.getName()) )
				return evictionStrategy;
		
		return null;
	}

	/**
	 * Given an array of names, find the eviction strategies registered with this cache whose names match
	 * the given names.
	 * Duplicates names in the given array are ignored (i.e. only show up once in the result)
	 * Unknown names in the given array are ignored (i.e. do not show up in the result)
	 * 
	 * @param evictionStrategyNames
	 * @return
	 */
	public EvictionStrategy[] getEvictionStrategies(String[] evictionStrategyNames)
	{
		if(evictionStrategyNames == null)
			return null;
		
		// make it a Set so that duplicates are effectively ignored 
		Set<EvictionStrategy> matchedEvictionStrategies = new HashSet<EvictionStrategy>();
		
		for(String evictionStrategyName : evictionStrategyNames)
		{
			EvictionStrategy matchedEvictionStrategy = getEvictionStrategy(evictionStrategyName);
			if(matchedEvictionStrategy != null)
				matchedEvictionStrategies.add(matchedEvictionStrategy);
		}
		
		return matchedEvictionStrategies.size() > 0 ? 
			matchedEvictionStrategies.toArray(new EvictionStrategy[matchedEvictionStrategies.size()]) : 
			null;
	}
	
	// =================================================================================================================================
	// Region 
	// =================================================================================================================================
	@Override
	public Region createRegion(String name, String[] evictionStrategyNames) 
	throws RegionInitializationException
	{
		EvictionStrategy[] evictionStrategies = getEvictionStrategies(evictionStrategyNames);
		return getRegionFactory().create(this, name, evictionStrategies);
	}

	@Override
	public Region createRegion(RegionMemento regionMemento) 
	throws RegionInitializationException
	{
		EvictionStrategy[] regionEvictionStrategies = getEvictionStrategies(regionMemento.getEvictionStrategyNames());
		return getRegionFactory().create(this, regionMemento.getName(), regionEvictionStrategies);
	}

	@Override
	public void addRegion(Region region) 
	throws CacheException
	{
		regions.add(region);
		notifyCacheStructureChangeListeners();
	}

	@Override
	public void addRegions(Collection<? extends Region> regions) 
	throws CacheException
	{
		for(Region region : regions)
			this.addRegion(region);
	}

	@Override
	public Region getRegion(String name)
	{
		if(name == null)
			return null;
		
		for(Region region : getRegions())
			if( name.equals(region.getName()) )
				return region;
		
		return null;
	}

	@Override
	public Collection<? extends Region> getRegions()
	{
		return Collections.unmodifiableCollection(this.regions);
	}
	
	// =================================================================================================================================
	// Instance Management
	// =================================================================================================================================
	@Override
	public void clear() 
	throws CacheException
	{
		for(Region region : getRegions())
			region.deleteAllChildGroups(false);
	}

	@Override
	public Instance getInstance(String regionName, String[] group, String key) 
	throws CacheException
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		return region.getInstance(group, key);
	}

	@Override
	public void deleteInstance(String regionName, String[] group, String key, boolean forceDelete) 
	throws CacheException
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		region.deleteInstance(group, key, forceDelete);
	}
	
	@Override
	public Instance getOrCreateInstance(String regionName, String[] group, String key) 
	throws CacheException
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		return region.getOrCreateInstance(group, key);
	}

	// =================================================================================================================================
	// Group Management
	// =================================================================================================================================
	
	@Override
	public Group getOrCreateGroup(String regionName, String[] group)
	throws CacheException 
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		return region.getOrCreateGroup(group);
	}

	@Override
	public Group getGroup(String regionName, String[] group)
	throws CacheException 
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		return region.getGroup(group);
	}

	@Override
	public void deleteGroup(String regionName, String[] group, boolean forceDelete) 
	throws CacheException
	{
		Region region = getRegion(regionName);
		
		if(region == null)
			throw new RegionDoesNotExistException();
		
		region.deleteGroup(group, forceDelete);
	}
	

	// =================================================================================================================================
	// Management and Monitoring
	// =================================================================================================================================
	/**
	 * @see gov.va.med.imaging.storage.cache.Cache#registerCacheStructureChangeListener(gov.va.med.imaging.storage.cache.CacheStructureChangeListener)
	 */
	@Override
	public void registerCacheStructureChangeListener(CacheStructureChangeListener listener)
	{
		cacheStructureChangeListeners.add(listener);
	}

	/**
	 * @see gov.va.med.imaging.storage.cache.Cache#unregisterCacheStructureChangeListener(gov.va.med.imaging.storage.cache.CacheStructureChangeListener)
	 */
	@Override
	public void unregisterCacheStructureChangeListener(CacheStructureChangeListener listener)
	{
		cacheStructureChangeListeners.remove(listener);
	}

	public void notifyCacheStructureChangeListeners()
	{
		for(CacheStructureChangeListener listener : cacheStructureChangeListeners)
			listener.cacheStructureChanged(this);
	}
	
	// =================================================================================================================================
	// Container Lifecycle Notification
	// =================================================================================================================================
	/**
	 * @see gov.va.med.imaging.storage.cache.CacheLifecycleListener#cacheLifecycleEvent(gov.va.med.imaging.storage.cache.CacheLifecycleEvent)
	 */
	@Override
	public void cacheLifecycleEvent(CacheLifecycleEvent event) 
	throws CacheStateException
	{
		for(Region region : getRegions())
			region.cacheLifecycleEvent(event);
	}
	
	class LifecycleLogger
	implements InstanceLifecycleListener, GroupLifecycleListener
	{

		/* (non-Javadoc)
		 * @see gov.va.med.imaging.storage.cache.events.InstanceLifecycleListener#notify(gov.va.med.imaging.storage.cache.events.InstanceLifecycleEvent)
		 */
		@Override
		public void notify(InstanceLifecycleEvent event)
		{
			System.out.println("Instance event [" + event.getLifecycleEvent() + "] " + event.getName() );
		}

		/* (non-Javadoc)
		 * @see gov.va.med.imaging.storage.cache.events.GroupLifecycleListener#notify(gov.va.med.imaging.storage.cache.events.GroupLifecycleEvent)
		 */
		@Override
		public void notify(GroupLifecycleEvent event)
		{
			System.out.println("Group event [" + event.getLifecycleEvent() + "] " + event.getName() );
		}
		
	}

	@Override
	public Boolean isImageFilesCached(String studyid, String imageid, String siteid,
			String PatientIdentifier, String regionName) {
		return true;
	}
}
