package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.Group;
import gov.va.med.imaging.storage.cache.InstanceByteChannelFactory;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.CacheInternalException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.impl.PersistentGroupSet;

import java.lang.ref.SoftReference;
import java.net.MalformedURLException;
import java.net.UnknownHostException;
import java.util.Iterator;

import jcifs.smb.SmbFile;
import jcifs.smb.SmbFileFilter;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * This class encapsulates the collection of child groups in a FileSystemCacheGroup and in
 * a FileSystemCacheRegion.
 * It is the responsibility of this class to ensure that the persistent (filesystem)
 * and the transient (memory) views of the Groups in a group are consistent.
 * It is a requirement that this class NOT keep references to child groups that 
 * would prevent garbage collection of groups that are no longer referenced outside
 * the cache (i.e. if the application does not have a reference then the cache should not
 * prevent garbage collection).
 * This class should be the sole modifier of the groups referenced within.
 * 
 * @author       BECKEC
 *
 */
class JcifsGroupSet
extends PersistentGroupSet
{
	private static final long serialVersionUID = 1L;
	@SuppressWarnings("unused")
	private Logger log = LogManager.getLogger(this.getClass());
	private SmbFile rootDirectory = null;			// the directory in which all of our instances reside in persistent storage
	private SmbUtilities smbUtilities = new SmbUtilities();
	
	/**
	 * 
	 * @param rootDirectory
	 * @param byteChannelFactory
	 * @param secondsReadWaitsForWriteCompletion
	 * @param setModificationTimeOnRead
	 */
	JcifsGroupSet(
		SmbFile rootDirectory, 
		InstanceByteChannelFactory byteChannelFactory,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead)
	{
		super(byteChannelFactory, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
		if(rootDirectory == null)
			throw new IllegalArgumentException("RootDirectory must be a valid directory.");
		this.rootDirectory = rootDirectory;
	}
	
	public SmbFile getRootDirectory()
	{
		return rootDirectory;
	}
	
	/**
	 * Override this to reduce type check warnings
	 */
	@Override
	@SuppressWarnings("unchecked")
	public InstanceByteChannelFactory<SmbFile> getByteChannelFactory()
	{
		return super.getByteChannelFactory();
	}
	
	/**
	 * Get or create a Group mapped to persistent storage.
	 * 
	 * @param name - the group name to get or create
	 * @param create - true if the groups should be created if it does not exist\
	 */
	@Override
	protected Group getOrCreate(String name, boolean create) 
	throws CacheException
	{
		SmbFile childGroupDir;
		try
		{
			childGroupDir = new SmbFile(getRootDirectory().getCanonicalPath() + "/" + name + "/");
		} 
		catch (MalformedURLException x)
		{
			throw new PersistenceIOException(
					"The SMB URL built from directory '" + getRootDirectory().getPath() + "' and file name '" + name + "' is invalid.", 
					x);
		} 
		//catch (UnknownHostException x)
		//{
		//	throw new PersistenceIOException(
		//			"The host specified in SMB URL '" + getRootDirectory().getPath() + "' and file name '" + name + "' is invalid.", 
		//			x);
		//}
		JcifsGroup child = create ? 
				JcifsGroup.getOrCreate(childGroupDir, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead()) :
				JcifsGroup.get(childGroupDir, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead());
		
		return child;
	}
	
	/**
	 * Assure that the internal represenation of child instances matches
	 * what is in the persistent storage (file system). 
	 */
	protected void internalSynchronizeChildren()
	throws CacheException
	{
		String name = null;		// declared here so tha the exception handling can report it
		
		try
		{
			// prune unused references
			for( Iterator<SoftReference<? extends Group>> iter=iterator();
				iter.hasNext(); )
			{
				SoftReference<? extends Group> groupRef = iter.next();
				if( groupRef.get() == null )
					iter.remove();
			}
	
			// get a list of all the child files
			for( SmbFile childDir : smbUtilities.getPersistentChildren(getRootDirectory(), true, false) )
			{
				name = childDir.getName();		// the file name and the Instance name are the same
				Group childGroup = getTransient(name);		// get an existing reference by name
				if(childGroup == null)
				{
					childGroup = getChild(name, false);
					SmbFile groupDir = new SmbFile(this.getRootDirectory(), name);
					childGroup =  JcifsGroup.get(groupDir, getByteChannelFactory(), getSecondsReadWaitsForWriteCompletion(), isSetModificationTimeOnRead());
					SoftReference<JcifsGroup> groupRef = new SoftReference<JcifsGroup>( (JcifsGroup)childGroup);
					add( groupRef );
				}
			}
		} 
		catch (MalformedURLException x)
		{
			throw new CacheInternalException(
					"The SMB URL built from directory '" + getRootDirectory().getPath() + "' and file name '" + name + "' is invalid.", 
					x);
		} 
		catch (UnknownHostException x)
		{
			throw new PersistenceIOException(
					"The host specified in the SMB URL '" + getRootDirectory().getPath() + "' cannot be contacted.", 
					x);
		}
	}
}
