package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.Group;
import gov.va.med.imaging.storage.cache.Instance;
import gov.va.med.imaging.storage.cache.InstanceByteChannelFactory;
import gov.va.med.imaging.storage.cache.exceptions.CacheException;
import gov.va.med.imaging.storage.cache.exceptions.GroupDoesNotExistException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceException;
import gov.va.med.imaging.storage.cache.exceptions.PersistenceIOException;
import gov.va.med.imaging.storage.cache.impl.PersistentGroup;
import gov.va.med.imaging.storage.cache.impl.PersistentGroupSet;
import gov.va.med.imaging.storage.cache.impl.PersistentInstanceSet;

import java.util.Date;
import java.util.Iterator;

import jcifs.smb.SmbException;
import jcifs.smb.SmbFile;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * 
 * 
 */
public class JcifsGroup 
extends PersistentGroup
{
	private Logger log = LogManager.getLogger(this.getClass());
	private SmbFile groupDirectory = null;
	private JcifsInstanceSet childInstances = null;
	private JcifsGroupSet childGroups = null;
	
	/**
	 * 
	 * @param groupDirectory
	 * @param instanceFactoryChannel
	 * @return
	 * @throws CacheException
	 */
	public static JcifsGroup getOrCreate(
		SmbFile groupDirectory, 
		InstanceByteChannelFactory<SmbFile> instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead)
	throws CacheException
	{
		try
		{
			if(! groupDirectory.exists())
			{
				LogManager.getLogger(JcifsGroup.class).debug("Creating persistence for group '" + groupDirectory.getPath() + "'.");
				groupDirectory.mkdir();
			}
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error determining the existence of, or creating the SMB directory '" + groupDirectory.getPath() + "'.", x);
		}
		
		return new JcifsGroup(groupDirectory, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
	}
	
	/**
	 * 
	 * @param groupDirectory
	 * @param instanceFactoryChannel
	 * @return
	 * @throws CacheException
	 */
	public static JcifsGroup get(
		SmbFile groupDirectory, 
		InstanceByteChannelFactory<SmbFile> instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead)
	throws CacheException
	{
		try
		{
			if(! groupDirectory.exists())
				return null;
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error determining the existence of the SMB directory '" + groupDirectory.getPath() + "'.", x);
		}
		
		return new JcifsGroup(groupDirectory, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
	}
	
	/**
	 * 
	 * @param groupDirectory
	 * @param createIfNotExist
	 * @param instanceFactoryChannel
	 * @throws CacheException
	 */
	private JcifsGroup(
		SmbFile groupDirectory, 
		InstanceByteChannelFactory<SmbFile> instanceFactoryChannel,
		int secondsReadWaitsForWriteCompletion,
		boolean setModificationTimeOnRead
	)
	throws CacheException
	{
		super(instanceFactoryChannel);
		this.groupDirectory = groupDirectory;
		if(! isPersistent())
			throw new GroupDoesNotExistException();
		childInstances = new JcifsInstanceSet(groupDirectory, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
		childGroups = new JcifsGroupSet(groupDirectory, instanceFactoryChannel, secondsReadWaitsForWriteCompletion, setModificationTimeOnRead);
	}

	// ======================================================================================================
	// Persistence type specific abstract method overrides
	// This is the stuff that makes a group a file system persisted group
	// ======================================================================================================
	
	public boolean isPersistent() 
	throws PersistenceException
	{
		try
		{
			return getGroupDirectory().exists();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Unable to determine the existence of '" + getGroupDirectory().getPath() + "'", x);
		}
	}
	
	protected PersistentInstanceSet getPersistentInstanceSet()
	{
		return childInstances;
	}
	
	@Override
	protected PersistentGroupSet getPersistentGroupSet()
	{
		return childGroups;
	}

	@Override
	public String getName()
	{
		return getGroupDirectory().getName();
	}

	protected SmbFile getGroupDirectory()
	{
		return groupDirectory;
	}
	
	/**
	 * The last accessed date is the last modified of the directory or of
	 * any file in the directory, otherwise the directory gets deleted
	 * right after it was created.
	 */
	@Override
	public Date getLastAccessed() 
	throws CacheException
	{
		long dirModified = 0L;
		try
		{
			dirModified = getGroupDirectory().lastModified();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException("Error getting the last modified date of the SMB directory '" + getGroupDirectory().getPath() + "'.", x);
		}
		Date instancesLastAccessed = super.getLastAccessed();
		
		return instancesLastAccessed.getTime() > dirModified ? 
				instancesLastAccessed : new Date(dirModified);
	}	
	
	// ======================================================================================================
	// Child Instance Management
	// ======================================================================================================
	/**
	 * It is important that any operation that creates child groups or instances be synchronized
	 * using the childInstancesLock to assure that there is exactly one representation of the group or instance.
	 * 
	 * @param key
	 * @return
	 * @throws CacheException
	 */
	@Override
	public Instance getOrCreateChildInstance(String key)
	throws CacheException
	{
		return childInstances.getChild(key, true);
	}
	
	@Override
	public Instance getChildInstance(String key) 
	throws CacheException
	{
		return childInstances.getChild(key, false);
	}

	@Override
	public Iterator<? extends Instance> getInstances() 
	throws CacheException
	{
		return childInstances.hardReferenceIterator();
	}

	@Override
	public void deleteChildInstance(String key, boolean forceDelete)
	throws CacheException 
	{
		Instance child = childInstances.getChild(key, false);
		if(child != null)
			childInstances.deleteChild(child, forceDelete);
	}

	@Override
	public void deleteAllChildInstances(boolean forceDelete)
	throws CacheException 
	{
		childInstances.deleteAll(forceDelete);
	}

	@Override
	public long getSize() 
	throws CacheException
	{
		return childInstances.getSize();
	}
	
	// ======================================================================================================
	// Child Group Management
	// ======================================================================================================
	/**
	 * 
	 */
	@Override
	public Group getChildGroup(String groupName) 
	throws CacheException
	{
		return childGroups.getChild(groupName, false);
	}

	/**
	 * 
	 */
	@Override
	public Group getOrCreateChildGroup(String groupName) 
	throws CacheException
	{
		return childGroups.getChild(groupName, true);
	}

	/**
	 * Remove a child group.
	 * @throws CacheException 
	 */
	@Override
	public void deleteChildGroup(Group childGroup, boolean forceDelete) 
	throws CacheException
	{
		childGroups.deleteChild(childGroup, forceDelete);
	}

	@Override
	public void deleteAllChildGroups(boolean forceDelete) 
	throws CacheException 
	{
		childGroups.deleteAll(forceDelete);
	}

	/**
	 * All subdirectories of ourselves are child groups.
	 * @throws CacheException 
	 */
	@Override
	public Iterator<? extends Group> getGroups() 
	throws CacheException
	{
		return childGroups.hardReferenceIterator();
	}
	
	/**
	 * Remove ourselves, along with all children
	 * @throws CacheException 
	 */
	@Override
	public void delete(boolean forceDelete) 
	throws CacheException
	{
		log.info("Group '" + this.getName() + "' is removing itself...");
		deleteAllChildGroups(forceDelete);
		deleteAllChildInstances(forceDelete);
		
		// remove the persistent group
		try
		{
			getGroupDirectory().delete();
		} 
		catch (SmbException x)
		{
			throw new PersistenceIOException(
				"Error deleting the SMB directory '" + getGroupDirectory().getPath() + "'.", 
				x);
		}
		log.info("Group '" + this.getName() + "' is removed.");
	}

	public int compareTo(JcifsGroup that)
	{
		if(this.getGroupDirectory().equals(that.getGroupDirectory()))
			return 0;
		return this.getGroupDirectory().getCanonicalPath().compareTo(that.getGroupDirectory().getCanonicalPath());
	}


	@Override
	public boolean equals(Object obj)
	{
		if(obj instanceof JcifsGroup)
		{
			JcifsGroup that = (JcifsGroup)obj;
			return this.getGroupDirectory().equals(that.getGroupDirectory());
		}
		return false;
	}


	@Override
	public String toString()
	{
		return this.getClass().getSimpleName() + " - " + this.getGroupDirectory().getCanonicalPath();
	}

	
}
