package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.InstanceWritableByteChannel;
import gov.va.med.imaging.storage.cache.exceptions.SimultaneousWriteException;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.zip.Checksum;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import jcifs.smb.SmbFile;

/**
 * A class that simply wraps a FileChannel that will be used for writing,
 * and releases a lock when the channel is closed.
 * 
 * @author       BECKEC
 *
 */
class InstanceWritableByteChannelImpl 
implements InstanceWritableByteChannel
{
	/**
	 * 
	 */
	private final JcifsByteChannelFactory factory;
	private final SmbFile file;
	private final OutputStream outStream;
	private final WritableByteChannel wrappedChannel;
	private long openedTime = 0L;					// keep this so that we could close the files ourselves if the client does not
	private long lastAccessedTime = 0L;
	private Checksum checksum;
	private StackTraceElement[] instantiatingStackTrace = null;
	
	InstanceWritableByteChannelImpl(JcifsByteChannelFactory factory, SmbFile file) 
	throws IOException, SimultaneousWriteException
	{
		this(factory, file, null);
	}
	
	InstanceWritableByteChannelImpl(JcifsByteChannelFactory factory, SmbFile file, Checksum checksum) 
	throws IOException, SimultaneousWriteException
	{
		this.factory = factory;
		this.file = file;
		this.checksum = checksum;
		
		this.factory.log.debug("InstanceWritableByteChannelImpl, opening '" + file.getPath() + "'" );
		if(this.factory.isTraceChannelInstantiation())
			instantiatingStackTrace = Thread.currentThread().getStackTrace();
		outStream = file.getOutputStream();
		wrappedChannel = Channels.newChannel(outStream);
		
		openedTime = System.currentTimeMillis();
		lastAccessedTime = openedTime;
	}
	
	SmbFile getFile()
	{
		return this.file;
	}
	
	public StackTraceElement[] getInstantiatingStackTrace()
	{
		return instantiatingStackTrace;
	}
	
	public long getLastAccessedTime()
	{
		return lastAccessedTime;
	}
	
	public Checksum getChecksum()
	{
		return this.checksum;
	}

	/**
	 * Write from an NIO ByteBuffer to a stream
	 */
	public int write(ByteBuffer src) 
	throws IOException
	{
		Checksum localChecksumRef = getChecksum();		// just for performance
		if(localChecksumRef != null)
			for(ByteBuffer localBuffer = src.asReadOnlyBuffer(); localBuffer.hasRemaining(); localChecksumRef.update(localBuffer.get()) );
		
		lastAccessedTime = System.currentTimeMillis();

		return wrappedChannel.write(src);
	}
	
	public void close() 
	throws IOException
	{
		close(false);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceWritableByteChannel#error()
	 */
	public void error() 
	throws IOException
	{
		close(true);
	}
	
	private void close(boolean errorClose) 
	throws IOException
	{
		this.factory.log.debug("InstanceWritableByteChannelImpl, closing '" + file.getPath() + "' " + (errorClose ? "WITH" : "without") + " delete");
		IOException ioX = null;
		
		try{ wrappedChannel.close(); }
		catch(IOException e)
		{this.factory.log.warn(e);} // the channel may already be closed through some error or other timeout, log it but keep going
		
		try{ outStream.close(); }
		catch(IOException e)
		{this.factory.log.warn(e);}
		
		if( errorClose )
			this.file.delete();
		
		// the following operation really must occur regardless of the 
		// success of the previous IO operations, else threads will lock and byte channels will repeatedly be closed when they are already closed
		this.factory.writableByteChannelClosed(this, errorClose);
		
		this.factory.log.debug("InstanceWritableByteChannelImpl - '" + file.getPath() + "' closed " + (errorClose ? "WITH" : "without") + " delete");
		if(ioX != null)
			throw ioX;
	}
	
	public boolean isOpen()
	{
		return wrappedChannel != null && wrappedChannel.isOpen();
	}
}