/**
 * Ax Web App Rest service definitions
 * Date Created: May 7, 2017
 * Developer:  vacotittoc
 */
package gov.va.med.imaging.ax.webservices.rest;

import gov.va.med.imaging.ax.web.AxDoGetDocument;
import gov.va.med.imaging.ax.webservices.commands.AxGetDocumentListCommand;
import gov.va.med.imaging.ax.webservices.commands.AxGetLucky;
import gov.va.med.imaging.ax.webservices.rest.types.Bundle;
import gov.va.med.imaging.ax.webservices.rest.types.LuckyType;
import gov.va.med.imaging.ax.webservices.rest.types.OperationOutcome;
import gov.va.med.imaging.ax.webservices.translator.AxTranslator;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.documents.DocumentSetResult;
import gov.va.med.imaging.exchange.enums.ImageQuality;
import gov.va.med.imaging.exchange.translation.exceptions.TranslationException;
import gov.va.med.imaging.transactioncontext.TransactionContext;
// import gov.va.med.imaging.encryption.exceptions.AesEncryptionException;
// import gov.va.med.imaging.mix.webservices.rest.types.v1.DiagnosticReports;
// import gov.va.med.imaging.rest.types.RestBooleanReturnType;
// import gov.va.med.imaging.rest.types.RestStringType;
// import gov.va.med.imaging.tomcat.vistarealm.encryption.EncryptionToken;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.transactioncontext.TransactionContextHttpHeaders;

import java.net.URLDecoder;
// import java.text.ParseException;
// import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
// import javax.ws.rs.Consumes;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.PathParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
// import javax.ws.rs.core.Request;
import javax.ws.rs.core.Response;
// import javax.ws.rs.core.Response.ResponseBuilder;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.StreamingOutput;
import javax.ws.rs.core.UriInfo;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

/**
 * @author vacotittoc
 *
 */
@Path("ax")
public class AxServices
{
	private static final String VAOID = "2.16.840.1.113883.4.349|";
	private static final String APPJSONFHIR = "application/json+fhir"; // useless, no body writer for this MIME type with genson-x.x

	private final static Logger logger = LogManager.getLogger(AxServices.class);

	protected Logger getLogger()
	{
		return logger;
	}

	// Pass 1 -- get list of patient documents by date range
	//     DocumentReference?subject.identifier=2.16.840.1.113883.4.349|{ICN}&[type={LOINC}][&created=>={YYYY-MM-DD}&created=<={YYYY-MM-DD}]&
	//						 requestor={system}&transactionId={queryId}&purposeOfUse=Treatment
	//      for the record: 2.16.840.1.113883.3.42.10001.100001.12|{EDIPI} for DoD
	@GET
	@Path("DocumentReference") 
//	@Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	@Produces({MediaType.APPLICATION_JSON, APPJSONFHIR})
	public Response getDocumentReferenceListForPatient(
			@Context HttpServletRequest request,
			@Context HttpServletResponse response,
			@Context UriInfo uriInfo,
			@QueryParam("subject.identifier") String patientId,
			@QueryParam("type") String LOINC,
			@QueryParam("created=>") String fromDate,
			@QueryParam("created=<") String toDate,
			@QueryParam("requestor") String requestor,
			@QueryParam("transactionID") String transactionId,
			@QueryParam("purposeOfUse") String purposeOfUse,
			@HeaderParam("Accept") String acceptHeader)
	throws MethodException, ConnectionException
	{
        @SuppressWarnings("deprecation")
		String requestURL = URLDecoder.decode(uriInfo.getRequestUri().toString()); // , "ISO-8859-1");
		getLogger().debug("AX GET DocumentReference request received: " + requestURL + " with Accept header: " + ((acceptHeader==null)?"":acceptHeader));
		
		// 2.16.840.1.113883.4.349|ICN, [LOINC], [yyyy-MM-dd]-s, requestor, transactionId and purposeOfUse ("Treatment") are expected
		String begDate=null;
		String endDate=null;
		begDate = convertDate(fromDate); // start of day
		endDate = convertDate(toDate); // later set end of the day!
		if ((LOINC==null) || LOINC.isEmpty())
			LOINC=null;
		// Are 2.16.840.1.113883.4.349|ICN, [LOINC], [yyyy-MM-dd]-s, requestor, transactionId and purposeOfUse ("Treatment") passed? 
		// (LOINC and dates can be null)
		if ((patientId==null) || (!patientId.startsWith(VAOID)) || (!patientId.contains("V")) ||
			(requestor==null) || requestor.isEmpty() || (transactionId==null) || transactionId.isEmpty() || 
			(purposeOfUse==null) || (!purposeOfUse.equalsIgnoreCase("Treatment"))) {
			String message = "Invalid DocumentReference request: subject.identifier must start with '2.16.840.1.113883.4.349|' and must be folowed by a correlated ICN; " +
							 "'requestor', 'transactionID' and 'purposeOfUse' must be explicitly defined and 'purposeOfUse' must be 'Treatment'!";
			return wrapErrorResponse(Status.NOT_ACCEPTABLE, message);
		}
		String patientIcn = patientId.replace(VAOID, ""); // remove OID, keep ICN

		// *** mocking with ICNs is temporary for Integration Tests ***
		// patientIcn = checkToSwapICN(patientIcn);

		if ((acceptHeader!=null) && !acceptHeader.isEmpty() && acceptHeader.contains(APPJSONFHIR)) {
			String newAccept = request.getHeader("Accept").replace(APPJSONFHIR, "application/json"); // make sure "application/json+fhir" disappears from Accept header
			request.setAttribute("Accept", newAccept); // overwrite ???!!!
			getLogger().debug("AX GET DR Accept header: '" + acceptHeader + "'; altered to: '" + newAccept + "'; from request: '" + request.getHeader("Accept") + "' got.");
//			response.setHeader("Accept", newAccept);
		}
		
		return wrapOkResponse(convertDocumentSetResult(new AxGetDocumentListCommand(patientIcn, LOINC, begDate, endDate, purposeOfUse).execute(), requestor, transactionId) );
		// DiagnosticReports class is returned
	}
	
// Pass 2, BLOB retrieval
//         Binary/{documentURN}/?transactionId={uniqueID}&requestor={JLV|HAIMS|Etc.}&purposeOfUse=Treatment
	@GET
	@Path("Binary/{documentURN}/")
	@Produces({MediaType.APPLICATION_OCTET_STREAM, MediaType.APPLICATION_OCTET_STREAM}) // Data is streamed binary, error is FHIR JSON
	public Response getDocumentBLOB(
			@Context UriInfo uriInfo,
			@Context HttpServletRequest request,
			@Context HttpServletResponse response,
			@PathParam("documentURN") String documentURN,
			@QueryParam("transactionID") String transactionId,
			@QueryParam("requestor") String requestor,
			@QueryParam("purposeOfUse") String purposeOfUse)
	throws MethodException, ConnectionException
	{
		// String theRIRequest = (request!=null)?(request.getPathInfo()  + "?" + request.getQueryString()):"n/a";
        @SuppressWarnings("deprecation")
		String requestURL = URLDecoder.decode(uriInfo.getRequestUri().toString()); // , "ISO-8859-1");
		getLogger().debug("Ax GET a Binary document request received: " +  requestURL);
		
		String message="";
		if ( (documentURN==null) || documentURN.isEmpty() ||
			 (requestor==null) || requestor.isEmpty() || (transactionId==null) || transactionId.isEmpty() || 
				(purposeOfUse==null) || !purposeOfUse.equalsIgnoreCase("Treatment")) {
			message="Invalid document retrieval request: 'documentURN', 'requestor', 'transactionID' and 'purposeOfUse' must be explicitly defined and 'purposeOfUse' must be 'Treatment'";
			return wrapErrorResponse(Status.NOT_ACCEPTABLE, message);
		}
		// temp error (until fully implemented):
		// message = "RetrieveImage is not fully implemented yet. Please use WADO-URI format !";
		// return wrapResponse(Status.INTERNAL_SERVER_ERROR, message);
		
		// 1 documentURN; MixGetAndstreamDicomJ2KImage will returns a java  OutputStream!
		return streamResponse(AxGetAndStreamDocument(request, response, documentURN));
	}
	
	// ------------------------------------------ test only!!! --------------------------------------------
	//     Lucky?word={a-word}&number={a-number}[&accept=application/json]
	@GET
	@Path("Lucky")
	@Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	public Response getLucky(
			@Context UriInfo uriInfo,
			@QueryParam("word") String aWord,
			@QueryParam("number") String aNumber)
	throws MethodException
	{
        @SuppressWarnings("deprecation")
		String requestURL = URLDecoder.decode(uriInfo.getRequestUri().toString()); // , "ISO-8859-1");
		getLogger().debug("MIX 'GET Lucky' request received. URI: " + requestURL);

		AxGetLucky getLucky = new AxGetLucky(aWord, aNumber);
		LuckyType luckyType = getLucky.crunch();
		return wrapOkResponse(luckyType);
	}
	// test only!!!
	//     LuckyPP/{word}/{number}[?accept=application/json]
	@GET
	@Path("LuckyPP/{word}/{number}")
	@Produces({MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	public Response getLuckyPP(
			@Context UriInfo uriInfo,
			@PathParam("word") String aWord,
			@PathParam("number") String aNumber)
	throws MethodException
	{
		String requestURL = URLDecoder.decode(uriInfo.getRequestUri().toString()); // , "ISO-8859-1");
		getLogger().debug("MIX 'GET LuckyPP' request received. URI: " + requestURL);

		AxGetLucky getLucky = new AxGetLucky(aWord, aNumber);
		LuckyType luckyType = getLucky.crunch();
		return wrapOkResponse(luckyType);
	}
	
// ------------------------------------------ support methods --------------------------------------------

	private String checkToSwapICN(String theICNgot)
	{
		String theICN = theICNgot;
		
		// trick to swap IPO 4&5 test ICNs to IPO 1&2 mirrors
		if (theICNgot.equalsIgnoreCase("1008689559V859134"))
			theICN = "1012740021V495560"; 
		else if (theICNgot.equalsIgnoreCase("1008689424V686541"))
			theICN = "1012740381V796853"; 
		else if (theICNgot.equalsIgnoreCase("1008689409V873033"))
			theICN = "1012740369V059445"; 
		else if (theICNgot.equalsIgnoreCase("1008689511V752466"))
			theICN = "1012740020V387031"; 
		
		if (!theICN.equalsIgnoreCase(theICNgot))
			getLogger().debug("AX GET DocumentReference -- ICN swap from '" + theICNgot + "' to '" + theICN + "' !!!");
		 
		return theICN;
	}

	private String convertDate(String webDate)
	{
		String dicomDate=null;

		// convert "yyyy-mm-dd" to "yyyymmdd" or return null
		if ((webDate!=null) && (!webDate.isEmpty())) {
			// dicomDate = webDate;
			dicomDate = webDate.replace("-", "");
		}
		
		return dicomDate;
	}
		
	protected Bundle convertDocumentSetResult(DocumentSetResult documentSetresult, String requestor, String transactionId)
	throws MethodException
	{
		Bundle bundle = null;
		try {
			bundle = AxTranslator.convertDocumentReferences(documentSetresult, requestor, transactionId);
		}
		catch (TranslationException te) {
			throw new MethodException("DocumentSetResult translation exception: " + te.getMessage());
		}

		if (bundle==null)
			bundle = new Bundle();

		bundle.setId(transactionId);
		
		return bundle;
	}

	private StreamingOutput AxGetAndStreamDocument(HttpServletRequest request, HttpServletResponse response,
			String documentURN) {
		AxDoGetDocument axDoGetImage = new AxDoGetDocument();
		return axDoGetImage.streamDocument(request, response, documentURN);
	}

	private Response wrapErrorResponse(Status status, String message)
	{		
		OperationOutcome operationOutcome = AxTranslator.createOperationOutcome(Integer.toString(status.getStatusCode()), message);
		
		return Response.status(status).header(TransactionContextHttpHeaders.httpHeaderMachineName, 
				TransactionContextFactory.get().getMachineName()).entity(operationOutcome).build();
	}

	private Response wrapOkResponse(Object result)
	{
		Response response=null;
		try {
			response = Response.status(Status.OK).header(TransactionContextHttpHeaders.httpHeaderMachineName, 
					    TransactionContextFactory.get().getMachineName()).entity(result).build();
		} 
		catch (Exception ex)
		{
			return wrapErrorResponse(Status.INTERNAL_SERVER_ERROR, ex.getMessage());
		}
		return response;
	}

	private Response streamResponse(StreamingOutput stream)
//	throws MethodException
	{
//		Response response=null;
//		try {
//			response = Response.ok(stream).header(TransactionContextHttpHeaders.httpHeaderMachineName, 
//				TransactionContextFactory.get().getMachineName()).build();
		return Response.ok(stream).header(TransactionContextHttpHeaders.httpHeaderMachineName, 
				TransactionContextFactory.get().getMachineName()).build();
//			TransactionContext transactionContext = TransactionContextFactory.get();
//			if (transactionContext.getResponseCode().equals(Status.NOT_ACCEPTABLE))
//			{
//				String msg = "Ax: URNFormatException (on documentURN) before streaming, " + transactionContext.getErrorMessage();
//				return wrapErrorResponse(Status.NOT_ACCEPTABLE, msg);
//			}
//		}
//		catch (Exception ex)
//		{
//			return wrapErrorResponse(Status.INTERNAL_SERVER_ERROR, ex.getMessage());
//		}
//		return response;
	}

}
