/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Jan 11, 2012
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        WERFEJ
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.awiv.server.translator;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.SERIALIZATION_FORMAT;
import gov.va.med.WellKnownOID;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.awiv.business.AwivArtifactResults;
import gov.va.med.imaging.awiv.business.AwivAuthenticationSitesInformation;
import gov.va.med.imaging.awiv.business.AwivImage;
import gov.va.med.imaging.awiv.business.AwivMeansTestResult;
import gov.va.med.imaging.awiv.business.AwivPatientSensitivity;
import gov.va.med.imaging.awiv.business.AwivServerInformation;
import gov.va.med.imaging.awiv.business.AwivSite;
import gov.va.med.imaging.awiv.business.AwivStudy;
import gov.va.med.imaging.awiv.business.AwivStudyDetails;
import gov.va.med.imaging.awiv.business.AwivStudyStatus;
import gov.va.med.imaging.awiv.business.AwivWelcomeMessage;
import gov.va.med.imaging.awiv.business.Patient;
import gov.va.med.imaging.awiv.configuration.AwivClientConfiguration;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.ArtifactResultError;
import gov.va.med.imaging.exchange.business.ArtifactResultErrorComparator;
import gov.va.med.imaging.exchange.business.ArtifactResults;
import gov.va.med.imaging.exchange.business.Image;
import gov.va.med.imaging.exchange.business.PatientMeansTestResult;
import gov.va.med.imaging.exchange.business.PatientSensitiveValue;
import gov.va.med.imaging.exchange.business.Series;
import gov.va.med.imaging.exchange.business.Site;
import gov.va.med.imaging.exchange.business.Study;
import gov.va.med.imaging.exchange.business.StudySetResult;
import gov.va.med.imaging.exchange.business.WelcomeMessage;
import gov.va.med.imaging.exchange.business.documents.Document;
import gov.va.med.imaging.exchange.business.documents.DocumentSet;
import gov.va.med.imaging.exchange.business.documents.DocumentSetResult;
import gov.va.med.imaging.exchange.business.util.ExchangeUtil;
import gov.va.med.imaging.exchange.enums.ArtifactResultErrorCode;
import gov.va.med.imaging.exchange.enums.ObjectStatus;
import gov.va.med.imaging.exchange.enums.PatientSensitivityLevel;
import gov.va.med.imaging.exchange.enums.VistaImageType;
import gov.va.med.imaging.translator.AbstractClinicalTranslator;
import gov.va.med.imaging.webservices.clinical.AbstractClinicalWebAppConfiguration;

/**
 * Translator for converting VIX business objects into AWIV serializable objects
 * 
 * @author       WERFEJ
 *
 */
public class AwivClientTranslator
extends AbstractClinicalTranslator
{
	private final static Logger logger = LogManager.getLogger(AwivClientTranslator.class);

	protected static Logger getLogger()
	{
		return logger;
	}
	
	public static AwivArtifactResults translate(ArtifactResults artifactResults) 
	throws URNFormatException, MethodException
	{
		AwivArtifactResults result = new AwivArtifactResults();
		List<AwivStudy> awivStudies = new ArrayList<AwivStudy>();
		
		// fix for TFS #54876
		if(artifactResults == null)
		{
			getLogger().warn("ArtifactResults is null, this should not happen. Returning empty result with error.");
			result.setAwivStudies(new AwivStudy[0]);
			return result;
		}
		
		// fix for TFS #54876
		if(artifactResults.getStudySetResult() == null && artifactResults.getDocumentSetResult() == null)
		{
			getLogger().warn("Both StudySetResult and DocumentSetResult of ArtifactResults are null, probably indicates patient not actually seen at site. Returning empty result without error.");
			result.setAwivStudies(new AwivStudy[0]);
			return result;
		}
		
		boolean isPartial = artifactResults.isPartialResult();
		StudySetResult studySetResult = artifactResults.getStudySetResult();
		boolean gotAResult = false;
		SortedSet<ArtifactResultError> errors = 
			new TreeSet<ArtifactResultError>(new ArtifactResultErrorComparator());
		if(studySetResult != null)
		{
			if(studySetResult.getArtifacts() != null)
			{
				for(Study study : studySetResult.getArtifacts())
				{
					awivStudies.add(translateStudy(study));
					gotAResult = true;
				}
			}
			if(studySetResult.getArtifactResultErrors() != null)
			{
				for(ArtifactResultError error : studySetResult.getArtifactResultErrors())
				{
					errors.add(error);
				}
			}
		}
		DocumentSetResult documentSetResult = artifactResults.getDocumentSetResult();
		if(documentSetResult != null)
		{
			// this will only be populated if getting documents from the DoD, otherwise VA documents 
			// will be included in the StudySetResult
			if(documentSetResult.getArtifacts() != null)
			{
				for(DocumentSet documentSet : documentSetResult.getArtifacts())
				{
					awivStudies.addAll(translate(documentSet));
					gotAResult = true;
				}
			}
			if(documentSetResult.getArtifactResultErrors() != null)
			{
				for(ArtifactResultError error : documentSetResult.getArtifactResultErrors())
				{
					errors.add(error);
				}
			}
		}
		StringBuilder partialResultErrorMessage = new StringBuilder();
		if(!gotAResult)
		{
			// if no result was included, then send an exception message from the error code
			if(errors.size() > 0)
			{
				// taking the highest priority error code to throw
				ArtifactResultError error = errors.first();
				ArtifactResultErrorCode errorCode = error.getErrorCode();
				if(errorCode == ArtifactResultErrorCode.timeoutException)
					throw new MethodException("java.net.SocketTimeoutException: Read timed out");
				else
					
					throw new MethodException(errorCode.name() + ": " + error.getCodeContext());
			}			
		}
		else
		{
			// got a valid result but want to include the details of the error message
			//if((isPartial) && ( errors.size() > 0))
			// JMW 5/23/2011 P104 - if there are any errors, include them in the error message
			if(errors.size() > 0)
			{
				partialResultErrorMessage.append("Recevieved '" + errors.size() + "' errors: ");
				for(ArtifactResultError error : errors)
				{					
					partialResultErrorMessage.append(error.getCodeContext());
					partialResultErrorMessage.append("; ");
				}
			}
		}
		
		result.setPartialResult(isPartial);
		result.setPartialResultMessage(partialResultErrorMessage.toString());
		result.setAwivStudies(awivStudies.toArray(new AwivStudy[result.size()]));
		
		return result;
	}
	
	private static AwivStudy translateStudy(
			Study study) 
	throws URNFormatException 
	{
		if(study == null)
			return null;
		
		String siteAbbr = study.getSiteAbbr();
		if(ExchangeUtil.isSiteDOD(study.getSiteNumber()))
		{
			siteAbbr = study.getSiteAbbr()+ (study.getSiteName() != null ? "-" + study.getSiteName() : "");			
		}
		String studyClass = study.getStudyClass() == null ? "" : study.getStudyClass();
		
		
		
		AwivStudy result = 
			new AwivStudy(study.getStudyUrn().toString(SERIALIZATION_FORMAT.CDTP), study.getSiteNumber(), 
					study.getPatientId(), study.getPatientName(), study.getProcedure(), 
					study.getProcedureDate(), siteAbbr, study.getImageCount(), study.getDescription(), 
					study.getImagePackage(), studyClass, study.getImageType(), study.getSpecialty(),
					study.getFirstImage().getImageUrn().toString(SERIALIZATION_FORMAT.CDTP), false, 
					study.isStudyImagesHaveAnnotations(), translate(study.getStudyViewStatus()),
					study.isSensitive(), translate(study.getFirstImage()));			
		return result;
	}
	
	private static AwivStudyStatus translate(ObjectStatus objectStatus)
	{
		return new AwivStudyStatus(objectStatus.getValue(), objectStatus.getDescription());
	}
	
	private static List<AwivStudy> translate(DocumentSet documentSet)
	{
		if(documentSet == null)
			return null;
		
		List<AwivStudy> result = 
			new ArrayList<AwivStudy>();
		
		for(Document document : documentSet)
		{
			VistaImageType vistaImageType = getImageType(getWebAppConfiguration(), document);
			if(vistaImageType == null)
			{
				vistaImageType = VistaImageType.UNKNOWN_IMAGING_TYPE;
				getLogger().debug("Document with media type '" + document.getMediaType() + "', returning VistaImageType of '" + vistaImageType + "' for Clinical Display.");						
			}		
			String id = document.getGlobalArtifactIdentifier().toString(SERIALIZATION_FORMAT.CDTP);
			String siteNumber = document.getRepositoryId();
			if((WellKnownOID.HAIMS_DOCUMENT.isApplicable(document.getGlobalArtifactIdentifier().getHomeCommunityId()) || 
					(ncatRepositoryId.equals(document.getRepositoryId()))))
			{
				siteNumber = "200";
			}
			
			String procedure = document.getName();
			String description = "";
			if(vistaImageType == VistaImageType.NCAT)
			{
				// NCAT reports don't have a name value
				description = "Neurocognitive Assessment Tool";
			}
			AwivStudy awivStudy = new AwivStudy(id, siteNumber, documentSet.getPatientIcn(), 
					documentSet.getPatientName(), procedure, 
					documentSet.getProcedureDate() != null ? documentSet.getProcedureDate() : document.getCreationDate(), 
					"DoD", 1, description, "", "", "", "", id, true, false, translate(ObjectStatus.VIEWABLE),
					false, translate(documentSet, document, vistaImageType));
			
			result.add(awivStudy);		
		}
		return result;
	}
	
	private static AwivImage translate(DocumentSet documentSet, 
			Document document, VistaImageType vistaImageType)
	{
		return new AwivImage(document.getGlobalArtifactIdentifier().toString(SERIALIZATION_FORMAT.CDTP),
				vistaImageType.getImageType(), "", "");
	}

	private static AbstractClinicalWebAppConfiguration getWebAppConfiguration()
	{
		return AwivClientConfiguration.getConfiguration();
	}
	
	public static AwivSite [] translatePatientSelectionSites(List<ResolvedArtifactSource> resolvedArtifactSources)
	{
		List<AwivSite> result = new ArrayList<AwivSite>();
		
		for(ResolvedArtifactSource resolvedArtifactSource : resolvedArtifactSources)
		{
			if(resolvedArtifactSource.getArtifactSource() instanceof Site)
			{
				Site site = (Site)resolvedArtifactSource.getArtifactSource();
				
				if(site.isSitePatientLookupable())
				{
					result.add(new AwivSite(site.getSiteName(), site.getSiteNumber()));
				}
			}
		}
		return result.toArray(new AwivSite[result.size()]);
	}
	
	private static AwivSite [] translateUserAuthenticationSites(List<ResolvedArtifactSource> resolvedArtifactSources)
	{
		List<AwivSite> result = new ArrayList<AwivSite>();
		
		for(ResolvedArtifactSource resolvedArtifactSource : resolvedArtifactSources)
		{
			if(resolvedArtifactSource.getArtifactSource() instanceof Site)
			{
				Site site = (Site)resolvedArtifactSource.getArtifactSource();
				if(site.isSiteUserAuthenticatable())
				{
					result.add(new AwivSite(site.getSiteName(), site.getSiteNumber()));
				}
			}
		}
		return result.toArray(new AwivSite[result.size()]);
	}
	
	public static AwivAuthenticationSitesInformation translate(List<ResolvedArtifactSource> resolvedArtifactSources, 
			AwivClientConfiguration awivClientConfiguration, String version)
	{
		AwivSite [] awivSites = translateUserAuthenticationSites(resolvedArtifactSources);
		AwivServerInformation awivServerInformation = translate(awivClientConfiguration, version);
		
		return new AwivAuthenticationSitesInformation(awivSites, awivServerInformation);
	}
	
	public static AwivServerInformation translate(AwivClientConfiguration awivClientConfiguration, String version)
	{
		int idleTimeoutMinutes = awivClientConfiguration.getIdleTimeoutMinutes();
		boolean ieOnly = awivClientConfiguration.isIeOnly();
		boolean checkMeansTest = awivClientConfiguration.isCheckMeansTest();
		return new AwivServerInformation(version, idleTimeoutMinutes, ieOnly, checkMeansTest);
	}
	
	public static Patient [] translatePatients(List<gov.va.med.imaging.exchange.business.Patient> patients)
	{
		Patient [] result = new Patient[patients.size()];
		for(int i = 0; i < patients.size(); i++)
		{			
			gov.va.med.imaging.exchange.business.Patient patient = patients.get(i);
			result[i] = translate(patient);				
		}
		return result;
	}
	
	public static Patient translate(gov.va.med.imaging.exchange.business.Patient patient)
	{
		if(patient == null)
			return null;
		Date dob = patient.getDob();
		if(dob.after(new Date()))
		{
			getLogger().debug("DOB is after today, subtracting 100 years from DOB");
			Calendar dobCalendar = Calendar.getInstance();
			dobCalendar.setTime(dob);
			dobCalendar.add(Calendar.YEAR, -100);
			dob = dobCalendar.getTime();
		}
		return new Patient(patient.getPatientName(), patient.getPatientIcn(), 
				patient.getSsn(), patient.getVeteranStatus(), dob, patient.getSensitive());
	}
	
	public static AwivStudyDetails translate(Study study)
	throws URNFormatException
	{
		if(study == null)
			return null;
		
		AwivStudyDetails result = new AwivStudyDetails();
		AwivStudy awivStudy = translateStudy(study);
		
		result.setAwivStudy(awivStudy);
		result.setAwivImages(translateImages(study));
		return result;
	}
	
	private static AwivImage[] translateImages(Study study)
	{
		List<AwivImage> result = new ArrayList<AwivImage>();
		for(Series series : study)
		{
			for(Image image : series)
			{
				result.add(translate(image));
			}
		}
		return result.toArray(new AwivImage[result.size()]);
	}
	
	private static AwivImage translate(Image image)
	{
		if(image == null)
			return null;
		return new AwivImage(image.getImageUrn().toString(SERIALIZATION_FORMAT.CDTP),
				image.getImgType(), image.getAbsLocation(), image.getFullLocation());
	}
	
	public static AwivPatientSensitivity translate(PatientSensitiveValue patientSensitivityValue)
	{
		int code = patientSensitivityValue.getSensitiveLevel().getCode();
		String warningMessage = translateNewLinesToBreaks(patientSensitivityValue.getWarningMessage());
		return new AwivPatientSensitivity(code, warningMessage);
	}
	
	public static String translateNewLinesToBreaks(String msg)
	{
		if(msg == null)
			return "";
		return msg.replace(new String("\n"), new String("<br>"));
	}
	
	public static PatientSensitivityLevel translatePatientSensitiveCode(int code)
	{
		return PatientSensitivityLevel.getPatientSensitivityLevel(code);
	}
	
	public static AwivWelcomeMessage translate(WelcomeMessage welcomeMessage)
	{
		if(welcomeMessage == null)
			return null;
		
		String message = welcomeMessage.getMessageText();
		message = message.replace(new String("\n"), new String("<br>"));		
		
		return new AwivWelcomeMessage(message);
	}
	
	public static AwivMeansTestResult translate(PatientMeansTestResult patientMeansTest)
	{
		if(patientMeansTest == null)
			return null;
		String meansTestMessage = patientMeansTest.getMessage();
		if(meansTestMessage != null)
		{
			meansTestMessage = meansTestMessage.replace(new String("\n"), new String("<br>"));
		}
		return new AwivMeansTestResult(patientMeansTest.getCode(), 
				meansTestMessage);
	}
}
