/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created: Dec 9, 2011
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:        WERFEJ
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */
package gov.va.med.imaging.awiv.client.ui.widgets;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import gov.va.med.imaging.awiv.business.AwivArtifactResults;
import gov.va.med.imaging.awiv.business.AwivMeansTestResult;
import gov.va.med.imaging.awiv.business.AwivPatientSensitivity;
import gov.va.med.imaging.awiv.business.AwivSite;
import gov.va.med.imaging.awiv.business.AwivStudy;
import gov.va.med.imaging.awiv.business.AwivUserInformation;
import gov.va.med.imaging.awiv.business.Patient;
import gov.va.med.imaging.awiv.business.UUID;
import gov.va.med.imaging.awiv.client.AwivHelper;
import gov.va.med.imaging.awiv.client.dao.PatientLookupServiceDao;
import gov.va.med.imaging.awiv.client.dao.PatientStudyServiceDao;
import gov.va.med.imaging.awiv.client.dao.PatientTreatingSitesServiceDao;
import gov.va.med.imaging.awiv.client.events.PatientSelectedEvent;
import gov.va.med.imaging.awiv.client.events.PatientSensitivityCheckCompleteEvent;
import gov.va.med.imaging.awiv.client.events.StudySelectedEvent;
import gov.va.med.imaging.awiv.client.ui.data.SiteConnectivityState;
import gov.va.med.imaging.awiv.client.ui.data.StudyRecord;
import gov.va.med.imaging.awiv.client.ui.widgets.dialogs.WaitDialog;
import gov.va.med.imaging.awiv.exceptions.AwivInsufficientPatientSensitivityException;

import com.google.gwt.http.client.Request;
import com.google.gwt.user.client.Window.Location;
import com.google.gwt.user.client.rpc.AsyncCallback;
import com.smartgwt.client.util.BooleanCallback;
import com.smartgwt.client.util.SC;
import com.smartgwt.client.widgets.Canvas;
import com.smartgwt.client.widgets.Label;
import com.smartgwt.client.widgets.events.ClickEvent;
import com.smartgwt.client.widgets.events.ClickHandler;
import com.smartgwt.client.widgets.grid.ListGridRecord;
import com.smartgwt.client.widgets.layout.HLayout;
import com.smartgwt.client.widgets.layout.VLayout;
import com.smartgwt.client.widgets.tab.Tab;

/**
 * The tab that shows the studies the patient has
 * @author       WERFEJ
 *
 */
public class PatientStudiesTab
extends Tab
implements AwivTab
{

	private Patient patient = null;
	private PatientStudiesListGrid studiesGrid = null;
	private VLayout layout = new VLayout();
	private StudySelectedEvent studySelectedEvent = null;
	private static Logger logger = LogManager.getLogger("");
	private final AwivUserInformation awivUserInformation;
	private final PatientInformationPane patientInformationPane;
	private final PatientSelectedEvent patientSelectedEvent;
	
	/**
	 * 
	 * @param awivUserInformation The information about the logged in user
	 * @param patientSelectedEvent Event handler for when a patient is selected, necessary so this tab can clear the patient (and itself) if a sensitive check fails
	 */
	public PatientStudiesTab(AwivUserInformation awivUserInformation, 
			PatientSelectedEvent patientSelectedEvent, boolean canClose)
	{
		super();
		this.awivUserInformation = awivUserInformation;
		this.patientSelectedEvent = patientSelectedEvent;
		this.setTitle("Selected Patient");
		this.setCanClose(canClose);
		studiesGrid = new PatientStudiesListGrid(awivUserInformation);
		patientInformationPane = new PatientInformationPane();
		patientInformationPane.setAwivUserInformation(awivUserInformation);
		HLayout mainLayout = new HLayout();
		mainLayout.setRedrawOnResize(false);
		
		Label ccowWarning = new Label("WARNING: The AWIV Web Application does not use CCOW (a means to ensure each window presents data from the same patient). Users <u>must</u> use caution to ensure that all data being viewed is from the correct patient.");
		ccowWarning.setWidth100();
		ccowWarning.setStyleName("CCOW-Warning");
		ccowWarning.setHeight(10);
		
		
		layout.setRedrawOnResize(false);
		layout.addMember(studiesGrid);
		layout.addMember(ccowWarning);
		mainLayout.setMembers(patientInformationPane, layout);
		this.setPane(mainLayout);
	}

	public StudySelectedEvent getStudySelectedEvent()
	{
		return studySelectedEvent;
	}

	public void setStudySelectedEvent(StudySelectedEvent studySelectedEvent)
	{
		this.studySelectedEvent = studySelectedEvent;
		studiesGrid.setStudySelectedEvent(this.studySelectedEvent);
	}

	/**
	 * Display the specified patient
	 * @param patient
	 */
	public void displayPatient(Patient patient)
	{
		this.patient = patient;
		//this.setTitle(patient.getName());
		this.setTitle("<span>" + Canvas.imgHTML(Location.getProtocol() + "//" + Location.getHost() + "/Awiv/images/ccow_broken16.gif") + " " + patient.getName() + "</span>");
		//this.patientInformationPane.displayPatient(patient);
		studiesGrid.setData((ListGridRecord[])null);
		studiesGrid.setPatient(patient);

		if(patient.isSensitive())
		{
			// don't display patient information until after sensitive warning
			displayPatientSensitivityWarning();
		}
		else
		{		
			this.patientInformationPane.displayPatient(patient);
			//loadTreatingSites();
			checkMeansTest();
		}
	}
	
	private void displayPatientSensitivityWarning()
	{
		PatientSensitivityCheck.checkSensitivePatient(awivUserInformation, patient, 
				new PatientSensitivityCheckCompleteEvent()
		{			
			@Override
			public void onPatientSensitivityCheckComplete(boolean allowedToContinue, int sensitiveCode)
			{
				if(allowedToContinue)
				{
					if(sensitiveCode > 0)
					{
						logSensitivePatientAccess();
					}
					patientInformationPane.displayPatient(patient);
					patient.updateAgreedSensitiveCode(sensitiveCode);
					checkMeansTest();
					//loadTreatingSites();
					
				}
				else
				{
					clearPatient();
				}
			}
		});
	}
	
	private void logSensitivePatientAccess()
	{
		String transactionId = UUID.uuid();
		logger.info("Logging access to sensitive patient '" + patient.getIcn() + "' from site '" + awivUserInformation.getPatientLookupSiteNumber() + "', with transaction Id '" + transactionId + "'.");
		PatientLookupServiceDao.patientLookupServiceDao.logPatientSensitiveAccess(transactionId, awivUserInformation.getPatientLookupSiteNumber(), patient.getIcn(), new AsyncCallback<Boolean>()
		{

			/* (non-Javadoc)
			 * @see com.google.gwt.user.client.rpc.AsyncCallback#onFailure(java.lang.Throwable)
			 */
			@Override
			public void onFailure(Throwable arg0)
			{
				// error occurred, not really doing anything about it
				logger.error("Error logging access to sensitive patient, " + arg0.getMessage());
				if(!AwivExceptionHandler.handleServiceException(arg0))
				{
					// don't do anything - don't show this to the user (although it's not great)
				}
			}

			/* (non-Javadoc)
			 * @see com.google.gwt.user.client.rpc.AsyncCallback#onSuccess(java.lang.Object)
			 */
			@Override
			public void onSuccess(Boolean arg0)
			{
				logger.info("Successfully logged access to sensitive patient");
			}
			
		});
	}
	
	private void checkMeansTest()
	{
		if(awivUserInformation.getAwivServerInformation().isCheckMeansTest())
		{
			String transactionId = UUID.uuid();
			logger.info("Checking means test for patient '" + patient.getIcn() + "' from site '" + awivUserInformation.getPatientLookupSiteNumber() + "', with transaction Id '" + transactionId + "'.");
			WaitDialog.displayWaitDialog("Checking Means Test", "Checking means test information for selected patient");
			PatientLookupServiceDao.patientLookupServiceDao.getPatientMeansTest(transactionId, 
					awivUserInformation.getPatientLookupSiteNumber(), patient.getIcn(), new AsyncCallback<AwivMeansTestResult>()
					{					
						@Override
						public void onSuccess(AwivMeansTestResult awivMeansTestResult)
						{
							WaitDialog.hideWaitDialog();
							if(awivMeansTestResult.isMeansTestRequired())
							{
								logger.info("Means test is required for patient '" + patient.getIcn() + "' from site '" + awivUserInformation.getPatientLookupSiteNumber() + ".");
								SC.say("MEANS TEST REQUIRED !!!", 
										awivMeansTestResult.getMessage() + "<br><br>A Means Test is Required for this patient.",
										new BooleanCallback()
										{										
											@Override
											public void execute(Boolean value)
											{
												// doesn't matter if they click ok or not
												loadTreatingSites();				
											}
										});
							}
							else
							{
								loadTreatingSites();
							}
						}
						
						@Override
						public void onFailure(Throwable arg0)
						{
							WaitDialog.hideWaitDialog();
							logger.error("Error loading means test for patient, " + arg0.getMessage());
							if(!AwivExceptionHandler.handleServiceException(arg0))
							{
								AwivHelper.displayErrorMessage("Error checking means test for patient", arg0);
							}
							loadTreatingSites();
						}
					});
		}
		else
		{
			// not checking means test, 
			loadTreatingSites();
		}
	}
	
	private void loadTreatingSites()
	{
		final TreatingSitesToolStrip toolStrip = new TreatingSitesToolStrip();
		layout.addMember(toolStrip, 0);
				
		String transactionId = UUID.uuid();
		logger.info("Retrieving treating sites for patient '" + patient.getIcn() + "' from site '" + awivUserInformation.getPatientLookupSiteNumber() + "', with transaction Id '" + transactionId + "'.");
		WaitDialog.displayWaitDialog("Loading Patient", "Loading patient treating sites");
		PatientTreatingSitesServiceDao.patientTreatingSitesServiceDao.getPatientSites(transactionId, 
				awivUserInformation.getPatientLookupSiteNumber(), patient.getIcn(), 
				new AsyncCallback<AwivSite[]>()
		{

			@Override
			public void onFailure(Throwable arg0)
			{
				WaitDialog.hideWaitDialog();
				logger.error("Error loading treating sites for patient, " + arg0.getMessage());
				if(!AwivExceptionHandler.handleServiceException(arg0))
				{
					AwivHelper.displayErrorMessage("Error loading treating sites for patient", arg0);
				}
			}

			@Override
			public void onSuccess(AwivSite[] awivSites)
			{
				WaitDialog.hideWaitDialog();
				logger.info("Found '" + awivSites.length  + "' sites for patient '" + patient.getIcn() + "'.");
				for(AwivSite awivSite : awivSites)
				{
					SiteButton button = new SiteButton(awivSite.getSiteName(), awivSite.getSiteNumber());
					addTreatingSiteClickEvent(button);
					toolStrip.addButton(button);
					logger.info("Adding button for site '" + awivSite.toString() + "'.");
				}
			}
		});
	}
	
	private void addTreatingSiteClickEvent(final SiteButton siteButton)
	{
		siteButton.addClickHandler(new ClickHandler()
		{			
			@Override
			public void onClick(ClickEvent event)
			{
				siteButtonClicked(siteButton);			
			}
		});		
	}
	
	private void siteButtonClicked(final SiteButton siteButton)
	{
		if(siteButton.getSiteConnectivityState() == SiteConnectivityState.connected)
		{
			// already connected, disconnect and remove the results
			if(siteButton.getRecords() != null)
			{
				logger.info("Disconnecting site '" + siteButton.getSiteName() + "', removing studies from this site.");
				studiesGrid.removeStudies(siteButton.getRecords());
				siteButton.setSiteConnectivityState(SiteConnectivityState.disconnected);
			}
		}
		else if(siteButton.getSiteConnectivityState() == SiteConnectivityState.connecting)
		{
			if(siteButton.getRequest() != null)
			{
				logger.info("Cancelling site '" + siteButton.getSiteName() + "' request for studies");
				siteButton.getRequest().cancel();
				siteButton.setSiteConnectivityState(SiteConnectivityState.disconnected);
			}
		}
		else
		{
			// disconnected, connect now			
			loadSiteStudies(siteButton);
		}		
	}
	
	private void loadSiteStudies(final SiteButton siteButton)
	{		
		siteButton.setSiteConnectivityState(SiteConnectivityState.connecting);
		siteButton.setTooltip("");
		
		String transactionId = UUID.uuid();
		String userViewableImageClass = AwivHelper.getUserViewableImageClass(this.awivUserInformation);
		logger.info("Retrieving studies for patient '" + patient.getIcn() + "' from site '" + siteButton.getSiteNumber() + "' with class '" + userViewableImageClass + "' with transaction Id '" + transactionId + "'.");		
		Request request = PatientStudyServiceDao.patientStudyServiceDao.getPatientStudies(transactionId, patient.getIcn(), 
				siteButton.getSiteNumber(), patient.getAgreedSensitiveCode(), 
				userViewableImageClass,
				new AsyncCallback<AwivArtifactResults>()
		{
			
			@Override
			public void onSuccess(AwivArtifactResults result)
			{
				siteButton.setRequest(null);
				AwivStudy [] awivStudies = result.getAwivStudies();
				StudyRecord [] records = new StudyRecord[awivStudies.length];
				for(int i = 0; i < awivStudies.length; i++)
				{
					records[i] = new StudyRecord(awivStudies[i]);
				}
				siteButton.setRecords(records);
				studiesGrid.addStudies(records);
				siteButton.setStudyCount(records.length);
				siteButton.setPartial(result.isPartialResult());
				if(result.getPartialResultMessage() != null)
					logger.info("Got partial result message '" + result.getPartialResultMessage() + "' from site '" + siteButton.getSiteNumber() + "'.");
				siteButton.setSiteConnectivityState(SiteConnectivityState.connected);
			}
			
			@Override
			public void onFailure(Throwable caught)
			{
				siteButton.setRequest(null);
				siteButton.setRecords(null);
				siteButton.setSiteConnectivityState(SiteConnectivityState.disconnected);
				logger.error("Error loading studies for patient from site '" + siteButton.getSiteNumber() + "', " + caught.getMessage());
				if(!AwivExceptionHandler.handleServiceException(caught))
				{					
					if(caught instanceof AwivInsufficientPatientSensitivityException)
					{
						final AwivInsufficientPatientSensitivityException awivInsufficientPatientSensitivityException = 
							(AwivInsufficientPatientSensitivityException)caught;
						logger.warn("InsufficientPatientSensitivityException for patient at site '" + siteButton.getSiteNumber() + "'.");
						PatientSensitivityCheck.displayPatientSensitivityWarning(
								new AwivPatientSensitivity(awivInsufficientPatientSensitivityException.getCode(), 
										awivInsufficientPatientSensitivityException.getWarningMessage()), 
								new PatientSensitivityCheckCompleteEvent()
								{
									
									@Override
									public void onPatientSensitivityCheckComplete(boolean allowedToContinue, int sensitiveCode)
									{
										if(allowedToContinue)
										{
											patient.updateAgreedSensitiveCode(sensitiveCode);
											// reload the data for the site
											loadSiteStudies(siteButton);
										}
										else
										{
											logger.warn("User did not agree to sensitive level for patient, will not display studies from '" + siteButton.getSiteNumber() + "'.");
										}
									}
								});
					}
					else
					{
						
						siteButton.setTooltip(caught.getMessage());
					}
				}
			}
		});	
		siteButton.setRequest(request);
	}
	
	@Override
	public int getCreationIndex()
	{
		return 2;
	}
	
	private void clearPatient()
	{
		patientSelectedEvent.onPatientSelected(null, true);
	}
}
