using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using System.IO;
using System.Threading;
using gov.va.med.imaging.exchange.VixInstaller.business;

namespace gov.va.med.imaging.exchange.VixInstaller.ui
{
    public partial class PrerequisiteChecklistPage : InteriorWizardPage
    {
        private System.Diagnostics.Process externalProcess = null;
        private VixCredentialsDialog tomcatDialog = new VixCredentialsDialog();
        private LaurelBridgeInfoDialog laurelBridgeDialog = new LaurelBridgeInfoDialog();
        private FederationInfoDialog federationInfoDialog = new FederationInfoDialog();
        private ServiceAccountDialog serviceAccountDialog = new ServiceAccountDialog();
        private ImagingServiceAccountDialog ImagingServiceAccountDialog = new ImagingServiceAccountDialog();
        private ZFViewerInfoDialog viewerRenderInfoDialog = new ZFViewerInfoDialog();
        private MSSQLServerInstallDialog mssqlServerInstallDialog = new MSSQLServerInstallDialog();
        private bool reapplyServiceAccount = (TomcatFacade.IsActiveTomcatVersionInstalled() == false) &&
            WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName);
        private bool ImagingServiceAccountActive { get; set; }

        public PrerequisiteChecklistPage()
        {
            InitializeComponent();
        }

        public PrerequisiteChecklistPage(IWizardForm wizForm, int pageIndex)
            : base(wizForm, pageIndex)
        {
            InitializeComponent();
            if (this.reapplyServiceAccount)
            {
                this.buttonTomcatServiceAccount.Text = "Assign";
            }
        }

        //protected override String Info(String infoMessage)
        //{
        //    this.InteriorPageSubHeader = infoMessage;
        //    return infoMessage;
        //}

        #region IWizardPage Members
        public override void Initialize()
        {
            this.InitializeBusinessFacadeDelegates();
            //BusinessFacade.InfoDelegate = null; // we dont want this output in the subheader
            //LaurelBridgeFacade.InfoDelegate = null; // we dont want this output in the subheader
            // restore the page header
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();

            if (config.VixRole == VixRoleType.DicomGateway)
            {
                this.InteriorPageHeader = "Install the HDIG Prerequisites.";
            }
            else
            {
                this.InteriorPageHeader = "Install the VIX Prerequisites.";
            }

            this.SetAdministratorPrerequisiteState();
            this.SetOSPrerequisiteState();
            this.SetJavaPrerequisiteState();
            this.SetTomcatPrerequisiteState();
            this.SetLaurelBridgePrerequisiteState();
            this.SetTomcatServiceAccountPrerequisiteState();
            this.SetVixCertificatePrerequisiteState();
            this.SetZFViewerPrerequisiteState();
            
            // set the sub header for the page
            this.SetPageSubHeader();

            // set wizard form button state
            if (this.Visible)
            {
                if (this.IsComplete())
                {
                    this.WizardForm.EnableNextButton(true);
                }
                else
                {
                    this.WizardForm.EnableNextButton(false);
                }
                this.WizardForm.EnableBackButton(true);
                this.WizardForm.EnableCancelButton(true);
            }
        }

        public override bool IsComplete()
        { 
            bool isComplete = false;
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            bool devMode = this.WizardForm.IsDeveloperMode();
            if (BusinessFacade.IsLoggedInUserAnAdministrator() && BusinessFacade.IsOperatingSystemApproved(config, devMode) &&
                JavaFacade.IsJavaInstalled(devMode) && TomcatFacade.IsActiveTomcatVersionInstalled() &&
                (LaurelBridgeFacade.IsLaurelBridgeRequired(config) == false || (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled() == false && LaurelBridgeFacade.IsLaurelBridgeLicensed())) && 
                WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) &&
                this.reapplyServiceAccount == false && VixFacade.IsFederationCertificateInstalledOrNotRequired(config)
                && ZFViewerFacade.IsZFViewerInstalled())
            {
                isComplete = true;
            }
            return isComplete;
        }

        #endregion

        #region events

        private void buttonJava_Click(object sender, EventArgs e)
        {
            this.buttonJava.Enabled = false;
            this.InstallJava();
        }

        /// <summary>
        /// Get the tomcat administrator credentials from the user and then install tomcat.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonTomcat_Click(object sender, EventArgs e)
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            if (config.TomcatAdminPassword == null)
            {
                if (this.tomcatDialog.ShowDialog() != DialogResult.OK) // ask for password associated with the admin account
                {
                    return;
                }
                config.TomcatAdminPassword = this.tomcatDialog.Password;
            }
            this.buttonTomcat.Enabled = false;
            Application.DoEvents();
            this.InstallTomcat();
        }

        private void buttonLaurelBridge_Click(object sender, EventArgs e)
        {
            //TODO: need to consider license type here. If a previous installation was removed (renamed) and the 
            // old license can be used without prompting then do so. Directory will need to be renamed to a known value
            // with the version number in it. This functionality is not needed for the initial move to 3.3.22c.

            string licensePath = null;
            bool cancel = false;
            if (this.WizardForm.IsDeveloperMode())
            {
                DialogResult result = MessageBox.Show("Are you installing a MAC based license? A MAC based license is usually a temporary license or a DCF SDK license.",
                    "Developer Mode: Specify your DCF license type", MessageBoxButtons.YesNo);
                if (result == DialogResult.Yes)
                {
                    if (this.laurelBridgeDialog.ShowDialog() == DialogResult.OK)
                    {
                        licensePath = this.laurelBridgeDialog.LicensePath;
                    }
                    else
                    {
                        cancel = true;
                    }
                }
            }
            else if (DcfLicenseType.MAC == LaurelBridgeFacade.GetActiveDcfLicenseType()) // not used now but added for completeness
            {
                if (this.laurelBridgeDialog.ShowDialog() == DialogResult.OK)
                {
                    licensePath = this.laurelBridgeDialog.LicensePath;
                }
                else
                {
                    cancel = true;
                }
            }

            if (cancel == false)
            {
                this.buttonLaurelBridge.Enabled = false;
                Application.DoEvents();
                // if licensePath is null it will indicate that this is an enterprise license install
                this.InstallAndLicenseLaurelBridgeToolkit(licensePath, LaurelBridgeFacade.GetActiveDcfRootFromManifest());
            }
        }

        private void buttonTomcatServiceAccount_Click(object sender, EventArgs e)
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            bool isTomcatAccountReady = false;
            bool isImagingAccountReady = false;

            isTomcatAccountReady = this.SetTomcatServiceAccount(config);
            isImagingAccountReady = this.SetImagingServiceAccount(config);


            if (isTomcatAccountReady == true && isImagingAccountReady == true)
            {
                this.buttonTomcatServiceAccount.Enabled = false;
                Application.DoEvents();
                this.ConfigureServiceAccounts();
            }
        }

        private void buttonVixCertificate_Click(object sender, EventArgs e)
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            if (VixFacade.IsFederationCertificateInstalledOrNotRequired(config) == false)
            {
                DialogResult result = this.federationInfoDialog.ShowDialog();

                if ( result == DialogResult.OK) // Cancel will skip this
                {
                    this.buttonVixCertificate.Enabled = false;
                    Application.DoEvents();

                    string certStoreDir = VixFacade.GetVixCertificateStoreDir();
                    if (!Directory.Exists(certStoreDir))
                    {
                        Directory.CreateDirectory(certStoreDir);
                    }
                    ZipUtilities.UnZip(this.federationInfoDialog.CertificatePath, certStoreDir);
                }
                this.Initialize();
            }
        }

        private void buttonZFViewer_Click(object sender, EventArgs e)
        {
            //IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            //if (config.TomcatAdminPassword == null)
            //{
            //config.TomcatAdminPassword = this.tomcatDialog.Password;
            //}
            this.buttonZFViewer.Enabled = false;
            //Application.DoEvents();
            //call installViewerRender() method below.
            this.InstallZFViewer();
            ZFViewerFacade.CreateZFViewerWindowsService();

            //this.InstallListener();
            //this.ConfigureZFViewerWindowServices();
            //this.SetZFViewerFailureActions();
        }

        #endregion

        #region install methods
        private void ConfigureServiceAccounts()
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            try
            {
                ConfigureImagingServiceAccount(config);
                ConfigureTomcatServiceAccount(config);
                Application.DoEvents();
            }
            finally
            {
                Cursor.Current = Cursors.Default;
                this.Initialize();
            }
        }

        private void ConfigureTomcatServiceAccount(IVixConfigurationParameters config)
        {
            Cursor.Current = Cursors.WaitCursor;
            String username = TomcatFacade.TomcatServiceAccountName;
            String password = config.ApacheTomcatPassword;
            try
            {
                if (WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) == false)
                {
                    this.Logger().Info(this.Info("Creating service account " + username + "."));
                    if (WindowsUserUtilities.CreateServiceAccount(username, password, "VIX service account") == true)
                    {
                        Application.DoEvents();
                        this.Logger().Info("Setting service account " + username + " privileges.");
                        LsaUtilities.SetServiceAccountPrivileges(username);
                        Application.DoEvents();
                    }
                    else
                    {
                        DialogResult result = MessageBox.Show("The password that you supplied does not meet the complexity requirements of this computer. Please try again.",
                            "Service account creation failed!", MessageBoxButtons.OK);
                        config.ApacheTomcatPassword = null;
                    }
                }
                if (WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) == true)
                {
                    if (!this.ImagingServiceAccountActive)
                    {
                        this.Logger().Info("Setting " + TomcatFacade.TomcatServiceName + " service to run under " + username + " account.");
                        string domain = Environment.MachineName;
                        ServiceUtilities.SetServiceCredentials(TomcatFacade.TomcatServiceName, domain, username, password, ClusterFacade.IsServerClusterNode());
                        if (this.reapplyServiceAccount)
                        {
                            this.reapplyServiceAccount = false;
                        }
                        Application.DoEvents();
                    }
                }
            }
            finally
            {
                Cursor.Current = Cursors.Default;
                this.Initialize();
            }
        }


        private void ConfigureImagingServiceAccount(IVixConfigurationParameters config)
        {
            Cursor.Current = Cursors.WaitCursor;
            if (config.ImagingServiceAccountUsername == null)
            {
                return;
            }
            String[] substrings = config.ImagingServiceAccountUsername.Split('\\');
            String domain = substrings[0];
            String username = substrings[1];
            String password = config.ImagingServiceAccountPassword;

            if (WindowsUserUtilities.RemoteAccountExist(domain, username) == true)
            {
                this.Logger().Info("Setting service account " + username + " privileges.");
                LsaUtilities.SetServiceAccountPrivileges(config.ImagingServiceAccountUsername);

                this.Logger().Info("Setting " + TomcatFacade.TomcatServiceName + " service to run under " + config.ImagingServiceAccountUsername + " account.");
                ServiceUtilities.SetServiceCredentials(TomcatFacade.TomcatServiceName, domain, username, password, ClusterFacade.IsServerClusterNode());

                this.ImagingServiceAccountActive = true;
            }
        }

        private void InstallAndLicenseLaurelBridgeToolkit(string macBasedLicenseFilespec, string dcfPath)
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            try
            {
                Cursor.Current = Cursors.WaitCursor;
                this.WizardForm.EnableBackButton(false);
                this.WizardForm.EnableCancelButton(false);
                this.WizardForm.EnableNextButton(false);

                // uninstall Laurel Bridge if necessary - safety net for HDIG - VIX/CVIX will do this on the uninstall page
                if (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled())
                {
                    this.Info("Removing existing Laurel Bridge DCF toolkit version " + LaurelBridgeFacade.GetDeprecatedLaurelBridgeVersion());
                    LaurelBridgeFacade.RemoveLaurelBridgeInstallation(config);
                    LaurelBridgeFacade.SaveRenamedDeprecatedDcfRoot(config.RenamedDeprecatedDcfRoot); // save only the deprecated root to disk so we dont lose it
                }

                if (LaurelBridgeFacade.IsLaurelBridgeInstalled() == false)
                {
                    string errorMessage = "";
                    if (LaurelBridgeFacade.InstallLaurelBridgeDcfToolkit(dcfPath, ref errorMessage) == false)
                    {
                        string message = @"The Laurel Bridge DCF toolkit could not be installed due to the following error: " + errorMessage;
                        MessageBox.Show(message, "Error installing DCF toolkit", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    }
                }

                if (LaurelBridgeFacade.IsLaurelBridgeInstalled() == true && LaurelBridgeFacade.IsLaurelBridgeLicensed() == false)
                {
                    if (macBasedLicenseFilespec != null) // MAC based licensing
                    {
                        string errorMessage = "";
                        bool isLicensed = LaurelBridgeFacade.LicenseLaurelBridgeDcfToolkitWithMacBasedKeyFile(macBasedLicenseFilespec, ref errorMessage);
                        if (isLicensed == false)
                        {
                            String message = "The provided Laurel Bridge License is not valid for this computer. The dcf_info utility returned the following:\n";
                            MessageBox.Show(message + errorMessage, "Invalid Laurel Bridge License", MessageBoxButtons.OK, MessageBoxIcon.Error);
                        }
                    }
                    else
                    {
                        // License toolkit with a VA enterprise license. The UI will provide feedback about errors.
                        string macAddress = null;
                        if (config.RenamedDeprecatedDcfRoot != null)
                        {
                            macAddress = LaurelBridgeFacade.GetMacAddressFromDeprecatedKeyFile(config.RenamedDeprecatedDcfRoot);
                        }
                        // last chance at getting the mac address
                        if (macAddress == null)
                        {
                            string deprecatedDcfRoot = LaurelBridgeFacade.GetDeprecatedRenamedRootDirspec(LaurelBridgeFacade.GetActiveDcfRootFromManifest());
                            macAddress = LaurelBridgeFacade.GetMacAddressFromDeprecatedKeyFile(deprecatedDcfRoot);
                        }

                        LaurelBridgeFacade.LicenseLaurelBridgeDcfToolkitWithEnterpriseLicense(macAddress);
                    }
                }
            }
            finally
            {
                Cursor.Current = Cursors.Default;
                this.Initialize();
            }
        }

        private void InstallTomcat()
        {
                Cursor.Current = Cursors.WaitCursor;
                this.Logger().Info("Installing Apache Tomcat version " + TomcatFacade.ActiveTomcatVersion);
                this.InteriorPageHeader = "Installing Apache Tomcat version " + TomcatFacade.ActiveTomcatVersion;
                this.InteriorPageSubHeader = "This usually takes under a minute to complete.";
                this.WizardForm.EnableBackButton(false);
                this.WizardForm.EnableCancelButton(false);
                this.WizardForm.EnableNextButton(false);
                Cursor.Current = Cursors.WaitCursor;

                String exe = TomcatFacade.InstallerFilespec;
                this.Logger().Info("Tomcat Installer: " + exe);
                
                externalProcess = new System.Diagnostics.Process();
                // Handle the Exited event that the Process class fires.
                externalProcess.Exited += new System.EventHandler(this.TomcatInstallComplete);
                externalProcess.EnableRaisingEvents = true;
                externalProcess.SynchronizingObject = this;
                externalProcess.StartInfo.FileName = exe;
                externalProcess.StartInfo.Arguments = "/S";
                externalProcess.Start();
        }

        private void InstallJava()
        {
            bool devMode = this.WizardForm.IsDeveloperMode();

            if (JavaFacade.IsJavaInstalledInWrongDirectory(devMode)) // this can happen on HDIG installs when installing for the first time over a legacy gateway
            {
                this.Logger().Warn("Java Runtime Environment version " + JavaFacade.ActiveJavaVersion + " is already installed in " + JavaFacade.GetActiveJrePathLocationIndependent());
                this.Logger().Warn("Performing uninstall so that the JRE may be re-installed in " + JavaFacade.GetActiveJavaPath(true)); // true means JRE not JDK
                JavaFacade.UninstallActiveJre();
            }

            this.Logger().Info("Installing the Java Runtime Environment version " + JavaFacade.ActiveJavaVersion);
            this.Logger().Info("Installer is located at: " + JavaFacade.GetInstallerFilespec());
            this.InteriorPageHeader = "Installing the Java Runtime Environment version " + JavaFacade.ActiveJavaVersion;
            this.InteriorPageSubHeader = "This usually takes a couple of minutes to complete.";
            this.WizardForm.EnableBackButton(false);
            this.WizardForm.EnableCancelButton(false);
            this.WizardForm.EnableNextButton(false);
            Cursor.Current = Cursors.WaitCursor;

            String exe = JavaFacade.GetInstallerFilespec();
            this.Logger().Info("Installer Filespec: " + exe);

            externalProcess = new System.Diagnostics.Process();
            // Handle the Exited event that the Process class fires.
            externalProcess.Exited += new System.EventHandler(this.JavaInstallComplete);
            externalProcess.EnableRaisingEvents = true;
            externalProcess.SynchronizingObject = this;
            externalProcess.StartInfo.FileName = exe;
            
            //JRE6
            //externalProcess.StartInfo.Arguments = "/s /v \"/qn REBOOT=Suppress JAVAUPDATE=0 INSTALLDIR=\\\"" + JavaFacade.GetActiveJavaPath(true) + "\\\"";
            //JRE8
            externalProcess.StartInfo.Arguments = "/s INSTALLDIR=\\\"" + JavaFacade.GetActiveJavaPath(true) + "\\\"";
            
            //externalProcess.StartInfo.Arguments = " /s STATIC=1 INSTALLDIR=\"" + JavaFacade.GetActiveJavaPath(true) + "\"";
            this.Logger().Info("Arguments: " + externalProcess.StartInfo.Arguments);
            externalProcess.Start();
        }

        private void InstallZFViewer()
        {
            Cursor.Current = Cursors.WaitCursor;
            this.WizardForm.EnableBackButton(false);
            this.WizardForm.EnableCancelButton(false);
            this.WizardForm.EnableNextButton(false);


            //unzip Viewer and Render services
            if(!ZFViewerFacade.IsZFViewerInstalled()){
                ZFViewerFacade.InstallZFViewer();
                if (ZFViewerFacade.isZFViewerCfgBackupFilesExist())
                {
                    ZFViewerFacade.RestoreZFViewerCfgFiles();
                }
                    if (this.viewerRenderInfoDialog.ShowDialog() != DialogResult.OK)
                    {
                        return;
                    }
                }
            Cursor.Current = Cursors.Default;
            this.Initialize();
        }

        private void ReInstallZFViewer()
        {
            Cursor.Current = Cursors.WaitCursor;

            ZFViewerFacade.StopZFViewerServices();

            bool isBackedUp = ZFViewerFacade.BackupZFViewerCfgFiles();
            if (!isBackedUp)
            {
                this.Info("Failed to backup Image Viewer/Render Configuration files");
            }
            
            bool isServicesRemoved = ZFViewerFacade.DestroyZFViewerWindowsServices();
            if (!isServicesRemoved)
            {
                this.Info("Failed to remove Image Viewer and/or Image Render Windows Services.");
            }
            
            if (!ZFViewerFacade.DeleteViewerPaths())
            {
                Info("Failed to remove existing Listener installation");
            }

            if (!ZFViewerFacade.PurgeZFViewerRenderDatabaseAndCache())
            {
                Info("Failed to purge existing Image Render Database.");
                Info("Please manually purge ViewRender database");
            }

            //if (!ZFViewerFacade.PurgeZFViewerCache())
            //{
            //    this.Info("Failed to purge Viewer Cache.");
            //}
            
            ZFViewerFacade.InstallZFViewer();
            
            if (ZFViewerFacade.isZFViewerCfgBackupFilesExist())
            {
                ZFViewerFacade.RestoreZFViewerCfgFiles();
            }

            this.viewerRenderInfoDialog.ShowDialog();

            this.Info("Create Viewer/Render Services");
            ZFViewerFacade.CreateZFViewerWindowsService();
            
            Cursor.Current = Cursors.Default;
        }


        private void JavaInstallComplete(object sender, System.EventArgs e)
        {
            Process p = sender as Process;
            // TODO: check for process error
            p.Close();
            p = null;


            // append to the path
            String javaPath = JavaFacade.GetActiveJavaPath(JavaFacade.IsActiveJreInstalled());
            String path = Environment.GetEnvironmentVariable("path", EnvironmentVariableTarget.Machine);
            if (path == null)
            {
                Environment.SetEnvironmentVariable("path", javaPath, EnvironmentVariableTarget.Machine);
            }
            else // prepend to path to get so that this java is found before MS java in the Windows\system32 dir
            {
                if (!path.Contains(javaPath))
                {
                    path = javaPath + @"\bin;" + path;
                    path = path.Replace(";;", ";"); // case sensisitive
                    this.Logger().Info("Adding " + path + " to the Windows PATH environment variable");
                    Environment.SetEnvironmentVariable("path", path, EnvironmentVariableTarget.Machine);
                }
            }

            Cursor.Current = Cursors.Default;
            this.Initialize();
        }

        private void TomcatInstallComplete(object sender, System.EventArgs e)
        {
            Process p = sender as Process;
            // TODO: check for process error
            p.Close();
            p = null;
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            TomcatFacade.ConfigureTomcatUsers(config);
            TomcatFacade.ConfigureTomcatService(config);

            Cursor.Current = Cursors.Default;
            this.Initialize();
        }

        #endregion

        #region initialize methods

        private bool DisableAllPageButtons()
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            return (!BusinessFacade.IsLoggedInUserAnAdministrator() || !BusinessFacade.IsOperatingSystemApproved(config, this.WizardForm.IsDeveloperMode())) ? true : false;
        }

        private void SetAdministratorPrerequisiteState()
        {
            String loggedInUser = BusinessFacade.GetLoggedInUserName();
            // report on administrator role prerequisite
            if (BusinessFacade.IsLoggedInUserAnAdministrator())
            {
                labelUserIsAdmin.Text = loggedInUser + " has the Administrator role.";
                pictureBoxUserIsAdmin.Image = global::VixInstaller.Properties.Resources.check;
            }
            else
            {
                labelUserIsAdmin.Text = BusinessFacade.GetLoggedInUserName() + " does not have Administrator role.";
                pictureBoxUserIsAdmin.Image = global::VixInstaller.Properties.Resources.error;
            }
        }

        private void SetOSPrerequisiteState()
        {
            String info = null;
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();

            if (BusinessFacade.IsOperatingSystemApproved(ref info, config, this.WizardForm.IsDeveloperMode()))
            {
                pictureBoxOS.Image = global::VixInstaller.Properties.Resources.check;
            }
            else
            {
                pictureBoxOS.Image = global::VixInstaller.Properties.Resources.error;
            }
            labelOS.Text = info;
        }

        private void SetJavaPrerequisiteState()
        {
            String info = null;
            // report on Java install status and set install button state
            if (JavaFacade.IsJavaInstalled(ref info, this.WizardForm.IsDeveloperMode()))
            {
                pictureBoxJava.Image = global::VixInstaller.Properties.Resources.check;
                buttonJava.Enabled = false;
            }
            else
            {
                pictureBoxJava.Image = global::VixInstaller.Properties.Resources.error;
                buttonJava.Enabled = this.DisableAllPageButtons() ? false : true;
                if (buttonJava.Enabled == true)
                {
                    buttonJava.Focus();
                }
            }
            labelJava.Text = info;
        }

        private void SetTomcatPrerequisiteState()
        {
            Logger().Info("Entered SetTomcatPrerequisiteState() method.");
            // report on Tomcat install status and set install button state
            if (TomcatFacade.IsActiveTomcatVersionInstalled())
            {
                labelTomcat.Text = "Apache Tomcat version " + TomcatFacade.ActiveTomcatVersion + " is installed.";
                pictureBoxTomcat.Image = global::VixInstaller.Properties.Resources.check;
                buttonTomcat.Enabled = false;
            }
            else
            {
                labelTomcat.Text = "Apache Tomcat version " + TomcatFacade.ActiveTomcatVersion + " is not installed.";
                pictureBoxTomcat.Image = global::VixInstaller.Properties.Resources.error;
                if (JavaFacade.IsJavaInstalled(this.WizardForm.IsDeveloperMode()))
                {
                    buttonTomcat.Enabled = this.DisableAllPageButtons() ? false : true;
                    if (buttonTomcat.Enabled == true)
                    {
                        buttonTomcat.Focus();
                    }
                }
                else
                {
                    buttonTomcat.Enabled = false; // Java must be installed first
                }
            }
        }

        private void SetLaurelBridgePrerequisiteState()
        {
            // report on Laurel Bridge status
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();

            if (LaurelBridgeFacade.IsLaurelBridgeRequired(config) == false)
            {
                labelLaurelBridge.Text = "The Laurel Bridge DICOM toolkit is not required.";
                pictureBoxLaurelBridge.Image = global::VixInstaller.Properties.Resources.check;
                buttonLaurelBridge.Enabled = false;
            }
            else if (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled())
            {
                labelLaurelBridge.Text = "Deprecated Laurel Bridge DICOM toolkit is installed.";
                pictureBoxLaurelBridge.Image = global::VixInstaller.Properties.Resources.error;
                buttonLaurelBridge.Enabled = true;
            }
            else if (LaurelBridgeFacade.IsLaurelBridgeLicensed()) // installed and licensed
            {
                labelLaurelBridge.Text = "The Laurel Bridge DICOM toolkit is installed.";
                pictureBoxLaurelBridge.Image = global::VixInstaller.Properties.Resources.check;
                buttonLaurelBridge.Enabled = false;
            }
            else if (LaurelBridgeFacade.IsLaurelBridgeInstalled()) // installed but not licensed
            {
                labelLaurelBridge.Text = "The Laurel Bridge DICOM toolkit is not licensed.";
                pictureBoxLaurelBridge.Image = global::VixInstaller.Properties.Resources.error;
                buttonLaurelBridge.Enabled = true;
            }
            else
            {
                labelLaurelBridge.Text = "The Laurel Bridge DICOM toolkit is not installed.";
                pictureBoxLaurelBridge.Image = global::VixInstaller.Properties.Resources.error;
                if (JavaFacade.IsJavaInstalled(this.WizardForm.IsDeveloperMode()) && TomcatFacade.IsActiveTomcatVersionInstalled())
                {
                    buttonLaurelBridge.Enabled = this.DisableAllPageButtons() ? false : true;
                    if (buttonLaurelBridge.Enabled == true)
                    {
                        buttonLaurelBridge.Focus();
                    }
                }
                else
                {
                    buttonLaurelBridge.Enabled = false;
                }
            }
        }

        private void SetTomcatServiceAccountPrerequisiteState()
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            if (WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) && (this.reapplyServiceAccount == false))
            {
                labelTomcatServiceAccount.Text = "The service account has been configured.";
                pictureBoxTomcatServiceAccount.Image = global::VixInstaller.Properties.Resources.check;
                buttonTomcatServiceAccount.Enabled = false;
            }
            else
            {
                if (reapplyServiceAccount)
                {
                    labelTomcatServiceAccount.Text = "The service account must be reassigned to the Apache Tomcat service.";
                }
                else
                {
                    labelTomcatServiceAccount.Text = "The service account has not been created yet.";
                }
                pictureBoxTomcatServiceAccount.Image = global::VixInstaller.Properties.Resources.error;
                if (JavaFacade.IsJavaInstalled(this.WizardForm.IsDeveloperMode()) && TomcatFacade.IsActiveTomcatVersionInstalled() &&
                    (LaurelBridgeFacade.IsLaurelBridgeRequired(config) == false || (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled() == false && LaurelBridgeFacade.IsLaurelBridgeLicensed())))
                {
                    buttonTomcatServiceAccount.Enabled = this.DisableAllPageButtons() ? false : true;
                    if (buttonTomcatServiceAccount.Enabled == true)
                    {
                        buttonTomcatServiceAccount.Focus();
                    }
                }
                else
                {
                    buttonTomcatServiceAccount.Enabled = false;
                }
            }
        }

        private void SetVixCertificatePrerequisiteState()
        {
            // report on VIX Federation certificate status
            // either the certificate not required or is already installed or we know the location of the certificate installation zip filespec
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();

            if (config.VixRole == VixRoleType.DicomGateway)
            {
                labelVixCertificate.Text = "A security certificate is not required.";
                pictureBoxVixCertificate.Image = global::VixInstaller.Properties.Resources.check;
                buttonVixCertificate.Enabled = false;
            }
            else if (VixFacade.IsFederationCertificateInstalledOrNotRequired(config))
            {
                labelVixCertificate.Text = "The VIX security certificate is installed.";
                pictureBoxVixCertificate.Image = global::VixInstaller.Properties.Resources.check;
                buttonVixCertificate.Enabled = false;
            }
            else
            {
                labelVixCertificate.Text = "The VIX security certificate is not installed.";
                pictureBoxVixCertificate.Image = global::VixInstaller.Properties.Resources.error;
                if (JavaFacade.IsJavaInstalled(this.WizardForm.IsDeveloperMode()) && TomcatFacade.IsActiveTomcatVersionInstalled() &&
                    (LaurelBridgeFacade.IsLaurelBridgeRequired(config) == false || (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled() == false && LaurelBridgeFacade.IsLaurelBridgeLicensed())) && 
                    WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) && (this.reapplyServiceAccount == false))
                {
                    buttonVixCertificate.Enabled = this.DisableAllPageButtons() ? false : true;
                    if (buttonVixCertificate.Enabled == true)
                    {
                        buttonVixCertificate.Focus();
                    }
                }
                else
                {
                    buttonVixCertificate.Enabled = false;
                }
            }
        }

        private void SetZFViewerPrerequisiteState()
        {
            String info = null;
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();

            // report on ZFViewer install status and set install button state
            if (ZFViewerFacade.IsZFViewerInstalled(ref info))
            {
                labelZFViewer.Text = "Viewer and Render Services is installed.";
                pictureBoxZFViewer.Image = global::VixInstaller.Properties.Resources.check;
                buttonZFViewer.Enabled = false;
            }
            else
            {
                labelZFViewer.Text = "Viewer and Render Services is not installed.";
                pictureBoxZFViewer.Image = global::VixInstaller.Properties.Resources.error;
                if (JavaFacade.IsJavaInstalled(this.WizardForm.IsDeveloperMode()) && TomcatFacade.IsActiveTomcatVersionInstalled() &&
                    (LaurelBridgeFacade.IsLaurelBridgeRequired(config) == false || (LaurelBridgeFacade.IsDeprecatedLaurelBridgeInstalled() == false && LaurelBridgeFacade.IsLaurelBridgeLicensed())) &&
                    WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) && (this.reapplyServiceAccount == false)
                    && (VixFacade.IsFederationCertificateInstalledOrNotRequired(config)))
                {
                    buttonZFViewer.Enabled = this.DisableAllPageButtons() ? false : true;
                    if (buttonZFViewer.Enabled == true)
                    {
                        buttonZFViewer.Focus();
                    }
                }
                else
                {
                    buttonZFViewer.Enabled = false;
                }
            }
            this.labelZFViewer.Text = info;
        }


        private void SetSQLServerPrerequisiteState()
        {
            if (!MSSQLServerFacade.IsSQLServerInstalled())
            {
            String message = "The MS SQL Server Express Installation will take some time.\nPlease be patient.";
            MessageBox.Show(message, "MS SQL Server Installation", MessageBoxButtons.OK, MessageBoxIcon.Information);

                if (this.mssqlServerInstallDialog.ShowDialog() == DialogResult.OK)
                {
                    //MSSQLServerFacade.SQLServerCurrentZipFileLocalLocation = mssqlServerInstallDialog.sqlServerZipFilePath;
                    MSSQLServerFacade.InstallSQLServer();
                }
            }
        }

        private void SetListenerPrerequisiteState()
        {
            //add calls to install the Listener Tool (piggyback on the Tomcat install)
            if (!ListenerFacade.IsListenerInstalled())
            {
                if (!ListenerFacade.InstallListener())
                {
                    Logger().Info("Listener Tool failed to install.  But the Installation process will continue.");
                }
                else
                {
                    ListenerFacade.CreateListenerWindowsService();
                    //ServiceUtilities.SetServiceCredentials(ListenerFacade.LISTENER_SERVICE_NAME, "LocalService", "", ClusterFacade.IsServerClusterNode());
                }
            }
        }



        private void SetPageSubHeader()
        {
            IVixConfigurationParameters config = this.WizardForm.GetVixConfigurationParameters();
            if (this.IsComplete())
            {
                if (config.VixRole == VixRoleType.DicomGateway)
                {
                    this.InteriorPageSubHeader = "All prerequisites for the HDIG installation have been met. To continue, click Next..";
                }
                else
                {
                    this.SetSQLServerPrerequisiteState();
                    this.SetListenerPrerequisiteState();
                    this.InteriorPageSubHeader = "All prerequisites for the VIX installation have been met. To continue, click Next..";
                }
            }
            else if (this.DisableAllPageButtons())
            {
                if (!BusinessFacade.IsOperatingSystemApproved(config, this.WizardForm.IsDeveloperMode()))
                {
                    if (config.VixRole == VixRoleType.DicomGateway)
                    {
                        this.InteriorPageSubHeader = "The HDIG is only approved for installation on Windows Server 2003. Click Cancel to exit the wizard.";
                    }
                    else
                    {
                        this.InteriorPageSubHeader = "The VIX is only approved for installation on Windows Server 2003. Click Cancel to exit the wizard.";
                    }
                }
                else
                {
                    if (config.VixRole == VixRoleType.DicomGateway)
                    {
                        this.InteriorPageSubHeader = "The HDIG Installer must be run by an Administrator. " + BusinessFacade.GetLoggedInUserName() + " does not have the Administrator role on this computer. Click Cancel to exit the wizard.";
                    }
                    else
                    {
                        this.InteriorPageSubHeader = "The VIX Installer must be run by an Administrator. " + BusinessFacade.GetLoggedInUserName() + " does not have the Administrator role on this computer. Click Cancel to exit the wizard.";
                    }
                }
            }
            else
            {
                //TODO: build a sub header based on next task.
                if (this.buttonJava.Enabled)
                {
                    this.InteriorPageSubHeader = "Click Install to install the Java Runtime Environment.";
                }
                else if (this.buttonTomcat.Enabled)
                {
                    this.InteriorPageSubHeader = "Click Install to install Apache Tomcat. You will be prompted to enter a password for the Tomcat admin account.";
                }
                else if (this.buttonLaurelBridge.Enabled)
                {
                    //if (this.IsDcfToolkitRefreshRequired())
                    //{
                    //    this.InteriorPageSubHeader = "Click Install to reinstall the Laurel Bridge DICOM toolkit.";
                    //}
                    //else
                    //{
                    this.InteriorPageSubHeader = "Click Install to install the Laurel Bridge DICOM toolkit. You will be prompted to activate the toolkit license.";
                    //}
                }
                else if (this.buttonTomcatServiceAccount.Enabled)
                {
                    this.InteriorPageSubHeader = "Click Create to create and configure the Tomcat service account. You will be prompted to enter a password for the service account.";
                }
                else if (this.buttonVixCertificate.Enabled)
                {
                    this.InteriorPageSubHeader = "Click Install to specify the location of the VIX security certificate.";
                }
            }
        }

        #endregion

        private void PrerequisiteChecklistPage_Load(object sender, EventArgs e)
        {

        }

        private bool SetTomcatServiceAccount(IVixConfigurationParameters config)
        {
            bool invokeServiceDialog = false;

            if (config.ApacheTomcatPassword == null)
            {
                invokeServiceDialog = true;
            }
            else if (this.reapplyServiceAccount)
            {
                DialogResult result = MessageBox.Show("If "+TomcatFacade.TomcatServiceAccountName +" service account password has not changed then answer Yes, otherwise No.",
                    "Reapply previous service account password?", MessageBoxButtons.YesNo);
                if (result == DialogResult.No)
                {
                    invokeServiceDialog = true;
                }
            }
            else if (WindowsUserUtilities.LocalAccountExists(TomcatFacade.TomcatServiceAccountName) == false)
        {
                invokeServiceDialog = true;
            }

            if (invokeServiceDialog && this.serviceAccountDialog.ShowDialog() == DialogResult.OK) // ask for password associated with the service account
            {
                config.ApacheTomcatPassword = this.serviceAccountDialog.Password;
            }

            if (config.ApacheTomcatPassword != null)
            {
                return true;
            }
            return false;
        }


        private bool SetImagingServiceAccount(IVixConfigurationParameters config)
        {
            bool invokeServiceDialog = false;

            if (config.ImagingServiceAccountUsername == null || config.ImagingServiceAccountPassword == null)
        {
                invokeServiceDialog = true;
        }

            else if (WindowsUserUtilities.RemoteAccountExist(config.ImagingServiceAccountUsername) == true)
        {
                DialogResult result = MessageBox.Show("If " + config.ImagingServiceAccountUsername + " service account password has not changed then answer Yes, otherwise No.",
                    "Reapply previous service account password?", MessageBoxButtons.YesNo);
                if (result == DialogResult.No)
                {
                    invokeServiceDialog = true;
                }
        }

            if (invokeServiceDialog && this.ImagingServiceAccountDialog.ShowDialog() == DialogResult.OK) // ask for password associated with the imaging service account
        {
                config.ImagingServiceAccountUsername = this.ImagingServiceAccountDialog.Username;
                config.ImagingServiceAccountPassword = this.ImagingServiceAccountDialog.Password;
        }

            if (config.ImagingServiceAccountUsername != null && config.ImagingServiceAccountPassword != null)
            {
                return true;
            }
            return false;
        }
    }
}
