using System;
using System.Collections.Generic;
using System.Text;
using log4net;
using gov.va.med.imaging.exchange.VixInstaller.business;
using System.IO;
using System.ServiceProcess;
using System.Threading;

namespace VixInstallerNightly
{
    class Program
    {
        static void Main(string[] args)
        {
            if (args.Length != 1)
            {
                usage();
                System.Environment.ExitCode = 1;
                return;
            }

            string logConfigFile = Path.Combine(args[0], "VixInstallerLog.xml");
            try
            {
                log4net.Config.XmlConfigurator.Configure(new System.IO.FileInfo(logConfigFile));
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);
                System.Environment.ExitCode = 1;
                return;
            }

            VixManifest manifest = new VixManifest(args[0]);
            VixFacade.Manifest = manifest;
            TomcatFacade.Manifest = manifest;
            JavaFacade.Manifest = manifest;
            LaurelBridgeFacade.Manifest = manifest;
            BusinessFacade.Manifest = manifest;
            VixFacade.InfoDelegate = LogInfo;
            JavaFacade.InfoDelegate = LogInfo;
            LaurelBridgeFacade.InfoDelegate = LogInfo;
            System.Environment.ExitCode = 0;

            // unzip the VIX distribution payload if necessary
            String payloadJar = Path.Combine(manifest.PayloadPath, @"server\jars");
            if (!Directory.Exists(payloadJar))
            {
                String unzipPath = Path.Combine(args[0], "VIX");
                String vixZipFile = BusinessFacade.GetPayloadZipPath(args[0]);
                try
                {
                    ZipUtilities.UnZip(vixZipFile, unzipPath);
                }
                catch (Exception ex)
                {
                    LogError(ex.Message);
                    System.Environment.ExitCode = 1;
                }
            }

            // reality check against the ViX distribution zip file being built incorrectly
            if (System.Environment.ExitCode == 0)
            {
                if (!Directory.Exists(payloadJar))
                {
                    String message = "The VIX distribution files contained in the VixDistribution.zip file cannot be installed.";
                    LogError(message);
                    System.Environment.ExitCode = 1;
                }
                else
                {
                    try
                    {
                        DeployVix(args[0], manifest);
                    }
                    catch (Exception ex)
                    {
                        LogInfo(ex.Message);
                        Logger().Error(ex.Message);
                        System.Environment.ExitCode = 1;
                    }
                    finally
                    {
                        if (Directory.Exists(manifest.PayloadPath))
                        {
                            Directory.Delete(manifest.PayloadPath, true); // uninstall will clean up the zip file
                        }
                    }
                }
            }
        }

        #region private methods
        private static void DeployVix(String cwd, VixManifest manifest)
        {
            IVixConfigurationParameters config = VixConfigurationParameters.FromXml(VixFacade.GetVixConfigurationDirectory());
            UnDeployVix(cwd, manifest, config);
            if (System.Environment.ExitCode == 0)
            {
                VixFacade.InstallVix(config);
            }
        }

        private static void UnDeployVix(String cwd, VixManifest manifest, IVixConfigurationParameters config)
        {
            ServiceControllerStatus status = ServiceUtilities.GetLocalServiceStatus(TomcatFacade.TomcatServiceName);
            LogInfo("The ApacheTomcat service is " + status.ToString("g"));
            if (status != ServiceControllerStatus.Stopped)
            {
                LogInfo("Stopping ApacheTomcat service.");
                // attempt to stop the service
                ServiceUtilities.StopLocalService(TomcatFacade.TomcatServiceName);
                do
                {
                    Thread.Sleep(500);
                    status = ServiceUtilities.GetLocalServiceStatus(TomcatFacade.TomcatServiceName);
                }
                while (status == System.ServiceProcess.ServiceControllerStatus.StopPending);
                status = ServiceUtilities.GetLocalServiceStatus(TomcatFacade.TomcatServiceName);
                LogInfo("The ApacheTomcat service is " + status.ToString("g"));
                if (status != ServiceControllerStatus.Stopped)
                {
                    string message = "Cannot stop the ApacheTomcat service";
                    LogError(message);
                    System.Environment.ExitCode = 1;
                    return;
                }
                VixFacade.UndeployVixWebApplications(config);
            }
        }

        private static ILog Logger()
        {
            return LogManager.GetLogger("VixInstallerNightly");
        }

        private static String LogError(String infoMessage)
        {
            Console.WriteLine(infoMessage);
            Logger().Error(infoMessage);
            return infoMessage;
        }
        
        private static String LogInfo(String infoMessage)
        {
            Console.WriteLine(infoMessage);
            Logger().Info(infoMessage);
            return infoMessage;
        }

        private static void usage()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine("VixInstallerNightly - the VIX nightly installation utility");
            sb.AppendLine("Usage:");
            sb.AppendLine("VixInstallerNightly");
            sb.AppendLine("\tdisplays this help information");
            sb.AppendLine("VixInstallerNightly \"application dirspec\"");
            sb.AppendLine("\tinstalls the nightly build using the specified application dirspec.");
            Console.Write(sb.ToString());
        }

        #endregion

    }
}
