using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Security.Principal;
using System.Security.AccessControl;
using System.Diagnostics;
using log4net;

namespace gov.va.med.imaging.exchange.VixInstaller.business
{
    public static class AccessContolUtilities
    {
        private static ILog Logger()
        {
            return LogManager.GetLogger(typeof(AccessContolUtilities).Name);
        }

        public static void SetRootDirectoryAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            if (Directory.Exists(directory))
            {
                String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
                NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
                FileSystemRights rights = FileSystemRights.FullControl;
                InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
                PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files

                FileSystemAccessRule ace = GetExplicitDirectoryAceForAccount(accountName, directory);
                if (ace == null) // dont overwrite
                {
                    ace = new FileSystemAccessRule(ntAccount, rights,
                        inheritance, propagate, AccessControlType.Deny);
                    ApplyAceToDirectory(ace, directory);
                    Logger().Info("Directory access control applied for " + directory + ".");
                }
                else
                {
                    Logger().Info("Directory access control already exists for " + directory + ".");
                }
            }
            else
            {
                Logger().Error(directory + " does not exist - cannot apply access control.");
            }
        }
        
        public static void SetDcfDirectoryAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            if (Directory.Exists(directory))
            {
                String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
                NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
                FileSystemRights rights = FileSystemRights.Traverse | FileSystemRights.ListDirectory | FileSystemRights.ReadAttributes |
                    FileSystemRights.CreateFiles | FileSystemRights.CreateDirectories | FileSystemRights.WriteAttributes |
                    FileSystemRights.WriteExtendedAttributes | FileSystemRights.DeleteSubdirectoriesAndFiles | FileSystemRights.Delete |
                    FileSystemRights.ReadPermissions | FileSystemRights.ReadExtendedAttributes;
                InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
                PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files
                FileSystemAccessRule ace = null;

                ace = GetInheritedDirectoryAceForAccount(accountName, directory);
                if (ace != null) // sever relationship with parent directory if necessary
                {
                    //DeleteAllExplicitAce(directory);
                    DisableAclInheritance(accountName, directory); // copies inherited ACE to explicit ACE
                    ace = GetExplicitDirectoryAceForAccount(accountName, directory);
                    Debug.Assert(ace != null);
                    DeleteAccountExplicitAce(directory, ntAccount);
                    Logger().Info("Directory access control inheritance severed for " + directory + ".");
                }
                else
                {
                    Logger().Info("Directory access control inheritance already severed for " + directory + ".");
                }

                ace = GetExplicitDirectoryAceForAccount(accountName, directory);
                if (ace == null) // dont overwrite
                {
                    // build the ACE that will grant accountName privleges in the directory sub tree
                    ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);
                    // apply the new ACE
                    ApplyAceToDirectory(ace, directory);
                    Logger().Info("Directory access control applied for " + directory + ".");
                }
                else
                {
                    Logger().Info("Directory access control already exists for " + directory + ".");
                }
            }
            else
            {
                Logger().Error(directory + " does not exist - cannot apply access control.");
            }
        }

        public static void SetFullDirectoryAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
            NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
            FileSystemRights rights = FileSystemRights.FullControl;
            InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
            PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files
            FileSystemAccessRule ace = null;

            ace = GetInheritedDirectoryAceForAccount(accountName, directory);
            if (ace != null) // sever relationship with parent directory if necessary
            {
                DisableAclInheritance(accountName, directory); // copies inherited ACE to explicit ACE
                //EnsureAcesAreCanonical(directory); // ocasionally, they are not after the inheritance is severed
                ace = GetExplicitDirectoryAceForAccount(accountName, directory);
                Debug.Assert(ace != null);
                DeleteAccountExplicitAce(directory, ntAccount);
                Logger().Info("Directory access control inheritance severed for " + directory + ".");
            }
            else
            {
                Logger().Info("Directory access control inheritance already severed for " + directory + ".");
            }

            ace = GetExplicitDirectoryAceForAccount(accountName, directory);
            if (ace == null) // dont overwrite
            {
                // build the ACE that will grant accountName privleges in the directory sub tree
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);
                // apply the new ACE
                ApplyAceToDirectory(ace, directory);
                Logger().Info("Directory access control applied for " + directory + ".");
            }
            else
            {
                Logger().Info("Directory access control already exists for " + directory + ".");
            }
        }

        public static void SetJavaDirectoryAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
            NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
            FileSystemRights rights = FileSystemRights.ReadAndExecute | FileSystemRights.ListDirectory | FileSystemRights.Read;
            InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
            PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files
            FileSystemAccessRule ace = null;

            ace = GetExplicitDirectoryAceForAccount(accountName, directory);
            if (ace == null)
            {
                // build the ACE that will grant accountName privleges in the directory sub tree
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);
                // apply the new ACE
                ApplyAceToDirectory(ace, directory);
                Logger().Info("Directory access control applied for " + directory + ".");
            }
            else
            {
                Logger().Info("Directory access control already exists for " + directory + ".");
            }

        }

        public static void SetTomcatAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
            NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
            String subDirectory = null;
            FileSystemRights rights = FileSystemRights.ReadAndExecute | FileSystemRights.ListDirectory | FileSystemRights.Read;
            InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
            PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files
            FileSystemAccessRule ace = null;

            ace = GetExplicitDirectoryAceForAccount(accountName, directory);
            if (ace == null)
            {
                // build the ACE that will grant accountName privleges in the directory sub tree
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);

                // apply the new ACE to the tomcat root directory
                ApplyAceToDirectory(ace, directory);
                Logger().Info("Directory access control applied for " + directory + ".");

                // *** do the common, server, and shared sub-directories
                // first make the ACE
                rights = FileSystemRights.Write;
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Deny);
                // common - tomcat 5.5
                subDirectory = Path.Combine(directory, "common");
                if (Directory.Exists(subDirectory))
                {
                    ApplyAceToDirectory(ace, subDirectory);
                    Logger().Info("Directory access control applied for " + subDirectory + ".");
                }
                // server - tomcat 5.5
                subDirectory = Path.Combine(directory, "server");
                if (Directory.Exists(subDirectory))
                {
                    ApplyAceToDirectory(ace, subDirectory);
                    Logger().Info("Directory access control applied for " + subDirectory + ".");
                }
                // shared - tomcat 5.5
                subDirectory = Path.Combine(directory, "shared");
                if (Directory.Exists(subDirectory))
                {
                    ApplyAceToDirectory(ace, subDirectory);
                    Logger().Info("Directory access control applied for " + subDirectory + ".");
                }
                // lib - tomcat 6.0
                subDirectory = Path.Combine(directory, "lib");
                if (Directory.Exists(subDirectory))
                {
                    ApplyAceToDirectory(ace, subDirectory);
                    Logger().Info("Directory access control applied for " + subDirectory + ".");
                }
                // *** do the webapps sub-directories
                // first make the ACE
                rights = FileSystemRights.Write | FileSystemRights.Delete | FileSystemRights.DeleteSubdirectoriesAndFiles; // for clarity - rights == FileSystemRights.Write from above
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);
                // webapps
                subDirectory = Path.Combine(directory, "webapps");
                Debug.Assert(Directory.Exists(subDirectory));
                ApplyAceToDirectory(ace, subDirectory);
                Logger().Info("Directory access control applied for " + subDirectory + ".");

                // *** do the conf, logs, temp, work sub-directories
                // first make the ACE
                rights = FileSystemRights.Write | FileSystemRights.Modify;
                ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);
                // conf
                subDirectory = Path.Combine(directory, "conf");
                Debug.Assert(Directory.Exists(subDirectory));
                ApplyAceToDirectory(ace, subDirectory);
                Logger().Info("Directory access control applied for " + subDirectory + ".");
                // logs
                subDirectory = Path.Combine(directory, "logs");
                Debug.Assert(Directory.Exists(subDirectory));
                ApplyAceToDirectory(ace, subDirectory);
                Logger().Info("Directory access control applied for " + subDirectory + ".");
                // temp
                subDirectory = Path.Combine(directory, "temp");
                Debug.Assert(Directory.Exists(subDirectory));
                ApplyAceToDirectory(ace, subDirectory);
                Logger().Info("Directory access control applied for " + subDirectory + ".");
                // work
                subDirectory = Path.Combine(directory, "work");
                Debug.Assert(Directory.Exists(subDirectory));
                ApplyAceToDirectory(ace, subDirectory);
                Logger().Info("Directory access control applied for " + subDirectory + ".");
            }
            else
            {
                Logger().Info("Directory access control already exists for " + directory + ".");
            }
        }

        public static void Tester()
        {
            //FileSystemAccessRule rootExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"c:\");
            //FileSystemAccessRule rootInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"c:\");
            //FileSystemAccessRule dcfExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"c:\dcf");
            //FileSystemAccessRule dcfInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"c:\dcf");
            //FileSystemAccessRule javaExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Java\jdk1.5.0_07\jre");
            //FileSystemAccessRule javaInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Java\jdk1.5.0_07\jre");
            //FileSystemAccessRule vixExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"C:\ViX");
            //FileSystemAccessRule vixInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"C:\ViX");
            //FileSystemAccessRule installExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"c:\install");
            //FileSystemAccessRule installInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"c:\install");
            //FileSystemAccessRule tomcatExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Apache Software Foundation\Tomcat 5.5");
            //FileSystemAccessRule tomcatInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Apache Software Foundation\Tomcat 5.5");
            //FileSystemAccessRule tomcatServerExplicit = GetExplicitDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Apache Software Foundation\Tomcat 5.5\server");
            //FileSystemAccessRule tomcatServerInherit = GetInheritedDirectoryAceForAccount("apachetomcat", @"C:\Program Files\Apache Software Foundation\Tomcat 5.5\server");
            //AccessContolUtilities.SetVixDirectoryAccessControl("apachetomcat", @"c:\ViX");
        }

        #region private methods

        private static FileSystemAccessRule GetExplicitDirectoryAceForAccount(String accountName, String directory)
        {
            return GetDirectoryAceForAccountHelper(accountName, directory, true, false);
        }

        private static FileSystemAccessRule GetInheritedDirectoryAceForAccount(String accountName, String directory)
        {
            return GetDirectoryAceForAccountHelper(accountName, directory, false, true);
        }

        private static void EnsureAcesAreCanonical(String directory)
        {
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));

            DirectorySecurity sd = Directory.GetAccessControl(directory, AccessControlSections.Access);
            if (!sd.AreAccessRulesCanonical)
            {
                Logger().Info(directory + " access rules are not in canonical order - correcting");
                // put the existing aces in canonical order
                AuthorizationRuleCollection aces = sd.GetAccessRules(true, false, typeof(NTAccount)); // explicit aces only
                DirectorySecurity canonicalOrder = new DirectorySecurity();
                foreach (FileSystemAccessRule ace in aces)
                {
                    canonicalOrder.AddAccessRule(ace);
                }
                Directory.SetAccessControl(directory, canonicalOrder);
                sd = Directory.GetAccessControl(directory, AccessControlSections.Access);
                Debug.Assert(sd.AreAccessRulesCanonical);
            }
        }
        
        private static FileSystemAccessRule GetDirectoryAceForAccountHelper(String accountName, String directory, bool includeExplicit, bool includeInherited)
        {
            FileSystemAccessRule accountAce = null;
            int aceCount = 0;
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
            NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
            SecurityIdentifier ntAccountSid = (SecurityIdentifier)ntAccount.Translate(typeof(SecurityIdentifier));
            Debug.Assert(ntAccountSid.IsAccountSid());

            DirectorySecurity sd = Directory.GetAccessControl(directory, AccessControlSections.Access);
            AuthorizationRuleCollection aces = sd.GetAccessRules(includeExplicit, includeInherited, typeof(NTAccount));
            foreach (FileSystemAccessRule ace in aces)
            {

                SecurityIdentifier aceSid = (SecurityIdentifier)ace.IdentityReference.Translate(typeof(SecurityIdentifier));
                if (!aceSid.IsAccountSid())
                {
                    continue;
                }

                if (aceSid == ntAccountSid)
                {
                    accountAce = ace;
                    aceCount++;
                }
            }

            Debug.Assert(aceCount <= 1);
            return accountAce;
        }

        private static void ApplyAceToDirectory(FileSystemAccessRule ace, String directory)
        {
            DirectorySecurity sd = Directory.GetAccessControl(directory);
            sd.AddAccessRule(ace);
            Directory.SetAccessControl(directory, sd);
        }

        private static void DisableAclInheritance(String accountName, String directory)
        {
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            // break ACL inheritance with parent folder
            DirectorySecurity sd = Directory.GetAccessControl(directory);
            AuthorizationRuleCollection aces = sd.GetAccessRules(true, false, typeof(NTAccount));
            foreach (FileSystemAccessRule ace in aces)
            {
                sd.RemoveAccessRule(ace);
            }
            sd.SetAccessRuleProtection(true, true);
            Directory.SetAccessControl(directory, sd);
            EnsureAcesAreCanonical(directory); // ocasionally, they are not after the inheritance is severed
        }

        private static void DeleteAccountExplicitAce(String directory, NTAccount ntAccount)
        {
            Debug.Assert(directory != null);
            Debug.Assert(Directory.Exists(directory));
            DirectorySecurity sd = Directory.GetAccessControl(directory);
            sd.PurgeAccessRules(ntAccount);
            Directory.SetAccessControl(directory, sd);
        }


        #endregion

        #region deprecated
        private static void DeprecatedSetDcfDirectoryAccessControl(String accountName, String directory)
        {
            Debug.Assert(accountName != null);
            Debug.Assert(directory != null);
            String fullyQualifiedAccountName = Environment.MachineName + @"\" + accountName;
            NTAccount ntAccount = new NTAccount(fullyQualifiedAccountName);
            //            SecurityIdentifier sid = (SecurityIdentifier)ntAccount.Translate(typeof(SecurityIdentifier));

            // break ACL inheritance with parent folder
            DirectorySecurity sd = Directory.GetAccessControl(directory);
            sd.SetAccessRuleProtection(true, true); // break ACL inheritance, copy current permissions obtained from partent folder            
            Directory.SetAccessControl(directory, sd);

            // TODO: add exception handling
            // TODO: perform dcfDirectory integrity checks

            // build the ACE that will grant accountName privleges in the directory sub tree
            FileSystemRights rights = FileSystemRights.Traverse | FileSystemRights.ListDirectory | FileSystemRights.ReadAttributes |
                FileSystemRights.CreateFiles | FileSystemRights.CreateDirectories | FileSystemRights.WriteAttributes |
                FileSystemRights.WriteExtendedAttributes | FileSystemRights.DeleteSubdirectoriesAndFiles | FileSystemRights.Delete |
                FileSystemRights.ReadPermissions | FileSystemRights.ReadExtendedAttributes;
            InheritanceFlags inheritance = InheritanceFlags.ContainerInherit | InheritanceFlags.ObjectInherit;
            PropagationFlags propagate = PropagationFlags.None; // this folder, subfolders, and files
            FileSystemAccessRule ace = new FileSystemAccessRule(ntAccount, rights, inheritance, propagate, AccessControlType.Allow);

            // remove any existing ACEs for accountName then apply the new ACE
            sd = Directory.GetAccessControl(directory); // refresh
            sd.PurgeAccessRules(ntAccount);
            sd.AddAccessRule(ace);
            Directory.SetAccessControl(directory, sd);
        }
        #endregion
    }
}
