/**
 * 
 * Property of ISI Group, LLC
 * Date Created: Oct 6, 2014
 * Developer: Julian Werfel
 */
package gov.va.med.imaging.study.web.commands;

import gov.va.med.PatientIdentifier;
import gov.va.med.PatientIdentifierType;
import gov.va.med.URNFactory;
import gov.va.med.exceptions.RoutingTokenFormatException;
import gov.va.med.imaging.CprsIdentifier;
import gov.va.med.imaging.StudyURN;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.RoutingTokenHelper;
import gov.va.med.imaging.exchange.business.ArtifactResults;
import gov.va.med.imaging.exchange.business.Study;
import gov.va.med.imaging.exchange.business.StudyFilter;
import gov.va.med.imaging.study.web.ViewerStudyFacadeRouter;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.web.commands.WebserviceInputParameterTransactionContextField;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Julian
 *
 */
public abstract class AbtractGetStudiesByCprsIdentifierCommand<E extends Object>
extends AbstractViewerStudyWebCommand<List<Study>, E>
{
	private final String siteId;
	private final String patientIcn;
	private final String cprsIdentifierString;
	
	public AbtractGetStudiesByCprsIdentifierCommand(String siteId, String patientIcn,
		String cprsIdentifierString)
	{
		super("getStudiesByCprsIdentifier");
		this.siteId = siteId;
		this.patientIcn = patientIcn;
		this.cprsIdentifierString = cprsIdentifierString;
	}

	/**
	 * @return the siteId
	 */
	public String getSiteId()
	{
		return siteId;
	}

	/**
	 * @return the patientIdentifier
	 */
	public String getPatientIcn()
	{
		return patientIcn;
	}

	/**
	 * @return the cprsIdentifierString
	 */
	public String getCprsIdentifierString()
	{
		return cprsIdentifierString;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#executeRouterCommand()
	 */
	@Override
	protected List<Study> executeRouterCommand()
	throws MethodException, ConnectionException
	{
		ViewerStudyFacadeRouter router = getRouter();
		try
		{
			String cprsIdentifier = getCprsIdentifierString();
			if(cprsIdentifier.startsWith("urn:vastudy") || cprsIdentifier.startsWith("urn:bhiestudy"))
			{
				StudyURN studyUrn = URNFactory.create(cprsIdentifier, StudyURN.class);
				StudyFilter filter = new StudyFilter(studyUrn);
				PatientIdentifier patientIdentifier = new PatientIdentifier(patientIcn, PatientIdentifierType.icn);
				TransactionContext transactionContext = TransactionContextFactory.get();
				transactionContext.setPatientID(studyUrn.getPatientId());
				transactionContext.setUrn(cprsIdentifier);
				ArtifactResults artifactResults = router.getStudyWithImagesByStudyURN(RoutingTokenHelper.createSiteAppropriateRoutingToken(getSiteId()), 
						patientIdentifier, filter, true, false);
				List<Study> result = new ArrayList<Study>(artifactResults.getStudySetResult().getArtifacts());
				return result;
			}
			else 
			{
				return router.getStudiesByCprsIdentifier(getPatientIcn(), 
					RoutingTokenHelper.createSiteAppropriateRoutingToken(getSiteId()), 
					new CprsIdentifier(getCprsIdentifierString()));
			}
		}
		catch (RoutingTokenFormatException rtfX)
		{
			throw new MethodException("RoutingTokenFormatException, unable to get studies by CPRS identifier", rtfX);
		}
		catch(URNFormatException urnfX)
		{
			throw new MethodException("URNFormatException, unable to get studies by CPRS study URN", urnfX);
		}
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getMethodParameterValuesString()
	 */
	@Override
	protected String getMethodParameterValuesString()
	{
		return "for patient [" + getPatientIcn() + "] to site [" + getSiteId() + "], CPRS identifier [" + getCprsIdentifierString() + "]";
	}
	
	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getTransactionContextFields()
	 */
	@Override
	protected Map<WebserviceInputParameterTransactionContextField, String> getTransactionContextFields()
	{
		Map<WebserviceInputParameterTransactionContextField, String> transactionContextFields = 
			new HashMap<WebserviceInputParameterTransactionContextField, String>();
		
		transactionContextFields.put(WebserviceInputParameterTransactionContextField.quality, transactionContextNaValue);
		transactionContextFields.put(WebserviceInputParameterTransactionContextField.urn, transactionContextNaValue);
		transactionContextFields.put(WebserviceInputParameterTransactionContextField.patientId, getPatientIcn());
		transactionContextFields.put(WebserviceInputParameterTransactionContextField.queryFilter, transactionContextNaValue);
	
		return transactionContextFields;
	}
}
