/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * @date May 12, 2010
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author vhaiswbeckec
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (301) 734-0100.
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */

package gov.va.med.imaging.dicom.dcftoolkit.startup;

import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.core.router.facade.InternalContext;
import gov.va.med.imaging.core.router.storage.StorageConfigurationDataSource;
import gov.va.med.imaging.core.router.storage.providers.StorageProviderFactory;
import gov.va.med.imaging.dicom.DicomContext;
import gov.va.med.imaging.dicom.DicomRouter;
import gov.va.med.imaging.dicom.common.SpringContext;
import gov.va.med.imaging.dicom.common.stats.DicomServiceStats;
import gov.va.med.imaging.dicom.dcftoolkit.common.license.info.DCFLicenseInfo;
import gov.va.med.imaging.dicom.dcftoolkit.listen.CINFO;
import gov.va.med.imaging.dicom.dcftoolkit.listen.DicomDataService_a;
import gov.va.med.imaging.dicom.dcftoolkit.listen.Listen;
import gov.va.med.imaging.dicom.dcftoolkit.listen.Log4JOutput;
import gov.va.med.imaging.exchange.business.AuditEvent;
import gov.va.med.imaging.exchange.business.dicom.DGWEmailInfo;
import gov.va.med.imaging.exchange.business.dicom.DicomServerConfiguration;
import gov.va.med.imaging.exchange.business.dicom.InstrumentConfig;
import gov.va.med.imaging.exchange.business.dicom.ModalityConfig;
import gov.va.med.imaging.exchange.business.dicom.UIDActionConfig;
import gov.va.med.imaging.exchange.business.storage.StorageServerDatabaseConfiguration;
import gov.va.med.imaging.notifications.NotificationFacade;
import gov.va.med.imaging.notifications.NotificationTypes;
import gov.va.med.imaging.transactioncontext.InvalidTransactionContextMementoException;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;
import gov.va.med.imaging.transactioncontext.TransactionContextMemento;
import gov.va.med.server.ServerAgnosticEngine;
import gov.va.med.server.ServerAgnosticEngineAdapter;
import gov.va.med.server.ServerLifecycleEvent;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.security.Principal;
import java.util.ArrayList;
import java.util.List;

import javax.naming.NamingException;

import org.apache.catalina.Server;
import org.apache.catalina.core.StandardServer;
import org.apache.log4j.Logger;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.lbs.APC.APCException;
import com.lbs.APC_a.AppControl_a;
import com.lbs.CDS.CDSException;
import com.lbs.CDS_a.CFGDB_a;
import com.lbs.DCF.DCFVersion;
import com.lbs.DCS.DCSException;
import com.lbs.DDS.DDSException;
import com.lbs.LOG.LOGClient;
import com.lbs.LOG_a.LOGClient_a;

/**
 * This is a test implementation of a ServerAgnosticEngine, useful only for testing
 * the ServerAgnosticEngineAdapter.
 * 
 * @author vhaiswbeckec
 *
 */
public class DicomEngineAdapter
implements ServerAgnosticEngine
{
	private static Logger logger = Logger.getLogger(DicomEngineAdapter.class);
    private static Logger simpleLogger = Logger.getLogger("Summary");
	private static ServerAgnosticEngineAdapter engineAdapter;

	private static ApplicationContext springFactoryContext = null;
	
	private static List<Listen> dicomListeners = new ArrayList<Listen>();
	private static DicomEngineAdapter dicomEngineAdapter;
	
	@Override
	public void serverEvent(ServerLifecycleEvent event)
	{
		// Only respond to events if DICOM support is enabled...
		if (DicomServerConfiguration.getConfiguration().isDicomEnabled())
		{
			// If this is the after-start event for the container,
			// start up the DICOM server
			if (event.getEventType().equals(ServerLifecycleEvent.EventType.START))
			{
				try
				{
					dicomEngineAdapter = this;
					initializeSpring();
					initializeDcf();
					startDicomServices();

				}
				catch (Exception e)
				{
					logger.error("An error occurred during DICOM initialization.", e);
				}
			}

			// If this is the before-stop event for the container,
			// shut down the DICOM server.
			if (event.getEventType().equals(ServerLifecycleEvent.EventType.STOP))
			{
				logger.info("Stopping the DICOM Engine and Listeners in VISA.");
				
				stopListeners();
			}
		} 
		else 
		{
			TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
			try
			{
				DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
				if (config.isDicomDebugDumpEnabled()) 
				{
					config.setDicomDebugDumpStartMillies(System.currentTimeMillis());
				}

				logger.info("Starting Authorization process.");
				engineAdapter.authenticate(config.getAccessCodeString(), config.getVerifyCodeString().getBytes());
				loadPartialDicomConfiguration(); // this is needed for Icon creation when Listener is not started
			} 
			catch (Exception e)
			{
				logger.error("An error occurred while loading partial DICOM Configuration.", e);
			}
			finally
			{
				TransactionContextFactory.restoreTransactionContext(memento);
			}
		}
	}

	public static void startDicomServices() throws MethodException, ConnectionException, DCSException, CDSException
	{
		TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
		try
		{
			DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();

			// Authenticate on the main thread.
			Principal principal = engineAdapter.authenticate(config.getAccessCodeString(), config.getVerifyCodeString().getBytes());
			
			// If a principal was returned, authentication was successful, so we can 
			// go ahead and start the DICOM listeners. If it's null, however, the 
			// credentials were invalid. 
			if (principal != null)
			{
				logger.info("Starting the DICOM Engine and Listeners in VISA.");
				simpleLogger.info("Starting DICOM listeners.");
				
				// Clear the list of listeners, in case this is a restart.
				dicomListeners.clear();
		
				logger.info("Dicom Toolkit Layer: " + "...Starting Authorization process.");
					
				if(config.isLoadFalseStats()){
					DicomServiceStats.getInstance().loadFalseStatistics();
				}
					
				if (config.isDicomDebugDumpEnabled()) 
				{
					config.setDicomDebugDumpStartMillies(System.currentTimeMillis());
				}
		
				loadDicomGatewayConfig();
				
				// Loop over all the instruments and create a listener on each configured port
				for (InstrumentConfig instrument : DicomServerConfiguration.getConfiguration().getInstruments())
				{
					DicomServiceStats.getInstance().setCurrentPortStatus(instrument.getPort(), DicomServiceStats.DOWN);
					dicomListeners.add(createListener(instrument));
				}
		
				//P116 -Have Q/R isolated to a different port#.  
				//	Use entry from DicomServiceConfiguration to set the Port#, Site ID, and hostname.
				InstrumentConfig hDIGListener = new InstrumentConfig();
				hDIGListener.setPort(DicomServerConfiguration.getConfiguration().getDicomListenerPort());
				hDIGListener.setDescription("Primary HDIG Listener -- for non-Store SCP Service Role(s).");
				hDIGListener.setSiteId(DicomServerConfiguration.getConfiguration().getSiteId());
				hDIGListener.setHostName(DicomServerConfiguration.getConfiguration().getHostName());
				hDIGListener.setService("All");
				DicomServiceStats.getInstance().setCurrentPortStatus(hDIGListener.getPort(), DicomServiceStats.DOWN);
				dicomListeners.add(createListener(hDIGListener));
		
				//Load up the storage server configuration now as well
				StorageServerDatabaseConfiguration.setConfigurationDataSource(new StorageConfigurationDataSource());
				StorageServerDatabaseConfiguration.setProviderFactory(new StorageProviderFactory());
				StorageServerDatabaseConfiguration.getConfiguration();

				config.setDicomStarted(true);
		        try 
		        {
		        	String message = "Starting DICOM Server.";
					InternalContext.getRouter().postAuditEvent(
							null,
							new AuditEvent(AuditEvent.STARTUP,
							DicomServerConfiguration.getConfiguration().getHostName(),
							DicomServerConfiguration.getConfiguration().getApplicationName(),
							message));
				}   
		        catch (ConnectionException cX) {
		            logger.error(cX.getMessage());
		            logger.error("Exception thrown posting to Audit Log.");			
				} 
			    catch (MethodException mX) {
		            logger.error(mX.getMessage());
		            logger.error("Exception thrown posting to Audit Log.");			
				}
			}
			else
			{
				// Send notification
				String subject = "DICOM listener startup failure: invalid service account credentials";
				String message = "The system was unable to start DICOM listeners because the service account credentials" +
								 " are invalid.";
				NotificationFacade.sendNotification(NotificationTypes.InvalidServiceAccountCredentials, subject, message);
			}
		}
		finally
		{
			TransactionContextFactory.restoreTransactionContext(memento);
		}

	}
	public static void stopListeners()
	{
		simpleLogger.info("Stopping DICOM listeners.");
		DicomServerConfiguration.getConfiguration().setDicomStarted(false);
		
		// First, shut down each of the listeners
		for (Listen dicomListener : dicomListeners) 
		{
			try 
			{
				dicomListener.shutdownServer();
			} 
			catch (Exception e) 
			{
				logger.error("An error occurred while shutting down a DICOM listener.", e);
			}
			
		}

		// Now attempt to write out the application log entry for the shutdown event. First
		// authenticate on the main thread, and clear the context when done. Note that if this
		// is being called due to the detection of invalid user credentials, we won't be able to
		// write the event log entry...
		TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
		try
		{
			// Authenticate on the main thread.
			DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
			Principal principal = engineAdapter.authenticate(config.getAccessCodeString(), config.getVerifyCodeString().getBytes());
			
			// If a principal was returned, authentication was successful, so we can 
			// make the call to the audit log. 
			if (principal != null)
			{
				String message = "Shutdown DICOM Server.";
				AuditEvent auditEvent = new AuditEvent(
						AuditEvent.SHUTDOWN,
						DicomServerConfiguration.getConfiguration().getHostName(),
						DicomServerConfiguration.getConfiguration().getApplicationName(), message);
					
				InternalContext.getRouter().postAuditEvent(null, auditEvent);
			}
		}
		catch (ConnectionException cX) 
		{
			logger.error(cX.getMessage());
			logger.error("Exception thrown posting to Audit Log.", cX);
		} 
		catch (MethodException mX) 
		{
			logger.error(mX.getMessage());
			logger.error("Exception thrown posting to Audit Log.", mX);
		}
		finally
		{
			// Make sure we always restore the transaction context
			TransactionContextFactory.restoreTransactionContext(memento);
		}
	}
	private static Listen createListener(InstrumentConfig instrument) throws DCSException, CDSException
	{
		// Create a DICOM listener on the configured port and start listening
		// on a separate thread.
		logger.info("Starting a listener on port " + instrument.getPort() 
				+ " for instrument '" + instrument.getDescription() + "' (" + instrument.getNickName() + ")"
				+ " located at site '" + instrument.getSite() + "' (" + instrument.getSiteId() + ")");

		Listen dicomListener = new Listen(instrument, engineAdapter);
		Thread listenerThread = new Thread(dicomListener, "DICOM Listener on port " + instrument.getPort() + ": " + instrument.getDescription() + "' (" + instrument.getNickName() + ")");
		listenerThread.start();

		// return the listener instance so it can be stored for later reference
		return dicomListener;

	}

	/**
	 * Initialize Spring
	 */
	private void initializeSpring()
	{
		// Load spring configuration from the VixConfig directory
		String SpringXMLFile = "dicomContext.xml";
		logger.info("Dicom Toolkit Layer: " + "...loading SpringFactoryContext using " + SpringXMLFile + ".");
		springFactoryContext = new ClassPathXmlApplicationContext(SpringXMLFile);
		SpringContext.setContext(springFactoryContext);
	}

	/**
	 * Setup all DCF Adapters necessary.  This includes Configuration and Log Adapters.  This
	 * method must be called before the createListener method in this object.
	 * @throws MethodException 
	 * @throws APCException 
	 * @throws CDSException 
	 *  
	 */
	private void initializeDcf() throws MethodException, APCException, CDSException, DDSException
	{
		this.logProperties();
		if (!this.isDCFLicenseValid())
		{
			throw new MethodException("Invalid DCF License.");
		}
		
		//For JMX
		DCFLicenseInfo.getInstance();
		DicomServiceStats.getInstance();
		
		String[] emptyArgs = new String[0];

		//The following is a basic setup necessary for DCF to become operational.
		logger.debug("Setting up basic services");
		AppControl_a.setupORB(emptyArgs);
		CFGDB_a.setFSysMode(true);

		logger.debug("Setting up DCF Configuration Adapter.");
		CFGDB_a.setup(emptyArgs);

		logger.debug("Setting up DCF AppControl Adapter.");
		AppControl_a.setup(emptyArgs, CINFO.instance());
		LOGClient_a.setConsoleMode(false);

		logger.debug("Setting up DCF LOGClient Adapter.");
		LOGClient_a.setup(emptyArgs);
		LOGClient.instance().addLOGOutput(new Log4JOutput());
		
		logger.debug("Setting up DCF Data Service Adapter." );
        //Setup the DicomDataService adapter.  This is where DCF Toolkit invokes
        //  VA implemented code, via an DCF Interface, for handling of DICOM
        //  Services.
        DicomDataService_a.setup(emptyArgs);
	}
	/**
	 * Populate the DicomGatewayConfig information 
	 */
	private static void loadDicomGatewayConfig() throws MethodException, ConnectionException
	{

		DicomRouter router = DicomContext.getRouter();

		// Load DICOM Gateway Configuration from Vista HIS.
		try {
			router.getDicomGatewayConfig();
		} catch (Exception e) {
			logger.error("Failed to load Legacy DICOM Gateway Configuration.\n"
					+"This will hinder the attempt to reconstitute CT objects.");
		}
				
		String hostName = DicomServerConfiguration.getConfiguration().getHostName();

		List<InstrumentConfig> instruments = router.getDgwInstrumentList(hostName);
		logger.info("Loading DICOM instrument configuration. Found " + instruments.size() + " entries.");
		// TODO: check if number of instruments defined is greater > 0! If not shut down this app.
		if (instruments.size() < 1) {
			simpleLogger.error("There are no DICOM instruments defined.");
			throw new MethodException("DICOM instrument configuration is empty");
		}

		List<ModalityConfig> modalities = router.getDgwModalityList(hostName);
		logger.info("Loading DICOM modality configuration. Found " + modalities.size() + " entries.");
		
		DGWEmailInfo dgwEmailInfo = router.getDgwEmailInfo(hostName);
		logger.info("Loading DGW E-mail/config Info. Found " + ((dgwEmailInfo.getDgwSiteID().length() < 1)?"NO":"1") + " entry.");
		// check if site entered on M side and Java side match! If not shut down this app.
		//FIXME This should not be comparing to the Email object.  It should be between the DicomServerConfiguration and the legacy DGW.
		if (!DicomServerConfiguration.getConfiguration().getSiteId().equalsIgnoreCase(dgwEmailInfo.getDgwSiteID())) {
			throw new MethodException("HDIG site ID=" + DicomServerConfiguration.getConfiguration().getSiteId() + 
					" does not match DGW site ID=" + dgwEmailInfo.getDgwSiteID());
		}

		List<UIDActionConfig> actions = router.getDgwUIDActionTable("SOP Class", "Storage", "Storage SCP");
		logger.info("Loading the DICOM SOP Class UID Actions configuration. Found " + actions.size() + " entries.");
		
		DicomServerConfiguration.getConfiguration().setInstruments(instruments);
		DicomServerConfiguration.getConfiguration().setModalities(modalities);
		DicomServerConfiguration.getConfiguration().setDgwEmailInfo(dgwEmailInfo);
		DicomServerConfiguration.getConfiguration().setModalities(modalities);
		DicomServerConfiguration.getConfiguration().setUidActions(actions);
	}

	/**
	 * Load the SOP Class UID Actions Configuration to memory 
	 */
	private void loadPartialDicomConfiguration() throws MethodException, ConnectionException
	{
		DicomRouter router = DicomContext.getRouter();

		List<UIDActionConfig> actions = router.getDgwUIDActionTable("SOP Class", "Storage", "Storage SCP");
		logger.info("Loading the DICOM SOP Class UID Actions configuration. Found " + actions.size() + " entries.");
		
		DicomServerConfiguration.getConfiguration().setUidActions(actions);
	}

	private void logProperties()
	{
		//Get the Java version from properties and put it in the Logger.
		//  This lets us make sure which version the site is running be looking at the log.
		java.util.Properties p = System.getProperties();
		String javaName = p.getProperty("java.runtime.name");
		String javaVersion = p.getProperty("java.runtime.version");
		String classPath = p.getProperty("java.class.path");
		String osName = p.getProperty("os.name");
		String osVersion = p.getProperty("os.version");

		logger.debug("Java Runtime Name= " + javaName);
		logger.debug("Java Runtime Version= " + javaVersion);
		logger.debug("OS Name= " + osName);
		logger.debug("OS Version= " + osVersion);
		logger.debug("Java Classpath= " + classPath);

		logger.debug("DCF Version= " + DCFVersion.version());
		logger.debug("DCF BuildNumber= " + DCFVersion.buildNumber());

		simpleLogger.debug("Java Runtime Name= " + javaName);
		simpleLogger.debug("Java Runtime Version= " + javaVersion);
		simpleLogger.debug("OS Name= " + osName);
		simpleLogger.debug("OS Version= " + osVersion);
		simpleLogger.debug("Java Classpath= " + classPath);
		simpleLogger.debug("DCF Version= " + DCFVersion.version());
		simpleLogger.debug("DCF BuildNumber= " + DCFVersion.buildNumber());

	}

	public boolean isDCFLicenseValid()
	{
		// A Runtime object has methods for dealing with the OS
		Runtime r = Runtime.getRuntime();
		BufferedReader is = null;
		String aLine = "";

		// A process object tracks one external running process
		Process p = null;

		// file contains unsorted data
		try
		{
			p = r.exec("dcf_info -c");

			// getInputStream gives an Input stream connected to
			// the process p's standard output (and vice versa). We use
			// that to construct a BufferedReader so we can readLine() it.
			is = new BufferedReader(new InputStreamReader(p.getInputStream()));

			aLine = is.readLine();
		}
		catch (IOException e)
		{
			logger.error("Could not determine if DCF License is valid.", e);
		}
		
		logger.info("DCF License Check: " + aLine);
		simpleLogger.info("DCF License Check: " + aLine);
		if (aLine.equalsIgnoreCase("valid"))
		{
			return true;
		}
		return false;
	}

	@Override
	public void setServerAgnosticEngineAdapter(ServerAgnosticEngineAdapter engineAdapter)
	{
		this.engineAdapter = engineAdapter;		
	}

	public ServerAgnosticEngineAdapter getEngineAdapter() 
	{
		return engineAdapter;
	}
	
	/**
	 * Checks to see if the currently configured Access/Verify codes (in DicomServerConfig) are valid
	 * @return
	 */
	public static boolean isValidCredentials()
	{
		DicomServerConfiguration config = DicomServerConfiguration.getConfiguration();
		return isValidCredentials(config.getAccessCodeString(), config.getVerifyCodeString());
	}

	/**
	 * Checks to see if a set of access and verify codes are valid
	 * @param accessCode
	 * @param verifyCode
	 * @return
	 */
	public static boolean isValidCredentials(String accessCode, String verifyCode)
	{
		TransactionContextMemento memento = TransactionContextFactory.get().getMemento();
		try
		{
			Principal principal = engineAdapter.authenticate(accessCode, verifyCode.getBytes());
			return (principal != null);
		}
		finally
        {
			TransactionContextFactory.restoreTransactionContext(memento);
        }

	}

}
