/**
 * 
  Property of ISI Group, LLC
  Date Created: May 13, 2014
  Developer:  Julian Werfel
 */
package gov.va.med.imaging.hydra.commands;

import java.util.HashMap;
import java.util.Map;

import gov.va.med.PatientIdentifier;
import gov.va.med.RoutingToken;
import gov.va.med.exceptions.RoutingTokenFormatException;
import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exchange.ProcedureFilter;
import gov.va.med.imaging.exchange.RoutingTokenHelper;
import gov.va.med.imaging.exchange.business.ArtifactResults;
import gov.va.med.imaging.exchange.enums.PatientSensitivityLevel;
import gov.va.med.imaging.exchange.enums.ProcedureFilterMatchMode;
import gov.va.med.imaging.exchange.translation.exceptions.TranslationException;
import gov.va.med.imaging.hydra.rest.types.HydraRestTranslator;
import gov.va.med.imaging.hydra.rest.types.HydraStudiesType;
import gov.va.med.imaging.web.commands.WebserviceInputParameterTransactionContextField;

/**
 * @author Julian Werfel
 *
 */
public class GetPatientRadiologyStudiesCommand
extends AbstractHydraCommand<ArtifactResults, HydraStudiesType>
{
	private final String siteId;
	private final String patientIcn;

	/**
	 * @param methodName
	 */
	public GetPatientRadiologyStudiesCommand(String siteId, String patientIcn)
	{
		super("getPatientRadiologyStudies");
		this.siteId = siteId;
		this.patientIcn = patientIcn;
	}
	
	public String getSiteId() 
	{
		return siteId;
	}

	public String getPatientIcn() 
	{
		return patientIcn;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#executeRouterCommand()
	 */
	@Override
	protected ArtifactResults executeRouterCommand() 
	throws MethodException, ConnectionException
	{
		try
		{
			ProcedureFilter filter = new ProcedureFilter(ProcedureFilterMatchMode.existInProcedureList);	
			filter.setMaximumAllowedLevel(PatientSensitivityLevel.DISPLAY_WARNING_REQUIRE_OK);
			RoutingToken routingToken = RoutingTokenHelper.createSiteAppropriateRoutingToken(getSiteId());

			return getRouter().getStudyOnlyArtifactResultsFromSite(routingToken, 
				PatientIdentifier.icnPatientIdentifier(getPatientIcn()), filter, true, false);
		}
		catch(RoutingTokenFormatException rtfX)
		{
			throw new MethodException(rtfX);
		}
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getMethodParameterValuesString()
	 */
	@Override
	protected String getMethodParameterValuesString()
	{
		return "for patient [" + getPatientIcn() + "] from site [" + getSiteId() + "]";
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#translateRouterResult(java.lang.Object)
	 */
	@Override
	protected HydraStudiesType translateRouterResult(
			ArtifactResults routerResult)
	throws TranslationException, MethodException
	{
		return HydraRestTranslator.translateStudies(routerResult);
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getResultClass()
	 */
	@Override
	protected Class<HydraStudiesType> getResultClass()
	{
		return HydraStudiesType.class;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getTransactionContextFields()
	 */
	@Override
	protected Map<WebserviceInputParameterTransactionContextField, String> getTransactionContextFields()
	{
		Map<WebserviceInputParameterTransactionContextField, String> transactionContextFields = 
				new HashMap<WebserviceInputParameterTransactionContextField, String>();
		
		transactionContextFields.put(WebserviceInputParameterTransactionContextField.patientId, getPatientIcn());
		
		return transactionContextFields;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.web.commands.AbstractWebserviceCommand#getEntriesReturned(java.lang.Object)
	 */
	@Override
	public Integer getEntriesReturned(HydraStudiesType translatedResult)
	{
		return translatedResult == null ? 0 : translatedResult.getStudy().length;
	}

}
