/**
 * 
 */
package gov.va.med.imaging.mix.business.fhir;

import gov.va.med.GlobalArtifactIdentifier;
import gov.va.med.ImageURNFactory;
import gov.va.med.PatientIdentifier;
import gov.va.med.PatientIdentifierType;
import gov.va.med.RoutingToken;
import gov.va.med.imaging.ImageURN;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.ComparableUtil;
import gov.va.med.imaging.exchange.enums.ObjectOrigin;
// import gov.va.med.imaging.exchange.enums.ObjectStatus;

import java.io.Serializable;
import java.util.Date;

import org.apache.log4j.Logger;

/**
 * @author VACOTITTOC
 *
 * This Instance class is for FHIR ImagingStudy model support (referring to the equivalent DICOM Instance term)
 * Cardinality: Patient 1..* DiagnosticReport 0..* ImagingStudy 0..* Series 0..* Instance
 */
public class Instance 
implements Serializable, Comparable<Instance> 
{
	private static final long serialVersionUID = -4029416178345334605L;
	private final static Logger logger = Logger.getLogger(Instance.class);
	
	private GlobalArtifactIdentifier globalArtifactIdentifier; // eventually imageURN
	// fields that make up the image identifier(s)
    //protected String patientICN;
    
    protected String number; // DICOM Instance sequence number in Series
	protected String uid; // R! DICOM SOP Instance UID (for VA a '.' separated imageURN carrying site (Station) Number segment!)
    protected String sopClassUid; // R! DICOM SOP class UID -- how to get it from VA legacy? - null?
    protected String type; // type of instance (image, ...) - unused?
    protected String title; // Description of instance - unused?
//    protected String content; // attachment - contente of instance - unused!
    private ObjectOrigin objectOrigin; // might be used for VA/DOD differentiation...
    
    /**
     * 
     */
    public static Instance create(String originatingSiteId, String imageId, String studyId, 
    		PatientIdentifier patientIdentifier,
    		String imageModality) 
    throws URNFormatException
    {
    	ImageURN imageUrn = ImageURNFactory.create(originatingSiteId, imageId, studyId, 
    			patientIdentifier.getValue(), imageModality, ImageURN.class);
    	imageUrn.setPatientIdentifierTypeIfNecessary(patientIdentifier.getPatientIdentifierType());
    	
    	return create( imageUrn );
    }

    /**
     * 
     * @param globalArtifactIdentifier
     * @return
     */
    public static Instance create(GlobalArtifactIdentifier globalArtifactIdentifier)
    {
    	return new Instance(globalArtifactIdentifier);
    }
    
    /**
     * 
     * @param globalArtifactIdentifier
     * @return
     */
    public static Instance create(ImageURN imageUrn)
    {
    	return new Instance(imageUrn);
    }
    
    /**
     * 
     * @param globalArtifactIdentifier
     */
    private Instance(GlobalArtifactIdentifier globalArtifactIdentifier)
    {
    	this.globalArtifactIdentifier = globalArtifactIdentifier;
        number = title = "";
        type = "image";
    }
    
	/**
	 * @return
	 */
	public final GlobalArtifactIdentifier getGlobalArtifactIdentifier()
	{
		return globalArtifactIdentifier; 
	}

    public String getNumber() {
		return number;
	}

	public void setNumber(String number) {
		this.number = number;
	}

    /**
     * @return the title
     */
    public String getTitle() {
        return title;
    }

    /**
     * @param title the title (description) to set
     */
    public void setTitle(String title) {
        this.title = title;
    }


    /**
     * @return the image IEN if this is a VistA Imaging image, else return null
     */
    public String getIen() 
    {
        return this.globalArtifactIdentifier instanceof ImageURN ? 
        	((ImageURN)(this.globalArtifactIdentifier)).getImageId() :
        	null;
    }

    /**
     * @return the sopClassUid
     */
    public String getSopClassUid() {
        return sopClassUid;
    }

    /**
     * @param sopClassUid the sopClassUid to set
     */
    public void setSopClassUid(String sopClassUid) {
        this.sopClassUid = sopClassUid;
    }

    /**
     * @return the type
     */
    public String getType() {
        return type;
    }

    /**
     * @param type the type of inst to set
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * @return the patientICN
     */
    public String getPatientId() 
    {
    	return this.globalArtifactIdentifier instanceof ImageURN ? 
            	((ImageURN)(this.globalArtifactIdentifier)).getPatientId() :
            	null;
    }
    
    public PatientIdentifier getPatientIdentifier()
	{
		String patientId = getPatientId();
		PatientIdentifierType patientIdentifierType = getPatientIdentifierType();
		if(patientId == null || patientIdentifierType == null)
			return null;
		
		return new PatientIdentifier(patientId, patientIdentifierType);
	}
	
	public PatientIdentifierType getPatientIdentifierType()
	{
		return this.globalArtifactIdentifier instanceof ImageURN ? 
            	((ImageURN)(this.globalArtifactIdentifier)).getPatientIdentifierTypeOrDefault() :
            	null;
	}

    /**
     * @return the siteNumber
     */
    public String getSiteNumber() 
    {
        return globalArtifactIdentifier.getRepositoryUniqueId();
    }

    /**
     * @param siteNumber the siteNumber to set
     */
    //public void setSiteNumber(String siteNumber) 
    //{
    //	try
	//	{
	//		if(this.globalArtifactIdentifier != null)
	//			this.globalArtifactIdentifier = 
	//				GlobalArtifactIdentifierFactory.create(this.globalArtifactIdentifier.getHomeCommunityId(), siteNumber, this.globalArtifactIdentifier.getDocumentUniqueId());
	//		else
	//			this.globalArtifactIdentifier = 
	//				GlobalArtifactIdentifierFactory.create(null, siteNumber, null);
	//	}
	//	catch (Throwable x)
	//	{
	//		x.printStackTrace();
	//	}
    //}

	/**
	 * @return the studyIen
	 */
	public String getStudyIen() 
	{
		return this.globalArtifactIdentifier instanceof ImageURN ? 
	        	((ImageURN)(this.globalArtifactIdentifier)).getStudyId() :
	        	null;
	}

	/**
	 * @return the uid
	 */
	public String getUid() {
		return uid;
	}

	/**
	 * @param uid the uid of the Instance to set
	 */
	public void setUid(String uid) {
		this.uid = uid;
	}

	/**
	 * @return the objectOrigin
	 */
	public ObjectOrigin getObjectOrigin() {
		return objectOrigin;
	}

	/**
	 * @param objectOrigin the objectOrigin to set
	 */
	public void setObjectOrigin(ObjectOrigin objectOrigin) {
		this.objectOrigin = objectOrigin;
	}
	
	public ImageURN getImageUrn() 
	{
		return this.globalArtifactIdentifier instanceof ImageURN ?
			(ImageURN)(this.globalArtifactIdentifier) : null;
	}

	public String getImageModality() 
	{
		return this.globalArtifactIdentifier instanceof ImageURN ? 
            	((ImageURN)(this.globalArtifactIdentifier)).getImageModality() :
            	null;
	}

	@Override
	public String toString() 
	{
		String output = "";
		output += "GlobalArtifactIdentifier[" + this.getGlobalArtifactIdentifier().toString() + "]\n"; 
		output += "Instance Properties:\n";
		output += "uid: " + uid + "\n";
		output += "sopClassUid: " + sopClassUid + "\n";
		output += "type: " + type + "\n";
		output += "title: " + title + "\n";
		output += "objectOrigin: " + objectOrigin + "\n";
		
		return output;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object arg0) 
	{
		if(arg0 instanceof Instance) 
		{
			Instance that = (Instance)arg0;
			
			// if the study, group and image IENs are equal then the procedure date had better
			// be equal, nonetheless it is compared here to assure that .equals and .compareTo
			// are strictly compatible
			if(	this.globalArtifactIdentifier.equalsGlobalArtifactIdentifier(that.getGlobalArtifactIdentifier()) &&
				this.uid != null && this.uid.equals(that.uid) &&
				this.getSiteNumber() != null && this.getSiteNumber().equalsIgnoreCase(that.getSiteNumber())) 
			{
				return true;
			}
		}
		return false;
	}
	
	/**
	 * The natural sort order of Image instances is:
	 * 1.) decreasing by procedure date
	 * 2.) increasing by site number
	 * 3.) increasing by study IEN
	 * 4.) increasing by group IEN (series)
	 * 5.) increasing by image IEN
	 * 
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	@Override
	public int compareTo(Instance that) 
	{
		int cumulativeCompare = 0;
		
		cumulativeCompare = ComparableUtil.compare(this.uid, that.uid, false);
		if(cumulativeCompare != 0)
			return cumulativeCompare;
		
		cumulativeCompare = ComparableUtil.compare(this.getSiteNumber(), that.getSiteNumber(), true);
		if(cumulativeCompare != 0)
			return cumulativeCompare;
			
		cumulativeCompare = ComparableUtil.compare(this.getStudyIen(), that.getStudyIen(), true);
		if(cumulativeCompare != 0)
			return cumulativeCompare;
		
		return ComparableUtil.compare(
			this.getGlobalArtifactIdentifier().getDocumentUniqueId(), 
			that.getGlobalArtifactIdentifier().getDocumentUniqueId(), 
			true
		);
	}
	
}
