package gov.va.med.imaging.mix.business.fhir;

import gov.va.med.imaging.mix.business.ImagingStudyComparator;

import java.io.Serializable;
import java.util.Iterator;
// import java.util.Date;
import java.util.SortedSet;
import java.util.TreeSet;

/**
 * @author VACOTITTOC
 *
 * This DiagnosticReport class is for FHIR DiagnosticReport model support
 * Cardinality: Patient 1..* DiagnosticReport 0..* ImagingStudy 0..* Series 0..* Instance
 *
 */
public class DiagnosticReport implements Serializable, Iterable<ImagingStudy>, Comparable<DiagnosticReport>
{	
	private static final long serialVersionUID = -5185851367113539915L;
	private String formattedTextReport;	// The actual report (to/from ImagingStudy's additional text extension element) -- never null or empty!
	private Integer dRID;				// Diagnostic Report ID -- if null from the enterprise, it will be a hashcode of the full report
	private String status="final";		// R! registered | partial | final | corrected | appended | cancelled | entered-in-error -- always final
	private String category;			// Service category - radiology, "ophthalmology", etc. -- the proper discipline that provided the report
	private String code;				// R! Code|text for this diagnostic report  CPT Or LOINC (by order filler)
	private String subject;				// R! reference to the subject of the report -- in our use case the enterprises patient ID
	private String encounter;			// Health care event when  test ordered (date-time) - can be provided (YYYYMMDD.HHMISS), but not required
	private String effectiveDateTime;	// when study was done & read (YYYYMMDD.HHMISS) -- for date range the start value
	private String issued;				// R!  DateTime (YYYYMMDD.HHMISS) the report was released -- ask HAIMS!
	private String performer; 			// R!  Responsible Diagnostic Service  radiology, etc. -- must be identified!
	private String request; 			// What was requested - not maintained by image exchange
	private String specimen; 			// Specimens this report is based on -- - might be used for Pathology - not currently received
    private String result;				// Observations - simple, or complex nested groups - not maintained by image exchange
    private String image=null;			// Key images associated with this report {comment, link //R!} - not maintained for image exchange
	private String conclusion;			// Clinical Interp. of test results  Concise/clinically contextualized narrative interp. of the Diag.Report - ??
	private String codedDiagnosis;		// Codes for the conclusion - not maintained by image exchange
	private String presentedForm;		// Entire report as issued (attachment) -- not maintained by image exchange, see formattedTextReport field!
	private Integer numberOfImagingStudies;	// read only field -- the ImagingStudies SET size
	private SortedSet<ImagingStudy> imagingStudies = new TreeSet<ImagingStudy>(new ImagingStudyComparator());

	/**
	 * Create a new ImagingStudy
	 * @param dTextReport The actual report from FHIR ImagingStudy's additional text extension element; should never be null
	 * @param dRID Diagnostic Report ID -- auto set report hashcode if null is given -- supposed to be localID from creator enterprise
	 * @param dCategory Service category - radiology, "ophthalmology", etc.
	 * @param dCode Code|text For this diagnostic report  CPT Or LOINC (by order filler)
	 * @param dPatID Patient ID of study creator enterprise (ICN for VA, EDI_PI for DOD)
	 * @param dEnc Health care event when test ordered (date-time) - can be provided (YYYYMMDD.HHMISS), but not required
	 * @param dEDT When study was done & read (YYYYMMDD.HHMISS) -- for date range the start value
	 * @param dIssued DateTime (YYYYMMDD.HHMISS) the report was released
	 * @param dPerfr Responsible Diagnostic Service  radiology, etc.
	 * @param dReq What was requested - not maintained by image exchange
	 * @param dSpecimen Specimens this report is based on -- - could be used for Pathology
	 * @param dResult Observations - simple, or complex nested groups - not maintained by image exchange
	 * @param dConcl Clinical Interp. of test results  concise/clinically contextualized narrative interp. of the Diag.Report
	 * @param dCodeDiag Codes for the conclusion - not maintained by image exchange
	 * @param dPresForm attachment is not maintained by image exchange, see formattedTextReport field!
	 */
	public DiagnosticReport(String dTextReport, Integer dRID, String dCategory, String dCode, String dPatID, String dEnc, String dEDT, String dIssued, 
			     String dPerfr, String dReq, String dSpecimen, String dResult, String dConcl, String dCodeDiag, String dPresForm)
			     throws Exception
	{
		if ((dTextReport == null) || dTextReport.isEmpty()) {
			throw new Exception("Formatted Text Report cannot be empty");
		}
		if ((dPatID == null) || dPatID.isEmpty()) {
			throw new Exception("Report's subject(patientID) cannot be empty");
		}
		this.formattedTextReport = dTextReport;
		if (this.dRID == null) { // set report hashcode here
			this.dRID= this.formattedTextReport.hashCode();
		} else
			this.dRID = dRID;
		this.status = "final"; // hardcoded
		this.category = dCategory;
		this.code = dCode;
		this.subject = dPatID;
		this.encounter = dEnc;
		this.effectiveDateTime = dEDT;
		this.issued = dIssued;
		this.performer = dPerfr;
		this.request = dReq;
		this.specimen = dSpecimen;
		this.result = dResult;
		this.image = null; // hardcoded
		this.conclusion = dConcl;
		this.codedDiagnosis = dCodeDiag;
		this.presentedForm = dPresForm;
	}

	public String getFormattedTextReport() {
		return formattedTextReport;
	}

	public void setFormattedTextReport(String formattedTextReport) {
		this.formattedTextReport = formattedTextReport;
	}

	public Integer getdRID() {
		return dRID;
	}

	public void setdRID(Integer dRID) {
		this.dRID = dRID;
	}

	public String getStatus() {
		return status;
	}

	public String getCategory() {
		return category;
	}

	public void setCategory(String category) {
		this.category = category;
	}

	public String getCode() {
		return code;
	}

	public void setCode(String code) {
		this.code = code;
	}

	public String getSubject() {
		return subject;
	}

	public void setSubject(String subject) {
		this.subject = subject;
	}

	public String getEncounter() {
		return encounter;
	}

	public void setEncounter(String encounter) {
		this.encounter = encounter;
	}

	public String getEffectiveDateTime() {
		return effectiveDateTime;
	}

	public void setEffectiveDateTime(String effectiveDateTime) {
		this.effectiveDateTime = effectiveDateTime;
	}

	public String getIssued() {
		return issued;
	}

	public void setIssued(String issued) {
		this.issued = issued;
	}

	public String getPerformer() {
		return performer;
	}

	public void setPerformer(String performer) {
		this.performer = performer;
	}

	public String getRequest() {
		return request;
	}

	public void setRequest(String request) {
		this.request = request;
	}

	public String getSpecimen() {
		return specimen;
	}

	public void setSpecimen(String specimen) {
		this.specimen = specimen;
	}

	public String getResult() {
		return result;
	}

	public void setResult(String result) {
		this.result = result;
	}

	public String getImage() {
		return image;
	}

	public String getConclusion() {
		return conclusion;
	}

	public void setConclusion(String conclusion) {
		this.conclusion = conclusion;
	}

	public String getCodedDiagnosis() {
		return codedDiagnosis;
	}

	public void setCodedDiagnosis(String codedDiagnosis) {
		this.codedDiagnosis = codedDiagnosis;
	}

	public String getPresentedForm() {
		return presentedForm;
	}

	public void setPresentedForm(String presentedForm) {
		this.presentedForm = presentedForm;
	}

	public Integer getNumberOfImagingStudies() {
		return numberOfImagingStudies;
	}

	public SortedSet<ImagingStudy> getImagingStudies() {
		return this.imagingStudies;
	}

	public void setImagingStudies(SortedSet<ImagingStudy> imagingStudies) {
		this.imagingStudies = imagingStudies;
	}
	
	/**
	 * Add an ImagingStudy to the DiagnosticReport as a child; auto increments counter
	 * @param imagingStudy as a single ImagingStudy
	 */
	public void addimagingStudy(ImagingStudy imagingStudy) 
	{
		synchronized(this.imagingStudies)
		{
			this.imagingStudies.add(imagingStudy);
			numberOfImagingStudies++;
		}
	}

	/**
	 * Add all of the series in the given Set to the Study as children; auto increments counter
	 * @param imagingStudies is a SortedSet of ImagingStudy
	 */
	public void addimagingStudies(SortedSet<ImagingStudy> imagingStudies) 
	{
		synchronized(this.imagingStudies)
		{
			this.imagingStudies.addAll(imagingStudies);
		}
	}

	@Override
	public Iterator<ImagingStudy> iterator()
	{
		return imagingStudies.iterator();
	}



	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() 
	{
		return "ReportID[" + this.dRID + "] of patient[" + this.subject + "]; category=" + this.category + "; Order Date.Time=" + this.effectiveDateTime + 
				"; Release Date.Time=" + this.issued + "; Conclusion=" + this.conclusion + 
				"#ImagingStudies=" + this.numberOfImagingStudies;
	}

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result
				+ ((dRID == null) ? 0 : dRID.hashCode());
		result = prime * result
				+ ((subject == null) ? 0 : subject.hashCode());
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final DiagnosticReport other = (DiagnosticReport) obj;
		if (dRID == null) {
			if (other.dRID != null)
				return false;
		} else if (!issued.equals(other.issued))
			return false;
		if (subject == null) {
			if (other.subject != null)
				return false;
		} else if (!subject.equals(other.subject))
			return false;
		return true;
	}	
	
	@Override
	public int compareTo(DiagnosticReport that) 
	{
		return (this.formattedTextReport.compareTo(that.formattedTextReport));
	}
}
