package gov.va.med.imaging.storage.cache.impl.jcifs;

import gov.va.med.imaging.storage.cache.InstanceReadableByteChannel;
import gov.va.med.imaging.storage.cache.exceptions.InstanceUnavailableException;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.util.zip.Checksum;

import jcifs.smb.SmbFile;

/**
 * A class that simply wraps a FileChannel that will be used for reading,
 * and releases a lock when the channel is closed.
 * 
 * @author VHAISWBECKEC
 *
 */
public class InstanceReadableByteChannelImpl 
implements InstanceReadableByteChannel
{
	/**
	 * 
	 */
	private final JcifsByteChannelFactory factory;
	private final SmbFile file;
	private final ReadableByteChannel wrappedChannel;
	private final InputStream inStream;
	private long openedTime = 0L;					// keep this so that we could close the files ourselves if the client does not
	private long lastAccessedTime = 0L;
	private Checksum checksum;
	private StackTraceElement[] instantiatingStackTrace = null;
	
	InstanceReadableByteChannelImpl(JcifsByteChannelFactory factory, SmbFile file) 
	throws IOException, InstanceUnavailableException
	{
		this(factory, file, null);
	}

	InstanceReadableByteChannelImpl(JcifsByteChannelFactory factory, SmbFile file, Checksum checksum) 
	throws IOException, InstanceUnavailableException
	{
		this.factory = factory;
		this.file = file;
		
		this.factory.log.debug("InstanceReadableByteChannelImpl, opening '" + file.getPath() + "'" );
		// trace the channel instantiation so we can tattle-tale later if its not closed
		if(this.factory.isTraceChannelInstantiation())
			instantiatingStackTrace = Thread.currentThread().getStackTrace();
		
		inStream = file.getInputStream();
		wrappedChannel = Channels.newChannel(inStream);
		
		openedTime = System.currentTimeMillis();
		lastAccessedTime = openedTime;
		
		// try to acquire a shared lock on the file
		//lock = wrappedChannel.tryLock(0L, Long.MAX_VALUE, true);
	}
	
	SmbFile getFile()
	{
		return this.file;
	}

	public File getCacheFile()
	{
		return null;
	}
	
	public StackTraceElement[] getInstantiatingStackTrace()
	{
		return instantiatingStackTrace;
	}
	
	public java.util.zip.Checksum getChecksum()
	{
		return this.checksum;
	}

	/* (non-Javadoc)
	 * @see gov.va.med.imaging.storage.cache.impl.filesystem.InstanceReadableByteChannel#error()
	 */
	public void error() 
	throws IOException
	{
		close(true);
	}
	
	/**
	 * A normal close, the cache item contents are valid.
	 */
	public void close() 
	throws IOException
	{
		close(false);
	}
	
	private void close(boolean errorClose) 
	throws IOException
	{
		IOException ioX = null;
		this.factory.log.debug("InstanceReadableByteChannelImpl, closing '" + file.getPath() + "' " + (errorClose ? "WITH" : "without") + " delete");

		try{ wrappedChannel.close(); }
		catch(IOException e)
		{this.factory.log.warn(e);} // the channel may already be closed through some error or other timeout, log it but keep going

		try{ inStream.close(); }
		catch(IOException e)
		{this.factory.log.warn(e);} // the stream may already be closed through some error or other timeout, log it but keep going
		
		if( errorClose )
			this.file.delete();
		
		// the following two operations really must occur regardless of the 
		// success of the previous IO operations, else channels will repeatedly be closed when they are already closed
		factory.readableByteChannelClosed(this, errorClose);
		
		this.factory.log.debug("InstanceReadableByteChannelImpl - '" + file.getPath() + "' closed " + (errorClose ? "WITH" : "without") + " delete");
		if(ioX != null)
			throw ioX;
	}
	

	public boolean isOpen()
	{
		return wrappedChannel != null && wrappedChannel.isOpen();
	}

	public int read(ByteBuffer dst) 
	throws IOException
	{
		Checksum localChecksumRef = getChecksum();		// just for performance
		if(localChecksumRef != null)
			for(ByteBuffer localBuffer = dst.asReadOnlyBuffer(); localBuffer.hasRemaining(); localChecksumRef.update(localBuffer.get()) );
		
		lastAccessedTime = System.currentTimeMillis();
		return wrappedChannel.read(dst);
	}

	public long getLastAccessedTime()
	{
		return lastAccessedTime;
	}
}