package gov.va.med.imaging.ihe;

import gov.va.med.GlobalArtifactIdentifier;
import gov.va.med.GlobalArtifactIdentifierFactory;
import gov.va.med.MediaType;
import gov.va.med.PatientArtifactIdentifierImpl;
import gov.va.med.exceptions.GlobalArtifactIdentifierFormatException;
import gov.va.med.imaging.DocumentURN;
import gov.va.med.imaging.GUID;
import gov.va.med.imaging.artifactsource.ResolvedArtifactSource;
import gov.va.med.imaging.channels.ChecksumValue;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.ArtifactResultError;
import gov.va.med.imaging.exchange.business.DocumentFilter;
import gov.va.med.imaging.exchange.business.documents.Document;
import gov.va.med.imaging.exchange.business.documents.DocumentSet;
import gov.va.med.imaging.exchange.business.documents.DocumentSetResult;
import gov.va.med.imaging.exchange.enums.ArtifactResultErrorCode;
import gov.va.med.imaging.exchange.enums.ArtifactResultStatus;
import gov.va.med.imaging.exchange.translation.TranslationMethod;
import gov.va.med.imaging.ihe.exceptions.ParameterFormatException;
import gov.va.med.imaging.ihe.exceptions.TranslationException;
import gov.va.med.imaging.ihe.request.CrossGatewayQueryRequest;
import gov.va.med.imaging.ihe.request.CrossGatewayRetrieveRequest;
import gov.va.med.imaging.ihe.request.StoredQueryParameter;
import gov.va.med.imaging.ihe.request.StoredQueryParameterList;
import gov.va.med.imaging.ihe.xca.DocumentStatus;
import gov.va.med.imaging.ihe.xca.XcaTimeZone;
import gov.va.med.imaging.ihe.xds.ErrorCode;
import gov.va.med.imaging.terminology.ClassifiedValue;
import gov.va.med.imaging.terminology.CodingScheme;
import gov.va.med.imaging.terminology.SchemeTranslationSPI;
import gov.va.med.imaging.terminology.SchemeTranslationServiceFactory;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;

import java.math.BigInteger;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;
import org.apache.log4j.Logger;
import org.ihe.iti.xdsb.xmlbeans.RetrieveDocumentSetRequestDocument;
import org.ihe.iti.xdsb.xmlbeans.RetrieveDocumentSetRequestType;
import org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryRequestDocument;
import org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument;
import org.oasis.ebxml.regrep.query.xmlbeans.ResponseOptionType;
import org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument.AdhocQueryResponse;
import org.oasis.ebxml.regrep.rim.xmlbeans.*;
import org.oasis.ebxml.regrep.rs.xmlbeans.RegistryErrorDocument;
import org.oasis.ebxml.regrep.rs.xmlbeans.RegistryErrorDocument.RegistryError;
import org.oasis.ebxml.regrep.rs.xmlbeans.RegistryErrorListDocument.RegistryErrorList;

/**
 * The translator from IHE XCA form into internal form. This includes
 * translation from the code scheme used in the SOAP request to the internal
 * codes.
 * 
 * @author vhaiswbeckec
 * 
 */
public class XCATranslatorXmlBeans
extends AbstractXCATranslator
{
	private static final String DEFAULT_LANGUAGE = "en-us";
	private static final String DEFAULT_CHARSET = "UTF-8";

	public static Logger logger = Logger.getLogger(XCATranslatorXmlBeans.class);

	private static ReferenceURI responseStatusSuccess;
	private static ReferenceURI responseStatusFailure;
	private static ReferenceURI responseStatusPartialSuccess;

	private static ReferenceURI documentStatusSubmitted;
	private static ReferenceURI documentStatusApproved;
	private static ReferenceURI documentStatusDeprecated;

	private static ReferenceURI objectTypeClassification;

	static
	{
		try
		{
			responseStatusSuccess = ReferenceURI.Factory.newInstance();
			responseStatusFailure = ReferenceURI.Factory.newInstance();
			responseStatusPartialSuccess = ReferenceURI.Factory.newInstance();

			documentStatusSubmitted = ReferenceURI.Factory.newInstance();
			documentStatusApproved = ReferenceURI.Factory.newInstance();
			documentStatusDeprecated = ReferenceURI.Factory.newInstance();

			objectTypeClassification = ReferenceURI.Factory.newInstance();
		}
		catch (Throwable t)
		{
			t.printStackTrace();
			throw new ExceptionInInitializerError(t);
		}

		responseStatusSuccess.setStringValue( EbXMLRsUUID.RESPONSE_STATUS_SUCCESS.getUrnAsString() );
		responseStatusFailure.setStringValue(EbXMLRsUUID.RESPONSE_STATUS_FAILURE.getUrnAsString());
		responseStatusPartialSuccess.setStringValue(EbXMLRsUUID.RESPONSE_STATUS_PARTIALSUCCESS.getUrnAsString());

		documentStatusSubmitted.setStringValue(EbXMLRimUUID.DOCUMENT_STATUS_SUBMITTED.getUrnAsString());
		documentStatusApproved.setStringValue(EbXMLRimUUID.DOCUMENT_STATUS_APPROVED.getUrnAsString());
		documentStatusDeprecated.setStringValue(EbXMLRimUUID.DOCUMENT_STATUS_DEPRECATED.getUrnAsString());

		objectTypeClassification.setStringValue("urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification");
	}

	/**
	 * 
	 * @return
	 */
//	public static boolean isHaimsCompatibilityMode()
//	{
//		TransactionContext tc = TransactionContextFactory.get();
//		return tc == null || tc.isHaims1Client() == null ? false : tc.isHaims1Client().booleanValue();
//	}

	/**
	 * 
	 */
	public XCATranslatorXmlBeans()
	{
		super();
	}

	/**
	 * @param patientIcn
	 * @param filter
	 * @return
	 */
	public static AdhocQueryRequestDocument translate(DocumentFilter documentFilter) 
	throws MalformedURLException
	{
		String queryId = (new GUID()).toString();
		AdhocQueryRequestDocument queryRequestDoc = AdhocQueryRequestDocument.Factory.newInstance();

		AdhocQueryRequestDocument.AdhocQueryRequest queryRequest = 
			queryRequestDoc.addNewAdhocQueryRequest();

		org.oasis.ebxml.regrep.query.xmlbeans.ResponseOptionType responseOptionType = 
			queryRequest.addNewResponseOption();
		// not sure about these two parameters
		responseOptionType.setReturnComposedObjects(true);
		responseOptionType.setReturnType(ResponseOptionType.ReturnType.LEAF_CLASS);
		queryRequest.setResponseOption(responseOptionType);

		AdhocQueryType adhocQuery = queryRequest.addNewAdhocQuery();
		adhocQuery.setId(queryId);
		populateAdhocQuerySlots(adhocQuery, documentFilter);

		// adhocQueryType.setLid(new URI("http://localhost/testclient"));
		// find document query Id
		adhocQuery.setId(IheUUID.FIND_DOCUMENTS_UUID.getUrnAsString());
		
		// home community Id - probably should not be this value since this is
		// VA...
		// don't set home community Id for document query call
		// adhocQueryType.setHome(new URI("urn:oid:2.16.840.1.113883.3.166"));

		// 
		queryRequest.setMaxResults(BigInteger.valueOf(documentFilter.getMaxResultsCount()));

		// not sure if putting data in the correct place
		// adhocQueryType.setSlot(param)

		queryRequest.setId("http://testUri");

		return queryRequestDoc;
	}

	@TranslationMethod(unmatchedMethod=true)
	public static CrossGatewayRetrieveRequest[] translate(RetrieveDocumentSetRequestDocument retrieveDocumentSetRequest)
	{
		RetrieveDocumentSetRequestType docSetRequest = retrieveDocumentSetRequest.getRetrieveDocumentSetRequest();
		return translate(docSetRequest);
	}
	
	@TranslationMethod(unmatchedMethod=true)
	public static CrossGatewayRetrieveRequest[] translate(
		RetrieveDocumentSetRequestType docSetRequest)
	{
		RetrieveDocumentSetRequestType.DocumentRequest[] documentRequests = docSetRequest.getDocumentRequestArray();
		return translate(documentRequests);
	}
	
	/**
	 * @param documentRequests
	 * @return
	 */
	public static CrossGatewayRetrieveRequest[] translate(RetrieveDocumentSetRequestType.DocumentRequest[] documentRequests)
	{
		CrossGatewayRetrieveRequest[] result = new CrossGatewayRetrieveRequest[documentRequests.length];
		
		int index=0;
		for(RetrieveDocumentSetRequestType.DocumentRequest documentRequest : documentRequests)
			result[index++] = translate(documentRequest);
		
		return result;
	}

	/**
	 * @param documentRequest
	 * @return
	 */
	public static CrossGatewayRetrieveRequest translate(
		RetrieveDocumentSetRequestType.DocumentRequest documentRequest)
	{
		String homeCommunityId = documentRequest.getHomeCommunityId();
		String repositoryUniqueId = documentRequest.getRepositoryUniqueId();
		String documentUniqueId = documentRequest.getDocumentUniqueId();
		
		CrossGatewayRetrieveRequest request = new CrossGatewayRetrieveRequest(
			homeCommunityId, repositoryUniqueId, documentUniqueId);
		
		return request;
	}

	/**
	 * 
	 * @param filter
	 * @return
	 */
	private static void populateAdhocQuerySlots(AdhocQueryType adhocQuery, DocumentFilter filter)
	{
		SlotType1 patientIdSlot = adhocQuery.addNewSlot();
		// added addition of HL7 formatting to patient ID
		String hl7PatientId = AbstractXCATranslator.createHL7PatientID(filter.getPatientId());
		populateSingleValueSlot(patientIdSlot,
			FindDocumentStoredQueryParameterNames.patientId.toString(),
			hl7PatientId,
			SlotFormat.STRING
		);

		if (filter.getCreationTimeFrom() != null)
		{
			SlotType1 creationTimeFromSlot = adhocQuery.addNewSlot();
			populateSingleValueSlot(creationTimeFromSlot,
				FindDocumentStoredQueryParameterNames.creationTimeFrom.toString(), 
				translateToIHEDate(filter.getCreationTimeFrom().getTime()),
				SlotFormat.DATE
			);
		}
		if (filter.getCreationTimeTo() != null)
		{
			SlotType1 creationTimeToSlot = adhocQuery.addNewSlot();
			populateSingleValueSlot(creationTimeToSlot,
				FindDocumentStoredQueryParameterNames.creationTimeTo.toString(), 
				translateToIHEDate(filter.getCreationTimeTo().getTime()),
				SlotFormat.DATE
			);
		}

		SlotType1 entryStatusSlot = adhocQuery.addNewSlot();
		populateSingleValueSlot(entryStatusSlot,
			FindDocumentStoredQueryParameterNames.entryStatus.toString(),
			DocumentStatus.approved.toString(),
			SlotFormat.STRING
		);

		return;
	}
	
	// Temp CVIX fix
//	private static String getPatientId(DocumentFilter filter)
//	{
//		if (("200".equals(filter.getSiteNumber())) && (filter.isUseAlternatePatientId()))
//		{
//			//translate through properties file
//			Properties patientIdTranslator = loadPatientIdTranslationFile();
//			String mappedPatientId = patientIdTranslator.getProperty(filter.getPatientId(), filter.getPatientId());
//			logger.info(("Patient id translated from " + filter.getPatientId() + " to " + mappedPatientId));
//			return mappedPatientId;
//		}	
//		else
//		{
//			return filter.getPatientId();
//		}
//	}
//
//	// Temp CVIX fix
//	private static Properties loadPatientIdTranslationFile()
//	{
//		Properties props = new Properties();
//		try
//		{
//			String patientIdPropertiesFile = "";
//			String vixConfigDir = System.getenv("vixconfig");
//			if(vixConfigDir == null)
//			{
//				patientIdPropertiesFile = "C:\\VixConfig\\PatientIdTranslation.properties";
//			}
//			else
//			{
//				patientIdPropertiesFile = vixConfigDir + "\\PatientIdTranslation.properties"; 
//			}					
//			props.load(new FileInputStream(patientIdPropertiesFile));
//		}
//		catch (IOException ioe)
//		{
//			ioe.printStackTrace();
//		}
//		return props;
//	}

	/**
	 * @param queryResponse
	 * @param patientIcn
	 * @param resolvedArtifactSource
	 * @return
	 * @throws TranslationException
	 */
	@TranslationMethod(unmatchedMethod=true)
	public static DocumentSetResult translate(
		AdhocQueryResponseDocument queryResponseDocument, 
		String patientIcn,
		ResolvedArtifactSource resolvedArtifactSource,
		boolean allowPartialSuccess) 
	throws TranslationException
	{
		logger.info("translating document query response into Document Sets");
		AdhocQueryResponse queryResponse = queryResponseDocument.getAdhocQueryResponse();
		queryResponse.getResponseSlotList();
		//BigInteger resultCount = queryResponse.getTotalResultCount();
		//BigInteger startIndex = queryResponse.getStartIndex();
		String responseStatus = queryResponse.getStatus();
		
		SortedSet<DocumentSet> result = new TreeSet<DocumentSet>();
		DocumentSet documentSet = null;
		ArtifactResultStatus artifactResultStatus = ArtifactResultStatus.fullResult;
		List<ArtifactResultError> artifactResultErrors = null;
		
		logger.info("Query response has status of '" + responseStatus + "'.");
		TransactionContext transactionContext = TransactionContextFactory.get();
		transactionContext.addDebugInformation("Query has a response status of '" + responseStatus + "'.");
		if( EbXMLRsUUID.RESPONSE_STATUS_SUCCESS.getUrnAsString().equals(responseStatus) ||
			(allowPartialSuccess && EbXMLRsUUID.RESPONSE_STATUS_PARTIALSUCCESS.getUrnAsString().equals(responseStatus)) )
		{
			RegistryObjectListType rolt = queryResponse.getRegistryObjectList();
			if(rolt != null)
			{
				IdentifiableType[] documents = rolt.getIdentifiableArray();
				if (documents != null)
				{
					for (IdentifiableType it : documents)
					{
						if(ExtrinsicObjectType.class.isInstance(it) )
						{
							ExtrinsicObjectType eot = (ExtrinsicObjectType) it;
							Document document = translate(eot, patientIcn, resolvedArtifactSource);
	
							if (documentSet == null)
							{
								documentSet = new DocumentSet(
									resolvedArtifactSource.getArtifactSource().getHomeCommunityId(),
									resolvedArtifactSource.getArtifactSource().getRepositoryId(),
									document.getDocumentSetIen());
								documentSet.setPatientIcn(patientIcn);
							}
							documentSet.add(document);
						}
					}
				}
				else
					logger.info("Query response has no document sets included.");
			}
			else
				logger.info("Query response has no registry object list element.");
			
			if(EbXMLRsUUID.RESPONSE_STATUS_PARTIALSUCCESS.getUrnAsString().equals(responseStatus))
				artifactResultStatus = ArtifactResultStatus.partialResult;			

			// if partial success is allowed then we may have errors listed here
			// that the client should know about
			RegistryErrorList rel = queryResponse.getRegistryErrorList();
			if(rel != null)
			{
				artifactResultErrors = new ArrayList<ArtifactResultError>();
				for( RegistryErrorDocument.RegistryError registryError : rel.getRegistryErrorArray() )
				{
					/* 
					 * This attribute indicates the severity of error that was encountered. The value of
					 * the severity attribute MUST be a reference to a ClassificationNode within the canonical
					 * ErrorSeverityType ClassificationScheme as described in [ebRIM].
					 * The following canonical values are defined for the ErrorSeverityType ClassificationScheme:
					 * Error  An Error is a fatal error encountered by the registry while processing a
					 * request. A registry MUST return a status of Failure in the RegistryResponse for a
					 * request that encountered Errors during its processing.
					 * Warning  A Warning is a non-fatal error encountered by the registry while
					 * processing a request. A registry MUST return a status of Success in the
					 * RegistryResponse for a request that only encountered Warnings during its processing
					 * and encountered no Errors.
					registryError.getSeverity();
					
					 * This attribute specifies a string that indicated where in the code the error occurred. 
					 * Implementations SHOULD show the stack trace and/or, code module and line number information 
					 * where the error was encountered in code.
					registryError.getLocation();
					
					 * This attribute specifies a string that indicates contextual text that provides
					 * additional detail to the errorCode. For example, if the errorCode is
					 * InvalidRequestException the codeContext MAY provide the reason why the request was
					 * invalid. 
					registryError.getCodeContext();
					
					 * This attribute specifies a string that indicates the error that was encountered.
					 * Implementations MUST set this attribute to the Exception or Error as defined by this
					 * specification (e.g. InvalidRequestException).
					 * 
					 * AuthorizationException: Indicates that the requestor attempted to perform an operation 
					 * for which he or she was not authorized.
					 * InvalidRequestException: Indicates that the requestor attempted to perform an 
					 * operation that was semantically invalid.
					 * SignatureValidationException: Indicates that a Signature specified for the request failed 
					 * to validate.
					 * TimeoutException: Indicates that the processing time for the request exceeded a
					 * registry specific limit.
					 * UnsupportedCapabilityException: 
					registryError.getErrorCode();
					
					registryError.getStringValue();
					
                     * This attribute indicates the severity of error that was encountered. The value of
                     * the severity attribute MUST be a reference to a ClassificationNode within the canonical
                     * ErrorSeverityType ClassificationScheme as described in [ebRIM].
                     * The following canonical values are defined for the ErrorSeverityType ClassificationScheme:
                     * Error  An Error is a fatal error encountered by the registry while processing a
                     * request. A registry MUST return a status of Failure in the RegistryResponse for a
                     * request that encountered Errors during its processing.
                     * Warning  A Warning is a non-fatal error encountered by the registry while
                     * processing a request. A registry MUST return a status of Success in the
                     * RegistryResponse for a request that only encountered Warnings during its processing
                     * and encountered no Errors.
                    registryError.getSeverity();
                    
                     * This attribute specifies a string that indicated where in the code the error occurred. 
                     * Implementations SHOULD show the stack trace and/or, code module and line number information 
                     * where the error was encountered in code.
                    registryError.getLocation();
                    
                     * This attribute specifies a string that indicates contextual text that provides
                     * additional detail to the errorCode. For example, if the errorCode is
                     * InvalidRequestException the codeContext MAY provide the reason why the request was
                     * invalid. 
                    registryError.getCodeContext();
                    
                     * This attribute specifies a string that indicates the error that was encountered.
                     * Implementations MUST set this attribute to the Exception or Error as defined by this
                     * specification (e.g. InvalidRequestException).
                     * 
                     * AuthorizationException: Indicates that the requestor attempted to perform an operation 
                     * for which he or she was not authorized.
                     * InvalidRequestException: Indicates that the requestor attempted to perform an 
                     * operation that was semantically invalid.
                     * SignatureValidationException: Indicates that a Signature specified for the request failed 
                     * to validate.
                     * TimeoutException: Indicates that the processing time for the request exceeded a
                     * registry specific limit.
                     * UnsupportedCapabilityException:
 
                    registryError.getErrorCode();
                     */

					
					artifactResultErrors.add(new XcaArtifactResultError(registryError));
				}
			}
		}
		else
		{
			logger.info("Query response has status of error");
			RegistryErrorList errorListType = queryResponse.getRegistryErrorList();
			artifactResultErrors = new ArrayList<ArtifactResultError>();
			if(errorListType != null && errorListType.getRegistryErrorArray() != null)
				for(RegistryErrorDocument.RegistryError error : errorListType.getRegistryErrorArray() )
				{
					logger.info(
						error.getSeverity() + 
						" :" + (error.getErrorCode() == null ? "<null error code>" : error.getErrorCode()) + 
						"@" + (error.getCodeContext() == null ? "<null code context>" : error.getCodeContext()) + 
						"'" + error.getStringValue() + "'.");
					artifactResultErrors.add(new XcaArtifactResultError(error));
				}
			else
				logger.info("Error status in query response.  No further information was provided.");
		}

		if (documentSet != null)
			result.add(documentSet);

		return DocumentSetResult.create(result, artifactResultStatus, artifactResultErrors);
	}

	/**
	 * The various identifiers must end up in the correct fields:
	 * 
	 * For a client (initiatingGateway) to successfully make a retrieve request back to 
	 * this server (respondingGateway) requires that it have the home community, the repository
	 * and the document identifiers.
	 * The home community ID is an attribute of the extrinsic object element.
	 * The repository and document IDs are external references.
	 * NOTE: the patient ID is not part of the retrieve request, for VA documents that identifier
	 * is included in the document ID.
	 * 
	 * documentSetIen = document Id from XCA query
	 * 
	 * @param extrinsicObject
	 * @return
	 * @throws XCADataSourceTranslatorException
	 */
	@TranslationMethod(unmatchedMethod=true)
	public static Document translate(
		ExtrinsicObjectType extrinsicObject,
		String patientIcn, 
		ResolvedArtifactSource resolvedArtifactSource)
	throws TranslationException
	{
		String documentSetId = "XCA";	//		
		SchemeTranslationServiceFactory translatorFactory = SchemeTranslationServiceFactory.getFactory();

		String name = null;
		String description = null;

		// one of the three identifiers we need to build a GlobalArtifactIdentifier
		String homeCommunityId = extrinsicObject.getHome();
		
		// The internal (to the XML) identifier of the extrinsic object.  This is NOT the
		// document identifier, but it might be.
		// As if any further proof is needed that XDS is a hacked over kludge comes this
		// gem from the RIM schema as provided from IHE.
        // "If id is provided and is not in proper URN syntax then it is used for
        // linkage within document and is ignored by the registry. In this case the
        // registry generates a UUID URN for id attribute.
        // id must not be null when object is retrieved from the registry."
		String documentReferenceIdentifier = extrinsicObject.getId();
		
		MediaType mediaType = MediaType.lookup(extrinsicObject.getMimeType());
		String objectType = extrinsicObject.getObjectType();		// the "objectType" attribute
		String status = extrinsicObject.getStatus();
		
		// is the extrinsic type really a document
		if( objectType != null && ! IheUUID.DOC_ENTRY.getUrnAsString().equals(objectType) )
			throw new TranslationException("The object type '" + objectType + "' is not the accepted value for a document entry");
		// is the extrinsic object in an Approved state
		if( status != null && ! EbXMLRimUUID.DOCUMENT_STATUS_APPROVED.getUrnAsString().equals(status) )
			throw new TranslationException("The status of the extrinsic object '" + documentReferenceIdentifier + 
				"' is '" + extrinsicObject.getStatus() + 
				"' but must be '" + EbXMLRimUUID.DOCUMENT_STATUS_APPROVED.getUrnAsString() + "'.");

		// get all of the slot values into a map
		SlotType1[] slots = extrinsicObject.getSlotArray();
		XCAQueryResponseDocumentProperties properties = XCAQueryResponseDocumentProperties.translate(slots);
		
		//String sourcePatientID = properties.getSourcePatientId();
		
		if(extrinsicObject.getName() != null && 
			(extrinsicObject.getName().getLocalizedStringArray() != null) && 
			(extrinsicObject.getName().getLocalizedStringArray().length > 0))
				name = extrinsicObject.getName().getLocalizedStringArray(0).getValue();
		
		if(extrinsicObject.getDescription() != null && 
			(extrinsicObject.getDescription().getLocalizedStringArray() != null) && 
			(extrinsicObject.getDescription().getLocalizedStringArray().length > 0))
					description = extrinsicObject.getDescription().getLocalizedStringArray(0).getValue();
		
		ExternalReferenceValueObjectList externalReferences = translate(extrinsicObject.getExternalIdentifierArray());

		String repositoryId = properties.getRepositoryId();
		//externalReferences.getExternalIdentifierValue(REPOSITORY_UNIQUE_ID);
			
		if(repositoryId == null)
			logger.error("Repository ID (external identifier 'XDSDocumentEntry.repositoryUniqueId') is null.  Retrieve needs this to work.");
		
		GlobalArtifactIdentifier documentGlobalIdentifier = null;
		try
		{
			// do not leave this here, DELETE THIS NEXT LINE BEFORE PRODUCTION!!!!
			//repositoryId = repositoryId == null ? "2.16.840.1.113883.3.198.2" : repositoryId;
			documentGlobalIdentifier = GlobalArtifactIdentifierFactory.create(homeCommunityId, repositoryId, documentReferenceIdentifier);
			// a PatientArtifactIdentifier includes the patient ID as an extra identifier
			documentGlobalIdentifier = PatientArtifactIdentifierImpl.create(documentGlobalIdentifier, patientIcn);
		}
		catch (GlobalArtifactIdentifierFormatException x)
		{
			logger.error(x);
			throw new TranslationException("Fatal error in translation, unable to build a global artifact ID", x);
		}
		catch (URNFormatException urnfX)
		{
			logger.error(urnfX);
			throw new TranslationException("Fatal error in translation, unable to build a patient artifact ID", urnfX);
		}

		if(documentGlobalIdentifier instanceof DocumentURN)
			logger.debug("Global artifact identifier'" + documentGlobalIdentifier.toString() + "' is a valid DocumentURN (i.e. is a VA document).");
		else
			logger.debug("Global artifact identifier'" + documentGlobalIdentifier.toString() + "' is NOT a valid DocumentURN (i.e. is NOT a VA document).");
			
		// translate the rest of the data
		String clinicalType = null;
		int confidentialityCode = 0;
		Date creationDate = null;
		Long contentLength = 0L;
		ChecksumValue checksumValue = null;
		String languageCode = null;

		try
		{
			creationDate = properties.getCreationTime();
		}
		catch (ParseException pX)
		{
			logger.warn(pX);
		}

		try
		{
			contentLength = properties.getSize();
		}
		catch (NumberFormatException nfX)
		{
			logger.warn("Error converting document size to long", nfX);
		}

		try
		{
			BigInteger checksum = properties.getHash();
			if(checksum != null)
				checksumValue = new ChecksumValue("SHA1", checksum);
		}
		catch (NumberFormatException nfX)
		{
			logger.warn( "Unable to parse 'hash' slot value.", nfX );
		}
		
		languageCode = properties.getLanguageCode();
		
		if (extrinsicObject.getClassificationArray() != null)
		{
			for (ClassificationType classification : extrinsicObject.getClassificationArray())
			{
				// node representation is the value in the classification scheme	(e.g. 51851-4, a LOINC document code)
				String nodeRepresentation = classification.getNodeRepresentation();
				// coding scheme is the identifier of the coding used for the value (e.g. 2.16.840.1.113883.6.1, the LOINC UUID)
				String codingSchemeUrnAsString = null;

				// find the slot with the name 'codingScheme'
				// the value of this slot is the coding scheme
				if (classification.getSlotArray() != null)
					for (SlotType1 classificationSlot : classification.getSlotArray())
						if ("codingScheme".equals(classificationSlot.getName()))
						{
							ValueListType codingSchemeValueList = classificationSlot.getValueList();
							String[] codingSchemeValue = codingSchemeValueList.getValueArray();
							if(codingSchemeValue != null && codingSchemeValue.length > 0)
								codingSchemeUrnAsString = codingSchemeValue[0];
						}
				
				// the classification scheme is a reference to a system of
				// categories that the extrinsic object is classified into
				// e.g. 41a5887f-8865-4c09-adf7-e362475b143a, Document entry class code UUID
				String classificationScheme = classification.getClassificationScheme();
				IheUUID classificationSchemeUuid = null;

				// the classificationSchemeUuid will indicate the classification ontology that the document is
				// being classified into
				try
				{
					classificationSchemeUuid = IheUUID.get(classificationScheme);
				}
				catch (URISyntaxException urisX)
				{
					throw new TranslationException( "Unable to parse '" + classificationScheme + "' classification scheme URI.", urisX );
				}

				CodingScheme codingScheme = null;
				try
				{
					codingScheme = CodingScheme.valueOf( codingSchemeUrnAsString );
				}
				catch (IllegalArgumentException iaX)
				{
					logger.warn("Error translating coding scheme '" + codingSchemeUrnAsString + "'.");
				}
				catch (NullPointerException npX)
				{
					logger.warn("Coding scheme '" + codingSchemeUrnAsString + "' is unknown.");
				}
				try{logger.info("Translating document classification '" + classificationSchemeUuid.toString() + "', coding scheme '" + codingScheme.toString() + "', value '" + nodeRepresentation + "'.");}
				catch (RuntimeException x1)
				{
					try{logger.info("Translating document classification '" + classificationSchemeUuid.toString() + "', coding scheme '" + codingSchemeUrnAsString + "', value '" + nodeRepresentation + "'.");}
					catch (RuntimeException x){}
					logger.info("Translating document classification '" + classificationScheme + "', coding scheme '" + codingSchemeUrnAsString + "', value '" + nodeRepresentation + "'.");
				}
				
				switch (classificationSchemeUuid)
				{
				case DOC_ENTRY_CLASS_CODE:
				{
					if(codingScheme != null)
					{
						SchemeTranslationSPI codingSchemeTranslator = translatorFactory.getSchemeTranslator(
								codingScheme, CodingScheme.VADOCUMENTCLASS);
						if(codingSchemeTranslator != null)
						{
							ClassifiedValue[] translatedValue = codingSchemeTranslator.translate(nodeRepresentation);
							if( CodingScheme.VADOCUMENTCLASS.equals(translatedValue[0].getCodingScheme()) )
								clinicalType = translatedValue[0].getCodeValue();
							else
								logger.warn("Unable to translate value '" + nodeRepresentation + "' from '" + codingScheme.toString() + "' to '" + CodingScheme.VADOCUMENTCLASS.toString() + "'.");
						}
						else
							try
							{
								clinicalType = nodeRepresentation;
							}
							catch (NumberFormatException x)
							{
								logger.error("Unable to translate and parse document type code '" + nodeRepresentation + "'.");
							}
					}
					else
						logger.warn("Unable to translate document entry classification code because the coding scheme was null.");
					
					break;
				}
				case DOC_ENTRY_CONFIDENTIALITY_CODE:
				{
					if(codingScheme != null)
					{
						SchemeTranslationSPI codingSchemeTranslator = translatorFactory.getSchemeTranslator(
								codingScheme, CodingScheme.VACONFIDENTIALITY);
						if(codingSchemeTranslator != null)
						{
							ClassifiedValue[] translatedValue = codingSchemeTranslator.translate(nodeRepresentation);
							confidentialityCode = Integer.parseInt(translatedValue[0].getCodeValue());
						}
						else
							try
							{
								confidentialityCode = Integer.parseInt(nodeRepresentation);
							}
							catch (NumberFormatException x)
							{
								logger.error("Unable to translate and parse confidentiality code '" + nodeRepresentation + "'.");
							}
					}
					else
						logger.warn("Unable to translate document entry confidentiality code because the coding scheme was null.");
								
					break;
				}
				}
			}
		}
		Document document = Document.create(
			documentSetId, 
			documentGlobalIdentifier
		); 
		if(mediaType == null)
		{
			getLogger().info("Unable to translate media type '" + extrinsicObject.getMimeType() + "' for object '" + documentGlobalIdentifier.toString() + "'.");
			TransactionContextFactory.get().addDebugInformation("Unable to translate media type '" + extrinsicObject.getMimeType() + "' for object '" + documentGlobalIdentifier.toString() + "'.");
		}
		document.setCreationDate(creationDate);
		document.setClinicalType(clinicalType);
		document.setMediaType(mediaType);
		document.setContentLength(contentLength == null ? 0L : contentLength.longValue());
		document.setLanguageCode(languageCode);
		document.setConfidentialityCode(confidentialityCode);
		document.setName(name);
		document.setDescription(description);

		return document;
	}

	/**
	 * Translate a bound XMLBeans representation of a cross gateway request into
	 * the internal transport-neutral form.
	 * 
	 * AdhocQueryRequest.AdhocQuery.Slot*
	 * 
	 * @param adhocQueryRequest
	 * @return
	 */
	public static CrossGatewayQueryRequest translate(AdhocQueryRequestDocument adhocQueryRequestDoc)
	throws TranslationException
	{
		AdhocQueryRequestDocument.AdhocQueryRequest adhocQueryRequest = adhocQueryRequestDoc.getAdhocQueryRequest();
		long maxResults = getMaxResults(adhocQueryRequest);

		boolean returnComposedObject = adhocQueryRequest.getResponseOption() == null ? 
				true : adhocQueryRequest.getResponseOption().getReturnComposedObjects();

		// A registry query.
		AdhocQueryType adhocQuery = adhocQueryRequest.getAdhocQuery();
		if (adhocQuery.getQueryExpression() != null)
			parseQuery(adhocQuery.getQueryExpression());

		// this is the query UUID and must indicate a FindDocumentQuery
		String queryUUID = adhocQuery.getId();
		if( ! IheUUID.FIND_DOCUMENTS_UUID.getUrnAsString().equals(queryUUID) )
			logger.warn("AdhocQueryRequest queryUUID is '" + queryUUID + "' and must be '" + IheUUID.FIND_DOCUMENTS_UUID.getUrnAsString() + "'.");

		// translate slot elements into the parameters list
		SlotType1[] slots = adhocQuery.getSlotArray();
		StoredQueryParameterList parameters = new StoredQueryParameterList();
		for (int slotIndex = 0; slotIndex < slots.length; ++slotIndex)
		{
			SlotType1 slot = slots[slotIndex];

			String name = slot.getName();
			ValueListType valueList = slot.getValueList();
			String[] values = valueList == null ? null : valueList.getValueArray();

			if(name == null)
				logger.warn("Null slot name received, slot is being ignored.");
			else if(values == null)
				logger.warn("Slot with name '" + name + "' received with null value list, slot is being ignored.");
			else
				parameters.add(new StoredQueryParameter(name, values));
		}

		// translate the external identifiers into the parameter list
		ExternalIdentifierType[] externalIdentifiers = adhocQuery.getExternalIdentifierArray();
		if(externalIdentifiers != null)
		{
			for(ExternalIdentifierType externalIdentifier : externalIdentifiers)
			{
				String name = "<unknown>";
				try
				{
					name = externalIdentifier.getName().toString();
					String identificationScheme = externalIdentifier.getIdentificationScheme().toString();
					String value = externalIdentifier.getValue();
					
					parameters.add( new StoredQueryParameter(name, new String[]{value}, identificationScheme) );
				}
				catch (NullPointerException npX)
				{
					// warn if any of the references are null, but continue to process
					logger.warn("Unable to get external identifier '" + name + "'.", npX);
				}
			}
		}
		
		// translate any parameters that are in a "foreign" coding scheme
		try
		{
			translateStoredQueryParameters(parameters);
		}
		catch (ParameterFormatException x)
		{
			throw new TranslationException(x);
		}

		return new CrossGatewayQueryRequest(maxResults, returnComposedObject, parameters);
	}

	/**
	 * Translate the internal transport-neutral form of a cross gateway request
	 * into a bound XMLBeans representation.
	 * 
	 * @param query
	 * @return
	 * @throws TranslationException
	 */
	public static AdhocQueryRequestDocument translate(CrossGatewayQueryRequest query) 
	throws TranslationException
	{
		AdhocQueryRequestDocument adhocQueryRequestDoc = AdhocQueryRequestDocument.Factory.newInstance();

		try
		{
			query.getAuthor();
		}
		catch (ParameterFormatException x)
		{
			throw new TranslationException(x);
		}
		
		return adhocQueryRequestDoc;
	}

	/**
	 * Parameters may be in any number of classifications schemes (e.g. SNOMED)
	 * This method translates the parameters into VA classification schemes.
	 * 
	 * @param parameters
	 * @throws ParameterFormatException 
	 */
	private static void translateStoredQueryParameters(StoredQueryParameterList parameters) 
	throws ParameterFormatException, TranslationException
	{
		StoredQueryParameter patientIdParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.patientId);
		String hl7Identifier = patientIdParameter.getValueAsString();
		String patientICN = parseHL7PatientID(hl7Identifier);
		patientIdParameter.setValue(patientICN);
		
		StoredQueryParameter classCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.classCode);
		translateCodeParameter(classCodeParameter, CodingScheme.VADOCUMENTCLASS);

		StoredQueryParameter confidentialityCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.confidentialityCodeList);
		translateCodeParameter(confidentialityCodeParameter, CodingScheme.VACONFIDENTIALITY);

		StoredQueryParameter eventCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.eventCodeList);
		translateCodeParameter(eventCodeParameter, CodingScheme.VAPROCEDURE);

		StoredQueryParameter facilityCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.healthcareFacilityTypeCode);
		translateCodeParameter(facilityCodeParameter, CodingScheme.VAFACILITYTYPE);

		StoredQueryParameter practiceSettingCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.practiceSettingCode);
		translateCodeParameter(practiceSettingCodeParameter, CodingScheme.VAPRACTICESETTING);
	}

	/**
	 * 
	 * @param queryExpression
	 */
	private static void parseQuery(QueryExpressionType queryExpression)
	{
		// if no query expression language specified then assume an XML query
		WellKnownUUID queryLanguageUUID = getDefaultQueryLanguageUUID();

		ReferenceURI queryLanguage = queryExpression.xgetQueryLanguage();
		if (queryLanguage != null)
			try
			{
				queryLanguageUUID = EbXMLRsUUID.get(queryLanguage.getStringValue());
			}
			catch (URISyntaxException x)
			{
				logger.error(x);
			}

	}

	/**
	 * Get the maximum allowed results, handling all the null and negative
	 * numbers and the conversion errors.
	 * 
	 * @param adhocQueryRequest
	 * @return
	 */
	private static long getMaxResults(AdhocQueryRequestDocument.AdhocQueryRequest adhocQueryRequest)
	{
		long maxResults = -1;
		BigInteger maximumResults = adhocQueryRequest.getMaxResults() == null ? BigInteger
				.valueOf(-1L)
				: adhocQueryRequest.getMaxResults();
		maxResults = maximumResults.longValue();
		if (maxResults < 0)
			maxResults = -1;
		if (!maximumResults.equals(BigInteger.valueOf(maxResults))) // if we
																	// can't
																	// convert
																	// to a
																	// long,
																	// assume
																	// unlimited
																	// results
			maxResults = -1;

		return maxResults;
	}

	/**
	 * Translate the List<DocumentSets> from the core into a valid query
	 * response.
	 * 
	 * e.g. <query:AdhocQueryResponse
	 * xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
	 * xsi:schemaLocation="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0
	 * ../schema/ebRS/query.xsd"
	 * xmlns:query="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0"
	 * xmlns:rim="urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0" status="Success">
	 * <rim:RegistryObjectList> <rim:ExtrinsicObject
	 * xmlns:q="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0"
	 * xmlns:rim="urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0"
	 * id="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf" // the unique
	 * identifier of the extrinsic object isOpaque="false" mimeType="text/xml" //
	 * the mime type of the extrinsic object
	 * objectType="urn:uuid:7edca82f-054d-47f2-a032-9b2a5b5186c1" // the UUID
	 * indicates that this is a document
	 * status="urn:oasis:names:tc:ebxml-regrep:StatusType:Approved"> // the URN
	 * indicates that the document is approved <rim:Slot name="URI"> // the URI
	 * where the document may be accessed from <rim:ValueList>
	 * <rim:Value>http://localhost:8080/XDS/Repository/08a15a6f-5b4a-42de-8f95-89474f83abdf.xml</rim:Value>
	 * </rim:ValueList> </rim:Slot> ... <rim:Name> <rim:LocalizedString
	 * charset="UTF-8" value="The Document Name" xml:lang="en-us"/> </rim:Name>
	 * <rim:Description> <rim:LocalizedString charset="UTF-8" value="The
	 * description of the document" xml:lang="en-us"/> </rim:Description>
	 * 
	 * <!-- Classification is --> <rim:Classification // Classification is a 3
	 * part coding; coding scheme, code value, code value display name
	 * classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a" //
	 * the UUID of XDSDocumentEntry.classCode
	 * classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf" //
	 * reference to the ExtrinsicObject element id
	 * id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf" // the unique
	 * identifier of this Classification nodeRepresentation="Education" // this
	 * is the code value
	 * objectType="urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification" >
	 * <rim:Slot name="codingScheme"> // this is the coding scheme name
	 * <rim:ValueList> <rim:Value>Connect-a-thon classCodes</rim:Value>
	 * </rim:ValueList> </rim:Slot> <rim:Name> // this is the code value display
	 * name <rim:LocalizedString charset="UTF-8" value="Education"
	 * xml:lang="en-us"/> </rim:Name> <rim:Description/> </rim:Classification>
	 * 
	 * @param documentSets
	 * @return
	 */
	public static AdhocQueryResponseDocument.AdhocQueryResponse translate(DocumentSetResult documentSetResult) 
	throws TranslationException
	{
		AdhocQueryResponseDocument.AdhocQueryResponse response = AdhocQueryResponseDocument.AdhocQueryResponse.Factory.newInstance();
		
		if(documentSetResult.getArtifactResultStatus() == ArtifactResultStatus.partialResult)
		{
			response.setStatus(responseStatusPartialSuccess.getStringValue());
		}
		else
		{
			response.setStatus(responseStatusSuccess.getStringValue());
		}
		
		if((documentSetResult.getArtifactResultErrors() != null) && (documentSetResult.getArtifactResultErrors().size() > 0))
		{
			RegistryErrorList rel = response.addNewRegistryErrorList(); 
			for(ArtifactResultError error : documentSetResult.getArtifactResultErrors())
			{
				RegistryError re = rel.addNewRegistryError();
				re.setCodeContext(error.getCodeContext());
				re.setErrorCode(translateErrorCode(error.getErrorCode()));				
				re.setLocation(error.getLocation());
				re.setSeverity(error.getSeverity().toString());
			}
		}
		

		RegistryObjectListType rolt = response.addNewRegistryObjectList();
		Collection<DocumentSet> documentSets = documentSetResult.getArtifacts();

		List<IdentifiableType> registryObjectIdentifiableList = new ArrayList<IdentifiableType>();
		int documentCount = 0;
		if(documentSets != null)
			for (DocumentSet documentSet : documentSets)
			{
				for (Document document : documentSet)
				{
					ExtrinsicObjectType eoType = ExtrinsicObjectType.Factory.newInstance();
	
					populateExtrinsicObject(eoType, document);
	
					registryObjectIdentifiableList.add(eoType);
					documentCount++;
				}
			}

		rolt.setIdentifiableArray(
			registryObjectIdentifiableList.toArray(
				new IdentifiableType[registryObjectIdentifiableList.size()]));
		response.setRegistryObjectList(rolt);
		
		logger.info("Translated into [" + documentCount + "] individual artifacts");
		TransactionContextFactory.get().addDebugInformation("Translated into [" + documentCount + "] individual artifacts");

		return response;
	}
	
	private static String translateErrorCode(ArtifactResultErrorCode errorCode)
	{
		String result = "";
		switch(errorCode)
		{		
			case unknownPatientId:
				result = ErrorCode.UNKNOWN_PATIENT_ID.getOpcode();
				break;
			case invalidRequestException:
				result = ErrorCode.REGISTRY_METADATA_ERROR.getOpcode();
				break;
			default: 
				result = ErrorCode.REGISTRY_ERROR.getOpcode();
		}
		return result;
	}

	/**
	 * For a client (initiatingGateway) to successfully make a retrieve request back to 
	 * this server (respondingGateway) requires that it have the home community, the repository
	 * and the document identifiers.
	 * The home community ID is an attribute of the extrinsic object element.
	 * The repository and document IDs are external references.
	 * NOTE: the patient ID is not part of the retrieve request, for VA documents that identifier
	 * is included in the document ID.
	 * 
	 * @param dateFormatter
	 * @param document
	 * @param eoType
	 * @throws TranslationException
	 */
	private static void populateExtrinsicObject(ExtrinsicObjectType eoType, Document document) 
	throws TranslationException
	{
		SimpleDateFormat dateFormatter = new SimpleDateFormat(XCATranslatorXmlBeans.iheDateFormat);
		dateFormatter.setTimeZone(XcaTimeZone.getTimeZone());

		// set home community ID attribute
		eoType.setHome(document.getGlobalArtifactIdentifier().getHomeCommunityId());

		// identify the extrinsic object as a document
		eoType.setObjectType(IheUUID.DOC_ENTRY.getUrnAsString());
		
		// set identifier attribute
		// The internal (to the XML) identifier of the extrinsic object, also may be the identifier
		// of the Identifiable object.
		// As if any further proof is needed that XDS is a hacked over kludge comes this
		// gem from the RIM schema as provided from IHE.
        // "If id is provided and is not in proper URN syntax then it is used for
        // linkage within document and is ignored by the registry. In this case the
        // registry generates a UUID URN for id attribute.
        // id must not be null when object is retrieved from the registry."
		// Note that no reference is given to the context of the operation but it sounds
		// like they are referring to a POST or PUT against the registry.  Whether an
		// extrinsic object identifier can be anything but a URN in a query response is
		// not stated, just that it cannot be null.
		String documentId = document.getGlobalArtifactIdentifier().getDocumentUniqueId(); 
		eoType.setId(documentId);

		// create a reference URI instance to be used by other elements that reference
		// this extrinsic object
		ReferenceURI documentReferenceUri = ReferenceURI.Factory.newInstance();
		documentReferenceUri.setStringValue(documentId);
		
		// Set name field
		InternationalStringType nameStringType = eoType.addNewName();
		LocalizedStringType nameString = nameStringType.addNewLocalizedString();
		nameString.setValue(document.getName());
		
		// Set description field
		InternationalStringType descriptionStringType = eoType.addNewDescription();
		LocalizedStringType descriptionString = descriptionStringType.addNewLocalizedString();
		descriptionString.setValue(document.getDescription());
		
		// set mime type attribute
		// JMW the documents don't have the media type set so
		// this was throwing an exception since getMediaType() was null
		// CTB - if no media type known, then send as application/octet-stream,
		// which is basically saying "uh, I dunno'"
		MediaType mediaType = document.getMediaType();
		eoType.setMimeType(
			mediaType == null ? MediaType.APPLICATION_OCTETSTREAM.toString() : mediaType.toString()
		);

		// set status attribute
		eoType.setStatus(documentStatusApproved.getStringValue());

		// The URI slot is populated with the toString() form of the 
		// document identifier.  For VA documents this will build a URN
		// with a vadoc namespace identifier.  This URN follows the URN rules
		// in that it is a permanent identifier of the document.
		SlotType1 uriSlot = eoType.addNewSlot();
		populateSingleValueSlot(uriSlot, QuerySlotName.URI.toString(), document.getGlobalArtifactIdentifier().toString(),
			SlotFormat.NONE);

		// set creation date slot
		if (document.getCreationDate() != null)
		{
			SlotType1 creationTimeSlot = eoType.addNewSlot();
			populateSingleValueSlot(creationTimeSlot, QuerySlotName.creationTime.toString(),
					dateFormatter.format(document.getCreationDate()),
					SlotFormat.DATE);
		}

		// set language code slot
		SlotType1 languageCodeSlot = eoType.addNewSlot();
		populateSingleValueSlot(languageCodeSlot, QuerySlotName.languageCode.toString(), DEFAULT_LANGUAGE,
			SlotFormat.NONE);

		// set source patient ID slot
		if (document.getPatientId() != null)
		{
			SlotType1 sourcePatientIdSlot = eoType.addNewSlot();
			populateSingleValueSlot(sourcePatientIdSlot, QuerySlotName.sourcePatientId.toString(),
				document.getPatientId(),
				SlotFormat.NONE);
		}

		//
		if (document.getContentLength() > 0)
		{
			SlotType1 sourceDocumentLengthSlot = eoType.addNewSlot();
			populateSingleValueSlot(
				sourceDocumentLengthSlot, 
				QuerySlotName.size.toString(), 
				Long.toString(document.getContentLength()),
				SlotFormat.NONE);
		}

		// if the checksum is available and is an SHA1 calculated value
		// then add the hash slot
		ChecksumValue checksum = document.getChecksumValue();
		if (checksum != null && "SHA1".equals(checksum.getAlgorithm()))
		{
			SlotType1 hashSlot = eoType.addNewSlot();
			populateSingleValueSlot(
				hashSlot, 
				QuerySlotName.hash.toString(), 
				Long.toString(checksum.getValue().longValue()),
				SlotFormat.NONE);
		}

		// Set the repository ID slot
		SlotType1 repositoryIdSlot = eoType.addNewSlot();
		populateSingleValueSlot(
			repositoryIdSlot, 
			QuerySlotName.repositoryUniqueId.toString(), 
			document.getGlobalArtifactIdentifier().getRepositoryUniqueId(),
			SlotFormat.NONE);
		
		// Set the repository ID external identifier
//		ExternalIdentifierType repositoryId = eoType.addNewExternalIdentifier();
//		populateExternalIdentifier(
//			repositoryId, 
//			documentId, 
//			REPOSITORY_UNIQUE_ID, 
//			document.getGlobalArtifactIdentifier().getRepositoryUniqueId(), 
//			"VASITENUMBER", 
//			document.getGlobalArtifactIdentifier().getRepositoryUniqueId());
		
		// Set the document ID external identifier
		ExternalIdentifierType documentIdExternalIdentifer = eoType.addNewExternalIdentifier();
		populateExternalIdentifier(
			documentIdExternalIdentifer, 
			documentId, 
			IheUUID.DOC_ENTRY_UNIQUE_IDENTIFICATION.getName(), 
			document.getGlobalArtifactIdentifier().getDocumentUniqueId(), 
			IheUUID.DOC_ENTRY_UNIQUE_IDENTIFICATION.getUrnAsString(), 
			document.getGlobalArtifactIdentifier().getDocumentUniqueId()
		);
		
		// Set the patient ID external identifier
		ExternalIdentifierType patientId = eoType.addNewExternalIdentifier();
		populateExternalIdentifier(
			patientId, 
			documentId, 
			IheUUID.DOC_ENTRY_PATIENT_IDENTIFICATION.getName(), 
			document.getPatientId(), 
			IheUUID.DOC_ENTRY_PATIENT_IDENTIFICATION.getUrnAsString(), 
			document.getPatientId()
		);
		
		// add the confidentiality code as a Classification element
		ClassificationType confidentialityClassification = eoType.addNewClassification();
		populateClassificationType(confidentialityClassification,
				documentReferenceUri,
				IheUUID.DOC_ENTRY_CONFIDENTIALITY_CODE.getUrn(),
				new ClassifiedValue(
						CodingScheme.VACONFIDENTIALITY, 
						Integer.toString(document.getConfidentialityCode())), 
						Integer.toString(document.getConfidentialityCode()));

		// add the document type as a Classification element
		// this element requires translation from VA to LOINC classification
		// before
		// sending
		ClassificationType docTypeClassification = eoType.addNewClassification();
		populateTranslatedClassificationType(docTypeClassification,
			documentReferenceUri, 
			IheUUID.DOC_ENTRY_TYPE_CODE.getUrn(), 
			CodingScheme.VADOCUMENTCLASS,
			new CodingScheme[]{CodingScheme.LOINC, CodingScheme.MHS}, 
			document.getClinicalType(),
			true							// use the source value for the localized-string, not the translated value
		);

		// NOTES:
		// <rim:Classification
		// classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a"
		// classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf"
		// id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf"
		// nodeRepresentation="Education"
		// objectType="Urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification">
		//
		// The classifiedObject references the URN of the document
	}

	/**
	 * 
	 * @param externalIdentifier
	 * @param registryObjectId
	 * @param externalIdentifierName
	 * @param externalIdentifierValue
	 * @param identificationScheme
	 * @param description
	 */
	private static void populateExternalIdentifier(
		ExternalIdentifierType externalIdentifier,
		String registryObjectId,
		String externalIdentifierName,		// e.g. "XDSDocumentEntry.uniqueId" 
		String externalIdentifierValue,		// e.g. "1.3.6.1.4.1.21367.2005.3.99.1.1010"
		String identificationScheme,		// e.g. "urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab"
		String description)
	{
		// create and set a unique identifier
		GUID id = new GUID();
		externalIdentifier.setId(id.toString());
		
		// set the reference to the registry object
		externalIdentifier.setRegistryObject(registryObjectId);
		
		// the name of the external identifier (i.e. the field/property name)
		if(externalIdentifierName != null)
		{
			InternationalStringType nameI18n = externalIdentifier.addNewName();
			addLocalizedString(nameI18n, externalIdentifierName);
		}
		
		// set the identification scheme
		externalIdentifier.setIdentificationScheme(identificationScheme);
		
		// set the value of the external identifier (e.g. the field/property value)
		externalIdentifier.setValue(externalIdentifierValue);
		
		// a human readable description
		if(description != null)
		{
			InternationalStringType descriptionI18n = externalIdentifier.addNewDescription();
			addLocalizedString(descriptionI18n, description);
		}
	}

	/**
	 * 
	 * @param valueI18n
	 * @param value
	 */
	private static void addLocalizedString(InternationalStringType valueI18n, String value)
	{
		addLocalizedString(valueI18n, value, DEFAULT_CHARSET, DEFAULT_LANGUAGE);
	}
	
	private static void addLocalizedString(InternationalStringType valueI18n, String value, String charset, String language)
	{
		LocalizedStringType descriptionL9d = valueI18n.addNewLocalizedString();
		descriptionL9d.setValue(value);
		descriptionL9d.addNewCharset().setStringValue(charset);
		descriptionL9d.xsetValue( FreeFormText.Factory.newInstance() );
		descriptionL9d.xgetValue().setStringValue(language);
		descriptionL9d.setValue(value);
	}
	
	/**
	 * Translate an array of ExternalIdentifierType, from an XCA document,
	 * into a list of ExternalReferenceValueObject, a simpler internal representation.
	 * 
	 * @param externalIdentifierArray
	 * @return
	 */
	public static ExternalReferenceValueObjectList translate(ExternalIdentifierType[] externalIdentifierArray)
	{
		ExternalReferenceValueObjectList externalReferences = new ExternalReferenceValueObjectList();
		
		for(ExternalIdentifierType externalIdentifier : externalIdentifierArray)
			externalReferences.add( translate(externalIdentifier) );
		
		return externalReferences;
	}

	/**
	 * Translate a single ExternalIdentifierType, from an XCA document,
	 * into a ExternalReferenceValueObject, a simpler internal representation.
	 * 
	 * @param externalIdentifier
	 * @return
	 */
	public static ExternalReferenceValueObject translate(ExternalIdentifierType externalIdentifier)
	{
		String registryObjectReference = externalIdentifier.getRegistryObject();
		InternationalStringType nameI9d = externalIdentifier.getName();
		String externalReferenceName = null;
		if(nameI9d != null)
			externalReferenceName = getLocalizedStringValue(nameI9d);
		
		String externalReferenceValue = externalIdentifier.getValue();
		
		InternationalStringType descriptionI9d = externalIdentifier.getDescription();
		String description = null;
		if(descriptionI9d != null)
			description = getLocalizedStringValue(nameI9d);
		
		String identificationScheme = externalIdentifier.getIdentificationScheme();
		
		return new ExternalReferenceValueObject(
			registryObjectReference, 
			externalReferenceName, 
			externalReferenceValue,
			identificationScheme,
			description);
	}
	
	/**
	 * Get the first localized string from an internalized string.
	 * 
	 * @param i9d
	 * @return
	 */
	private static String getLocalizedStringValue(InternationalStringType i9d)
	{
		return getLocalizedStringValue(i9d, null, null);
	}
	
	/**
	 * Translate the localized string with the given charset and language
	 * from the international string.
	 * If either the charset or language parameters is null then they will
	 * not be used in the matching (i.e. pass null for both to get the first
	 * localized string).
	 * 
	 * @param i9d
	 * @param charset
	 * @param language
	 * @return
	 */
	private static String getLocalizedStringValue(InternationalStringType i9d, String charset, String language)
	{
		for(LocalizedStringType localizedString : i9d.getLocalizedStringArray() )
		{
			String localizedCharset = localizedString.getCharset().getStringValue();
			String localizedLanguage = localizedString.xgetValue().getStringValue();
			if( ( charset == null || (charset != null && charset.equals(localizedCharset)) ) &&
				( language == null || (language != null && language.equals(localizedLanguage)) ) )
			{
				return localizedString.getValue();
			}
		}
		return null;
	}
	
	/**
	 * Create a Classification element, translating the classification value
	 * into the target classification scheme.
	 * 
	 * For each of the possible destination coding schemes, try to get a translator
	 * and then do the translation.
	 * The first coding scheme that results in a successful translation scheme is the
	 * resulting translating, i.e. the array of CodingScheme is order-sensitive
	 * 
	 * @param classifiedObjectUri
	 * @param classificationScheme
	 * @param sourceClassificationScheme
	 * @param destinationClassificationScheme
	 * @param sourceClassificationValue
	 * @return
	 * @throws TranslationException
	 */
	private static void populateTranslatedClassificationType(
		ClassificationType classificationType,
		ReferenceURI classifiedObjectUri, URI classificationScheme,
		CodingScheme sourceClassificationScheme,
		CodingScheme[] destinationCodingSchemes,
		String sourceClassificationValue,
		boolean useSourceValueAsLocalizedString) 
	throws TranslationException
	{
		// create the "translated" classified value with the untranslated values
		// these will be the default if the translation does not work
		ClassifiedValue[] translatedClassifiedValue = new ClassifiedValue[]
		{
			new ClassifiedValue(sourceClassificationScheme, sourceClassificationValue)
		};
		
		// try to translate to one of the destination coding schemes
		// if no translation can take place then the original classified value
		// is returned
		translatedClassifiedValue = AbstractXCATranslator.translate(translatedClassifiedValue, destinationCodingSchemes);
		
		// populate the classification type
		// CTB 15Nov2010
		// Note that the "sourceClassificationValue" will end up as the localized string
		// element in the classification.  This was a request from HAIMS for something human readable
		// and known to be consistent with the document.
		populateClassificationType(classificationType, classifiedObjectUri,
			classificationScheme, 
			translatedClassifiedValue[0],
			useSourceValueAsLocalizedString ? sourceClassificationValue:translatedClassifiedValue[0].getCodeValue()
		);
		classificationType.setClassifiedObject(classifiedObjectUri.toString());
	}

	/**
	 * Create a Classification that does not require any translation of the
	 * classification scheme.
	 * 
	 * @param classifiedObjectUri -
	 *            the extrinsic object being classified i.e. a document
	 *            reference
	 * @param classificationSchemeUri -
	 *            the classification scheme e.g. "healthcare facility type code"
	 *            as a URI
	 * @param classifiedValue -
	 *            the coding scheme and value e.g. LOINC 1234-4
	 * @param classificationDescription -
	 *            a human readable coding e.g. "hospital"
	 * @return
	 * @throws TranslationException
	 * 
	 * Example XML:
	 * 
	 * <rim:Classification
	 * classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a"
	 * classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf"
	 * id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf"
	 * nodeRepresentation="Education"
	 * objectType="Urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification" >
	 * <rim:Slot name="codingScheme"> <rim:ValueList> <rim:Value>Connect-a-thon
	 * classCodes</rim:Value> </rim:ValueList> </rim:Slot> <rim:Name>
	 * <rim:LocalizedString charset="UTF-8" value="Education" xml:lang="en-us"/>
	 * </rim:Name> <rim:Description/> </rim:Classification>
	 * 
	 */
	private static void populateClassificationType(
			ClassificationType classificationType,
			ReferenceURI classifiedObjectUri, URI classificationSchemeUri,
			ClassifiedValue classifiedValue, String classificationDescription)
			throws TranslationException
	{
		// create a unique identifier for the classification element
		String classificationId = (new GUID()).toString();
		classificationType.setId(classificationId);
		
		classificationType.setObjectType(objectTypeClassification.getStringValue());

		// set the reference to the classified object (the document)
		classificationType.setClassifiedObject(classifiedObjectUri.getStringValue());
		classificationType.setClassificationScheme(classificationSchemeUri.toString());
		classificationType.setNodeRepresentation(classifiedValue.getCodeValue());

		// ID is a required attribute
		classificationType.setId((new GUID()).toShortString());
		
		
		if (classificationDescription != null)
		{
			InternationalStringType name = classificationType.addNewName();
			LocalizedStringType localizedName = name.addNewLocalizedString();
			// localizedName.setLang("en-us");
			localizedName.setValue(classificationDescription);
		}

		SlotType1 codingSchemeSlot = classificationType.addNewSlot();
		populateSingleValueSlot(
			codingSchemeSlot, 
			"codingScheme",
			classifiedValue.getCodingScheme().getCanonicalIdentifier().toString(),
			SlotFormat.NONE
		);
	}

	private enum SlotFormat{NONE, STRING, DATE, NUMBER;};
	/**
	 * 
	 * @param value
	 * @return
	 * @throws URNFormatException
	 */
	private static void populateSingleValueSlot(
		SlotType1 slot, 
		String key,
		String value,
		SlotFormat slotFormat)		// if a String, adds single quotes
	{
		slot.setName(key);
		ValueListType slotValues = slot.addNewValueList();
		LongName slotValue = slotValues.addNewValue();
		
		// JMW 12/2/2009 - HAIMS 1 cannot support a quoted value here, take it out for now
//		if( isHaimsCompatibilityMode() )
//			slotValue.setStringValue(value);
//		else 
		if(SlotFormat.STRING == slotFormat)
			slotValue.setStringValue("'" + value + "'");
		else
			slotValue.setStringValue(value);

		return;
	}

	/**
	 * @param x
	 * @return
	 */
	public static org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument translate(
			Exception x)
	{
		org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument response = org.oasis.ebxml.regrep.query.xmlbeans.AdhocQueryResponseDocument.Factory
				.newInstance();

		response.addNewAdhocQueryResponse();
		response.getAdhocQueryResponse().setStatus(
				responseStatusFailure.getStringValue());

		return response;
	}
}
