package gov.va.med.imaging.ihe;


import gov.va.med.MediaType;
import gov.va.med.URNFactory;
import gov.va.med.imaging.DocumentURN;
import gov.va.med.imaging.GUID;
import gov.va.med.imaging.channels.ChecksumValue;
import gov.va.med.imaging.core.interfaces.exceptions.ImageNotFoundException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exceptions.URNFormatException;
import gov.va.med.imaging.exchange.business.ImageStreamResponse;
import gov.va.med.imaging.exchange.business.documents.Document;
import gov.va.med.imaging.exchange.business.documents.DocumentSet;
import gov.va.med.imaging.exchange.enums.ImageQuality;
import gov.va.med.imaging.exchange.storage.ByteBufferBackedImageInputStream;
import gov.va.med.imaging.exchange.translation.TranslationMethod;
import gov.va.med.imaging.ihe.exceptions.TranslationException;
import gov.va.med.imaging.ihe.request.CrossGatewayQueryRequest;
import gov.va.med.imaging.ihe.request.CrossGatewayRetrieveRequest;
import gov.va.med.imaging.ihe.request.StoredQueryParameter;
import gov.va.med.imaging.ihe.request.StoredQueryParameterList;
import gov.va.med.imaging.ihe.xds.ErrorCode;
import gov.va.med.imaging.terminology.ClassifiedValue;
import gov.va.med.imaging.terminology.CodingScheme;
import gov.va.med.imaging.terminology.SchemeTranslationSPI;
import gov.va.med.imaging.transactioncontext.TransactionContext;
import gov.va.med.imaging.transactioncontext.TransactionContextFactory;

import java.io.IOException;
import java.math.BigInteger;
import java.net.URISyntaxException;
import java.text.SimpleDateFormat;
import java.util.List;
import javax.activation.DataHandler;
import org.apache.axis2.databinding.types.URI.MalformedURIException;
import org.apache.log4j.Logger;
import org.ihe.iti.xdsb.adb.*;
import org.oasis.ebxml.regrep.query.adb.AdhocQueryRequest;
import org.oasis.ebxml.regrep.query.adb.AdhocQueryResponse;
import org.oasis.ebxml.regrep.rim.adb.*;
import org.oasis.ebxml.regrep.rs.adb.RegistryResponseType;

/**
 * The translator from IHE XCA form into internal form.
 * This includes translation from the code scheme used in the SOAP request
 * to the internal codes.
 * 
 * @author vhaiswbeckec
 *
 */
public class XCATranslatorAdb
extends AbstractXCATranslator
{
	public static String PARAMETER_PATIENT_ID;
	public static Logger logger = Logger.getLogger(XCATranslatorAdb.class);
	
	public static final String iheDateFormat = "yyyyMMddhhmm";

	private static ReferenceURI responseStatusSuccess;
	private static ReferenceURI responseStatusFailure;
	
	private static ReferenceURI documentStatusSubmitted;
	private static ReferenceURI documentStatusApproved;
	private static ReferenceURI documentStatusDeprecated;
	
	private static ReferenceURI objectTypeClassification;
	
	static
	{
		responseStatusSuccess = new ReferenceURI();
		responseStatusFailure = new ReferenceURI();
		
		documentStatusSubmitted = new ReferenceURI();
		documentStatusApproved = new ReferenceURI();
		documentStatusDeprecated = new ReferenceURI();
		
		objectTypeClassification = new ReferenceURI();
		
		try
		{
			responseStatusSuccess.setReferenceURI( 
				new org.apache.axis2.databinding.types.URI(EbXMLRsUUID.RESPONSE_STATUS_SUCCESS.getUrnAsString()) 
			);
			responseStatusFailure.setReferenceURI( 
				new org.apache.axis2.databinding.types.URI(EbXMLRsUUID.RESPONSE_STATUS_FAILURE.getUrnAsString()) 
			);
			
			documentStatusSubmitted.setReferenceURI(
				new org.apache.axis2.databinding.types.URI(EbXMLRimUUID.DOCUMENT_STATUS_SUBMITTED.getUrnAsString()) 
			);
			documentStatusApproved.setReferenceURI(
				new org.apache.axis2.databinding.types.URI(EbXMLRimUUID.DOCUMENT_STATUS_APPROVED.getUrnAsString()) 
			);
			documentStatusDeprecated.setReferenceURI(
				new org.apache.axis2.databinding.types.URI(EbXMLRimUUID.DOCUMENT_STATUS_DEPRECATED.getUrnAsString())
			);
			
			objectTypeClassification.setReferenceURI(
				new org.apache.axis2.databinding.types.URI("urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification") 
			);
		}
		catch (MalformedURIException x)
		{
			x.printStackTrace();
			throw new java.lang.ExceptionInInitializerError(x);
		}
	}

	// =============================================================================================
	public XCATranslatorAdb() 
	{
	}
	
	/**
	 * 
	 * @param adhocQueryRequest
	 * @return
	 */
	public CrossGatewayQueryRequest translate(AdhocQueryRequest adhocQueryRequest)
	throws TranslationException
	{
		long maxResults = getMaxResults(adhocQueryRequest);
		
		boolean returnComposedObject = 
			adhocQueryRequest.getResponseOption() == null ? true : 
			adhocQueryRequest.getResponseOption().getReturnComposedObjects();
		
		// A registry query.
		AdhocQueryType query = adhocQueryRequest.getAdhocQuery();
		if( query.getQueryExpression() != null )
			parseQuery(query.getQueryExpression());
		
		// this is the query UUID and must indicate a FindDocumentQuery
		org.apache.axis2.databinding.types.URI queryUUID = query.getId();
		org.apache.axis2.databinding.types.URI homeCommunityId = query.getHome();
		
		// all of the parameters, whether from the slots or external identifiers
		// will go into the parameter set
		StoredQueryParameterList parameters = new StoredQueryParameterList();
		
		// translate the slots into a parameter list
		SlotListType slotList = adhocQueryRequest.getRequestSlotList();
		SlotType1[] slots = slotList.getSlot();
		for(int slotIndex=0; slotIndex<slots.length; ++slotIndex)
		{
			SlotType1 slot = slots[slotIndex];
			
			String name = slot.getName().toString();
			ValueListType valueList = slot.getValueList();
			ValueListTypeSequence[] values = valueList.getValueListTypeSequence();
			String[] stringValues = new String[values.length];
			for(int valueIndex=0; valueIndex<values.length; ++valueIndex)
				stringValues[valueIndex] = values[valueIndex].getValue().getLongName();
			
			parameters.add( new StoredQueryParameter(name, stringValues) );
		}
		
		ExternalIdentifierType[] externalIdentifiers = query.getExternalIdentifier();
		if(externalIdentifiers != null)
		{
			for(ExternalIdentifierType externalIdentifier : externalIdentifiers)
			{
				String name = "<unknown>";
				try
				{
					name = externalIdentifier.getName().getInternationalStringTypeSequence()[0].getLocalizedString().getValue().getFreeFormText();
					String identificationScheme = externalIdentifier.getIdentificationScheme().getReferenceURI().toString();
					String value = externalIdentifier.getValue().getLongName();
					
					parameters.add( new StoredQueryParameter(name, new String[]{value}, identificationScheme) );
				}
				catch (RuntimeException npX)
				{
					// warn if any of the references are null, but continue to process
					logger.warn("Unable to get external identifier '" + name + "'.", npX);
				}
			}
		}
		
		translateStoredQueryParameters(parameters);
		
		return new CrossGatewayQueryRequest(
				maxResults, 
				returnComposedObject, 
				parameters);
	}
	
	/**
	 * @param retrieveDocumentSetRequest
	 * @return
	 */
	public CrossGatewayRetrieveRequest[] translate(RetrieveDocumentSetRequest retrieveDocumentSetRequest)
	{
		RetrieveDocumentSetRequestType rds = retrieveDocumentSetRequest.getRetrieveDocumentSetRequest();
		DocumentRequest_type0[] documentRequest = rds.getDocumentRequest();
		
		if(documentRequest == null)
			return null;
		
		CrossGatewayRetrieveRequest[] requests = new CrossGatewayRetrieveRequest[documentRequest.length];
		for(int index=0; index < documentRequest.length; ++index)
		{
			DocumentRequest_type0 request = documentRequest[index]; 
			requests[index] = new CrossGatewayRetrieveRequest( 
				request.getHomeCommunityId().getLongName(),
				request.getRepositoryUniqueId().getLongName(),
				request.getDocumentUniqueId().getLongName() );
		}		
		
		return requests;
	}

	/**
	 * 
	 * @param documentUrn
	 * @return
	 */
	public RetrieveDocumentSetRequest createRetrieveDocumentSetRequest(DocumentURN documentUrn)
	{
		RetrieveDocumentSetRequest request = new RetrieveDocumentSetRequest();
		
		DocumentRequest_type0 documentRequest = new DocumentRequest_type0();
		
		DocumentRequest_type0 [] documentRequestArray = new DocumentRequest_type0[1];
		documentRequestArray[0] = documentRequest;
		
		//TODO: need to set values from URN?
		LongName homeComunityId = new LongName();
		homeComunityId.setLongName("");
		documentRequest.setHomeCommunityId(homeComunityId);
		
		LongName repositoryId = new LongName();
		// this should be a parameter of the call, the originating site Id for DoD documents will be 200, but that 
		// is not what the repository Id is (it will be a part of the request)
		repositoryId.setLongName(documentUrn.getOriginatingSiteId());
		documentRequest.setRepositoryUniqueId(repositoryId);
		
		LongName documentUniqueId = new LongName();
		documentUniqueId.setLongName(documentUrn.toString());
		documentRequest.setDocumentUniqueId(documentUniqueId);
		
		RetrieveDocumentSetRequestType documentSetRequestType = new RetrieveDocumentSetRequestType();
		
		documentSetRequestType.setDocumentRequest(documentRequestArray);
		
		request.setRetrieveDocumentSetRequest(documentSetRequestType);
		
		
		return request;			
	}

	/**
	 * @param homeCommunityUid
	 * @param repositoryUniqueId
	 * @param documentId
	 * @return
	 */
	public RetrieveDocumentSetRequest createRetrieveDocumentSetRequest(String homeCommunityUid, String repositoryUniqueId, String documentId)
	{
		RetrieveDocumentSetRequest request = new RetrieveDocumentSetRequest();
		
		DocumentRequest_type0 documentRequest = new DocumentRequest_type0();
		
		DocumentRequest_type0 [] documentRequestArray = new DocumentRequest_type0[1];
		documentRequestArray[0] = documentRequest;
		
		//TODO: need to set values from URN?
		LongName homeComunityId = new LongName();
		homeComunityId.setLongName(homeCommunityUid);
		documentRequest.setHomeCommunityId(homeComunityId);
		
		LongName repositoryId = new LongName();
		// this should be a parameter of the call, the originating site Id for DoD documents will be 200, but that 
		// is not what the repository Id is (it will be a part of the request)
		repositoryId.setLongName(repositoryUniqueId);
		documentRequest.setRepositoryUniqueId(repositoryId);
		
		LongName documentUniqueId = new LongName();
		documentUniqueId.setLongName(documentId);
		documentRequest.setDocumentUniqueId(documentUniqueId);
		
		RetrieveDocumentSetRequestType documentSetRequestType = new RetrieveDocumentSetRequestType();
		
		documentSetRequestType.setDocumentRequest(documentRequestArray);
		
		request.setRetrieveDocumentSetRequest(documentSetRequestType);
		
		
		return request;			
	}

	
	/**
	 * 
	 * @param documentSetResponseType
	 * @return
	 * @throws IOException
	 */
	public ImageStreamResponse translate(RetrieveDocumentSetResponse documentSetResponseType)
	throws IOException, ImageNotFoundException, MethodException
	{
		RegistryResponseType responseType = 
			documentSetResponseType.getRetrieveDocumentSetResponse().getRegistryResponse();
		String responseString = responseType.getStatus().toString();
		TransactionContext transactionContext = TransactionContextFactory.get();
		transactionContext.addDebugInformation("Retrieve has a response of '" + responseString + "'.");
		if(responseString.equals(responseStatusSuccess.toString()))
		{
			logger.info("Document response status 'successful'");
			DocumentResponse_type0 [] documentResponses = 
				documentSetResponseType.getRetrieveDocumentSetResponse().getRetrieveDocumentSetResponseTypeSequence_type0().getDocumentResponse();
			
			// how are we handling multiple responses? probably won't ever happen since we only make one request at a time
			for(DocumentResponse_type0 documentResponse : documentResponses)
			{
				String mediaType = documentResponse.getMimeType().getLongName();
				
				DataHandler dataHandler = documentResponse.getDocument();
				
				// don't have the checksum or the length of the file
				ImageStreamResponse response = 
					new ImageStreamResponse(
						new ByteBufferBackedImageInputStream(dataHandler.getDataSource().getInputStream(), 0));
				response.setImageQuality(ImageQuality.DIAGNOSTICUNCOMPRESSED); // not sure...
				response.setMediaType(mediaType);
				transactionContext.setDataSourceImageFormatReceived(response.getImageFormat() == null ? "" : response.getImageFormat().toString());
				// This will always be DIAGNOSTICUNCOMPRESSED (see above) but oh well
				transactionContext.setDataSourceImageQualityReceived(response.getImageQuality().toString());
				
				return response;
			}
		}
		else
		{
			logger.info("Document response status 'error'");
			org.oasis.ebxml.regrep.rs.adb.RegistryErrorList_type0 errorList = responseType.getRegistryErrorList();
			if((errorList != null) && (errorList.getRegistryError() != null) && (errorList.getRegistryError().length > 0))
			{
				// just getting the first error to create the exception
				org.oasis.ebxml.regrep.rs.adb.RegistryError_type0 error = errorList.getRegistryError()[0];
				String errorCode = error.getErrorCode();				
				if(ErrorCode.MISSING_DOCUMENT.getOpcode().equals(errorCode))
				{
					throw new ImageNotFoundException(errorCode + " retrieving artifact, " + error.getCodeContext());
				}
				else
					throw new MethodException(errorCode + " retriving artifact, " + error.getCodeContext());
			}
			throw new MethodException("Received status of error retrieving artifact with no specified error description");			
		}
		
		return null;
	}
	
	
	/**
	 * Parameters may be in any number of classifications schemes (e.g. SNOMED)
	 * This method translates the parameters into VA classification schemes.
	 * 
	 * @param parameters
	 */
	private void translateStoredQueryParameters(StoredQueryParameterList parameters)
	{
		StoredQueryParameter classCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.classCode);
		translateCodeParameter(classCodeParameter, CodingScheme.VADOCUMENTCLASS);
		
		StoredQueryParameter confidentialityCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.confidentialityCodeList);
		translateCodeParameter(confidentialityCodeParameter, CodingScheme.VACONFIDENTIALITY);
		
		StoredQueryParameter eventCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.eventCodeList);
		translateCodeParameter(eventCodeParameter, CodingScheme.VAPROCEDURE);
		
		StoredQueryParameter facilityCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.healthcareFacilityTypeCode);
		translateCodeParameter(facilityCodeParameter, CodingScheme.VAFACILITYTYPE);

		StoredQueryParameter practiceSettingCodeParameter = parameters.getByName(FindDocumentStoredQueryParameterNames.practiceSettingCode);
		translateCodeParameter(practiceSettingCodeParameter, CodingScheme.VAPRACTICESETTING);
	}

	/**
	 * 
	 * @param queryExpression
	 */
	private void parseQuery(QueryExpressionType queryExpression)
	{
		// if no query expression language specified then assume an XML query
		WellKnownUUID queryLanguageUUID = getDefaultQueryLanguageUUID();
		
		ReferenceURI queryLanguage = queryExpression.getQueryLanguage();
		if(queryLanguage != null)
			try
			{
				queryLanguageUUID = EbXMLRsUUID.get(queryLanguage.toString());
			}
			catch (URISyntaxException x)
			{
				logger.error(x);
			}

		
	}

	/**
	 * Get the maximum allowed results, handling all the null and 
	 * negative numbers and the conversion errors.
	 * 
	 * @param adhocQueryRequest
	 * @return
	 */
	private long getMaxResults(AdhocQueryRequest adhocQueryRequest)
	{
		long maxResults = -1;
		BigInteger maximumResults = adhocQueryRequest.getMaxResults() == null ? 
				BigInteger.valueOf(-1L) :
				adhocQueryRequest.getMaxResults();
		maxResults = maximumResults.longValue();
		if(maxResults < 0)
			maxResults = -1;
		if( !maximumResults.equals(BigInteger.valueOf(maxResults)) )	// if we can't convert to a long, assume unlimited results
			maxResults = -1;
		
		return maxResults;
	}

	/**
	 * Translate the List<DocumentSets> from the core into a valid
	 * query response.
	 * 
	 * e.g.
	 * <query:AdhocQueryResponse 
	 * xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" 
	 * xsi:schemaLocation="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0 ../schema/ebRS/query.xsd"
	 * xmlns:query="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0" 
	 * xmlns:rim="urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0" 
	 * status="Success">
	 *   <rim:RegistryObjectList>
	 *     <rim:ExtrinsicObject 
	 *       xmlns:q="urn:oasis:names:tc:ebxml-regrep:xsd:query:3.0" 
	 *       xmlns:rim="urn:oasis:names:tc:ebxml-regrep:xsd:rim:3.0"
	 *       id="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf" 		// the unique identifier of the extrinsic object
	 *       isOpaque="false" 
	 *       mimeType="text/xml" 										// the mime type of the extrinsic object
	 *       objectType="urn:uuid:7edca82f-054d-47f2-a032-9b2a5b5186c1"	// the UUID indicates that this is a document
	 *       status="urn:oasis:names:tc:ebxml-regrep:StatusType:Approved">	// the URN indicates that the document is approved
	 *         <rim:Slot name="URI">									// the URI where the document may be accessed from
	 *           <rim:ValueList>
	 *             <rim:Value>http://localhost:8080/XDS/Repository/08a15a6f-5b4a-42de-8f95-89474f83abdf.xml</rim:Value>
	 *           </rim:ValueList>
	 *         </rim:Slot>
	 *         ...
	 *         <rim:Name>
	 *           <rim:LocalizedString charset="UTF-8" value="The Document Name" xml:lang="en-us"/>
	 *         </rim:Name>
	 *         <rim:Description>
	 *           <rim:LocalizedString charset="UTF-8" value="The description of the document" xml:lang="en-us"/>
	 *         </rim:Description>
	 *         
	 *         <!-- Classification is  -->
	 *         <rim:Classification 										// Classification is a 3 part coding; coding scheme, code value, code value display name
	 *           classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a"	// the UUID of XDSDocumentEntry.classCode
	 *           classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf" 		// reference to the ExtrinsicObject element id
	 *           id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf" 	// the unique identifier of this Classification
	 *           nodeRepresentation="Education"							// this is the code value
	 *           objectType="urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification"
	 *         >
	 *             <rim:Slot name="codingScheme">		// this is the coding scheme name
	 *               <rim:ValueList>
	 *                 <rim:Value>Connect-a-thon classCodes</rim:Value>
	 *               </rim:ValueList>
	 *             </rim:Slot>
	 *             <rim:Name>							// this is the code value display name
	 *               <rim:LocalizedString charset="UTF-8" value="Education" xml:lang="en-us"/>
	 *             </rim:Name>
	 *             <rim:Description/>
	 *         </rim:Classification>
	 * 
	 * @param documentSets
	 * @return
	 */
	public AdhocQueryResponse translate(List<DocumentSet> documentSets)
	throws TranslationException
	{
		AdhocQueryResponse response = new AdhocQueryResponse();

		response.setStatus(responseStatusSuccess);

		RegistryObjectListType rolt = new RegistryObjectListType();
		response.setRegistryObjectList(rolt);
		
		for(DocumentSet documentSet : documentSets)
		{
			for(Document document : documentSet)
			{
				ExtrinsicObjectType eoType = new ExtrinsicObjectType();
				
				populateExtrinsicObject(eoType, document);
				rolt.addIdentifiable(eoType);
			}
		}
		
		response.setRegistryObjectList(rolt);
		
		return response;
	}

	/**
	 * 
	 * @param dateFormatter
	 * @param document
	 * @param eoType
	 * @throws TranslationException
	 */
	private void populateExtrinsicObject(
		ExtrinsicObjectType eoType,
		Document document)
	throws TranslationException
	{
		SimpleDateFormat dateFormatter = new SimpleDateFormat(XCATranslatorAdb.iheDateFormat);
		
		ReferenceURI documentReferenceUri;
		try
		{
			// set home community ID attribute
			eoType.setHome( new org.apache.axis2.databinding.types.URI(document.getGlobalArtifactIdentifier().getHomeCommunityId()) );
			
			// set identifier attribute
			eoType.setId( new org.apache.axis2.databinding.types.URI(document.getGlobalArtifactIdentifier().getDocumentUniqueId()) );
			
			documentReferenceUri = new ReferenceURI();
			documentReferenceUri.setReferenceURI( new org.apache.axis2.databinding.types.URI(document.getGlobalArtifactIdentifier().toString()) );
		}
		catch (MalformedURIException x1)
		{
			logger.error("Error creating a home, ID or document reference URI", x1);
			throw new TranslationException(x1);
		}
		
		// set mime type attribute
		// JMW the documents don't have the media type set so
		// this was throwing an exception since getMediaType() was null
		// CTB - if no media type known, then send as application/octet-stream, which
		// is basically saying "uh, I dunno'"
		eoType.setMimeType( 
			createLongName(document.getMediaType() == null ? 
				MediaType.APPLICATION_OCTETSTREAM.toString() : 
				document.getMediaType().toString()
			)
		);
		
		// set status attribute
		eoType.setStatus(documentStatusApproved);

		// create the document URI slot
		SlotType1 uriSlot = new SlotType1();
		populateSingleValueSlot(uriSlot, "URI", document.getDocumentUrn().toString());
		eoType.addSlot(uriSlot);
		
		// set creation date slot
		if(document.getCreationDate() != null)
		{
			SlotType1 creationTimeSlot = new SlotType1();
			populateSingleValueSlot(creationTimeSlot, "creationTime", dateFormatter.format(document.getCreationDate()) );
			eoType.addSlot(creationTimeSlot);
		}
		
		// set language code slot
		SlotType1 languageCodeSlot = new SlotType1();
		populateSingleValueSlot(languageCodeSlot, "languageCode", "en-us" );
		eoType.addSlot(languageCodeSlot);
		
		// set source patient ID slot
		if(document.getPatientId() != null)
		{
			SlotType1 sourcePatientIdSlot = new SlotType1();
			populateSingleValueSlot(sourcePatientIdSlot, "sourcePatientId", document.getPatientId() );
			eoType.addSlot(sourcePatientIdSlot);
		}
		
		//
		if(document.getContentLength() > 0)
		{
			SlotType1 sourcePatientIdSlot = new SlotType1();
			populateSingleValueSlot(sourcePatientIdSlot, "size", Long.toString(document.getContentLength()) );
			eoType.addSlot(sourcePatientIdSlot);
		}

		// populate repository, patient and document ID here
		
		// if the checksum is available and is an SHA1 calculated value
		// then add the hash slot
		ChecksumValue checksum = document.getChecksumValue();
		if( checksum != null && "SHA1".equals(checksum.getAlgorithm()) ) 
		{
			SlotType1 hashSlot = new SlotType1();
			populateSingleValueSlot(hashSlot, "hash", checksum.getValue().toString(16) );
			eoType.addSlot(hashSlot);
		}

		// add the confidentiality code as a Classification element
		ClassificationType confidentialityClassification = 
			translate(
				documentReferenceUri,
				IheUUID.DOC_ENTRY_CONFIDENTIALITY_CODE.getUrn(),
				new ClassifiedValue( CodingScheme.VACONFIDENTIALITY, 
						Integer.toString(document.getConfidentialityCode()) ),
				Integer.toString(document.getConfidentialityCode()));
		eoType.addClassification(confidentialityClassification);
		
		// add the document type as a Classification element
		// this element requires translation from VA to LOINC classification before
		// sending
		ClassificationType docTypeClassification = translate(
			documentReferenceUri,
			IheUUID.DOC_ENTRY_TYPE_CODE.getUrn(),
			CodingScheme.VADOCUMENTCLASS, 
			CodingScheme.LOINC,
			document.getClinicalType()
		);
		eoType.addClassification(docTypeClassification);
		
		// NOTES:
		// <rim:Classification 
		// classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a" 
		// classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf" 
		// id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf" 
		// nodeRepresentation="Education" 
		// objectType="Urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification">
		//
		// The classifiedObject references the URN of the document
	}

	/**
	 * 
	 * @param externalIdentifier
	 * @param registryObjectId
	 * @param externalIdentifierName
	 * @param externalIdentifierValue
	 * @param identificationScheme
	 * @param description
	 * @throws MalformedURIException 
	 */
	private void populateExternalIdentifier(
		ExternalIdentifierType externalIdentifier,
		ReferenceURI registryObjectId,
		String externalIdentifierName,		// e.g. "XDSDocumentEntry.uniqueId" 
		String externalIdentifierValue,		// e.g. "1.3.6.1.4.1.21367.2005.3.99.1.1010"
		ReferenceURI identificationScheme,		// e.g. "urn:uuid:2e82c1f6-a085-4c72-9da3-8640a32e42ab"
		String description) 
	throws MalformedURIException
	{
		// create and set a unique identifier
		GUID id = new GUID();
		externalIdentifier.setId( new org.apache.axis2.databinding.types.URI(id.toString()) );
		
		// set the reference to the registry object
		externalIdentifier.setObjectType(registryObjectId);
		
		// the name of the external identifier (i.e. the field/property name)
		if(externalIdentifierName != null)
		{
			externalIdentifier.setName(new InternationalStringType());
			InternationalStringType nameI18n = externalIdentifier.getName();
			nameI18n.addInternationalStringTypeSequence(new InternationalStringTypeSequence());
			nameI18n.getInternationalStringTypeSequence()[0].setLocalizedString(new LocalizedStringType());
			LocalizedStringType nameL9d = nameI18n.getInternationalStringTypeSequence()[0].getLocalizedString();
			nameL9d.setValue(new FreeFormText());
			nameL9d.getValue().setFreeFormText(externalIdentifierName);
			//nameL9d.addNewCharset().setStringValue("UTF-8");
			//nameL9d.xgetValue().setStringValue("en-us");
		}
		
		// set the identification scheme
		externalIdentifier.setIdentificationScheme(identificationScheme);
		
		// set the value of the external identifier (e.g. the field/property value)
		externalIdentifier.setValue( createLongName(externalIdentifierValue) );
		
		// a human readable description
		if(description != null)
		{
			externalIdentifier.setDescription(new InternationalStringType());
			InternationalStringType descriptionI18n = externalIdentifier.getDescription();
			descriptionI18n.addInternationalStringTypeSequence(new InternationalStringTypeSequence());
			descriptionI18n.getInternationalStringTypeSequence()[0].setLocalizedString(new LocalizedStringType());
			LocalizedStringType decsriptionL9d = descriptionI18n.getInternationalStringTypeSequence()[0].getLocalizedString();
			decsriptionL9d.setValue(new FreeFormText());
			decsriptionL9d.getValue().setFreeFormText(externalIdentifierName);
		}
	}
	
	/**
	 * @param string
	 * @return
	 */
	public LongName createLongName(String string)
	{
		LongName longName = new LongName();
		longName.setLongName(string);
		return longName;
	}

	/**
	 * Create a Classification element, translating the classification value
	 * into the target classification scheme.
	 * 
	 * @param classifiedObjectUri
	 * @param classificationScheme
	 * @param sourceClassificationScheme
	 * @param destinationClassificationScheme
	 * @param sourceClassificationValue
	 * @return
	 * @throws TranslationException
	 */
	@TranslationMethod(unmatchedMethod=true)
	private ClassificationType translate(
		ReferenceURI classifiedObjectUri,
		java.net.URI classificationScheme,
		CodingScheme sourceClassificationScheme,
		CodingScheme destinationClassificationScheme,
		String sourceClassificationValue
	) 
	throws TranslationException
	{
		// create the "translated" classified value with the untranslated values
		// these will be the default if the translation does not work
		ClassifiedValue[] translatedClassifiedValue = new ClassifiedValue[]{ 
			new ClassifiedValue( sourceClassificationScheme, sourceClassificationValue )
		};
		try
		{
			SchemeTranslationSPI classificationSchemeTranslator = 
				getTranslationFactory().getSchemeTranslator(sourceClassificationScheme.toString(), destinationClassificationScheme.toString());
			if(classificationSchemeTranslator != null)
				translatedClassifiedValue = classificationSchemeTranslator.translate(sourceClassificationValue);
			else
				logger.error("No Installed translator for '" + sourceClassificationScheme + 
						"' to '" + destinationClassificationScheme + "', unable to translate classification types.");
		}
		catch (URISyntaxException x)
		{
			String msg = "Invalid URI syntax, either '" + sourceClassificationScheme.toString() + "'  or '" +
				destinationClassificationScheme.toString() + "' are invalid classification type URIs.";
			logger.error(msg, x);
			throw new TranslationException(msg, x);
		}
		
		ClassificationType classificationType = 
			translate(
				classifiedObjectUri,
				classificationScheme,
				translatedClassifiedValue[0],
				(translatedClassifiedValue[0]).getCodeValue());
		classificationType.setClassifiedObject(classifiedObjectUri);
		
		return classificationType;
	}
	
	/**
	 * Create a Classification that does not require any translation of 
	 * the classification scheme.
	 * 
	 * @param classifiedObjectUri - the extrinsic object being classified i.e. a document reference
	 * @param classificationSchemeUri - the classification scheme e.g. "healthcare facility type code" as a URI
	 * @param classifiedValue - the coding scheme and value e.g. LOINC 1234-4
	 * @param classificationDescription - a human readable coding  e.g. "hospital"
	 * @return
	 * @throws TranslationException
	 * 
	 * Example XML:
	 * 
	 * <rim:Classification 
	 * classificationScheme="urn:uuid:41a5887f-8865-4c09-adf7-e362475b143a"
	 * classifiedObject="urn:uuid:08a15a6f-5b4a-42de-8f95-89474f83abdf"
	 * id="urn:uuid:ac872fc0-1c6e-439f-84d1-f76770a0ccdf"
	 * nodeRepresentation="Education"
	 * objectType="Urn:oasis:names:tc:ebxml-regrep:ObjectType:RegistryObject:Classification"
	 * >
	 * 	<rim:Slot name="codingScheme">
	 * 		<rim:ValueList>
	 * 			<rim:Value>Connect-a-thon classCodes</rim:Value>
	 * 		</rim:ValueList>
	 * 	</rim:Slot>
	 * 	<rim:Name>
	 * 		<rim:LocalizedString charset="UTF-8" value="Education" xml:lang="en-us"/>
	 * 	</rim:Name>
	 * 	<rim:Description/>
	 * </rim:Classification>
	 */
	@TranslationMethod(unmatchedMethod=true)
	private ClassificationType translate(
		ReferenceURI classifiedObjectUri,
		java.net.URI classificationSchemeUri,
		ClassifiedValue classifiedValue,
		String classificationDescription
		) 
	throws TranslationException 
	{
		ClassificationType classificationType = new ClassificationType();
		
		org.apache.axis2.databinding.types.URI classSchemeUri = translate(classificationSchemeUri);
		classificationType.setObjectType(objectTypeClassification);
		
		// set the reference to the classified object (the document)
		classificationType.setClassifiedObject(classifiedObjectUri);

		ReferenceURI classificationReferenceUri = new ReferenceURI();
		classificationReferenceUri.setReferenceURI(classSchemeUri);
		classificationType.setClassificationScheme( classificationReferenceUri );
		classificationType.setNodeRepresentation( createLongName(classifiedValue.getCodeValue()) );
		
		// ID is a required attribute
		try
		{
			org.apache.axis2.databinding.types.URI identifier = 
				new org.apache.axis2.databinding.types.URI((String)null, (new GUID()).toShortString());
			classificationType.setId(identifier);
		}
		catch (MalformedURIException x)
		{
			logger.error(x);
			throw new TranslationException(x.getMessage());
		}
		
		if(classificationDescription != null)
		{
			InternationalStringType name = new InternationalStringType();
			LocalizedStringType localizedName = new LocalizedStringType();
			FreeFormText text = new FreeFormText();
			text.setFreeFormText(classificationDescription);
			localizedName.setValue( text );
			
			InternationalStringTypeSequence i18nTypeSequence = new InternationalStringTypeSequence();
			i18nTypeSequence.setLocalizedString(localizedName);
			
			name.addInternationalStringTypeSequence(i18nTypeSequence);
			//localizedName.setLang("en-us");
			
			 classificationType.setName(name);
		}
		
		SlotType1 codingSchemeSlot = new SlotType1();
		populateSingleValueSlot(codingSchemeSlot, "codingScheme", classifiedValue.getCodingScheme().toString());
		
		classificationType.addSlot(codingSchemeSlot);
		
		return classificationType;
	}

	/**
	 * 
	 * @param javaNetUri
	 * @return
	 * @throws TranslationException
	 */
	public org.apache.axis2.databinding.types.URI translate(java.net.URI javaNetUri) 
	throws TranslationException
	{
		org.apache.axis2.databinding.types.URI adbUri;
		try
		{
			adbUri = new org.apache.axis2.databinding.types.URI(javaNetUri.toString());
		}
		catch (MalformedURIException x)
		{
			logger.error("Error converting java net URI to ADB form", x);
			throw new TranslationException("Error converting java net URI to ADB form", x);
		}
		
		return adbUri;
	}
	
	public java.net.URI translate(org.apache.axis2.databinding.types.URI adbUri) 
	throws TranslationException
	{
		return java.net.URI.create( adbUri.toString() );
	}
	
	/**
	 * 
	 * @param value
	 * @return
	 * @throws URNFormatException
	 */
	private void populateSingleValueSlot(SlotType1 slot, String key, String value)
	{
		slot.setName( createLongName(key) );
		ValueListType slotValues = new ValueListType();
		slot.setValueList(slotValues);
		LongName slotValue = createLongName(value);
		ValueListTypeSequence vltSequence = new ValueListTypeSequence();
		vltSequence.setValue(slotValue);
		slotValues.addValueListTypeSequence(vltSequence);
		
		return;
	}

	/**
	 * @param x
	 * @return
	 */
	public AdhocQueryResponse translate(Exception x)
	{
		AdhocQueryResponse response = new AdhocQueryResponse();
		response.setStatus(XCATranslatorAdb.responseStatusFailure);
		
		return response;
	}

}
