/**
 * 
 */
package gov.va.med.imaging.ihe;

import gov.va.med.OID;
import gov.va.med.WellKnownOID;
import gov.va.med.imaging.exchange.translation.AbstractTranslator;
import gov.va.med.imaging.exchange.translation.TranslationMethod;
import gov.va.med.imaging.ihe.exceptions.TranslationException;
import gov.va.med.imaging.ihe.request.StoredQueryParameter;
import gov.va.med.imaging.ihe.xca.XcaTimeZone;
import gov.va.med.imaging.terminology.ClassifiedValue;
import gov.va.med.imaging.terminology.CodingScheme;
import gov.va.med.imaging.terminology.SchemeTranslationSPI;
import gov.va.med.imaging.terminology.SchemeTranslationServiceFactory;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

/**
 * @author DNS
 *
 */
public abstract class AbstractXCATranslator
extends AbstractTranslator
{
	public static final OID AUTHORITY_DOMAIN = WellKnownOID.VA_DOCUMENT.getCanonicalValue();
	public static final String AUTHORITY_DOMAIN_TYPE = "ISO";
	public static final String iheDateFormat = "yyyyMMddHHmm";
	
	public static String getIheDateFormat()
	{
		return iheDateFormat;
	}
	
	private final static DateFormat iheDateFormatter;//
	static
	{
		iheDateFormatter = new SimpleDateFormat(getIheDateFormat());
		iheDateFormatter.setTimeZone(XcaTimeZone.getTimeZone());
	}
	//= new SimpleDateFormat(getIheDateFormat());
	private final static SchemeTranslationServiceFactory translationFactory;
	private final static WellKnownUUID defaultQueryLanguageUUID = EbXMLRsUUID.QUERY_LANGUAGE_XQUERY;
	
	static
	{
		translationFactory = SchemeTranslationServiceFactory.getFactory();
	}
	
	protected static SchemeTranslationServiceFactory getTranslationFactory()
	{
		return translationFactory;
	}

	public static WellKnownUUID getDefaultQueryLanguageUUID()
	{
		return defaultQueryLanguageUUID;
	}
	
	public static String translateToIHEDate(long date)
	{
		synchronized(iheDateFormatter)
		{
			return iheDateFormatter.format(date);
		}
	}
	
	public static long translateFromIHEDate(String iheDate) 
	throws ParseException
	{
		synchronized(iheDateFormatter)
		{
			return iheDateFormatter.parse(iheDate).getTime();
		}
	}

	/**
	 * 
	 */
	public static String createHL7PatientID(String patientICN)
	{
		HL7PatientIdentifier hl7Identifier = new HL7PatientIdentifier(
			patientICN, null, null, 
			new HL7PatientIdentifier.AssigningAuthority(null, AUTHORITY_DOMAIN.toString(), AUTHORITY_DOMAIN_TYPE),
			null, null, null, null
			);
		
		return hl7Identifier.toString();
	}
	
	/**
	 * 
	 * @param hl7Identifier
	 * @return
	 * @throws TranslationException
	 */
	public static String parseHL7PatientID(String hl7Identifier) 
	throws TranslationException
	{
		try
		{
			HL7PatientIdentifier patientIdentifier = HL7PatientIdentifier.create(hl7Identifier);

			if( ! AUTHORITY_DOMAIN.toString().equals(patientIdentifier.getAssigningAuthority().getUniversalID()) )
				throw new TranslationException(
					"Patient identifier '" + 
					hl7Identifier + 
					"' has an alien authority domain.  Value is '" +
					patientIdentifier.getAssigningAuthority().getUniversalID() + 
					"' and must be '" + 
					AUTHORITY_DOMAIN.toString()
				);
			if( ! AUTHORITY_DOMAIN_TYPE.equals(patientIdentifier.getAssigningAuthority().getUniversalIDType()) )
				throw new TranslationException(
					"Patient identifier '" + 
					hl7Identifier + 
					"' has an alien authority domain type.  Value is '" +
					patientIdentifier.getAssigningAuthority().getUniversalIDType() + 
					"' and must be '" + 
					AUTHORITY_DOMAIN_TYPE.toString()
				);
			
			return patientIdentifier.getPatientId();
		}
		catch (ParseException x)
		{
			throw new TranslationException(x);
		}
	}
	
	/**
	 * 
	 * @param sourceClassifiedValues
	 * @param destinationCodingSchemes
	 * @return
	 * @throws TranslationException
	 */
	@TranslationMethod(unmatchedMethod=true)
	public static ClassifiedValue[] translate(
		gov.va.med.imaging.terminology.ClassifiedValue[] sourceClassifiedValues, 
		gov.va.med.imaging.terminology.CodingScheme[] destinationCodingSchemes)
	throws TranslationException
	{
		List<ClassifiedValue> results = new ArrayList<ClassifiedValue>();
		for(gov.va.med.imaging.terminology.ClassifiedValue sourceClassifiedValue : sourceClassifiedValues)
		{
			ClassifiedValue[] translationResults = translate(sourceClassifiedValue, destinationCodingSchemes);
			for(ClassifiedValue result : translationResults)
				results.add(result);
		}
		
		return results.toArray(new ClassifiedValue[results.size()]);
	}
	/**
	 * Translate a ClassifiedValue into a ClassifiedValue using one of the
	 * given destination coding schemes.
	 * 
	 * For each of the possible destination coding schemes, try to get a translator
	 * and then do the translation.
	 * The first coding scheme that results in a successful translation scheme is the
	 * resulting translating, i.e. the array of CodingScheme is order-sensitive
	 * 
	 * If no translation SPI to any destination coding scheme is available then the 
	 * original ClassifiedValue is returned, and an error is generated in the log.
	 * 
	 * If no translation in any destination coding scheme is available then the 
	 * original ClassifiedValue is returned, and an info is generated in the log.
	 * 
	 * @param classifiedValue
	 * @param destinationCodingSchemes
	 * @return
	 * @throws TranslationException
	 */
	@TranslationMethod(unmatchedMethod=true)
	public static ClassifiedValue[] translate(
		gov.va.med.imaging.terminology.ClassifiedValue sourceClassifiedValue, 
		gov.va.med.imaging.terminology.CodingScheme[] destinationCodingSchemes)
	throws TranslationException
	{
		if(destinationCodingSchemes == null)
			return new ClassifiedValue[]{sourceClassifiedValue};
		
		// For each of the possible destination coding schemes, try to get a
		// translator and then do the translation
		// The first coding scheme that results in a successful translation
		// scheme is the
		// resulting translating, i.e. the array of CodingScheme is
		// order-sensitive

		// note that we differentiate not finding any translator from not
		// finding
		// a translation code, the former is an error, the later just an info
		// message
		boolean translatorFound = false;

		for (CodingScheme destinationCodingScheme : destinationCodingSchemes)
		{
			SchemeTranslationSPI classificationSchemeTranslator = null;
			classificationSchemeTranslator = getTranslationFactory().getSchemeTranslator(
					sourceClassifiedValue.getCodingScheme(),
					destinationCodingScheme);

			if (classificationSchemeTranslator != null)
			{
				translatorFound = true;
				ClassifiedValue[] translatedClassifiedValues = 
					classificationSchemeTranslator.translate(sourceClassifiedValue.getCodeValue());

				// if the translation actually translated to the target coding scheme
				if(! sourceClassifiedValue.getCodingScheme().equals(translatedClassifiedValues[0].getCodingScheme()) )
					return translatedClassifiedValues;
			}
		}

		// print a message that the classification could not be translated
		StringBuilder sbMsg = new StringBuilder();
		sbMsg.append('[');
		for (CodingScheme destinationCodingScheme : destinationCodingSchemes)
		{
			if (sbMsg.length() > 0)
				sbMsg.append(',');
			sbMsg.append(destinationCodingScheme);
		}
		sbMsg.append(']');
		if (translatorFound)
			getLogger().info(	
				"Unable to translate '"
				+ sourceClassifiedValue.getCodingScheme()
				+ "' to '"
				+ sbMsg.toString()
				+ "', classification type remains in source scheme.");
		else
			getLogger().error(
					"No Installed translator for '"
							+ sourceClassifiedValue.getCodingScheme()
							+ "' to '" + sbMsg.toString()
							+ "', unable to translate classification types.");

		return new ClassifiedValue[]{sourceClassifiedValue};
	}

	/**
	 * Translate a parameter with an associated scheme parameter into a different
	 * code scheme.  If there is no translation available then the code
	 * values are left as is.
	 * 
	 * @param codeParameter
	 * @param codeSchemeParameter
	 * @param destinationCodeScheme
	 */
	public static void translateCodeParameter(
		StoredQueryParameter storedQueryParameter,
		CodingScheme destinationCodeScheme)
	{
		if(storedQueryParameter != null && destinationCodeScheme != null)
		{
			SchemeTranslationServiceFactory schemeTranslationFactory = SchemeTranslationServiceFactory.getFactory();
			
			storedQueryParameter.translateToClassificationScheme(schemeTranslationFactory, destinationCodeScheme);
		}
	}
}
