package gov.va.med.imaging.dicom.io;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataElementTag;
import gov.va.med.imaging.dicom.DataSet;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;

import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * 
 * @author       DNS
 *
 */
public class Part10DataSetReader
{
	// The list of element tags that may occur in a Part 10 header
	// The occurrence of any other element tag will end the reading of the
	// Part 10 header
	public final static DataElementTag[] part10ElementTags = new DataElementTag[]
    {
		new DataElementTag(0x0002, 0x0000),		// Group Length
		new DataElementTag(0x0002, 0x0001),		// File Meta Information Version
		new DataElementTag(0x0002, 0x0002),		// Media Storage SOP Class UID
		new DataElementTag(0x0002, 0x0003),		// Media Storage SOP Instance UID
		new DataElementTag(0x0002, 0x0010),		// Transfer Syntax UID
		new DataElementTag(0x0002, 0x0012),		// Implementation Class UID
		new DataElementTag(0x0002, 0x0013),		// Implementation Class UID
		new DataElementTag(0x0002, 0x0016),		// Source Application Entity Title
		new DataElementTag(0x0002, 0x0100),		// Private Information Creator UID
		new DataElementTag(0x0002, 0x0102)		// Private Information
    };
	
	private static boolean isPart10HeaderElementTag(DataElementTag elementTag)
	{
		for(DataElementTag headerTag : part10ElementTags)
			if(headerTag.equals(elementTag))
				return true;
		return false;
	}
	
	private final DataElementReader reader;
	private Logger logger = Logger.getLogger(this.getClass().getName());
	
	/**
	 * 
	 * @param inputStream
	 * @param dataElementFactory
	 */
	public Part10DataSetReader(DataElementLimitedInputStream inputStream)
	{
		logger.log(Level.FINE, "Part10DataSetReader instantiating... ");
		DataElementFactory headerElementFactory = DataElementFactory.getDefaultHeaderElementFactory();
		this.reader = new DataElementReader(inputStream, headerElementFactory);
		logger.log(Level.FINE, "Part10DataSetReader instantiated. ");
	}

	/**
	 * Read a DataSet, starting from the current position and stopping when an
	 * element tag not in the Part10 list is encountered.  The stream is left at the
	 * start of the first non-Part10 element.
	 * @return
	 * @throws IOException 
	 * @throws UnsupportedOperationException 
	 * @throws InvalidVRException 
	 * @throws InvalidVRModeException 
	 * @throws DicomFormatException 
	 */
	public DataSet readPart10DataSet() 
	throws 
		UnsupportedOperationException, IOException, 
		InvalidVRModeException, InvalidVRException, DicomFormatException
	{
		logger.entering(this.getClass().getName(), "readPart10DataSet");
		DataSet dataSet = new DataSet(reader.getDataElementFactory().getTransferSyntaxUid());
		
		for( DataElementTag dataElementTag = reader.getInputStream().peekNextDataElementTag();
			dataElementTag != null && isPart10HeaderElementTag(dataElementTag);
			dataElementTag = reader.getInputStream().peekNextDataElementTag() )
		{
			DataElement dataElement = reader.readNextDataElement();
			dataSet.add(dataElement);
			logger.log(Level.INFO, "Added data element " + dataElement.getDataElementTag().toString() + ", " +
					dataElement.getValueRepresentation()[0].name() + " to data set.");
		}
		
		logger.exiting(this.getClass().getName(), "readPart10DataSet");
		return dataSet;
	}
}
