/**
 * Package: MAG - VistA Imaging
 * WARNING: Per VHA Directive 2004-038, this routine should not be modified.
 * Date Created: Jul 23, 2008
 * Site Name:  Washington OI Field Office, Silver Spring, MD
 * @author       DNS
 * @version 1.0
 *
 * ----------------------------------------------------------------
 * Property of the US Government.
 * No permission to copy or redistribute this software is given.
 * Use of unreleased versions of this software requires the user
 * to execute a written test agreement with the VistA Imaging
 * Development Office of the Department of Veterans Affairs,
 * telephone (DNS
 * 
 * The Food and Drug Administration classifies this software as
 * a Class II medical device.  As such, it may not be changed
 * in any way.  Modifications to this software may result in an
 * adulterated medical device under 21CFR820, the use of which
 * is considered to be a violation of US Federal Statutes.
 * ----------------------------------------------------------------
 */
package gov.va.med.imaging.dicom.io;

import gov.va.med.imaging.dicom.DataElement;
import gov.va.med.imaging.dicom.DataElementFactory;
import gov.va.med.imaging.dicom.DataSet;
import gov.va.med.imaging.dicom.TransferSyntaxUid;
import gov.va.med.imaging.dicom.exceptions.DicomFormatException;
import gov.va.med.imaging.dicom.exceptions.IncompatibleValueLengthField;
import gov.va.med.imaging.dicom.exceptions.InvalidVRException;
import gov.va.med.imaging.dicom.exceptions.InvalidVRModeException;
import gov.va.med.imaging.dicom.exceptions.MissingDicmPreambleException;
import gov.va.med.imaging.dicom.exceptions.MissingDicmPrefixException;
import gov.va.med.imaging.dicom.exceptions.ValueRepresentationInterpretationException;
import gov.va.med.imaging.dicomviewer.SecurityContext;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * @author       DNS
 *
 */
public class Part10DataSetLoader
{
	private static Logger logger = Logger.getLogger(Part10DataSetLoader.class.getName());
	
	/**
	 * 
	 * @param dicomFile
	 * @return
	 * @throws InvalidVRException 
	 * @throws InvalidVRModeException 
	 * @throws DicomFormatException 
	 * @throws UnsupportedOperationException 
	 * @throws ValueRepresentationInterpretationException 
	 * @throws IncompatibleValueLengthField 
	 */
	public static DataSet load(File dicomFile) 
	throws UnsupportedOperationException, DicomFormatException, 
	InvalidVRModeException, InvalidVRException, ValueRepresentationInterpretationException, IncompatibleValueLengthField
	{
		DataInputStream din = null;
		try
        {
			System.out.println("Opening DICOM file '" + dicomFile.toString() + "'.");
	        din = new DataInputStream(  new BufferedInputStream( new FileInputStream(dicomFile) ) );
			System.out.println("DICOM file '" + dicomFile.toString() + "' opened.");
	        
	        DataSet dicomData = load(din);
	        
	        return dicomData;
        } 
		catch (FileNotFoundException e)
        {
	        e.printStackTrace();
        } 
		catch (IOException e)
        {
	        e.printStackTrace();
        }
		finally
		{
			try{din.close();}catch(Throwable t){}
		}
		
		return null;
	}
	
	/**
	 * 
	 * @param imageUrl
	 * @return
	 */
	public static DataSet load(URL imageUrl)
	{
		DataInputStream din = null;
		try
		{
			URLConnection connection = imageUrl.openConnection();
			HttpURLConnection httpConnection = null;
			if (connection instanceof HttpURLConnection)
			{
				httpConnection = (HttpURLConnection) connection;
				// httpConnection.setRequestMethod("GET");
				// httpConnection.setRequestProperty("Accept",
				// MimeTypes.buildHttpAcceptString(mimeType == null, mimeType)
				// );

				// Authorization: Basic QWxhZGRpbjpvcGVuIHNlc2FtZQ==
				SecurityContext securityContext = SecurityContext.get();
				String authentication = securityContext == null ? null : securityContext.getAsBase64();
				if (authentication != null)
					httpConnection.setRequestProperty("Authorization", "Basic " + authentication);
			}
			System.out.println("Connecting to URL '" + imageUrl.toString() + "'.");
			connection.connect();
			System.out.println("Connected to URL '" + imageUrl.toString() + "', reading from input stream.");

			// InputStream
			// BufferedInputStream
			// InputStream inS = urlConn.openStream();
			din = new DataInputStream( new BufferedInputStream(connection.getInputStream()) );
			
			// }else{
			// System.out.println("file");
			// FinS = new FileInputStream(imgURL);
			// inS = new BufferedInputStream(FinS);
			// din = new DataInputStream(inS);
			// System.out.println("file");
			// }

			return load(din);
		} 
		catch (EOFException eof)
		{
			System.out.println("DicomFile.EOFException: " + eof.getMessage());
		} 
		catch (IOException ioe)
		{
			System.out.println("DicomFile.IOException: " + ioe.getMessage());
		} 
		catch (Exception e)
		{
			System.out.println("DicomFile.Exception: " + e.getMessage());
		}
		finally
		{
			try{din.close();}catch(Throwable x){}
		}

		return null;
	}
	
	/**
	 * 
	 * @param inputStream
	 * @return
	 * @throws IOException
	 * @throws DicomFormatException
	 * @throws UnsupportedOperationException
	 * @throws InvalidVRModeException
	 * @throws InvalidVRException
	 * @throws ValueRepresentationInterpretationException
	 * @throws IncompatibleValueLengthField
	 */
    public static DataSet load(InputStream inputStream) 
    throws IOException, 
	    DicomFormatException, 
	    UnsupportedOperationException, 
	    InvalidVRModeException, InvalidVRException, ValueRepresentationInterpretationException, IncompatibleValueLengthField
    {
    	if(inputStream instanceof DataElementLimitedInputStream)
    		return load((DataElementLimitedInputStream)inputStream);
    	
    	return load(new DataElementLimitedInputStream( new DataInputStream(inputStream), TransferSyntaxUid.RAW_EXPLICIT_VR_LITTLEENDIAN));
    }
    
	/**
     * @param dicomInputStream
     * @throws IOException
	 * @throws DicomFormatException 
	 * @throws InvalidVRException 
	 * @throws InvalidVRModeException 
	 * @throws UnsupportedOperationException 
	 * @throws ValueRepresentationInterpretationException 
	 * @throws IncompatibleValueLengthField 
     */
    public static DataSet load(DataElementLimitedInputStream dicomInputStream) 
    throws IOException, 
    DicomFormatException, 
    UnsupportedOperationException, 
    InvalidVRModeException, InvalidVRException, ValueRepresentationInterpretationException, IncompatibleValueLengthField
    {
    	// read the header (the 128 bytes and the 'DICM')
    	readPreambleAndPrefix(dicomInputStream);
    	
    	Part10DataSetReader part10Reader = new Part10DataSetReader(dicomInputStream);
    	DataSet headerDataSet = part10Reader.readPart10DataSet();
    	DataElement<?> transferSyntaxElement = headerDataSet.getTransferSyntaxUID();
    	if(transferSyntaxElement == null)
    	{
    		logger.log(Level.WARNING, "The Part 10 header had no transfer syntax specified.");
    		return null;
    	}
    	
    	logger.log(Level.INFO, "Header data set read, reading data set using '" + 
    			new String(transferSyntaxElement.getRawValue()) + "' transfer syntax.");
    	TransferSyntaxUid transferSyntax = TransferSyntaxUid.getByUid( (String)transferSyntaxElement.getValue() );
    	
    	DataElementFactory dataElementFactory = DataElementFactory.getDataElementFactory(transferSyntax);
    	DataSetReader dataSetReader = new DataSetReader(dicomInputStream, dataElementFactory);

    	DataSet dataSet = dataSetReader.readDataSet();
    	
    	return dataSet;
    }

    /**
     * Except for the 128 byte preamble and the 4 byte prefix, the File Meta Information 
     * shall be encoded using the Explicit VR Little Endian Transfer Syntax (UID=1.2.840.10008.1.2.1) 
     * as defined in DICOM PS 3.5. 
     * Values of each File Meta Element shall be padded when necessary to achieve an even length as 
     * specified in PS 3.5 by their corresponding Value Representation. For compatibility with future versions 
     * of this Standard, any Tag (0002,xxxx) not defined in Table 7.1-1 shall be ignored.
     * @param dicomInputStream 
     * @param dataSet 
     * @throws IOException 
     */
    protected static void readPreambleAndPrefix(DataElementLimitedInputStream dicomInputStream) 
	throws IOException, DicomFormatException
    {
		byte[] preamble = new byte[128];
		try{dicomInputStream.readFully(preamble);}		// read this into a buffer, we may add some analysis later
		catch(IOException ioX){throw new MissingDicmPreambleException();}
		
		byte[] dicmPrefix = new byte[4];
		try{dicomInputStream.readFully(dicmPrefix);}
		catch(IOException ioX){throw new MissingDicmPrefixException();}
    }
}
