using System;
using System.Collections.Generic;
using System.Text;
using VixBuilderBusiness;
using log4net;
using System.IO;
using System.Diagnostics;
using System.Configuration;

namespace VixBuilderConsole
{
    class Program
    {
        static void Main()
        {
            string[] args = Environment.GetCommandLineArgs(); // note arg[0] contains .exe filespec
            if (args.Length < 3) // .exe, build config filespec, action at a minimum
            {
                usage();
                System.Environment.ExitCode = 1;
                return;
            }

            if (String.IsNullOrEmpty(args[0]))
            {
                Console.Write("Empty executable filepsec returned from Environment.GetCommandLineArgs()");
                System.Environment.ExitCode = 1;
                return;
            }

            string appPath = Path.GetDirectoryName(args[0]);

            try
            {
                string logConfigFile = Path.Combine(appPath, "VixBuilderLog.xml");
                string logFile = Path.Combine(appPath, "vix-build-log.txt");
                if (File.Exists(logFile))
                {
                    File.Delete(logFile);
                }
                if (!File.Exists(logConfigFile))
                {
                    throw new BuildConfigurationException("Error " + logConfigFile + " not found.");
                }
                log4net.Config.XmlConfigurator.Configure(new System.IO.FileInfo(logConfigFile));
            }
            catch (Exception ex)
            {
                LogError(ex.Message);
                System.Environment.ExitCode = 1;
                return;
            }

            // check to see that build configuration file is in the correct position in the argument list
            int firstActionIndex = GetFirstActionIndex(args); 
            if (firstActionIndex < 2)
            {
                usage();
                System.Environment.ExitCode = 1;
                return;
            }

            VixBuilderFacade vixBuilderFacade;
            try
            {
                // load the build configuation and override andy fields that have been specified on the command line
                BuildConfiguration config = null;
                if (!File.Exists(args[1]))
                {
                    throw new BuildConfigurationException("Specified build configuration file not found: " + args[1]);
                }
                config = BuildConfiguration.Open(args[1]);

                for (int i = 2; i < args.Length; i++)
                {
                    if (args[i].StartsWith("-"))
                    {
                        UpdateBuildConfiguration(config, args[i]);
                    }
                }

                vixBuilderFacade = new VixBuilderFacade(config);
                vixBuilderFacade.Info = LogInfo;
                vixBuilderFacade.Output = LogInfo;

                for (int i = firstActionIndex; i < args.Length; i++)
                {
                    if (args[i].StartsWith("-")) continue;
                    BuildActionEnum action = (BuildActionEnum)Enum.Parse(typeof(BuildActionEnum), args[i]);
                    System.Environment.ExitCode = vixBuilderFacade.ExecuteBuildAction(action, appPath);
                    if (System.Environment.ExitCode != 0)
                    {
                        break;
                    }
                }
            }
            catch (Exception ex)
            {
                LogError(ex.Message);
                System.Environment.ExitCode = 1;
            }
        }

        #region private methods

        private static void UpdateBuildConfiguration(BuildConfiguration config, string arg)
        {
            string[] splits = arg.Split(new char[] {'='});
            if (splits.Length != 2)
            {
                throw new BuildConfigurationException("Invalid command line parameter: " + arg);
            }
            string buildConfigurationPropery = splits[0].TrimStart(new char[] { '-' });
            string value = splits[1].Replace("\"", "");
            switch (buildConfigurationPropery.ToLower())
            {
                case "tfsworkspaceworkingfolder":
                    config.TfsWorkspaceWorkingFolder = value;
                    break;
                case "tfsrootdirspec":
                    config.TfsRootDirspec = value;
                    break;
                case "maven2repodirspec":
                    config.Maven2RepoDirspec = value;
                    break;
                case "vixpayloadrootdirspec":
                    config.VixPayloadRootDirspec = value;
                    break;
                case "vixinstallerrootdirspec":
                    config.VixInstallerRootDirspec = value;
                    break;
                case "msiversion":
                    config.MsiVersion = value;
                    break;
                case "deletepayloadrootcontents":
                    config.DeletePayloadRootContents = Boolean.Parse(value); // allow exception
                    break;
                default:
                    throw new BuildConfigurationException("Unrecognized build configuration property :" + buildConfigurationPropery);
            }
        }

        private static void SaveLastUsedBuildConfigurationFilespec(System.Configuration.Configuration systemConfig, string lastUsedFilespec)
        {
            KeyValueConfigurationElement keyValue = systemConfig.AppSettings.Settings["BuildConfigurationFilespec"];
            if (keyValue != null)
            {
                systemConfig.AppSettings.Settings.Remove("BuildConfigurationFilespec");
            }
            systemConfig.AppSettings.Settings.Add("BuildConfigurationFilespec", lastUsedFilespec);
            systemConfig.Save(ConfigurationSaveMode.Modified);
            ConfigurationManager.RefreshSection("appSettings"); // Force a reload of a changed section.
        }
        
        private static string GetLastUsedBuildConfigurationFilespec(System.Configuration.Configuration systemConfig)
        {
            string lastUsedFilespec = null;

            KeyValueConfigurationElement keyValue = systemConfig.AppSettings.Settings["BuildConfigurationFilespec"];
            if (keyValue != null)
            {
                lastUsedFilespec = keyValue.Value;
            }
            return lastUsedFilespec;
        }

        private static int GetFirstActionIndex(string[] args)
        {
            int index = -1;
            for (int i = 0; i < args.Length; i++)
            {
                try
                {
                    BuildActionEnum action = (BuildActionEnum)Enum.Parse(typeof(BuildActionEnum), args[i]);
                    index = i;
                    break;
                }
                catch (Exception) { }
            }
            return index;
        }
        
        private static ILog Logger()
        {
            return LogManager.GetLogger("VixBuilderConsole");
        }

        private static String LogError(String infoMessage)
        {
            Console.WriteLine(infoMessage);
            Logger().Error(infoMessage);
            return infoMessage;
        }

        private static String LogInfo(String infoMessage)
        {
            Console.WriteLine(infoMessage);
            Logger().Info(infoMessage);
            return infoMessage;
        }

        //private static String Info(String infoMessage)
        //{
        //    Console.WriteLine(infoMessage);
        //    return infoMessage;
        //}

        //private static String Output(String outputMessage)
        //{
        //    Console.WriteLine(outputMessage);
        //    return outputMessage;
        //}

        private static void usage()
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine("VIX Builder - the VIX build utility - console edition");
            sb.AppendLine("Usage:");
            sb.AppendLine("VixBuilderConsole");
            sb.AppendLine("\tdisplays this help information");
            
            sb.AppendLine("VixBuilderConsole buildConfigurationFilespec, -configField=value [-configField=value] action [action]");
            sb.AppendLine("\tRuns the specified action(s) using the specified configuration file.");
            sb.AppendLine("\tBe sure to enclose parameters with embedded spaces in double quotes.");
            sb.AppendLine("");

            sb.AppendLine("Supported configuration fields:");
            sb.AppendLine("\tTfsWorkspaceWorkingFolder - The local TFS folder mapped to java projects");
            sb.AppendLine("\tTfsRootDirspec - The local TFS folder mapped to HealthcareCollection.");
            sb.AppendLine("\tMaven2RepoDirspec - The Maven2 repository dirspec.");
            sb.AppendLine("\tVixPayloadRootDirspec - The root dirspec where the payload zip file is assembled.");
            sb.AppendLine("\tVixInstallerRootDirspec - The root dirspec where the MSI will be placed.");
            sb.AppendLine("\tMsiVersion - The MSI version number when building the VixInstallerSetup project. If specified MsiVersion.txt will nto be used.");
            sb.AppendLine("\tDeletePayloadRootContents - after a payload copy, perform a recursive delete on contents of VixPayloadRootDirspec.");
            sb.AppendLine("");

            sb.AppendLine("Supported build actions:");
            sb.AppendLine("\tmakeprod - composite action that performs getBuildProjects build getDeployProjects assembleAndStageMsiPayload buildMsi");
            sb.AppendLine("\tmakedev - composite action that performs getBuildProjects build deploytomcat");
            sb.AppendLine("\tbuildnightly - composite action that performs getBuildProjects build getDeployProjects assembleAndStageMsiPayload");
            sb.AppendLine("\tgetBuildProjects - Get all Java projects listed in the build manifest from the build repository");
            sb.AppendLine("\tbuild - Perform a Maven 2 build of all Java projects in the listed in the build manifest");
            sb.AppendLine("\tgetDeployProjects - Get all deploy projects listed in the build manifest from the deployment repository");
            sb.AppendLine("\tassembleAndStageMsiPayload - Build payload zip file then copy to the VIX installer setup project directory along with the install manifest");
            sb.AppendLine("\tbuildmsi - use Visual Studio 2005 to perform a release build on the VIX installer - generates a msi file");
            sb.AppendLine("\tdeploytomcat - copy build artifacts from the Maven 2 repository to Tomcat");
            sb.AppendLine("\tupdateMsiIdentityAndVersion - updates the MSI Identity (i.e. VIX, CVIX, HDIG, RVIX) and version number.");
            Console.Write(sb.ToString());
        }

        #endregion
    }
}
