using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using log4net;
using System.Diagnostics;
using Ionic.Zip;

namespace VixBuilderBusiness
{
    public static class ZipUtilities
    {
        private static ILog Logger()
        {
            return LogManager.GetLogger(typeof(ZipUtilities).Name);
        }

        public static ZipFile GetZipFile(string zipFileSpec)
        {
            if (!File.Exists(zipFileSpec))
            {
                throw new ApplicationException("Zip file does not exist: " + zipFileSpec);
            }
            return new ZipFile(zipFileSpec);
        }

        public static int FileCount(string zipFileSpec)
        {
            int fileCount = 0;
            Debug.Assert(zipFileSpec != null);
            Debug.Assert(zipFileSpec.Length != 0);

            ZipFile zipFile = null;
            try
            {
                zipFile = GetZipFile(zipFileSpec);
                foreach (ZipEntry entry in zipFile.Entries)
                {
                    if (entry.IsDirectory)
                        continue;//ignore directories
                    fileCount++;
                }
            }
            catch (Exception ex)
            {
                throw new ApplicationException("unable to unzip: " + zipFileSpec + "\nIt might be password protected.", ex);
            }
            finally
            {
                if (zipFile != null)
                {
                    zipFile.Dispose(); // release the zip file
                    zipFile = null;
                }
            }
            return fileCount;
        }

        /// <summary>
        /// Check a zip file to make sure that it only contains the files specified in filelist. This method is
        /// currently case sensistive on filename. Beware.
        /// </summary>
        /// <param name="zipFileSpec">The zip file to check.</param>
        /// <param name="filelist">A dictionay containing a list of allowed files</param>
        /// <returns>true if only allowed files are in the zip, false otherwise</returns>
        public static bool CheckZipContents(string zipFileSpec, string[] allowedFilesArray)
        {
            Debug.Assert(zipFileSpec != null);
            Debug.Assert(zipFileSpec.Length != 0);
            Debug.Assert(allowedFilesArray != null);

            ZipFile zipFile = GetZipFile(zipFileSpec);

            if (allowedFilesArray.Length == 0)
            {
                return false;
            }

            Dictionary<String, String> allowedFiles = new Dictionary<string, string>();

            for (int i = 0; i < allowedFilesArray.Length; i++)
            {
                string s = allowedFilesArray[i].ToUpper();
                allowedFiles.Add(s, s);
            }

            try
            {
                foreach (ZipEntry ze in zipFile.Entries)
                {
                    if (ze.IsDirectory)
                        continue; //dont consider directories
                    string fname = ze.FileName.ToUpper();
                    if (!allowedFiles.ContainsKey(fname))
                    {
                        return false;
                    }
                }
            }
            catch (Exception ex)
            {
                throw new ApplicationException("Unable to access: " + zipFileSpec + "\nIt might be password protected.", ex);
            }
            finally
            {
                if (zipFile != null)
                {
                    zipFile.Dispose(); // release the zip file
                    zipFile = null;
                }
            }
            return true;
        }

        public static void UnZip(string zipFileSpec, string targetFolder)
        {
            sbyte[] _buffer = new sbyte[2048];

            Debug.Assert(zipFileSpec != null);
            Debug.Assert(zipFileSpec.Length != 0);

            ZipFile zipFile = null;
            try
            {
                zipFile = GetZipFile(zipFileSpec);
                foreach (ZipEntry ze in zipFile.Entries)
                {
                    if (ze.IsDirectory)
                        continue;//ignore directories
                    string fname = ze.FileName;
                    ze.Extract(targetFolder);
                }
            }
            finally
            {
                if (zipFile != null)
                {
                    zipFile.Dispose(); // release the zip file
                    zipFile = null;
                }
            }
        }

        public static void CreateZipFile(string zipFilespec, string rootDirspecToZip, bool isRelativePath, string pathInArchive)
        {
            ZipFile zipFile = null;
            if (File.Exists(zipFilespec))
            {
                File.Delete(zipFilespec);
            }
            try
            {
                if (System.IO.File.Exists(rootDirspecToZip))
                {
                    zipFile = new ZipFile(zipFilespec);
                    if (isRelativePath)
                        zipFile.AddFile(rootDirspecToZip, "/");
                    else
                        zipFile.AddFile(rootDirspecToZip);
                }
                else if (Directory.Exists(rootDirspecToZip))
                {
                    zipFile = new ZipFile(zipFilespec);
                    DirectoryInfo info1 = new DirectoryInfo(rootDirspecToZip);
                    if (isRelativePath)
                    {
                        if (pathInArchive == null)
                            zipFile.AddDirectory(rootDirspecToZip, "/");
                        else
                            zipFile.AddDirectory(rootDirspecToZip, pathInArchive);
                    }
                    else
                    {
                        zipFile.AddDirectory(rootDirspecToZip);
                    }
                }
            }
            finally
            {
                if (zipFile != null)
                {
                    zipFile.Save();
                    zipFile.Dispose(); // release the zip file
                    zipFile = null;
                }
            }
        }

    }
}
