using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Text.RegularExpressions;
using System.Diagnostics;
using log4net;

namespace VixBuilderBusiness
{
    using System.Globalization;

    class VisualStudioSetupProjectHelper
    {
        // magic cookies - ugh - if the setup project (folder or configuration used) changes - this will need to change.
        // fortunately this should not be necessary - has been stable for 2 years - DKB
        // port to Visual 2010 caused magic cookie changes - 4/30/2011
        private const string APP_FOLDER_SECTION_IDENTIFIER = "\"{3C67513D-01DD-4637-8A68-80971EB9504F}:_37B99DEDD3584D62AB729835C549573C\"";
        private const string FINISH_DIALOG_SECTION_IDENTIFIER = "\"{688940B3-5CA9-4162-8DEE-2993FA9D8CBC}:_B910969B9541420F87214E56D79D686F\"";
        private const int MAX_MSI_BUILD_NUMBER = 65535;

        readonly string filespec = null;
        readonly BuildConfiguration config = null;
        readonly IBuildRepository buildRepository;
        readonly IDeployRepository deployRepository;
        readonly string vixInstallerBusinessAssemblyVersion = null;
        List<string> lines = null;
        int productNameIndex = -1;
        int productCodeIndex = -1;
        int packageCodeIndex = -1;
        int productVersionIndex = -1;
        int vixInstallerBusinessDisplayNameIndex = -1;
        int appFolderSectionIndex = -1;
        int appFolderIndex = -1;
        int productTitleIndex = -1;
        int shortcutSectionIndex = -1;
        int shortcutNameIndex = -1;
        int finishDialogSectionIndex = -1;
        int finishDialogValueIndex = -1;

        public static string GetNextVersion(string currentVersion)
        {
            StringBuilder sb = new StringBuilder();
            string[] splits = currentVersion.Split('.');
            if (splits.Length != 3)
            {
                throw new BuildException(LogError("Version string is malformed: " + currentVersion));
            }
            // convert parts to ints
            int[] intSplits = new int[splits.Length];
            for (int i = 0; i < splits.Length; i++)
            {
                intSplits[i] = Int32.Parse(splits[i]);
            }
            intSplits[splits.Length - 1]++;
            if (intSplits[splits.Length - 1] > MAX_MSI_BUILD_NUMBER) // this will never happen in practice
            {
                throw new BuildException(LogError("MAX_MSI_BUILD_NUMBER has been exceeded. Minor build number must be incremented and build number must be reset."));
            }

            ////// increment parts
            ////for (int i = splits.Length - 1; i > 0; i--)
            ////{
            ////    if (intSplits[i] >= 100)
            ////    {
            ////        intSplits[i] = 0;
            ////        intSplits[i - 1]++;
            ////    }
            ////}

            // reassemble
            for (int i = 0; i < splits.Length; i++)
            {
                sb.Append(intSplits[i].ToString(CultureInfo.InvariantCulture));
                if (i < splits.Length - 1)
                {
                    sb.Append(".");
                }
            }

            return sb.ToString();
        }

        public VisualStudioSetupProjectHelper(string filespec, BuildConfiguration config, IBuildRepository buildRepository, IDeployRepository deployRepository, string vixInstallerBusinessVersion)
        {
            this.filespec = filespec;
            this.config = config;
            this.buildRepository = buildRepository;
            this.deployRepository = deployRepository;
            this.vixInstallerBusinessAssemblyVersion = vixInstallerBusinessVersion;
        }

        public string RewriteSetupFile(bool performCheckout)
        {
            Load();
            ApplyProductName();
            ApplyProductCode();
            ApplyPackageCode();
            ApplyAssemblyVersion();
            string nextInstallerVersionNumber = ApplyProductversion(performCheckout);
            //ApplyNewJsharpMergePackageIdentifier();
            ApplyAppFolderName();
            ApplyShortcut();
            ApplyFinishDialogValue();
            ApplyProductTitle();
            Save();
            return nextInstallerVersionNumber;
        }

        #region private methods
        private void Save()
        {
            using (StreamWriter sw = File.CreateText(this.filespec)) // file will be overwritten
            {
                foreach (string line in lines)
                {
                    sw.WriteLine(line);
                }
                sw.Close();
            }
        }

        private void ApplyFinishDialogValue()
        {
            Debug.Assert(finishDialogValueIndex >= 0);
            string installerName = InstallationType.GetInstallationTypeByID(config.InstallationType).InstallerDisplayName;
            //lines[finishDialogValueIndex] = "                            \"Value\" = \"8:Click the Close button.\"";
            lines[finishDialogValueIndex] = "                            \"Value\" = \"8:To run the " + installerName + ", click Start | All Programs | VistA Imaging Programs | " + installerName + ".\"";
        }
        
        private void ApplyShortcut()
        {
            Debug.Assert(shortcutNameIndex >= 0);
            string installerName = InstallationType.GetInstallationTypeByID(config.InstallationType).InstallerDisplayName;
            lines[shortcutNameIndex] = "            \"Name\" = \"8:" + installerName + "\"";
        }

        private void ApplyAppFolderName()
        {
            Debug.Assert(appFolderIndex >= 0);
            string appFolderName = InstallationType.GetInstallationTypeByID(config.InstallationType).InstallerMsiApplicationFolder;
            lines[appFolderIndex] = "            \"DefaultLocation\" = \"8:[ProgramFilesFolder]\\\\Vista\\\\Imaging\\\\" + appFolderName + "\"";
        }

        private void ApplyProductName()
        {
            Debug.Assert(productNameIndex >= 0);
            string installerName = InstallationType.GetInstallationTypeByID(config.InstallationType).InstallerDisplayName;
            lines[productNameIndex] = "        \"ProductName\" = \"8:" + installerName + " " + this.buildRepository.GetBuildManifest().FullyQualifiedPatchNumber + "\"";
        }

        private void ApplyProductCode()
        {
            Debug.Assert(productCodeIndex >= 0);
            lines[productCodeIndex] = "        \"ProductCode\" = \"8:{" + Guid.NewGuid().ToString().ToUpper() + "}\"";
        }

        private void ApplyPackageCode()
        {
            Debug.Assert(packageCodeIndex >= 0);
            lines[packageCodeIndex] = "        \"PackageCode\" = \"8:{" + Guid.NewGuid().ToString().ToUpper() + "}\"";
        }

        private string ApplyProductversion(bool performCheckout)
        {
            Debug.Assert(productVersionIndex >= 0);

            Regex regex = new Regex("[0-9]*\\.[0-9]*\\.[0-9]*");
            Match match = regex.Match(lines[productVersionIndex]);
            Debug.Assert(match.Success);
            // attempt to get the next version number from the deploy repository falling back to the version number in the setup file if necessary
            string nextInstallerVersionNumber = config.MsiVersion != null ? config.MsiVersion : this.deployRepository.GetNextMsiVersion(performCheckout); 
            if (nextInstallerVersionNumber == null)
            {
                throw new BuildConfigurationException(LogError("VssSetupProjectHelper.ApplyProductVersion: Could not obtain next MSI version number."));
            }
            LogInfo("MSI Version Number is " + nextInstallerVersionNumber);            
            lines[productVersionIndex] = "        \"ProductVersion\" = \"8:" + nextInstallerVersionNumber + "\"";
            return nextInstallerVersionNumber;
        }

        private void ApplyProductTitle()
        {
            Debug.Assert(productTitleIndex >= 0);
            string installerName = InstallationType.GetInstallationTypeByID(config.InstallationType).InstallerDisplayName;
            lines[productTitleIndex] = "        \"Title\" = \"8:" + installerName + " Setup\"";
        }

        private void ApplyAssemblyVersion()
        {
            Debug.Assert(vixInstallerBusinessDisplayNameIndex >= 0);

            Regex regex = new Regex("Version=[0-9]*\\.[0-9]*\\.[0-9]*\\.[0-9]*");
            Match match = regex.Match(lines[vixInstallerBusinessDisplayNameIndex]);
            Debug.Assert(match.Success);
            string existingAssemblyVersion = match.Value.Replace("Version=", "");
            lines[vixInstallerBusinessDisplayNameIndex] = lines[vixInstallerBusinessDisplayNameIndex].Replace(existingAssemblyVersion, this.vixInstallerBusinessAssemblyVersion);
        }


        private void Load()
        {
            if (lines == null)
            {
                lines = new List<string>();
            }

            if (lines.Count == 0)
            {
                int index = 0;
                string line;
                //int mergeModuleIndex = -1;
                using (StreamReader sr = new StreamReader(filespec))
                {
                    while ((line = sr.ReadLine()) != null)
                    {
                        if (line.Contains("\"ProductName\""))
                        {
                            productNameIndex = index;
                        }
                        else if (productNameIndex >= 0 && line.Contains("\"ProductCode\""))
                        {
                            productCodeIndex = index;
                        }
                        else if (productNameIndex >= 0 && productCodeIndex >= 0 && line.Contains("\"PackageCode\""))
                        {
                            packageCodeIndex = index;
                        }
                        else if (productNameIndex >= 0 && productCodeIndex >= 0 && packageCodeIndex >= 0 && line.Contains("\"ProductVersion\""))
                        {
                            productVersionIndex = index;
                        }
                        else if (productNameIndex >= 0 && productCodeIndex >= 0 && packageCodeIndex >= 0 && productVersionIndex >= 0 && line.Contains("\"Title\""))
                        {
                            productTitleIndex = index;
                        }
                        //else if (line.Contains("\"MergeModule\""))
                        //{
                        //    mergeModuleIndex = index;
                        //}
                        //else if (mergeModuleIndex >= 0 && line.Contains("\"ProjectOutput\""))
                        //{
                        //    this.jsharpMergePackageIdentifier = GetJsharpMergeModuleIdentifier(mergeModuleIndex, index);
                        //}
                        else if (line.Contains("\"AssemblyAsmDisplayName\"") && line.Contains("VixInstallerBusiness"))
                        {
                            this.vixInstallerBusinessDisplayNameIndex = index;
                        }
                        else if (line.Contains(APP_FOLDER_SECTION_IDENTIFIER))
                        {
                            appFolderSectionIndex = index;
                        }
                        else if (appFolderSectionIndex >= 0 && line.Contains("\"DefaultLocation\""))
                        {
                            appFolderIndex = index;
                            appFolderSectionIndex = -1; // one default location only please
                        }
                        else if (line.Contains("\"Shortcut\""))
                        {
                            shortcutSectionIndex = index;
                        }
                        else if (shortcutSectionIndex >= 0 && line.Contains("\"Name\""))
                        {
                            shortcutNameIndex = index;
                            shortcutSectionIndex = -1; // one name only please
                        }
                        else if (line.Contains(FINISH_DIALOG_SECTION_IDENTIFIER))
                        {
                            finishDialogSectionIndex = index;
                        }
                        else if (finishDialogSectionIndex >= 0 && line.Contains("\"Value\""))
                        {
                            finishDialogValueIndex = index;
                            finishDialogSectionIndex = -1; // one value only please
                        }
 
                        lines.Add(line);
                        index++;
                    }
                    sr.Close();
                }
            }
        }

        private static void LogInfo(string message)
        {
            LogManager.GetLogger("VssSetupProjectHelper").Info(message);
        }

        private static string LogError(string message)
        {
            LogManager.GetLogger("VssSetupProjectHelper").Error(message);
            return message;
        }

        #endregion
    }
}