﻿using System;
using System.Collections.Generic;
using System.Text;
using log4net;
using System.Diagnostics;
using System.IO;
using System.Threading;


namespace VixBuilderBusiness
{

    internal class VssRepository : AbstractRepository, IDeployRepository
    {
        private IBuildRepository buildRepository;
        public VssRepository(BuildConfiguration config, IBuildRepository buildRepository) : base(config, RepositoryEnum.VisualSourceSafe)
        {
            this.buildRepository = buildRepository;
        }

        #region IDeployRepository methods

        /// <summary>
        /// Gets all the deployment projects listed in the build manifest from VSS.
        /// </summary>
        public void GetDeployProjects()
        {
            this.GetVssProjects();
        }

        /// <summary>
        /// Check out a file from VSS.
        /// </summary>
        /// <param name="filespec">The file to check out.</param>
        /// <returns>true if sucessful</returns>
        public bool CheckoutFile(string filespec)
        {
            int results = CheckoutVssProjectFile(filespec);
            return results == 0 ? true : false;
        }

        /// <summary>
        /// Undo a checkout file operation from VSS.
        /// </summary>
        /// <param name="fileSpec">A filespec constructed using the root of the workspace as the starting point.</param>
        /// <returns>true if sucessful, false otherwise</returns>
        /// <remarks>filespec should be constructed without regard to any working folder mappings that may be present.</remarks>
        public bool UndoFileCheckout(string fileSpec)
        {
            throw new NotImplementedException("VssRepository.CheckoutFile not implemented.");
        }

        /// <summary>
        /// Check in a file to VSS.
        /// </summary>
        /// <param name="filespec">The file to check in.</param>
        /// <param name="comment">The comment to associate with the file in VSS.</param>
        /// <returns>true if sucessful</returns>
        public bool CheckinFile(string filespec, string comment)
        {
            int results = CheckinVssProjectFile(filespec, comment);
            return results == 0 ? true : false;
        }

        /// <summary>
        /// Copies the payload structure and static files from the repository to the payload construction directory.
        /// </summary>
        /// <param name="payloadRootDirspec">The payload construction root dirspec.</param>
        /// <remarks></remarks>
        public void CopyPayloadArtifacts()
        {
            // copy payload skeleton and files from deployment repository
            string payloadRootDirspec = Path.Combine(config.VixPayloadRootDirspec, this.buildRepository.Manifest.FullyQualifiedPatchNumber);
            string repositoryPayloadRootDirspec = this.GetVixPayloadRootDirspec();
            Debug.Assert(Directory.Exists(repositoryPayloadRootDirspec));
            CopyAll(new DirectoryInfo(repositoryPayloadRootDirspec), new DirectoryInfo(payloadRootDirspec));
        }

        /// <summary>
        ///  // Gets the absolute dirspec of VixInstallerSolution2013.root
        /// </summary>
        /// <remarks></remarks>
        public string GetVixInstallerSolutionRootDirspec()
        {
            DeployProject installerSolutionRootProject = this.buildRepository.Manifest.GetVixInstallerSolutionRootDeployProject();
            string repositorySolutionRootDirspec = Path.Combine(config.VssRootDirspec, installerSolutionRootProject.VssRelativeProjectPath);
            repositorySolutionRootDirspec = Path.Combine(repositorySolutionRootDirspec, installerSolutionRootProject.ProjectName);
            return repositorySolutionRootDirspec;
        }

        /// <summary>
        ///  // Gets the absolute dirspec of Vix Installer payload directory
        /// </summary>
        /// <remarks></remarks>
        public string GetVixPayloadRootDirspec()
        {
            DeployProject installerPayloadRootProject = this.buildRepository.Manifest.GetPayloadDeployProject();
            string repositoryPayloadRootDirspec = Path.Combine(config.VssRootDirspec, installerPayloadRootProject.VssRelativeProjectPath);
            repositoryPayloadRootDirspec = Path.Combine(repositoryPayloadRootDirspec, installerPayloadRootProject.ProjectName);
            return repositoryPayloadRootDirspec;
        }

        /// <summary>
        /// Gets the next msi version from the file specified by the build manifest.
        /// </summary>
        /// <returns>The next MSI version number as a three part string.</returns>
        /// <remarks>As a side effect, the file that holds the last version number is checked out, rewritten with the new version number, and then checked in.</remarks>
        public string GetNextMsiVersion(bool performCheckout)
        {
            throw new NotImplementedException("VssRepository.GetNextMsiVersion not implemented.");
        }

        public string GetLocalFolderSpecFromRepositoryWorkspace(string repositoryFileSpec)
        {
            throw new NotImplementedException("ClearCaseRepository.GetNextMsiVersion not implemented.");
        }


        #endregion

        #region private methods

        /// <summary>
        /// Gets all the deployment VSS projects listed in the manifest from VSS
        /// </summary>
        private void GetVssProjects()
        {
            // set ssdir=\\vhaiswimgs15\vix_vss$
            // "C:\Program Files\Microsoft Visual SourceSafe\ss.exe" get "$\ViX\VixBuild\Patch119" -R -Yusername,password -I-N
            DeployProject[] deployProjects = this.buildRepository.Manifest.GetDeployProjects();
            LogInfo("Beginning VSS Get phase...");
            foreach (DeployProject deployProject in deployProjects)
            {
                LogInfo("\tPerforming VSS Get of project " + deployProject.ProjectName + (deployProject.Recurse ? " (recursive)" : string.Empty));
                int resultCode = this.GetVssProject(deployProject);

                if (resultCode == 100 && config.SkipWriteableFilesOnGet) // DKB - allow for VSS files to be locally modified - useful for VixInstaller development
                {
                    if (config.DisplayVssOutput == true)
                    {
                        DisplayOutput();
                    }
                }
                else if (resultCode > 0)
                {
                    DisplayOutput();
                    throw new BuildException(LogError("Get of project " + deployProject.ProjectName + " failed - aborting build process"));
                }
                else if (config.DisplayVssOutput == true)
                {
                    DisplayOutput();
                }

                if (CanRun == false)
                {
                    LogInfo("Build canceled by user.");
                    return;
                }
            }
            LogInfo("VSS Get phase complete");
        }


        /// <summary>
        /// Gets the deployment project from VSS.
        /// </summary>
        /// <param name="deployProject">The deployment project to get.</param>
        /// <returns>The exit code from the VSS executable. Anything non zero is considered an error.</returns>
        private int GetVssProject(DeployProject deployProject)
        {
            ResetStandardOutput();

            // set ssdir=\\vhaiswimgs15\vix_vss$
            // "C:\Program Files\Microsoft Visual SourceSafe\ss.exe" get "$\ViX\VixBuild\Patch119" -R -Yuser,password -I-N
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();
            string vssProjectRelativeDirspec = deployProject.GetProjectRepositoryRelativeDirspec(RepositoryEnum.VisualSourceSafe);
            string vssProjectIdentifier = @"$\" + vssProjectRelativeDirspec;
            string vssProjectDirspec = Path.Combine(config.VssRootDirspec, vssProjectRelativeDirspec);

            sb.AppendFormat("get \"{0}\" -Y{1},{2} -I-N", vssProjectIdentifier, config.VssUsername, config.VssPassword);
            if (deployProject.Recurse)
            {
                sb.Append(" -R");
            }

            if (!Directory.Exists(vssProjectDirspec))
            {
                Directory.CreateDirectory(vssProjectDirspec);
            }

            string arguments = sb.ToString();
            Process externalProcess = new Process
                {
                    StartInfo =
                        {
                            FileName = this.config.VssExecutable,
                            Arguments = arguments,
                            WorkingDirectory = vssProjectDirspec,
                            UseShellExecute = false,
                            CreateNoWindow = true,
                            RedirectStandardOutput = true,
                            RedirectStandardError = true
                        }
                };

            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);

            if (externalProcess.StartInfo.EnvironmentVariables.ContainsKey("ssdir"))
            {
                externalProcess.StartInfo.EnvironmentVariables.Remove("ssdir");
            }

            externalProcess.StartInfo.EnvironmentVariables.Add("ssdir", config.VssSharename);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                }
                while (!externalProcess.HasExited);

                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
            }

            return exitCode;
        }

        /// <summary>
        /// Checkout a file from VSS.
        /// </summary>
        /// <param name="filespec">The filespec.</param>
        /// <returns>The exit code from the VSS executable. Anything non zero is considered an error.</returns>
        private int CheckoutVssProjectFile(string filespec)
        {
            ResetStandardOutput();

            // set ssdir=\\vhaiswimgs15\vix_vss$
            // "C:\Program Files\Microsoft Visual SourceSafe\ss.exe" checkout "$\ViX\VixInstaller\VixInstallerSetup\VixInstallerSetup.vdproj" -Yuser,password
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("checkout \"{0}\" -I-Y -Y{1},{2}", this.TranslateFilespecToVssSemantics(filespec), this.config.VssUsername, this.config.VssPassword);
            string dirspec = Path.GetDirectoryName(filespec);
            if (dirspec == null)
            {
                throw new BuildConfigurationException(filespec + "was provided as a relative file specification,");
            }

            Process externalProcess = new System.Diagnostics.Process
                {
                    StartInfo =
                        {
                            FileName = this.config.VssExecutable,
                            Arguments = sb.ToString(),
                            WorkingDirectory = dirspec,
                            UseShellExecute = false,
                            CreateNoWindow = true,
                            RedirectStandardOutput = true,
                            RedirectStandardError = true
                        }
                };

            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);

            if (externalProcess.StartInfo.EnvironmentVariables.ContainsKey("ssdir"))
            {
                externalProcess.StartInfo.EnvironmentVariables.Remove("ssdir");
            }

            externalProcess.StartInfo.EnvironmentVariables.Add("ssdir", config.VssSharename);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } 
                while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
            }

            return exitCode;
        }

        /// <summary>
        /// Check in a file to VSS.
        /// </summary>
        /// <param name="filespec">The file to check in.</param>
        /// <param name="comment">The comment to associate with the file in VSS.</param>
        /// <returns>The exit code from the VSS executable. Anything non zero is considered an error.</returns>
        private int CheckinVssProjectFile(string filespec, string comment)
        {
            ResetStandardOutput();

            // set ssdir=\\vhaiswimgs15\vix_vss$
            // "C:\Program Files\Microsoft Visual SourceSafe\ss.exe" checkin "$\ViX\VixInstaller\VixInstallerSetup\VixInstallerSetup.vdproj" -Yuser,password -C"comment"
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("checkin \"{0}\" -Y{1},{2} -C\"{3}\"", this.TranslateFilespecToVssSemantics(filespec), config.VssUsername, config.VssPassword, comment);
            string dirspec = Path.GetDirectoryName(filespec);
            if (dirspec == null)
            {
                throw new BuildConfigurationException(filespec + "was provided as a relative file specification,");
            }

            Process externalProcess = new System.Diagnostics.Process
                {
                    StartInfo =
                        {
                            FileName = this.config.VssExecutable,
                            Arguments = sb.ToString(),
                            WorkingDirectory = dirspec,
                            UseShellExecute = false,
                            CreateNoWindow = true,
                            RedirectStandardOutput = true,
                            RedirectStandardError = true
                        }
                };
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);

            if (externalProcess.StartInfo.EnvironmentVariables.ContainsKey("ssdir"))
            {
                externalProcess.StartInfo.EnvironmentVariables.Remove("ssdir");
            }

            externalProcess.StartInfo.EnvironmentVariables.Add("ssdir", config.VssSharename);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                }
                while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
            }

            return exitCode;
        }

        /// <summary>
        /// Translates the filespec to VSS semantics. In practice this means replacing any existing drive specification with $
        /// </summary>
        /// <param name="filespec">The file specification to translate.</param>
        /// <returns>The translated file specification suitable for use on the VSS command line.</returns>
        private string TranslateFilespecToVssSemantics(string filespec)
        {
            string vssFilespec = filespec.Replace(config.VssRootDirspec, @"$");
            return vssFilespec;
        }
      
        #endregion

    }
}
