/**
 * 
  Package: MAG - VistA Imaging
  WARNING: Per VHA Directive 2004-038, this routine should not be modified.
  Date Created:
  Site Name:  Washington OI Field Office, Silver Spring, MD
  Developer:
  Description: 

        ;; +--------------------------------------------------------------------+
        ;; Property of the US Government.
        ;; No permission to copy or redistribute this software is given.
        ;; Use of unreleased versions of this software requires the user
        ;;  to execute a written test agreement with the VistA Imaging
        ;;  Development Office of the Department of Veterans Affairs,
        ;;  telephone (301) 734-0100.
        ;;
        ;; The Food and Drug Administration classifies this software as
        ;; a Class II medical device.  As such, it may not be changed
        ;; in any way.  Modifications to this software may result in an
        ;; adulterated medical device under 21CFR820, the use of which
        ;; is considered to be a violation of US Federal Statutes.
        ;; +--------------------------------------------------------------------+

 */

namespace VixBuilderBusiness
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Text;
    using System.Threading;
    using log4net;

    /// <summary>
    /// This class encompasses all the functionality needed to build and package a VISA deployment.
    /// </summary>
    public class VixBuilderFacade
    {
        /// <summary>
        /// the build repository.
        /// </summary>
        private readonly IBuildRepository buildRepository;

        /// <summary>
        /// the deployment repository.
        /// </summary>
        private readonly IDeployRepository deployRepository;

        /// <summary>
        /// the build configuration
        /// </summary>
        private readonly BuildConfiguration config;

        /// <summary>
        /// the build manifest, wrapped by the Manifest property
        /// </summary>
        //private BuildManifest manifest;

        /// <summary>
        /// wrapped by CanRun property.
        /// </summary>
        private bool canRun = true;

        /// <summary>
        /// wrapped by DoEvents property
        /// </summary>
        private DoEventsDelegate doEvents;

        /// <summary>
        /// wrapped by the Info property
        /// </summary>
        private InfoDelegate info;

        /// <summary>
        /// wrapped by the Output property
        /// </summary>
        private OutputDelegate output;

        /// <summary>
        /// used to accumulate output to be displayed in the User Interface
        /// </summary>
        private StringBuilder outputStringBuilder;

        /// <summary>
        /// Initializes a new instance of the <see cref="VixBuilderFacade"/> class.
        /// </summary>
        /// <param name="config">The build configuration.</param>
        public VixBuilderFacade(BuildConfiguration config)
        {
            this.config = config;

            switch (config.BuildRepository)
            {
                case RepositoryEnum.ClearCase:
                    this.buildRepository = new ClearCaseRepository(config);
                    break;
                case RepositoryEnum.ConcurrentVersionsSystem:
                    this.buildRepository = new CvsRepository(config);
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    this.buildRepository = new TfsRepository(config);
                    break;
                default:
                    throw new BuildException("Illegal Build Repository specified: " + config.BuildRepository.ToString());
            }

            switch (config.DeployRepository)
            {
                case RepositoryEnum.ClearCase:
                    this.deployRepository = new ClearCaseRepository(config);
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    this.deployRepository = new TfsRepository(config);
                    break;
                case RepositoryEnum.VisualSourceSafe:
                    this.deployRepository = new VssRepository(config, this.buildRepository); // build repository provides access to the build manifest
                    break;
                default:
                    throw new BuildException("Illegal Deploy Repository specified: " + config.DeployRepository.ToString());
            }
        }

        #region properties

        /// <summary>
        /// Gets the build manifest.
        /// </summary>
        /// <value>The build manifest.</value>
        /// <remarks>This property is delay initialized because the build manifest should not be accessed until after the ImagingExchange project
        /// has been retrieved from the build repository.</remarks>
        //private BuildManifest Manifest
        //{
        //    get
        //    {
        //        if (this.manifest == null)
        //        {
        //            this.manifest = this.GetBuildManifest(); // manifest may not exist until after ImagingExchange has been retrieved from the build repository
        //        }
        //        Debug.Assert(this.manifest != null, "No build manifest available");
        //        return this.manifest;
        //    }
        //}

        /// <summary>
        /// Gets or sets a value indicating whether the user interface has canceled the current operation.
        /// </summary>
        /// <value><c>true</c> if this instance can run; otherwise, <c>false</c>.</value>
        public bool CanRun
        {
            get
            {
                return this.canRun;
            }

            set
            {
                this.canRun = value;
                this.buildRepository.CanRun = value;
                this.deployRepository.CanRun = value;
            }
        }

        /// <summary>
        /// Sets a delegate used to display a line of text in the main form's info text box
        /// </summary>
        /// <value>The info delegate.</value>
        public InfoDelegate Info
        {
            set
            {
                this.info = value;
                this.buildRepository.Info = value;
                this.deployRepository.Info = value;
            }
        }

        /// <summary>
        /// Sets a delegate used to display combined standard output and error to output text box
        /// </summary>
        /// <value>The output delegate.</value>
        public OutputDelegate Output
        {
            set
            {
                this.output = value;
                this.buildRepository.Output = value;
                this.deployRepository.Output = value;
            }
        }

        /// <summary>
        /// Sets a delegate keep the user interface responsive
        /// </summary>
        /// <value>The do events delegate.</value>
        public DoEventsDelegate DoEvents
        {
            set
            {
                this.doEvents = value;
                this.buildRepository.DoEvents = value;
                this.deployRepository.DoEvents = value;
            }
        }
        #endregion

        #region public methods
        /// <summary>
        /// Gets all accumulated output strings.
        /// </summary>
        /// <returns>A string which contains all the output that has been generated.</returns>
        public string GetOutput()
        {
            return this.outputStringBuilder == null ? string.Empty : this.outputStringBuilder.ToString();
        }

        /// <summary>
        /// Executes the build action.
        /// </summary>
        /// <param name="action">The action.</param>
        /// <param name="applicationDirspec">The application dirspec.</param>
        /// <returns>An exit code consistent with batch file processing, namely 0 indicates successs and anything other than 0 indicates an error</returns>
        public int ExecuteBuildAction(BuildActionEnum action, string applicationDirspec)
        {
            this.CanRun = true;
            int exitCode = 0;

            try
            {
                this.config.ValidateBuildConfigurationForBuildAction(action); // this will throw a BuildCOnfigurationException if validation fails
                switch (action)
                {
                    case BuildActionEnum.makenightly:
                        this.DoMakeNightlyCompoundAction(applicationDirspec);
                        break;
                    case BuildActionEnum.makeprod:
                        this.DoMakeProdCompoundAction(applicationDirspec);
                        break;
                    case BuildActionEnum.makedev:
                        this.DoMakeDevCompoundAction(applicationDirspec);
                        break;
                    case BuildActionEnum.getBuildProjects:
                        if (this.CanRun)
                        {
                            this.buildRepository.GetBuildProjects();
                        }
                        break;
                    case BuildActionEnum.build:
                        if (this.CanRun)
                        {
                            this.BuildVisaWorkspace();
                        }
                        break;
                    case BuildActionEnum.getDeployProjects:
                        if (this.CanRun)
                        {
                            this.deployRepository.GetDeployProjects();
                        }
                        break;
                    case BuildActionEnum.assembleAndStageMsiPayload:
                        if (this.CanRun)
                        {
                            this.BuildAndDeployVixInstallerPayload();
                        }
                        break;
                    case BuildActionEnum.buildMsi:
                        if (this.CanRun)
                        {
                            this.BuildVixInstallerMsi(applicationDirspec);
                        }
                        break;
                    case BuildActionEnum.deploytomcat:
                        ////if (config.GenerateTomcatConfigFiles && (TomcatFacade.IsServerConfiguredForVix(config) == false || TomcatFacade.IsTomcatUsersConfiguredForVix(config) == false))
                        ////{
                        ////    List<DeployProjectType> typesToGet = new List<DeployProjectType>();
                        ////    typesToGet.Add(DeployProjectType.payload);
                        ////    GetDeployProjects(config, typesToGet);
                        ////}
                        if (this.CanRun)
                        {
                            this.DeployAndConfigureDevelopmentEnvironment();
                        }
                        break;
                    case BuildActionEnum.labelCvsWorkspace:
                        if (this.CanRun)
                        {
                            this.buildRepository.LabelBuildProjects();
                        }
                        break;
                    case BuildActionEnum.deleteLabelFromCvsWorkspace:
                        if (this.CanRun)
                        {
                            this.buildRepository.DeleteLabelFromBuildProjects();
                        }
                        break;
                    case BuildActionEnum.branchCvsWorkspace:
                        if (this.CanRun)
                        {
                            this.buildRepository.BranchBuildProjects();
                        }
                        break;
                    case BuildActionEnum.buildnightly:
                        if (this.CanRun)
                        {
                            this.BuildNightlyVixInstaller(applicationDirspec);
                        }
                        break;
                    case BuildActionEnum.updateMsiIdentityAndVersion: // team city command - can only be run from the console
                        this.UpdateMsiIdentityAndVersion();
                        break;
                }
            }
            catch (Exception ex)
            {
                LogInfo("Error: " + ex.Message);
                exitCode = 1;
            }
            return exitCode;
        }

        /// <summary>
        /// Builds the visa workspace.
        /// </summary>
        /// <remarks></remarks>
        public void BuildVisaWorkspace()
        {
            // delete any existing artifacts that we are going to rebuild. This will help detect any inconsistencies between the build and deploy elements
            //List<VixDependencyType> typesToDelete = new List<VixDependencyType>();
            //typesToDelete.Add(VixDependencyType.AppExtensionJar);
            //typesToDelete.Add(VixDependencyType.AppJar);
            //typesToDelete.Add(VixDependencyType.AppWar);
            //DeleteRepositoryArtifacts(typesToDelete);

            // Delete all imaging artifacts in the repository to ensure that deprecated build dependencies are not hanging around masking errors. 
            // Previously the strategy was to only delete artifacts called out in the build manifest. - DKB 9/26/2012
            if (String.IsNullOrEmpty(config.OverrideJarWarVersioning)) //5/9/2013 - if not on development machine, then delete Imaging directory in repository
            {
                string imagingRepoArtifactsDirspec = Path.Combine(config.Maven2RepoDirspec, "Imaging");
                if (Directory.Exists(imagingRepoArtifactsDirspec))
                {
                    LogInfo("\tDeleting " + config.Maven2RepoDirspec);
                    Directory.Delete(imagingRepoArtifactsDirspec, true); // recurse
                }
            }

            // build the java extensions
            List<VixDependencyType> typesToBuild = new List<VixDependencyType>();
            typesToBuild.Add(VixDependencyType.AppExtensionJar);
            BuildMaven2Projects(typesToBuild);
            // deploy the java extensions so that the rest of the build will work
            LogInfo("Copying Java extension jars");
            CopyDeploymentArtifacts(true, typesToBuild);
            LogInfo("Copying Java extension jars complete");
            // build the app jars and wars
            typesToBuild.Clear();
            typesToBuild.Add(VixDependencyType.AppJar);
            typesToBuild.Add(VixDependencyType.AppWar);
            BuildMaven2Projects(typesToBuild);
        }

        /// <summary>
        /// Builds the and deploy vix installer payload.
        /// </summary>
        /// <remarks></remarks>
        public void BuildAndDeployVixInstallerPayload()
        {
            LogInfo("Beginning construction and deployment of VIX payload zip file...");
            Debug.Assert(Directory.Exists(config.VixPayloadRootDirspec));

            // Prep an empty payload directory based on VIX version number
            if (CanRun == false)
            {
                LogInfo("Operation canceled by user.");
                return;
            }
            PreparePayloadDirectory();

            if (CanRun == false)
            {
                LogInfo("Operation canceled by user.");
                return;
            }
            // copy deployment repository based artifacts to the payload directory. This creates the payload directory structure.
            this.deployRepository.CopyPayloadArtifacts();

            // copy the remaining artifacts to the payload directory
            if (CanRun == false)
            {
                LogInfo("Operation canceled by user.");
                return;
            }
            CopyDeploymentArtifacts(false, null); // false=copy to payload directory, null= do all artifacts

            // zip up payload
            string payloadDirspec = GetPayloadRootDirspec();
            string zipFilespec = Path.Combine(config.VixPayloadRootDirspec, "VixDistribution.zip");
            LogInfo("\tCreating zip payload file " + zipFilespec);
            ZipUtilities.CreateZipFile(zipFilespec, payloadDirspec, true, @"/" + this.buildRepository.GetBuildManifest().FullyQualifiedPatchNumber); // true means use relative file paths

            // copy payload zip file to VixInstaller directories
            string VixInstallerDirspec = this.deployRepository.GetVixInstallerSolutionRootDirspec();
            VixInstallerDirspec = this.deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(VixInstallerDirspec);
            string VixInstallerSetupFilespec = Path.Combine(VixInstallerDirspec, @"VixInstallerSolution2013\VixInstallerSetup\VixDistribution.zip");
            string VixInstallerDebugBinVixDirspec = Path.Combine(VixInstallerDirspec, @"VixInstallerSolution2013\VixInstaller\bin\Debug\Vix");
            string VixInstallerDebugBinVixFilespec = Path.Combine(VixInstallerDebugBinVixDirspec, @"VixDistribution.zip");
            string VixInstallerNightlyBinVixDirspec = Path.Combine(VixInstallerDirspec, @"VixInstallerSolution2013\VixInstallerNightly\bin\Debug\Vix");
            string VixInstallerNightlyBinVixFilespec = Path.Combine(VixInstallerNightlyBinVixDirspec, @"VixDistribution.zip");

            LogInfo("\tCopying zip payload file to " + VixInstallerSetupFilespec);
            File.Copy(zipFilespec, VixInstallerSetupFilespec, true);

            if (config.DeployPayloadForInstallerDebug)
            {
                if (Directory.Exists(VixInstallerDebugBinVixDirspec))
                {
                    LogInfo("\tDeleting directory " + VixInstallerDebugBinVixDirspec);
                    Directory.Delete(VixInstallerDebugBinVixDirspec, true);
                }
                LogInfo("\tCreating directory " + VixInstallerDebugBinVixDirspec);
                Directory.CreateDirectory(VixInstallerDebugBinVixDirspec);
                LogInfo("\tCopying zip payload file to " + VixInstallerDebugBinVixFilespec);
                File.Copy(zipFilespec, VixInstallerDebugBinVixFilespec, true);
            }

            if (config.DeployPayloadForNightlyBuild)
            {
                if (Directory.Exists(VixInstallerNightlyBinVixDirspec))
                {
                    LogInfo("\tDeleting directory " + VixInstallerNightlyBinVixDirspec);
                    Directory.Delete(VixInstallerNightlyBinVixDirspec, true);
                }
                LogInfo("\tCreating directory " + VixInstallerNightlyBinVixDirspec);
                Directory.CreateDirectory(VixInstallerNightlyBinVixDirspec);
                LogInfo("\tCopying zip payload file to " + VixInstallerNightlyBinVixFilespec);
                File.Copy(zipFilespec, VixInstallerNightlyBinVixFilespec, true);
            }

            if (config.DeletePayloadRootContents)
            {
                this.RecreateDirectory(config.VixPayloadRootDirspec);
            }

            LogInfo("Construction and deployment of VIX payload zip file complete");
        }

        /// <summary>
        /// Updates the msi identity and version. Requires that the VIX Installer setup project and the MsiVersion.txt file be checked out ahead of time.
        /// </summary>
        /// <remarks></remarks>
        public void UpdateMsiIdentityAndVersion()
        {
            this.LogInfo("Beginning Team City driven update MSI Identity and version phase...");
            string msiSetupProjectFilespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerSetup\VixInstallerSetup.vdproj");
            this.LogInfo("\tUpdating version information in " + msiSetupProjectFilespec);
            string assemblyInfoFilespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerBusiness\Properties\AssemblyInfo.cs");
            string assemblyVersion = AssemblyInfoHelper.GetAssemblyVersion(assemblyInfoFilespec);
            VisualStudioSetupProjectHelper visualStudioSetupProjectHelper = new VisualStudioSetupProjectHelper(msiSetupProjectFilespec, config, buildRepository, deployRepository, assemblyVersion);
            visualStudioSetupProjectHelper.RewriteSetupFile(false); // do not check out files from version control
            this.LogInfo("Update MSI Identity and version phase complete.");
        }

        /// <summary>
        /// Builds the vix installer msi.
        /// </summary>
        /// <param name="applicationDirspec">The application directory specification.</param>
        public void BuildVixInstallerMsi(string applicationDirspec)
        {
            // "C:\Program Files\Microsoft Visual Studio 8\Common7\IDE\devenv.exe" "C:\VSS\ViX\VixInstaller\VixInstaller.sln" /Rebuild "Release"
            string msiSetupProjectFilespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerSetup\VixInstallerSetup.vdproj");
            string msiSetupProjectLocalFilespec = deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(msiSetupProjectFilespec);
            this.LogInfo("Beginning VIX Installer creation phase...");
            this.LogInfo("\tCheckout " + msiSetupProjectFilespec);
            if (this.deployRepository.CheckoutFile(msiSetupProjectFilespec) == false)
            {
                throw new BuildException(this.LogError("@Checkout of " + msiSetupProjectFilespec + " failed - aborting build process"));
            }

            string nextInstallerVersionNumber;
            try
            {
                this.LogInfo("\tUpdating version information in " + msiSetupProjectFilespec);
                string assemblyInfoFilespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerBusiness\Properties\AssemblyInfo.cs");
                string assemblyInfoLocalFilespec = deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(assemblyInfoFilespec);
                string assemblyVersion = AssemblyInfoHelper.GetAssemblyVersion(assemblyInfoLocalFilespec);
                string originalVersion = "original Assembly Version: " + assemblyVersion;
                LogInfo(originalVersion);
                VisualStudioSetupProjectHelper visualStudioSetupProjectHelper = new VisualStudioSetupProjectHelper(msiSetupProjectLocalFilespec, config, buildRepository, deployRepository, assemblyVersion);
                nextInstallerVersionNumber = visualStudioSetupProjectHelper.RewriteSetupFile(true); // checkout files from version control
                string nextVersion = "next Assembly Version: " + nextInstallerVersionNumber;
                LogInfo(nextVersion);
            }
            catch (Exception ex)
            {
                this.deployRepository.UndoFileCheckout(msiSetupProjectFilespec);
                throw new BuildException(LogError(ex.Message));
            }

            this.LogInfo("\tCheckin " + msiSetupProjectFilespec);
            string comment = "Vix Builder - VIX version " + this.buildRepository.GetBuildManifest().FullyQualifiedPatchNumber + " - VIX Installer version " + nextInstallerVersionNumber;
            if (deployRepository.CheckinFile(msiSetupProjectFilespec, comment) == false)
            {
                throw new BuildException(LogError("@Checkin of " + msiSetupProjectFilespec + " failed - aborting build process"));
            }

            string solutionFilespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerSolution2013.sln");
            string solutionLocalFilespec = deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(solutionFilespec);
            this.LogInfo("\tPerfoming a Release build of " + solutionLocalFilespec);
            int resultCode = this.BuildSolution(solutionLocalFilespec, "Release");
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("@Release build of " + solutionLocalFilespec + " failed - aborting build process"));
            }

            string sourceFilespec = this.GetVixInstallerSourceFilespec();
            string targetFilespec = this.GetVixInstallerTargetFilespec();
            string targetDirspec = Path.GetDirectoryName(targetFilespec);
            this.LogInfo("\tCopying VixInstaller.msi to  " + targetFilespec);
            if (!Directory.Exists(targetDirspec))
            {
                Directory.CreateDirectory(targetDirspec);
            }
            File.Copy(sourceFilespec, targetFilespec, true);
            string sourceLogFilespec = Path.Combine(applicationDirspec, "vix-build-log.txt");
            string targetLogFilespec = Path.Combine(targetDirspec, Path.GetFileNameWithoutExtension(targetFilespec) + "BuildLog.txt" );
            this.LogInfo("\tCopying " + sourceLogFilespec + " to " + targetLogFilespec);
            this.LogInfo("VIX Installer creation phase complete");
            File.Copy(sourceLogFilespec, targetLogFilespec, true);
        }

        /// <summary>
        /// Builds the nightly vix installer.
        /// </summary>
        /// <param name="applicationDirspec">The application dirspec.</param>
        /// <remarks></remarks>
        public void BuildNightlyVixInstaller(string applicationDirspec)
        {
            //"C:\Program Files\Microsoft Visual Studio 8\Common7\IDE\devenv.exe" "C:\VSS\ViX\VixInstaller\VixInstaller.sln" /Rebuild "Nightly"

            LogInfo("Beginning Nightly VIX Installer creation phase...");

            string solutionFilespec = Path.Combine(config.VssRootDirspec, @"ViX\VixInstallerSolution2013.root\VixInstallerSolution2013\VixInstallerSolution2013.sln");
            LogInfo("\tPerfoming a Nightly build of " + solutionFilespec);
            int resultCode = BuildSolution(solutionFilespec, "Nightly");
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("@Nightly build of " + solutionFilespec + " failed - aborting build process"));
            }

            LogInfo("Nightly VIX Installer creation phase complete");
        }

        /// <summary>
        /// Deploys the results of a build into the development environment. Specifically jars are copied into the apache tomcat directory, the
        /// server.xml file is regenerated, and an updated context.xml is deployed from VSS.
        /// </summary>
        /// <remarks>This method is unfinished. Support needs to be added to deploy web applications and to regenerate configuration files
        /// in a specified configuration directory.</remarks>
        public void DeployAndConfigureDevelopmentEnvironment()
        {
            this.LogInfo("Beginning development environment deployment.");

            ////if (config.DeployWebApplications)
            ////{
            ////    TomcatFacade.UndeployVixWebApplications(config, manifest);
            ////}
            
            // copy dependency files
            List<VixDependencyType> typesToDeploy = new List<VixDependencyType> {
                    VixDependencyType.AppJar,
                    VixDependencyType.JavaPropertyFile,
                    VixDependencyType.XmlConfigFile,
                    VixDependencyType.OpenSourceJar,
                    VixDependencyType.CommercialJar
                };

            if (this.config.DeployWebApplications) // invisible in the UI for now
            {
                typesToDeploy.Add(VixDependencyType.AppWar);
            }

            this.LogInfo("\tDeploying the following artifact types:" + this.GetDependenciesForDisplay(typesToDeploy));
            this.CopyDeploymentArtifacts(true, typesToDeploy);
            this.LogInfo("\tArtifact deployment complete.");

            // Update Tomcat configuration files if necessary
            if (this.config.GenerateTomcatConfigFiles)
            {
                if (TomcatFacade.IsServerConfiguredForVix(this.config) == false)
                {
                    List<DeployProjectType> typesToGet = new List<DeployProjectType> { DeployProjectType.payload };
                    DeployProject[] deployProjects = this.buildRepository.GetBuildManifest().GetDeployProjects();
                    this.LogInfo("\tCopying and configuring server.xml");
                    string serverXmlFilespec = TomcatFacade.CopyServerXmlFromVssToTomcat(this.config, this.deployRepository);
                    TomcatFacade.ConfigureServer(serverXmlFilespec);
                    this.LogInfo("\tCopying context.xml");
                    TomcatFacade.CopyContextXmlFromVssToTomcat(this.config, this.deployRepository);
                }
                if (TomcatFacade.IsTomcatUsersConfiguredForVix(this.config) == false)
                {
                    this.LogInfo("\tConfiguring tomcat users");
                    TomcatFacade.ConfigureTomcatUsers(this.config);
                }
            }

            // Deploy web applications if necessary
            ////if (config.DeployWebApplications)
            ////{
            ////    TomcatFacade.DeployVixWebApplications(config, manifest);
            ////}

            // create configuration files if necessary
            ////if (config.GenerateConfigFiles)
            ////{
            ////}

            this.LogInfo("Development environment deployment complete.");
        }

        #endregion

        #region private methods

        //private BuildManifest GetBuildManifest()
        //{
        //    string buildManifestFilespec = Path.Combine(config.CvsProjectRoot, ("ImagingExchange\\" + config.BuildManifestFilename));
        //    if (!File.Exists(buildManifestFilespec))
        //    {
        //        throw new BuildException("Missing build manifest " + buildManifestFilespec);
        //    }
        //    return new BuildManifest(buildManifestFilespec, config);
        //}

        /// <summary>
        /// Resets the standard output.
        /// </summary>
        /// <remarks></remarks>
        private void ResetStandardOutput()
        {
            this.outputStringBuilder = new StringBuilder();
        }

        /// <summary>
        /// Outputs the handler.
        /// </summary>
        /// <param name="sendingProcess">The sending process.</param>
        /// <param name="outLine">The <see cref="System.Diagnostics.DataReceivedEventArgs"/> instance containing the event data.</param>
        /// <remarks></remarks>
        private void OutputHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            // Collect the sort command output.
            if (!string.IsNullOrEmpty(outLine.Data))
            {
                this.outputStringBuilder.AppendLine(outLine.Data);
            }
        }

        /// <summary>
        /// Logs the info.
        /// </summary>
        /// <param name="message">The message.</param>
        /// <remarks></remarks>
        private void LogInfo(string message)
        {
            if (info != null)
            {
                info(message);
            }
            LogManager.GetLogger(this.GetType().Name).Info(message);
        }

        /// <summary>
        /// Displays the output.
        /// </summary>
        /// <remarks></remarks>
        private void DisplayOutput()
        {
            if (output != null)
            {
                output(GetOutput());
            }
        }

        /// <summary>
        /// Logs the error.
        /// </summary>
        /// <param name="message">The message.</param>
        /// <returns></returns>
        /// <remarks></remarks>
        private string LogError(string message)
        {
            LogManager.GetLogger(this.GetType().Name).Error(message);
            return message;
        }

        /// <summary>
        /// Yields to user interface.
        /// </summary>
        /// <remarks></remarks>
        private void YieldToUserInterface()
        {
            if (this.doEvents != null)
            {
                this.doEvents();
            }
        }

        ////private VixManifest GetDeploymentManifest(BuildConfiguration config)
        ////{
        ////    string deployManifestFilespec = Path.Combine(config.CvsProjectRoot, ("ImagingExchange\\" + config.DeploymentManifestFilename));
        ////    if (!File.Exists(deployManifestFilespec))
        ////    {
        ////        throw new BuildException("Missing build manifest " + deployManifestFilespec);
        ////    }
        ////    return new VixManifest(deployManifestFilespec);
        ////}

        /// <summary>
        /// Gets the dependencies for display.
        /// </summary>
        /// <param name="dependencyList">The dependency list.</param>
        /// <returns></returns>
        /// <remarks></remarks>
        private string GetDependenciesForDisplay(List<VixDependencyType> dependencyList)
        {
            StringBuilder sb = new StringBuilder();
            if (dependencyList == null)
            {
                sb.Append(" all");
            }
            else
            {
                foreach (VixDependencyType dependency in dependencyList)
                {
                    sb.AppendFormat(" {0}", dependency.ToString());
                }
            }
            return sb.ToString();
        }

        /// <summary>
        /// Gets the next version.
        /// </summary>
        /// <param name="currentVersion">The current version.</param>
        /// <returns></returns>
        /// <remarks></remarks>
        public string GetNextVersion(string currentVersion)
        {
            StringBuilder sb = new StringBuilder();
            string[] splits = currentVersion.Split('.');
            if (splits.Length != 3)
            {
                throw new BuildException("Version string is malformed: " + currentVersion);
            }

            // convert parts to ints
            int[] intSplits = new int[splits.Length];
            for (int i = 0; i < splits.Length; i++)
            {
                intSplits[i] = int.Parse(splits[i]);
            }

            intSplits[splits.Length - 1]++;

            // increment parts
            for (int i = splits.Length - 1; i > 0; i--)
            {
                if (intSplits[i] >= 100)
                {
                    intSplits[i] = 0;
                    intSplits[i - 1]++;
                }
            }

            // reassemble
            for (int i = 0; i < splits.Length; i++)
            {
                sb.Append(intSplits[i].ToString());
                if (i < splits.Length - 1)
                {
                    sb.Append(".");
                }
            }

            return sb.ToString();
        }

        #endregion

        #region private maven methods

        private void DeleteProjectFromMavenRepository(BuildProject buildProject)
        {
            string imagingRepoArtifactsDirspec = Path.Combine(config.Maven2RepoDirspec, "Imaging");
            string versionNumber = "0.1";
            if (buildProject.DependencyType == VixDependencyType.AppJar ||
                buildProject.DependencyType == VixDependencyType.AppWar)
            {
                versionNumber = config.OverrideJarWarVersioning.Trim();
            }
            imagingRepoArtifactsDirspec = Path.Combine(imagingRepoArtifactsDirspec, buildProject.ProjectName);
            imagingRepoArtifactsDirspec = Path.Combine(imagingRepoArtifactsDirspec, versionNumber);
            if (Directory.Exists(imagingRepoArtifactsDirspec))
            {
                LogInfo("\tDeleting " + imagingRepoArtifactsDirspec);
                Directory.Delete(imagingRepoArtifactsDirspec, true); // recurse
            }
        }


        /// <summary>
        /// Builds the maven2 project.
        /// </summary>
        /// <param name="config">The config.</param>
        /// <param name="maven2ProjectName">Name of the maven2 project.</param>
        /// <returns>An exit code consistent with batch file processing, namely 0 indicates successs and anything other than 0 indicates an error</returns>
        private int BuildMaven2Project(BuildProject buildProject)
        {

            int exitCode = 0; // external process executed ok - command shell convention

            if (!Directory.Exists(buildProject.WorkingFolderDirspec))
            {
                throw new BuildException("Maven2 project directory does not exist: " + buildProject.WorkingFolderDirspec);
            }

            StringBuilder infoSB = new StringBuilder();
            StringBuilder argumentsSB = new StringBuilder();

            infoSB.AppendFormat("\tPerforming Maven2 build of project {0}", buildProject.WorkingFolderDirspec);
            argumentsSB.Append("-f pom.xml clean install");
            if (String.IsNullOrEmpty(config.OverrideJarWarVersioning) == false)
            {
                DeleteProjectFromMavenRepository(buildProject);
                if (buildProject.DependencyType == VixDependencyType.AppJar ||
                    buildProject.DependencyType == VixDependencyType.AppWar)
                {
                    infoSB.AppendFormat(" : version {0}", config.OverrideJarWarVersioning);
                    argumentsSB.AppendFormat(" -DBUILD_VERSION={0}", config.OverrideJarWarVersioning);
                }
                else
                {
                    infoSB.AppendFormat(" : version {0}", "0.1");
                }
            }
            else
            {
                infoSB.AppendFormat(" : version {0}", "0.1");
            }

            if (config.DisableMavenUnitTests == true)
            {
                argumentsSB.Append(" -Dmaven.test.skip=true");
                infoSB.Append(" : skipping unit tests");
            }

            LogInfo(infoSB.ToString());

            exitCode = runExternalProcess(config.Maven2Executable, argumentsSB.ToString(), buildProject.WorkingFolderDirspec);

            /*
            Process externalProcess = new Process();
            externalProcess.StartInfo.FileName = config.Maven2Executable;
            externalProcess.StartInfo.Arguments = argumentsSB.ToString();
            externalProcess.StartInfo.WorkingDirectory = buildProject.WorkingFolderDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    this.YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }
            */
            return exitCode;
        }

        /// <summary>
        /// Returns the dirspec associated with the specofied BuildProject. If the build configuration has specified a workspace working folder
        /// then it will be used as the path to the project folder. Otherwise the build repository will be queried to provide the dirspec, which
        /// will be rooted on the top level workspace folder.
        /// </summary>
        /// <param name="buildProject">The project to build.</param>
        /// <returns>the dirspec of the VISA build project</returns>
        //private string GetMavenProjectDirspec(BuildProject buildProject)
        //{
        //    string maven2ProjectDirspec;

        //    //if (string.IsNullOrEmpty(config.WorkspaceWorkingFolder))
        //    //{
        //    //    maven2ProjectDirspec = this.buildRepository.GetBuildProjectDirspec(buildProject);
        //    //}
        //    //else
        //    //{
        //    //    maven2ProjectDirspec = Path.Combine(config.WorkspaceWorkingFolder, buildProject.ProjectName);
        //    //}

        //    maven2ProjectDirspec = Path.Combine(buildRepository.Manifest.WorkspaceWorkingFolder, buildProject.ProjectName);

        //    return maven2ProjectDirspec;
        //}

        /// <summary>
        /// Builds the maven2 projects.
        /// </summary>
        /// <param name="projectsToBuild">The projects to build.</param>
        private void BuildMaven2Projects(List<VixDependencyType> projectsToBuild)
        {
            //WFP-set VIXType environment variable.  This is necessary for specific WebApp project builds to select the correct 
            //  context.xml file.
            Environment.SetEnvironmentVariable(@"VIXType", Enum.GetName(typeof(InstallationTypeEnum), config.InstallationType), 
                EnvironmentVariableTarget.Machine);

            BuildProject[] buildProjects = this.buildRepository.GetBuildManifest().GetBuildProjects();

            LogInfo("Beginning Maven 2 build phase for the following project types:" + GetDependenciesForDisplay(projectsToBuild));
            foreach (BuildProject buildProject in buildProjects)
            {
                //ResetStandardOutput();
                if (projectsToBuild == null || projectsToBuild.Contains(buildProject.DependencyType))
                {
                    int resultCode = BuildMaven2Project(buildProject);
                    if (resultCode > 0)
                    {
                        // VIX cache build can fail occasionally - try again after clearing previous result set
                        //ResetStandardOutput();
                        resultCode = BuildMaven2Project(buildProject);
                    }
                    if (resultCode > 0)
                    {
                        DisplayOutput();
                        throw new BuildException(LogError("Maven2 build of project " + buildProject.WorkingFolderDirspec + " failed - aborting build process"));
                    }
                    if (CanRun == false)
                    {
                        LogInfo("Build canceled by user.");
                        return;
                    }
                    else if (config.DisplayMavenOutput == true)
                    {
                        DisplayOutput();
                    }
                }
            }
            LogInfo("Maven 2 build phase complete for the following project types:" + GetDependenciesForDisplay(projectsToBuild));
        }

        #endregion
 
    #region private deployment artifact methods

        /// <summary>
        /// Prepares the payload directory.
        /// </summary>
        /// <remarks></remarks>
        private void RecreateDirectory(string dirspec)
        {
            if (Directory.Exists(dirspec))
            {
                LogInfo("\t" + dirspec + " exists - deleting");
                Directory.Delete(dirspec, true); // recurse
            }
            LogInfo("\t Creating" + dirspec);
            Directory.CreateDirectory(dirspec);
        }
        
        /// <summary>
        /// Prepares the payload directory.
        /// </summary>
        /// <remarks></remarks>
        private void PreparePayloadDirectory()
        {
            string payloadRootDirspec = GetPayloadRootDirspec();
            if (Directory.Exists(payloadRootDirspec))
            {
                LogInfo("\t" + payloadRootDirspec + " exists - deleting");
                Directory.Delete(payloadRootDirspec, true); // recurse
            }
            LogInfo("\t Creating" + payloadRootDirspec);
            Directory.CreateDirectory(payloadRootDirspec);
        }

        /// <summary>
        /// Gets the payload root dirspec.
        /// </summary>
        /// <returns></returns>
        /// <remarks></remarks>
        private string GetPayloadRootDirspec()
        {
            string payloadRootDirspec = Path.Combine(config.VixPayloadRootDirspec, this.buildRepository.GetBuildManifest().FullyQualifiedPatchNumber);
            return payloadRootDirspec;
        }

        /// <summary>
        /// Gets the vix installer source filespec.
        /// </summary>
        /// <returns></returns>
        /// <remarks></remarks>
        private string GetVixInstallerSourceFilespec()
        {
            string filespec = Path.Combine(this.deployRepository.GetVixInstallerSolutionRootDirspec(), @"VixInstallerSolution2013\VixInstallerSetup\Release\VixInstallerSetup.msi");
            string localFilespec = deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(filespec);
            return localFilespec;
        }

        /// <summary>
        /// Gets the vix installer target filespec.
        /// </summary>
        /// <returns></returns>
        /// <remarks></remarks>
        private string GetVixInstallerTargetFilespec()
        {
            string payloadRootDirspec = Path.Combine(config.VixInstallerRootDirspec, this.buildRepository.GetBuildManifest().FullyQualifiedPatchNumber);
            string filespec = Path.Combine(payloadRootDirspec, this.buildRepository.GetBuildManifest().MsiFilename);
            return filespec;
        }

        /// <summary>
        /// Copies the deployment artifacts.
        /// </summary>
        /// <param name="isDevelopment">if set to <c>true</c> [is development].</param>
        /// <param name="typesToDeploy">The types to deploy.</param>
        /// <remarks></remarks>
        private void CopyDeploymentArtifacts(bool isDevelopment, List<VixDependencyType> typesToDeploy)
        {
            DeploymentArtifact[] deploymentArtifacts = this.buildRepository.GetBuildManifest().GetDeploymentArtifacts();
            DeploymentArtifact[] buildRepositoryDeploymentArtifacts = this.buildRepository.GetDeploymentArtifacts();
            List<DeploymentArtifact> artifacts = new List<DeploymentArtifact>();
            artifacts.AddRange(deploymentArtifacts);
            artifacts.AddRange(buildRepositoryDeploymentArtifacts);

            foreach (DeploymentArtifact deploymentArtifact in artifacts)
            {
                // note destination usually is a directory specification but can sometimes be a file specification (specification means fully qualified name)
                string destinationFilespec = isDevelopment ? deploymentArtifact.DevDestinationFilespec : deploymentArtifact.DestinationFilespec;

                if (string.IsNullOrEmpty(destinationFilespec))
                {
                    continue;
                }

                if (typesToDeploy == null || typesToDeploy.Contains(deploymentArtifact.DependencyType))
                {
                    // dont force a developer to deploy the webapps if they dont want to.
                    if (isDevelopment && deploymentArtifact.DependencyType == VixDependencyType.AppWar && config.DeployWebApplications == false)
                    {
                        continue;
                    }


                    string sourceFileSpec = deploymentArtifact.SourceFilespec;

                    if (deploymentArtifact.repositoryType == RepositoryEnum.TeamFoundationServer)
                    {
                        sourceFileSpec = deployRepository.GetLocalFolderSpecFromRepositoryWorkspace(sourceFileSpec);
                    }

                    //Developer override of artifact versioning versioning
                    if (isDevelopment && String.IsNullOrEmpty(config.OverrideJarWarVersioning) == false)
                    {
                        if (deploymentArtifact.DependencyType == VixDependencyType.AppJar ||
                            deploymentArtifact.DependencyType == VixDependencyType.AppWar)
                        {
                            sourceFileSpec = sourceFileSpec.Replace("0.1", config.OverrideJarWarVersioning);
                            destinationFilespec = destinationFilespec.Replace("0.1", config.OverrideJarWarVersioning);
                        }
                    }

                    File.Copy(sourceFileSpec, destinationFilespec, true);
                    File.SetAttributes(destinationFilespec, FileAttributes.Normal);
                    LogInfo("\tCopied " + sourceFileSpec + " to " + destinationFilespec);
                    if (CanRun == false)
                    {
                        return;
                    }
                }
            }
        }

        /// <summary>
        /// Deletes the repository artifacts.
        /// </summary>
        /// <param name="repoArtifactsToDelete">The repository artifacts to delete.</param>
        private void DeleteRepositoryArtifacts(List<VixDependencyType> repoArtifactsToDelete)
        {
            DeploymentArtifact[] deploymentArtifacts = this.buildRepository.GetBuildManifest().GetDeploymentArtifacts();

            this.LogInfo("Deleting repository artifact types:" + this.GetDependenciesForDisplay(repoArtifactsToDelete));
            foreach (DeploymentArtifact deploymentArtifact in deploymentArtifacts)
            {
                if (repoArtifactsToDelete.Contains(deploymentArtifact.DependencyType))
                {
                    if (File.Exists(deploymentArtifact.SourceFilespec))
                    {
                        this.LogInfo("\tDeleting " + deploymentArtifact.SourceFilespec);
                        File.Delete(deploymentArtifact.SourceFilespec);
                    }
                    if (this.CanRun == false)
                    {
                        this.LogInfo("Build canceled by user.");
                        return;
                    }
                }
            }
            this.LogInfo("Completed deletion of repository artifact types:" + this.GetDependenciesForDisplay(repoArtifactsToDelete));
        }

    #endregion

    #region private installer build methods
        /// <summary>
        /// Builds the solution.
        /// </summary>
        /// <param name="solutionFilespec">The solution filespec.</param>
        /// <param name="visualStudioBuildConfiguration">The visual studio build configuration.</param>
        /// <returns></returns>
        private int BuildSolution(string solutionFilespec, string visualStudioBuildConfiguration)
        {
            //"C:\Program Files\Microsoft Visual Studio 8\Common7\IDE\devenv.exe" "C:\VSS\ViX\VixBuilderSolution\VixBuilderSolution.sln" /Rebuild "Release"
            //ResetStandardOutput();

            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();
            //sb.AppendFormat("\"{0}\" /Build \"{1}\"", solutionFilespec, visualStudioBuildConfiguration);
            sb.AppendFormat("{0} /Rebuild {1}", solutionFilespec, visualStudioBuildConfiguration);
            string dirspec = Path.GetDirectoryName(solutionFilespec);

            exitCode = runExternalProcess(config.VisualStudioExeFilespec, sb.ToString(), dirspec);
    
            /*
            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.VisualStudioExeFilespec;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = dirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            catch
            {
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }
            */

            return exitCode;
        }


        #endregion

    #region private dev environment configure methods

        ///// <summary>
        ///// 
        ///// </summary>
        ///// <param name="commandParameters"></param>
        ///// <param name="payloadPath"></param>
        ///// <param name="config"></param>
        ////private void RunJavaUtility(String commandParameters, IVixConfigurationParameters config)
        ////{
        ////    String payloadJar = Path.Combine(VixFacade.Manifest.PayloadPath, @"server\jars");
        ////    Process externalProcess = new System.Diagnostics.Process();
        ////    String javaExeFilespec = Path.Combine(JavaFacade.GetActiveJavaPath(JavaFacade.IsActiveJreInstalled()), @"bin\java.exe");

        ////    externalProcess.StartInfo.FileName = javaExeFilespec;
        ////    externalProcess.StartInfo.Arguments = commandParameters;
        ////    externalProcess.StartInfo.WorkingDirectory = payloadJar;
        ////    externalProcess.StartInfo.UseShellExecute = false;
        ////    externalProcess.StartInfo.CreateNoWindow = true; // DKB - 5/25/10

        ////    //externalProcess.StartInfo.RedirectStandardError = true;
        ////    //externalProcess.StartInfo.RedirectStandardOutput = true;

        ////    if (externalProcess.StartInfo.EnvironmentVariables.ContainsKey("vixconfig"))
        ////    {
        ////        externalProcess.StartInfo.EnvironmentVariables.Remove("vixconfig");
        ////    }
        ////    externalProcess.StartInfo.EnvironmentVariables.Add("vixconfig", config.ConfigDir);

        ////    if (externalProcess.StartInfo.EnvironmentVariables.ContainsKey("vixcache"))
        ////    {
        ////        externalProcess.StartInfo.EnvironmentVariables.Remove("vixcache");
        ////    }
        ////    externalProcess.StartInfo.EnvironmentVariables.Add("vixcache", config.LocalCacheDir);

        ////    externalProcess.Start();
        ////    do
        ////    {
        ////        Thread.Sleep(500);
        ////        DoEvents();
        ////        externalProcess.Refresh();
        ////    } while (!externalProcess.HasExited);
        ////    //Console.WriteLine(externalProcess.StandardOutput.ReadToEnd());
        ////    //Console.WriteLine(externalProcess.StandardError.ReadToEnd());
        ////    // TODO: check return value
        ////    externalProcess = null;
        ////}

    #endregion

    #region compound actions

        /// <summary>
        /// Does the make nightly compound action.
        /// </summary>
        /// <param name="applicationDirspec">The application directory specification.</param>
        private void DoMakeNightlyCompoundAction(string applicationDirspec)
        {
            if (this.CanRun)
            {
                this.buildRepository.GetBuildProjects();
            }

            if (this.CanRun)
            {
                this.BuildVisaWorkspace();
            }
            if (this.CanRun)
            {
                this.deployRepository.GetDeployProjects();
            }

            if (this.CanRun)
            {
                this.BuildAndDeployVixInstallerPayload();
            }

            if (this.CanRun)
            {
                this.BuildNightlyVixInstaller(applicationDirspec);
            }
        }

        /// <summary>
        /// Does the makeprod compound action.
        /// </summary>
        /// <param name="applicationDirspec">The application directory specification.</param>
        private void DoMakeProdCompoundAction(string applicationDirspec)
        {
            if (this.CanRun)
            {
                this.buildRepository.GetBuildProjects();
            }

            if (this.CanRun)
            {
                this.BuildVisaWorkspace();
            }

            if (this.CanRun)
            {
                this.deployRepository.GetDeployProjects();
            }

            if (this.CanRun)
            {
                this.BuildAndDeployVixInstallerPayload();
            }

            if (this.CanRun)
            {
                this.BuildVixInstallerMsi(applicationDirspec);
            }
        }

        /// <summary>
        /// Does the make dev compound action.
        /// </summary>
        /// <param name="applicationDirspec">The application directory specification.</param>
        private void DoMakeDevCompoundAction(string applicationDirspec)
        {
            if (string.IsNullOrEmpty(this.config.ApacheTomcatDirspec))
            {
                throw new BuildConfigurationException("You must specify the location of the Tomcat directory before performing a development build.");
            }

            if (this.CanRun)
            {
                this.buildRepository.GetBuildProjects();
            }

            if (this.CanRun)
            {
                this.BuildVisaWorkspace();
            }

            if (this.config.GenerateTomcatConfigFiles && (TomcatFacade.IsServerConfiguredForVix(this.config) == false || TomcatFacade.IsTomcatUsersConfiguredForVix(this.config) == false))
            {
                List<DeployProjectType> typesToGet = new List<DeployProjectType>();
                typesToGet.Add(DeployProjectType.payload);
                if (this.CanRun)
                {
                    this.buildRepository.GetBuildProjects();
                }
            }

            if (this.CanRun)
            {
                this.DeployAndConfigureDevelopmentEnvironment();
            }
        }

        #endregion

        #region externalprocessing

        private int runExternalProcess(string filename, string arguments, string workingDirspec)
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention

            Process externalProcess = new Process();
            externalProcess.StartInfo.FileName = filename;
            externalProcess.StartInfo.Arguments = arguments;
            externalProcess.StartInfo.WorkingDirectory = workingDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    this.YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        #endregion
    }
}
