using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Xml;
using System.Diagnostics;
using log4net;

namespace VixBuilderBusiness
{

    public static class TomcatFacade
    {
        private static InfoDelegate info = null; // used to display a line of text in the main form's info text box
        private static DoEventsDelegate doEvents = null; // keep the UI responsive
        private static readonly string TOMCAT_ADMIN_ACCOUNT_NAME = "admin";
        private static readonly String VIX_KEYSTORE_FILENAME = "federation.keystore";
        private static readonly String VIX_TRUSTSTORE_FILENAME = "federation.truststore";


       #region properties
        public static InfoDelegate Info
        {
            set { info = value; }
        }

        public static DoEventsDelegate DoEvents
        {
            set { doEvents = value; }
        }
        #endregion
        
        #region public methods
        public static string CopyServerXmlFromVssToTomcat(BuildConfiguration config, IDeployRepository deployRepository)
        {
            string sourceFilespec = Path.Combine(deployRepository.GetVixPayloadRootDirspec(), @"common\Apache Tomcat\server.xml");
            string targetFilespec = Path.Combine(config.ApacheTomcatDirspec, @"conf\server.xml");
            File.Copy(sourceFilespec, targetFilespec, true);
            File.SetAttributes(targetFilespec, FileAttributes.Normal);
            return targetFilespec;
        }

        public static void CopyContextXmlFromVssToTomcat(BuildConfiguration config, IDeployRepository deployRepository)
        {
            string sourceFilespec = Path.Combine(deployRepository.GetVixPayloadRootDirspec(), @"common\Apache Tomcat\context.xml");
            string targetFilespec = Path.Combine(config.ApacheTomcatDirspec, @"conf\context.xml");
            File.Copy(sourceFilespec, targetFilespec, true);
            File.SetAttributes(targetFilespec, FileAttributes.Normal);
        }

        public static string GetTomcatConfigurationFolder(BuildConfiguration config)
        {
            return Path.Combine(config.ApacheTomcatDirspec, @"conf");
        }

        public static string GetTomcatWebApplicationFolder(BuildConfiguration config)
        {
            return Path.Combine(config.ApacheTomcatDirspec, @"webapps");
        }

        public static string GetTomcatWorkFolder(BuildConfiguration config)
        {
            return Path.Combine(config.ApacheTomcatDirspec, @"work\Catalina\660.URL       ");
        }

        public static bool IsTomcatUsersConfiguredForVix(BuildConfiguration config)
        {
            bool configured = false;
            string filespec = Path.Combine(config.ApacheTomcatDirspec, @"conf\tomcat-users.xml");
            string line;

            using (StreamReader sr = new StreamReader(filespec))
            {
                while ((line = sr.ReadLine()) != null)
                {
                    if (line.Contains("vixlog"))
                    {
                        configured = true;
                        break;
                    }
                }
                sr.Close();
            }

            return configured;
        }

        public static bool IsServerConfiguredForVix(BuildConfiguration config)
        {
            bool configured = false;
            string filespec = Path.Combine(config.ApacheTomcatDirspec, @"conf\server.xml");

            string line;

            using (StreamReader sr = new StreamReader(filespec))
            {
                while ((line = sr.ReadLine()) != null)
                {
                    if (line.Contains("Resource") && line.Contains("gov.va.med.server.CoreRouterFactory"))
                    {
                        configured = true;
                        break;
                    }
                }
                sr.Close();
            }

            return configured;
        }

        public static void ConfigureTomcatUsers(BuildConfiguration config)
        {
            StringBuilder sb = new StringBuilder();
            sb.AppendLine("<tomcat-users>");
            sb.AppendLine("\t<role rolename=\"clinical-display-user\"/>");
            sb.AppendLine("\t<role rolename=\"administrator\"/>");
            sb.AppendLine("\t<role rolename=\"developer\"/>");
            sb.AppendLine("\t<role rolename=\"tomcat\"/>");
            sb.AppendLine("\t<role rolename=\"manager\"/>");
            sb.AppendLine("\t<role rolename=\"peer-vixs\"/>");
            sb.AppendLine("\t<role rolename=\"admin\"/>");
            sb.AppendLine("\t<role rolename=\"vista-user\"/>");
            sb.AppendLine("\t<role rolename=\"tester\"/>");
            sb.AppendLine("\t<user username=\"alexdelarge\" password=\"655321\" roles=\"clinical-display-user\"/>");
            sb.AppendLine("\t<user username=\"vixlog\" password=\"tachik0ma\" roles=\"administrator,tester\"/>");
            //sb.AppendLine("\t<user username=\"vixdev\" password=\"tachik0ma\" roles=\"administrator,developer\"/>");
            sb.AppendLine("\t<user username=\"vixs\" password=\"vixs\" roles=\"peer-vixs\"/>");
            sb.AppendFormat("\t<user username=\"{0}\" password=\"{1}\" roles=\"admin,manager\" />", TOMCAT_ADMIN_ACCOUNT_NAME, "Rootbeer");
            sb.Append(Environment.NewLine);
            sb.AppendLine("</tomcat-users>");

            string filespec = Path.Combine(config.ApacheTomcatDirspec, @"conf\tomcat-users.xml");
            using (StreamWriter sw = File.CreateText(filespec))
            {
                sw.Write(sb.ToString());
            }

        }

        public static void ConfigureServer(string serverXmlFilespec)
        {
            Debug.Assert(File.Exists(serverXmlFilespec));
            XmlDocument xmlServerDom = new XmlDocument();
            xmlServerDom.Load(serverXmlFilespec);

            XmlNode xmlConnector = xmlServerDom.SelectSingleNode("Server/Service/Connector[@port='8443']");
            if (xmlConnector != null) // patch 34 doesnt have a federation interface
            {
                UpdateConnector(xmlConnector);
            }

            // update Engine element with default host name
            XmlNode xmlEngine = xmlServerDom.SelectSingleNode("Server/Service[@name='Catalina']/Engine");
            UpdateEngine(xmlEngine);

            // update Host element name attribute
            XmlNode xmlHost = xmlEngine.SelectSingleNode("./Host");
            UpdateHost(xmlHost);

            // update Realm element site information
            XmlNode xmlRealm = xmlHost.SelectSingleNode("./Realm");
            UpdateRealm(xmlRealm);

            // configure the DicomSCPService service child elements if present
            xmlEngine = xmlServerDom.SelectSingleNode("Server/Service[@name='DicomSCPService']/Engine");
            if (xmlEngine != null)
            {
                // update Engine element with default host name
                UpdateEngine(xmlEngine);
                // update Realm element site information
                xmlRealm = xmlEngine.SelectSingleNode("./Realm");
                UpdateRealm(xmlRealm);
                // update Host element name attribute
                xmlHost = xmlEngine.SelectSingleNode("./Host");
                UpdateHost(xmlHost);
            }

            // configure the PeriodicCommandService service child elements if present
            xmlEngine = xmlServerDom.SelectSingleNode("Server/Service[@name='PeriodicCommandService']/Engine");
            if (xmlEngine != null)
            {
                // update Engine element with default host name
                UpdateEngine(xmlEngine);
                // update Realm element site information
                xmlRealm = xmlEngine.SelectSingleNode("./Realm");
                UpdateRealm(xmlRealm);
                // update Host element name attribute
                xmlHost = xmlEngine.SelectSingleNode("./Host");
                UpdateHost(xmlHost);
            }

            xmlServerDom.Save(serverXmlFilespec);
        }

        public static void UndeployVixWebApplications(BuildConfiguration config, BuildManifest manifest)
        {
            if (config.DeployWebApplications == false)
            {
                return; // sanity check
            }

            VixWebApplication[] vixWebApplications = manifest.GetVixWebApplications();
            LogInfo("\tBegin uninstalling web applications.");

            foreach (VixWebApplication vixWebApplication in vixWebApplications)
            {
                LogInfo("\t\tUninstalling Web Application " + vixWebApplication.Path);
                string warFileSpec = Path.Combine(TomcatFacade.GetTomcatWebApplicationFolder(config), vixWebApplication.War);
                if (File.Exists(warFileSpec))
                {
                    File.Delete(warFileSpec);
                    LogInfo("\t\tDeleted " + warFileSpec);
                }

                string pathFolder = vixWebApplication.Path.StartsWith("/") ? vixWebApplication.Path.Substring(1) : vixWebApplication.Path;
                string webAppPathFolder = Path.Combine(TomcatFacade.GetTomcatWebApplicationFolder(config), pathFolder);
                if (Directory.Exists(webAppPathFolder))
                {
                    Directory.Delete(webAppPathFolder, true); // recurse
                    LogInfo("\t\tDeleted " + webAppPathFolder);
                }
                string webAppPathWorkFolder = Path.Combine(TomcatFacade.GetTomcatWorkFolder(config), pathFolder);
                if (Directory.Exists(webAppPathWorkFolder))
                {
                    Directory.Delete(webAppPathWorkFolder, true); // recurse
                    LogInfo("\t\tDeleted " + webAppPathWorkFolder);
                }

            }
            LogInfo("\tAll web applications have been uninstalled.");

        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="config"></param>
        public static void DeployVixWebApplications(BuildConfiguration config, BuildManifest manifest)
        {
            VixWebApplication[] vixWebApplications = manifest.GetVixWebApplications();

            string tomcatCatalinaHostPathFolder = Path.Combine(TomcatFacade.GetTomcatConfigurationFolder(config), @"Catalina\660.URL       ");
            if (!Directory.Exists(tomcatCatalinaHostPathFolder))
            {
                Directory.CreateDirectory(tomcatCatalinaHostPathFolder);
            }

            LogInfo("\tBegin deploying web applications.");
            foreach (VixWebApplication vixWebApplication in vixWebApplications)
            {
                if (vixWebApplication.WebApplicationOption == VixWebApplicationOption.Remove)
                {
                    continue; // dont deploy web applications marked as remove
                }

                LogInfo("\t\tDeploying Web Application " + vixWebApplication.Path);
                string warFileSpec = Path.Combine(TomcatFacade.GetTomcatWebApplicationFolder(config), vixWebApplication.War);
                Debug.Assert(File.Exists(warFileSpec));

                string pathFolder = vixWebApplication.Path.StartsWith("/") ? vixWebApplication.Path.Substring(1) : vixWebApplication.Path;
                string webAppPathFolder = Path.Combine(TomcatFacade.GetTomcatWebApplicationFolder(config), pathFolder);
                if (Directory.Exists(webAppPathFolder))
                {
                    Directory.Delete(webAppPathFolder, true);
                }
                Directory.CreateDirectory(webAppPathFolder);
                ZipUtilities.UnZip(warFileSpec, webAppPathFolder); // unzip the war to the specified path
                File.Delete(warFileSpec); // delete the war so that it does not auto deploy
                // copy context file to catalina host folder
                string contextTargetFilename = pathFolder + ".xml";
                string contextTargetPathFilespec = Path.Combine(tomcatCatalinaHostPathFolder, contextTargetFilename);
                string contextSourcePathFilespec = Path.Combine(webAppPathFolder, @"META-INF\context.xml");
                if (!File.Exists(contextSourcePathFilespec))
                {
                    contextSourcePathFilespec = @"c:\vix\vixbuilder\" + pathFolder + @"\context.xml"; ;
                }
                if (!File.Exists(contextSourcePathFilespec))
                {
                    throw new BuildException("Error: " + contextSourcePathFilespec + " not found.");
                }
                File.Copy(contextSourcePathFilespec, contextTargetPathFilespec, true);
            }
            LogInfo("\tWeb application deployment compete.");
        }
        #endregion

        #region private methods
        private static void UpdateConnector(XmlNode xmlConnector)
        {
            String s = Path.Combine("C:/VixCertStore", VIX_KEYSTORE_FILENAME);
            String federationKeystorePath = s.Replace(@"\", "/");
            s = Path.Combine("C:/VixCertStore", VIX_TRUSTSTORE_FILENAME);
            String federationTruststorePath = s.Replace(@"\", "/");

            xmlConnector.Attributes["keystoreFile"].Value = federationKeystorePath;
            xmlConnector.Attributes["keystorePass"].Value = "Marklar";
            xmlConnector.Attributes["truststoreFile"].Value = federationTruststorePath;
            xmlConnector.Attributes["truststorePass"].Value = "R00tbeer";
        }

        private static void UpdateEngine(XmlNode xmlEngine)
        {
            xmlEngine.Attributes["defaultHost"].Value = "660.URL       ";
        }

        private static void UpdateHost(XmlNode xmlHost)
        {
            xmlHost.Attributes["name"].Value = "660.URL       ";
        }

        private static void UpdateRealm(XmlNode xmlRealm)
        {
            xmlRealm.Attributes["siteNumber"].Value = "660";
            xmlRealm.Attributes["siteAbbreviation"].Value = "SLC";
            xmlRealm.Attributes["siteName"].Value = "Salt Lake City, UT";
            xmlRealm.Attributes["vistaServer"].Value = "localhost";
            xmlRealm.Attributes["vistaPort"].Value = "9300";
        }

        private static void LogInfo(string message)
        {
            if (info != null)
            {
                info(message);
            }
            LogManager.GetLogger("VixBuilder").Info(message);
        }

        private static string LogError(string message)
        {
            LogManager.GetLogger("VixBuilder").Error(message);
            return message;
        }

        private static void YieldToUI()
        {
            if (doEvents != null)
            {
                doEvents();
            }
        }


        #endregion

    }
}
