﻿using System;
using System.Collections.Generic;
using System.Text;
using log4net;
using System.Diagnostics;
using System.IO;
using System.Threading;
using System.Text.RegularExpressions;


namespace VixBuilderBusiness
{

    internal class TfsRepository : AbstractRepository, IBuildRepository, IDeployRepository
    {
        private BuildManifest manifest;
        private bool IsWorkspaceSet { get; set; }

        private const int RetryCount = 5;

        public TfsRepository(BuildConfiguration config) : base(config, RepositoryEnum.TeamFoundationServer)
        {
            
        }

        #region IBuildRepository

        public DeploymentArtifact[] GetDeploymentArtifacts()
        {
            return this.Manifest.GetRepositoryDeploymentArtifacts();
        }

        /// <summary>
        /// Gets the build manifest.
        /// </summary>
        /// <value>The build manifest.</value>
        /// <remarks>This property is delay initialized because the build manifest should not be accessed until after the ImagingExchange project
        /// has been retrieved from the build repository.</remarks>
        public BuildManifest Manifest
        {
            get
            {
                if (this.manifest == null)
                {
                    this.manifest = GetBuildManifest(); // manifest may not exist until after ImagingExchange or VisaBuildConfiguration project has been retrieved
                }
                Debug.Assert(this.manifest != null, "No build manifest available");
                return this.manifest;
            }
        }

        public void DeleteBuildProjects()
        {
            BuildProject[] buildProjects = Manifest.GetBuildProjects();

            LogInfo("Beginning TFS project deletion phase...");

            foreach (BuildProject buildProject in buildProjects)
            {
                DeleteTfsProject(buildProject);
            }
            LogInfo("TFS project deletion phase complete");
        }

        public void GetBuildProjects()
        {
            int resultCode;
            if (!IsWorkspaceSet)
            {
                LogInfo("Setting TFS workspace to " + config.TfsWorkspace);
                resultCode = SetWorkspace();
                if (resultCode > 0)
                {
                    DisplayOutput();
                    throw new BuildException(LogError("Setting TFS workspace to " + config.TfsWorkspace + "failed. Please see output for additional information."));
                }
                IsWorkspaceSet = true;
                this.LogInfo("");
            }
            if (CanRun) this.GetVisaConfiguration();
            if (this.config.DeleteBuildProjectsBeforeGet)
            {
                if (this.CanRun)
                {
                    this.DeleteBuildProjects();
                }
            }
            if (CanRun) this.GetTfsBuildProjects();
        }

        public void LabelBuildProjects()
        {
            throw new NotImplementedException("TfsRepository.LabelBuildProjects not implemented.");
        }

        public void BranchBuildProjects()
        {
            throw new NotImplementedException("TfsRepository.BranchBuildProjects not implemented.");
        }

        public void DeleteLabelFromBuildProjects()
        {
            throw new NotImplementedException("TfsRepository.DeleteLabelFromVisaWorkspace not implemented.");
        }

        public override BuildManifest GetBuildManifest()
        {
            string buildManifestFilespec = Path.Combine(Path.Combine(config.TfsRootDirspec, @"eHMP\IMAG_eHMP\IMAG_Build\Configuration\VisaBuildConfiguration"), config.BuildManifestFilename);
            if (!File.Exists(buildManifestFilespec))
            {
                throw new BuildException("Missing build manifest " + buildManifestFilespec);
            }
            return new BuildManifest(buildManifestFilespec, config);
        }

        //public string GetBuildProjectDirspec(BuildProject buildProject)
        //{
        //    return Path.Combine(this.config.TfsRootDirspec, buildProject.GetProjectRepositoryRelativeDirspec(RepositoryEnum.TeamFoundationServer));
        //}

        #endregion

        #region IDeployRepository methods
        public void GetDeployProjects()
        {
            DeployProject[] deployProjects = this.Manifest.GetDeployProjects();

            LogInfo("Beginning TFS Deploy project Get phase...");
            foreach (DeployProject deployProject in deployProjects)
            {
                string projectDirspec = Path.Combine(Path.Combine(config.TfsRootDirspec, deployProject.TfsRelativeProjectPath), deployProject.ProjectName);
                //string logInfoMessage = "\tPerforming TFS Get of project " + projectDirspec;
                //LogInfo(logInfoMessage);

                // Build the project dirspec without considering working folder mappings.
                int resultCode = this.GetTfsProjectWithRetry(projectDirspec, null, RetryCount);

                //if (resultCode == 1 && config.SkipTfsWriteableFilesOnGet) // DKB - allow for TFS files to be locally modified - useful for developers
                //{
                //    if (config.DisplayTfsOutput == true)
                //    {
                //        DisplayOutput();
                //    }
                //}
                if (resultCode > 0)
                {
                    Output(GetOutput());
                    throw new BuildException(LogError("Get of project " + projectDirspec + " failed - aborting build process"));
                }
                else if (config.DisplayTfsOutput == true)
                {
                    Output(GetOutput());
                }

                if (CanRun == false)
                {
                    LogInfo("Build canceled by user.");
                    return;
                }
            }
            LogInfo("TFS Deploy Project Get phase complete");
        }

        /// <summary>
        /// Checkout a file from TFS.
        /// </summary>
        /// <param name="fileSpec">A filespec constructed using the root of the workspace as the starting point.</param>
        /// <returns>true if sucessful, false otherwise</returns>
        /// <remarks>filespec should be constructed without regard to any working folder mappings that may be present.</remarks>
        public bool CheckoutFile(string fileSpec)
        {
            bool success = true;
            LogInfo("\tPerforming TFS Checkout of " + fileSpec);
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention

            StringBuilder sb = new StringBuilder();

            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" checkout $/VistA_Imaging/Dev/foo.txt /login:"username,password"
            sb.AppendFormat("checkout {0} /login:\"{1},{2}\"", this.TranslateFilespecToTfsSemantics(fileSpec), config.TfsUsername, config.TfsPassword);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            catch
            {
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            if (exitCode > 0)
            {
                DisplayOutput();
                success = false;
            }
            else if (config.DisplayTfsOutput == true)
            {
                Output(GetOutput());
            }
            return success;
        }


        /// <summary>
        /// Undo a checkout file operation from TFS.
        /// </summary>
        /// <param name="fileSpec">A filespec constructed using the root of the workspace as the starting point.</param>
        /// <returns>true if sucessful, false otherwise</returns>
        /// <remarks>filespec should be constructed without regard to any working folder mappings that may be present.</remarks>
        public bool UndoFileCheckout(string fileSpec)
        {
            bool success = true;
            LogInfo("\tUndoing the TFS Checkout of " + fileSpec);
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention

            StringBuilder sb = new StringBuilder();

            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" checkout $/VistA_Imaging/Dev/foo.txt /login:"username,password"
            sb.AppendFormat("undo {0} /login:\"{1},{2}\"", this.TranslateFilespecToTfsSemantics(fileSpec), config.TfsUsername, config.TfsPassword);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
            }
            catch
            {
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            if (exitCode > 0)
            {
                DisplayOutput();
                success = false;
            }
            else if (config.DisplayTfsOutput == true)
            {
                Output(GetOutput());
            }
            return success;
        }

        /// <summary>
        /// Checkin a file to TFS.
        /// </summary>
        /// <param name="fileSpec">A filespec constucted using the root of the workspace as the starting point.</param>
        /// <returns>true if sucessful, false otherwise</returns>
        /// <remarks>filespec should be constructed without regard to any working folder mappings that may be present.</remarks>
        public bool CheckinFile(string filespec, string comment)
        {
            bool success = true;
            LogInfo("\tPerforming TFS Checkin of " + filespec);
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention

            StringBuilder sb = new StringBuilder();

            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" checkout $/VistA_Imaging/Dev/foo.txt /login:"username,password"
            sb.AppendFormat("checkin {0} /login:\"{1},{2}\" /comment:\"{3}\" /noprompt", this.TranslateFilespecToTfsSemantics(filespec), config.TfsUsername, config.TfsPassword, comment);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
            }
            catch
            {
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            if (exitCode > 0)
            {
                DisplayOutput();
                success = false;
            }
            else if (config.DisplayTfsOutput == true)
            {
                Output(GetOutput());
            }
            return success;
        }

        /// <summary>
        /// Copies the payload structure and static files from the repository to the payload construction directory.
        /// </summary>
        /// <param name="payloadRootDirspec">The payload construction root dirspec.</param>
        /// <remarks></remarks>
        public void CopyPayloadArtifacts()
        {
            // copy payload skeleton and files from deployment repository
            string payloadRootDirspec = Path.Combine(config.VixPayloadRootDirspec, Manifest.FullyQualifiedPatchNumber);
            string repositoryPayloadRootDirspec = this.GetVixPayloadRootDirspec();
            Debug.Assert(Directory.Exists(repositoryPayloadRootDirspec));
            CopyAll(new DirectoryInfo(repositoryPayloadRootDirspec), new DirectoryInfo(payloadRootDirspec));
        }

        /// <summary>
        ///  // Gets the absolute dirspec of VixInstallerSolution20xx.root
        /// </summary>
        /// <remarks></remarks>
        public string GetVixInstallerSolutionRootDirspec()
        {
            DeployProject installerSolutionRootProject = Manifest.GetVixInstallerSolutionRootDeployProject();
            string repositorySolutionRootDirspec = Path.Combine(config.TfsRootDirspec, installerSolutionRootProject.TfsRelativeProjectPath);
            repositorySolutionRootDirspec = Path.Combine(repositorySolutionRootDirspec, installerSolutionRootProject.ProjectName);
            return repositorySolutionRootDirspec;
        }

        /// <summary>
        ///  // Gets the absolute dirspec of Vix Installer payload directory
        /// </summary>
        /// <remarks></remarks>
        public string GetVixPayloadRootDirspec()
        {
            DeployProject installerPayloadRootProject = Manifest.GetPayloadDeployProject();
            string repositoryPayloadRootDirspec = Path.Combine(config.TfsRootDirspec, installerPayloadRootProject.TfsRelativeProjectPath);
            string outputStr1 = "repositoryPayloadRootDirspec1: " + repositoryPayloadRootDirspec;
            LogInfo(outputStr1);
            string outputStr3 = "repositoryPayloadRootDirspec3: " + repositoryPayloadRootDirspec;
            LogInfo(outputStr3);
            string repositoryPayloadLocalFolder = this.GetLocalFolderSpecFromRepositoryWorkspace(repositoryPayloadRootDirspec);
            string outputStr4 = "repositoryPayloadLocalFolder: " + repositoryPayloadLocalFolder;
            LogInfo(outputStr4);
            return repositoryPayloadLocalFolder;
        }


        /// <summary>
        /// Gets the next msi version from the file specified by the build manifest.
        /// </summary>
        /// <returns>The next MSI version number as a three part string.</returns>
        /// <remarks>As a side effect, the file that holds the last version number is checked out, rewritten with the new version number, and then checked in.</remarks>
        public string GetNextMsiVersion(bool performCheckout)
        {
            string versionNumberFilespec = Manifest.TfsMsiVersionNumberFilespec;
            string versionNumberLocalFilespec = this.GetLocalFolderSpecFromRepositoryWorkspace(versionNumberFilespec);
            string nextMsiVersionNumber = null;
            if (String.IsNullOrEmpty(versionNumberLocalFilespec) == false && File.Exists(versionNumberLocalFilespec))
            {
                if (performCheckout)
                {
                    // checkout
                    if (this.CheckoutFile(versionNumberFilespec))
                    {
                        try
                        {
                            nextMsiVersionNumber = this.GetNextVersionNumberFromFile(versionNumberLocalFilespec);
                            this.CheckinFile(
                                versionNumberFilespec, "Increment VISA MSI version number to " + nextMsiVersionNumber);
                        }
                        catch (Exception ex)
                        {
                            this.UndoFileCheckout(versionNumberFilespec);
                            throw new BuildException(this.LogError(ex.Message));
                        }
                    }
                    else
                    {
                        throw new BuildException("Could not checkout " + versionNumberFilespec);
                    }
                }
                else
                {
                    nextMsiVersionNumber = this.GetNextVersionNumberFromFile(versionNumberLocalFilespec);
                }

            }

            return nextMsiVersionNumber;
        }

        /// <summary>
        /// Finds the corresponding local folder for a TFS workspace server folder.
        /// </summary>
        /// <returns>local folder string</returns>
        /// <remarks>error code uses DOS command line semantics</remarks>
        public string GetLocalFolderSpecFromRepositoryWorkspace(string repositoryFileSpec)
        {
            repositoryFileSpec = this.TranslateFilespecToTfsSemantics(repositoryFileSpec);

            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" workfold /workspace:TFSBuildWorkspace /login:"username,password"
            ResetStandardOutput();

            int exitCode = 0; // external process executed ok - command shell convention
            ////string dirspec = Path.Combine(config.TfsRootDirspec, Manifest.TfsRelativeBuildProjectPath);
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("workfold /workspace:{0} /login:\"{1},{2}\" {3}", config.TfsWorkspace, config.TfsUsername,
                    config.TfsPassword, repositoryFileSpec);

            Process externalProcess = new Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    this.YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            if (exitCode > 0)
            {
                return null;
            }
            //LogInfo(GetOutput()); // show tf.exe output on success
            string tfsLocalFolder = null;
            string tfsPath = null;

            string strOutput = GetOutput();
            if (strOutput == null)
            {
                return null;
            }
            //the result output consist of 4 lines.  Split lines using "\r\n" delimiter.  Need to grab the 4th line.
            //char[] delimiters = new char[] {'\n' };
            string[] lines = Regex.Split(strOutput, "\r\n");
            if (lines.Length < 4)
            {
                LogError("TFS response did not contain 4 lines.");
                return null;
            }
            string translateLine = lines[3];
            LogInfo("TFS Mapping: " + translateLine);
            //split 4th line using the ":" delimiter.
            if (!translateLine.Contains(":"))
            {
                LogError("TFS response did not contain a colon.");
                return null;
            }
            //Find 1st colon character and split at that index.
            int index = translateLine.IndexOf(':');
            //string[] translatePieces = translateLine.Split(':');
            //remove leading and trailing whitespace characters.
            tfsPath = translateLine.Substring(0, index).Trim();
            tfsLocalFolder = translateLine.Substring(index + 1, (translateLine.Length - 1) - index);
            string outputStrPath = "TFS repository path: " + tfsPath;
            //LogInfo(outputStrPath);
            string outputStrLocal = "TFS Local path: " + tfsLocalFolder;
            //LogInfo(outputStrLocal);

            //if (!repositoryFileSpec.Equals(tfsPath, StringComparison.OrdinalIgnoreCase))
            //{
            //    LogError("The TFS server folder did not match the given TFS server folder.");
            //    return null;
            //}
            //LogInfo("returning non-null string value.");
            return tfsLocalFolder;
        }


        #endregion

        #region private methods

        private void GetVisaConfiguration()
        {
            ResetStandardOutput();
            string projectDirspec = Path.Combine(config.TfsRootDirspec, @"eHMP\IMAG_eHMP\IMAG_Build\Configuration\VisaBuildConfiguration");

            int resultCode = this.GetTfsProjectWithRetry(projectDirspec, null, RetryCount);

            //if (resultCode == 1 && config.SkipTfsWriteableFilesOnGet) // DKB - allow for TFS files to be locally modified - useful for developers
            //{
            //    if (config.DisplayTfsOutput == true)
            //    {
            //        DisplayOutput();
            //    }
            //}
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("Error getting " + projectDirspec + " project. Please see output for additional information."));
            }
            else if (config.DisplayTfsOutput == true)
            {
                Output(GetOutput());
            }
        }
        
        private void DeleteTfsProject(BuildProject buildProject)
        {
            if (Directory.Exists(buildProject.WorkingFolderDirspec))
            {
                LogInfo("\tDeleting  " + buildProject.WorkingFolderDirspec);
                this.RemoveReadOnlyAttributes(buildProject.WorkingFolderDirspec); // TFS like to make files read only
                Directory.Delete(buildProject.WorkingFolderDirspec, true); // recursive
            }
        }

        /// <summary>
        /// Sets the TFS workspace to use for subsequent commands.
        /// </summary>
        /// <returns>0 for sucess, non zeo otherwise</returns>
        /// <remarks>error code uses DOS command line semantics</remarks>
        private int SetWorkspace()
        {
            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" workfold /workspace:TFSBuildWorkspace /login:"username,password"
            ResetStandardOutput();

            int exitCode = 0; // external process executed ok - command shell convention
            ////string dirspec = Path.Combine(config.TfsRootDirspec, Manifest.TfsRelativeBuildProjectPath);
            StringBuilder sb = new StringBuilder();
            sb.AppendFormat("workfold /workspace:{0} /login:\"{1},{2}\"", config.TfsWorkspace, config.TfsUsername, config.TfsPassword);

            Process externalProcess = new Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    this.YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            LogInfo(GetOutput()); // show tf.exe output on success
            return exitCode;
        }

        private int GetTfsProjectWithRetry(string repositoryDirspec, string workingFolderDirspec, int retryCount)
        {
            int resultCode = 0;
            Debug.Assert(retryCount > 0);

            for (int i = 0; i < retryCount; i++)
            {
                bool allowForceOption = (i == 0);

                string displayDirspec = (String.IsNullOrEmpty(workingFolderDirspec) ? repositoryDirspec : workingFolderDirspec);

                LogInfo(this.InfoGetMessage((displayDirspec), i, allowForceOption));    

                resultCode = this.GetTfsProject(repositoryDirspec, allowForceOption);
                if (resultCode == 0) 
                {
                    break;
                }
                //else if (resultCode == 1 && config.SkipTfsWriteableFilesOnGet)
                //{
                //    break;
                //}
                if (config.DisplayTfsOutput)
                {
                    Output(GetOutput());
                }
            }
            return resultCode;
        }

        private string InfoGetMessage(string repositoryDirspec, int retryPass, bool allowForceOption)
        {
                StringBuilder sb = new StringBuilder();
                sb.AppendFormat("\t{0} TFS Get of project {1}", (retryPass == 0 ? "Performing" : "Retrying"), repositoryDirspec);
                string label = GetTfsLabel();
                if (String.IsNullOrEmpty(label) == false)
                {
                    sb.AppendFormat(" : label {0}", label);
                }
                if (allowForceOption && UseForceOption())
                {
                    sb.Append(" : force");
                }
                return sb.ToString();
        }

        private void GetTfsBuildProjects()
        {
            BuildProject[] buildProjects = Manifest.GetBuildProjects();

            LogInfo("Beginning TFS Build Project Checkout phase...");
            foreach (BuildProject buildProject in buildProjects)
            {
                //string tfsRelativeBuildProjectPath = Manifest.TfsRelativeBuildProjectPath;
                //if (String.IsNullOrEmpty(tfsRelativeBuildProjectPath))
                //{
                //    throw new BuildConfigurationException("tfspath attribute has not been set in build manifest.");
                //}

                // Build the project dirspec without considering working folder mappings.
                //string projectDirspec = buildProject.RepositoryDirspec;
                int resultCode = this.GetTfsProjectWithRetry(buildProject.RepositoryDirspec, buildProject.WorkingFolderDirspec, RetryCount);

                //if (resultCode == 1 && config.SkipTfsWriteableFilesOnGet)
                //{
                //    if (config.DisplayTfsOutput == true)
                //    {
                //        DisplayOutput();
                //    }
                //}
                if (resultCode > 0)
                {
                    Output(GetOutput());
                    throw new BuildException(LogError("Get of project " + buildProject.WorkingFolderDirspec + " failed - aborting build process"));
                }
                else if (config.DisplayTfsOutput == true)
                {
                    Output(GetOutput());
                }

                if (CanRun == false)
                {
                    LogInfo("Build canceled by user.");
                    return;
                }
            }
            LogInfo("TFS Build Project Get phase complete");
        }

        private string GetTfsLabel()
        {
            string label = null;

            if (config.GetFromTfsLabel)
            {
                label = Manifest.BuildProjectLabel;
            }

            return label;
        }

        private bool UseForceOption()
        {
            return config.DeleteBuildProjectsBeforeGet || config.ApplyForceOption;
        }

        private int GetTfsProject(string repositoryDirspec, bool allowForceOption)
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();

            //"C:\Program Files\Microsoft Visual Studio 10.0\Common7\IDE\tf.exe" get "$/Root/Development/Dev" /force /recursive /noprompt /login:"username,password"
            sb.AppendFormat("get {0} /recursive /noprompt /login:{1},{2}", this.TranslateFilespecToTfsSemantics(repositoryDirspec), config.TfsUsername, config.TfsPassword);

            string label = GetTfsLabel();
            if (label != null)
            {
                sb.AppendFormat(" /version:L{0}", label);
            }

            if (allowForceOption && UseForceOption())
            {
                sb.Append(" /force");
            }

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.TfsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.TfsRootDirspec;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
            }
            catch
            {
                exitCode = externalProcess.ExitCode;
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        /// <summary>
        /// Translates the filespec to TFS semantics. In practice this means replacing any existing drive specification with $
        /// </summary>
        /// <param name="filespec">The file specification to translate.</param>
        /// <returns>The translated file specification suitable for use on the TFS command line.</returns>
        private string TranslateFilespecToTfsSemantics(string filespec)
        {
            string tfsFilespec = null;
            if (filespec.Contains(config.TfsWorkspaceWorkingFolder))
            {
                tfsFilespec = filespec.Replace(config.TfsWorkspaceWorkingFolder, @"$");
                tfsFilespec.Replace(@"\", "/");
                return tfsFilespec;
            }
            else if (filespec.Contains(config.TfsRootDirspec))
            {
                tfsFilespec = filespec.Replace(config.TfsRootDirspec, @"$");
                tfsFilespec.Replace(@"\", "/");
                return tfsFilespec;
            }
            tfsFilespec = filespec;
            return tfsFilespec;
        }


        #endregion
    }
}
