﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Threading;
using log4net;
using System.Diagnostics;
using System.IO;


namespace VixBuilderBusiness
{
    internal class CvsRepository : AbstractRepository, IBuildRepository
    {
        private BuildManifest manifest;

        public CvsRepository(BuildConfiguration config) : base(config, RepositoryEnum.ConcurrentVersionsSystem)
        {
            
        }

        #region IBuildRepository

        public DeploymentArtifact[] GetDeploymentArtifacts()
        {
            return this.Manifest.GetRepositoryDeploymentArtifacts();
        }

        /// <summary>
        /// Gets the build manifest.
        /// </summary>
        /// <value>The build manifest.</value>
        /// <remarks>This property is delay initialized because the build manifest should not be accessed until after the ImagingExchange project
        /// has been retrieved from the build repository.</remarks>
        public BuildManifest Manifest
        {
            get
            {
                if (this.manifest == null)
                {
                    this.manifest = GetBuildManifest(); // manifest may not exist until after ImagingExchange has been retrieved from the build repository
                }
                Debug.Assert(this.manifest != null, "No build manifest available");
                return this.manifest;
            }
        }

        public void DeleteBuildProjects()
        {
            BuildProject[] buildProjects = Manifest.GetBuildProjects();

            LogInfo("Beginning CVS project deletion phase...");

            //DeleteCvsProject("ImagingExchange");
            foreach (BuildProject cvsProject in buildProjects)
            {
                DeleteCvsProject(cvsProject.ProjectName);
            }
            LogInfo("CVS project deletion phase complete");
        }

        public void GetBuildProjects()
        {
            if (CanRun) this.GetCvsImagingExchangeProject();
            //if (this.config.DeleteBuildProjectsBeforeGet)
            //{
            //    if (this.CanRun)
            //    {
            //        this.DeleteBuildProjects();
            //    }
            //}
            if (CanRun) CheckoutCvsProjects();
        }

        public void LabelBuildProjects()
        {
            LogInfo("Beginning CVS workspace labeling phase...");
            int resultCode = LabelCvsWorkspace();
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("Labeling of CVS workspace cannot failed. Please see output for reason."));
            }
            DisplayOutput(); // always display this output
            LogInfo("CVS workspace has been labeled as: " + config.CvsLabelToApply);
            LogInfo("Please review the output to confirm that everything labeled as expected.");
        }

        public void DeleteLabelFromBuildProjects()
        {
            LogInfo("Beginning CVS workspace label deletion phase...");
            int resultCode = DeleteLabelFromCvsWorkspace(config);
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("Deleting Label from CVS workspace cannot failed. Please see output for reason."));
            }
            DisplayOutput(); // always display this output
            LogInfo("The following CVS workspace label has been deleted: " + config.CvsLabelToApply);
            LogInfo("Please review the output to confirm that worked as expected.");
        }

        public void BranchBuildProjects()
        {
            LabelBuildProjects();
            if (CanRun == false)
            {
                LogInfo("Build canceled by user.");
                return;
            }
            DeleteCvsProject("ImagingExchange"); // we never branch ImagingExchange
            if (CanRun == false)
            {
                LogInfo("Build canceled by user.");
                return;
            }
            LogInfo("Beginning CVS workspace branching phase...");
            int resultCode = BranchCvsWorkspaceFromLabel(config);
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("Branching of CVS workspace failed. Please see output for reason."));
            }
            DisplayOutput(); // always display this output
            LogInfo("CVS workspace has been Branched as: " + config.CvsBranchToApply);
            LogInfo("Please review the output to confirm that everything brancheded as expected.");
        }

        public override BuildManifest GetBuildManifest()
        {
            string buildManifestFilespec = Path.Combine(config.CvsProjectRoot, ("ImagingExchange\\" + config.BuildManifestFilename));
            if (!File.Exists(buildManifestFilespec))
            {
                throw new BuildException("Missing build manifest " + buildManifestFilespec);
            }
            return new BuildManifest(buildManifestFilespec, config);
        }

        //public string GetBuildProjectDirspec(BuildProject buildProject)
        //{
        //    return Path.Combine(this.config.CvsProjectRoot, buildProject.GetProjectRepositoryRelativeDirspec(RepositoryEnum.ConcurrentVersionsSystem));
        //}

        #endregion

        #region private methods
        private void GetCvsImagingExchangeProject()
        {
            LogInfo("Performing CVS Get of project ImagingExchange");
            int resultCode = this.GetCvsProject("ImagingExchange", null); // null means checkout from trunk
            ////if (resultCode == 2)
            ////{
            ////    CvsLogin(config);
            ////    resultCode = GetCvsProject("ImagingExchange", null);
            ////}
            if (resultCode > 0)
            {
                // retry once - CVS can be tempermental
                resultCode = this.GetCvsProject("ImagingExchange", null);
            }
            if (resultCode > 0)
            {
                DisplayOutput();
                throw new BuildException(LogError("Get of project ImagingExchange failed twice - aborting build process"));
            }
            else if (config.DisplayCvsOutput == true)
            {
                DisplayOutput();
            }
        }

        private void CheckoutCvsProjects()
        {
            BuildProject[] buildProjects = Manifest.GetBuildProjects();
            string branchOrLabel = GetCvsBranchOrLabel();

            LogInfo("Beginning CVS Get phase...");
            foreach (BuildProject buildProject in buildProjects)
            {
                string logInfoMessage = "\tPerforming CVS Get of project " + buildProject.ProjectName + (branchOrLabel == null ? "" : " - " + config.CvsCheckoutMode.ToString() + " (" + branchOrLabel + ")");
                LogInfo(logInfoMessage);
                int resultCode = this.GetCvsProject(buildProject.ProjectName, branchOrLabel);
                if (resultCode > 0)
                {
                    // retry once - CVS can be tempermental -- allow both result sets to accumulate
                    resultCode = this.GetCvsProject(buildProject.ProjectName, branchOrLabel);
                }
                if (resultCode > 0)
                {
                    Output(GetOutput());
                    throw new BuildException(LogError("Get of project " + buildProject.ProjectName + " failed twice - aborting build process"));
                }
                else if (config.DisplayCvsOutput == true)
                {
                    Output(GetOutput());
                }

                if (CanRun == false)
                {
                    LogInfo("Build canceled by user.");
                    return;
                }
            }
            LogInfo("CVS Get phase complete");
        }

        private int DeleteLabelFromCvsWorkspace(BuildConfiguration config)
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("-d :pserver:{0}:{1}@{2}:/cvs tag -d {3}", config.CvsUsername, config.CvsPassword, config.CvsServer, config.CvsLabelToApply);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.CvsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.CvsProjectRoot;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
                if (exitCode > 0)
                {
                    string stdout = sbOutput.ToString();
                    if (stdout.Contains("Empty password used"))
                    {
                        exitCode = 2; // use 2 to indicate cvs login required
                    }
                }
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        private int BranchCvsWorkspaceFromLabel(BuildConfiguration config)
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("-d :pserver:{0}:{1}@{2}:/cvs tag -b {3}", config.CvsUsername, config.CvsPassword, config.CvsServer, config.CvsBranchToApply);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.CvsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.CvsProjectRoot;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
                if (exitCode > 0)
                {
                    string stdout = sbOutput.ToString();
                    if (stdout.Contains("Empty password used"))
                    {
                        exitCode = 2; // use 2 to indicate cvs login required
                    }
                }
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        private void DeleteCvsProject(string projectName)
        {
            string dirspec = Path.Combine(config.CvsProjectRoot, projectName);
            if (Directory.Exists(dirspec))
            {
                LogInfo("\tDeleting  " + dirspec);
                Directory.Delete(dirspec, true); // recursive
            }
        }

        //private void CvsLogin(BuildConfiguration config)
        //{
        //    //string responseFilespec = GenerateCvsLoginResponseFile(config.CvsPassword);
        //    StringBuilder sb = new StringBuilder();
        //    sb.AppendFormat("-d :pserver:{0}:{1}@{2}:/cvs login", config.CvsUsername, config.CvsPassword, config.CvsServer);
        //    //sb.AppendFormat("-d :pserver:{0}@{1}:/cvs login <\"{2}\"", config.CvsUsername, config.CvsServer, responseFilespec);
        //    Process externalProcess = new System.Diagnostics.Process();
        //    externalProcess.StartInfo.FileName = config.CvsExecutable;
        //    externalProcess.StartInfo.Arguments = sb.ToString();
        //    //externalProcess.StartInfo.WorkingDirectory = config.CvsProjectRoot;
        //    externalProcess.StartInfo.UseShellExecute = false;
        //    externalProcess.StartInfo.CreateNoWindow = false;

        //    // cant get standard in redirection to work - cvs login exits immediately. This was will support prompting in the command window which is better than nothing.

        //    externalProcess.StartInfo.RedirectStandardInput = false;
        //    externalProcess.Start();

        //    try
        //    {
        //        do
        //        {
        //            Thread.Sleep(500);
        //            externalProcess.Refresh();
        //        } while (!externalProcess.HasExited);
        //    }
        //    finally
        //    {
        //        externalProcess.Close();
        //        externalProcess = null;
        //        //if (File.Exists(responseFilespec))
        //        //{
        //        //    File.Delete(responseFilespec);
        //        //}
        //    }
        //}

        private int GetCvsProject(string cvsProjectName, string branchOrLabel)
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("-d :pserver:{0}:{1}@{2}:/cvs checkout", config.CvsUsername, config.CvsPassword, config.CvsServer);

            if (branchOrLabel != null)
            {
                sb.AppendFormat(" -r \"{0}\"", branchOrLabel);
            }

            sb.AppendFormat(" {0}", cvsProjectName);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.CvsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.CvsProjectRoot;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
                if (exitCode > 0)
                {
                    string stdout = sbOutput.ToString();
                    if (stdout.Contains("Empty password used"))
                    {
                        exitCode = 2; // use 2 to indicate cvs loging required
                    }
                }
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        private int LabelCvsWorkspace()
        {
            ResetStandardOutput();
            int exitCode = 0; // external process executed ok - command shell convention
            StringBuilder sb = new StringBuilder();

            sb.AppendFormat("-d :pserver:{0}:{1}@{2}:/cvs tag -c ", config.CvsUsername, config.CvsPassword, config.CvsServer);

            if (config.MoveCvsLabelIfExists)
            {
                sb.Append("-F ");
            }

            sb.AppendFormat("{0}", config.CvsLabelToApply);

            Process externalProcess = new System.Diagnostics.Process();
            externalProcess.StartInfo.FileName = config.CvsExecutable;
            externalProcess.StartInfo.Arguments = sb.ToString();
            externalProcess.StartInfo.WorkingDirectory = config.CvsProjectRoot;
            externalProcess.StartInfo.UseShellExecute = false;
            externalProcess.StartInfo.CreateNoWindow = true;
            externalProcess.StartInfo.RedirectStandardOutput = true;
            externalProcess.OutputDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.StartInfo.RedirectStandardError = true;
            externalProcess.ErrorDataReceived += new DataReceivedEventHandler(OutputHandler);
            externalProcess.Start();
            externalProcess.BeginOutputReadLine();
            externalProcess.BeginErrorReadLine();

            try
            {
                do
                {
                    YieldToUserInterface();
                    Thread.Sleep(500);
                    externalProcess.Refresh();
                } while (!externalProcess.HasExited);

                //string stderr = externalProcess.StandardError.ReadToEnd();

                exitCode = externalProcess.ExitCode;
                if (exitCode > 0)
                {
                    string stdout = sbOutput.ToString();
                    if (stdout.Contains("Empty password used"))
                    {
                        exitCode = 2; // use 2 to indicate cvs loging required
                    }
                }
            }
            finally
            {
                externalProcess.Close();
                externalProcess = null;
            }

            return exitCode;
        }

        private string GetCvsBranchOrLabel()
        {
            string branchOrLabel = null;

            if (config.CvsCheckoutMode == CheckoutModeEnum.Branch)
            {
                branchOrLabel = Manifest.BuildProjectBranch;
            }
            else if (config.CvsCheckoutMode == CheckoutModeEnum.Label)
            {
                branchOrLabel = Manifest.BuildProjectLabel;
            }

            return branchOrLabel;
        }

        #endregion

    }
}
