using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.Diagnostics;
using System.Text.RegularExpressions;
using System.IO;

namespace VixBuilderBusiness
{
    public enum SubstitutionType { cvs, workspace, mavenrepo, payload, tomcat, vss, buildversion, jre, vixconfig, tfs, cc};
    public enum DeployProjectType { payload, installer, other };

    // types shared with VixInstaller
    public enum VixWebApplicationOption { AddUpdate, Remove };
    public enum VixDependencyType
    {
        AppExtensionJar, XmlConfigFile, OpenSourceDll, CommercialDll, OpenSourceJar,
        ModifiedOpenSourceJar, CommercialJar, AppJar, AppWar, JavaPropertyFile, ZipConfigFile, ResourceFile, TextFile, InHouseDll, Other, InstallerManifest
    };

    // see VixManifest.cs
    //public enum VixDependencyType
    //{
    //    AppExtensionJar, XmlConfigFile, OpenSourceDll, CommercialDll, OpenSourceJar,
    //    ModifiedOpenSourceJar, CommercialJar, AppJar, AppWar, JavaPropertyFile, ZipConfigFile, ResourceFile, TextFile, InHouseDll, Other
    //};

    public class BuildManifest
    {
        BuildConfiguration config = null;
        private XmlDocument manifest = null;

        public BuildManifest(string manifestFilespec, BuildConfiguration config)
        {
            this.manifest = new XmlDocument();
            this.manifest.Load(manifestFilespec);
            this.config = config;
        }

        #region properties
        public string FullyQualifiedPatchNumber
        {
            get
            {
                XmlNode xmlBuild = this.manifest.SelectSingleNode("Build/Patch");
                Debug.Assert(xmlBuild != null);
                string fullyQualifiedPatchNumber = xmlBuild.Attributes["number"].Value.Trim();
                return fullyQualifiedPatchNumber;
            }
        }

        public string MsiFilename
        {
            get
            {
                XmlNode xmlBuild = this.manifest.SelectSingleNode("Build/Patch");
                Debug.Assert(xmlBuild != null);
                string msiName = xmlBuild.Attributes["msiName"].Value.Trim();
                return msiName;
            }
        }

        public string TfsMsiVersionNumberFilespec
        {
            get
            {
                return config.TfsRootDirspec == null ? null : this.GetMsiVersionNumberFilespec(config.TfsRootDirspec, "tfsPath"); // sanity check - should not happen
            }
        }

        //public string VssMsiVersionNumberFilespec
        //{
        //    get
        //    {
        //        return config.VssRootDirspec == null ? null : this.GetMsiVersionNumberFilespec(config.VssRootDirspec, "vssPath"); // sanity check - should not happen
        //    }
        //}

        //public string ClearCaseMsiVersionNumberFilespec
        //{
        //    get
        //    {
        //        return config.ClearCaseRootDirspec == null ? null : this.GetMsiVersionNumberFilespec(config.ClearCaseRootDirspec, "ccPath"); // sanity check - should not happen
        //    }
        //}

        public int MajorPatchNumber
        {
            get
            {
                XmlNode xmlVistaPatch = this.manifest.SelectSingleNode("Build/Patch");
                Debug.Assert(xmlVistaPatch != null);
                string fullyQualifiedPatchNumber = xmlVistaPatch.Attributes["number"].Value.Trim();
                string[] splits = fullyQualifiedPatchNumber.Split('.'); //VistaReleaseNumber.PatchNumber.BuildNumber.IterationNumber
                Debug.Assert(splits.Length == 4);
                return Int32.Parse(splits[1]);
            }
        }

        public string BuildProjectBranch
        {
            get
            {
                XmlNode xmlBuildProjects = this.manifest.SelectSingleNode("Build/BuildProjects");
                Debug.Assert(xmlBuildProjects != null);
                string branch = xmlBuildProjects.Attributes["branch"].Value.Trim();
                return branch;
            }
        }

        public string BuildProjectLabel
        {
            get
            {
                XmlNode xmlBuildProjects = this.manifest.SelectSingleNode("Build/BuildProjects");
                Debug.Assert(xmlBuildProjects != null);
                string label = xmlBuildProjects.Attributes["label"].Value.Trim();
                return label;
            }
        }

        private string TfsRelativeBuildProjectPath
        {
            get
            {
                XmlNode xmlBuildProjects = this.manifest.SelectSingleNode("Build/BuildProjects");
                Debug.Assert(xmlBuildProjects != null);
                string path = xmlBuildProjects.Attributes["tfsPath"].Value.Trim();
                return path;
            }
        }

        private string ClearCaseRelativeBuildProjectPath
        {
            get
            {
                XmlNode xmlBuildProjects = this.manifest.SelectSingleNode("Build/BuildProjects");
                Debug.Assert(xmlBuildProjects != null);
                string path = xmlBuildProjects.Attributes["ccPath"].Value.Trim();
                return path;
            }
        }

        private string WorkspaceWorkingFolder
        {
            get
            {
                string workspaceDirspec;
                if (config.BuildRepository == RepositoryEnum.ConcurrentVersionsSystem)
                {
                    workspaceDirspec = config.CvsProjectRoot;
                }
                else if (config.BuildRepository == RepositoryEnum.TeamFoundationServer)
                {
                    if (String.IsNullOrEmpty(config.TfsWorkspaceWorkingFolder))
                    {
                        workspaceDirspec = Path.Combine(config.TfsRootDirspec, TfsRelativeBuildProjectPath);
                    }
                    else
                    {
                        workspaceDirspec = config.TfsWorkspaceWorkingFolder;
                    }
                }
                else
                {
                    throw new NotImplementedException("Manifest.WorkspaceWorkingFolder unsupported build repository " + config.BuildRepository.ToString());
                }

                return workspaceDirspec;
            }
        }


        #endregion
        
        #region public methods
        public BuildProject[] GetBuildProjects()
        {
            List<BuildProject> buildProjects = new List<BuildProject>();
            XmlNode buildProjectParent = this.manifest.SelectSingleNode("Build/BuildProjects");
            //string cvsProjectRelativePath = "";
            //string tfsProjectRelativePath = buildProjectParent.Attributes["tfsPath"].Value.Trim();
            //string ccProjectRelativePath = buildProjectParent.Attributes["ccPath"].Value.Trim();
            XmlNodeList xmlBuildProjects = this.manifest.SelectNodes("Build/BuildProjects/BuildProject");
            foreach (XmlNode xmlBuildProject in xmlBuildProjects)
            {
                string projectName = xmlBuildProject.InnerText.Trim();
                VixDependencyType dependencyType = (VixDependencyType)Enum.Parse(typeof(VixDependencyType), xmlBuildProject.Attributes["dependencyType"].Value.Trim());
                BuildProject buildProject = new BuildProject(projectName, dependencyType, this.GetProjectRepositoryDirspec(projectName), this.GetProjectWorkingFolderDirspec(projectName));
                buildProjects.Add(buildProject);
            }

            return buildProjects.ToArray();
        }

        public DeployProject[] GetDeployProjects()
        {
            List<DeployProject> vssProjects = new List<DeployProject>();
            XmlNodeList xmlVssProjects = this.manifest.SelectNodes("Build/DeployProjects/DeployProject");
            foreach (XmlNode xmlVssProject in xmlVssProjects)
            {
                string vssProjectString = xmlVssProject.InnerText.Trim();
                bool recurse = bool.Parse(xmlVssProject.Attributes["recurse"].Value.Trim());
                DeployProjectType deployProjectType = (DeployProjectType) Enum.Parse(typeof(DeployProjectType), xmlVssProject.Attributes["type"].Value.Trim());
                string vssProjectRelativePath = xmlVssProject.Attributes["vssPath"].Value.Trim();
                string tfsProjectRelativePath = xmlVssProject.Attributes["tfsPath"].Value.Trim();
                string ccProjectRelativePath = xmlVssProject.Attributes["ccPath"].Value.Trim();
                DeployProject deployProject = new DeployProject(vssProjectString, recurse, deployProjectType, vssProjectRelativePath, tfsProjectRelativePath, ccProjectRelativePath);
                vssProjects.Add(deployProject);
            }

            return vssProjects.ToArray();
        }

        public DeployProject GetPayloadDeployProject()
        {
            DeployProject deployProject = null;
            DeployProject[] deployProjects = this.GetDeployProjects();
            foreach (DeployProject project in deployProjects)
            {
                if (project.DeployProjectType == DeployProjectType.payload)
                {
                    deployProject = project;
                    break;
                }
            }
            return deployProject;
        }

        public DeployProject GetVixInstallerSolutionRootDeployProject()
        {
            DeployProject deployProject = null;
            DeployProject[] deployProjects = this.GetDeployProjects();
            foreach (DeployProject project in deployProjects)
            {
                if (project.DeployProjectType == DeployProjectType.installer)
                {
                    deployProject = project;
                    break;
                }
            }
            return deployProject;
        }


        public DeploymentArtifact[] GetDeploymentArtifacts()
        {
            XmlNodeList xmlDeploymentArtifacts = this.manifest.SelectNodes("Build/DeploymentArtifacts/DeploymentArtifact");
            return this.GetDeploymentArtifacts(xmlDeploymentArtifacts);
        }

        public DeploymentArtifact[] GetRepositoryDeploymentArtifacts()
        {
            XmlNodeList xmlDeploymentArtifacts = this.manifest.SelectNodes("Build/RepositoryDeploymentArtifacts/DeploymentArtifact");
            return this.GetDeploymentArtifacts(xmlDeploymentArtifacts);
        }

        private DeploymentArtifact[] GetDeploymentArtifacts(XmlNodeList xmlDeploymentArtifacts)
        {
            //<DeploymentArtifact source="%mavenrepo%" dest="%payload%\server\jars" devDest="%tomcat%\lib">Imaging\ImagingCommon\0.1\ImagingCommon-0.1.jar</DeploymentArtifact>
            List<DeploymentArtifact> deploymentArtifacts = new List<DeploymentArtifact>();
            //Debug.Assert(xmlDeploymentArtifacts != null);
            foreach (XmlNode xmlDeploymentArtifact in xmlDeploymentArtifacts)
            {
                string sourceFilespec = GetDeploymentArtifactSourceFilespec(xmlDeploymentArtifact);
                string destinationFilespec = GetDeploymentArtifactDestinationFilespec(xmlDeploymentArtifact, sourceFilespec);
                string devDestinationFilespec = GetDeploymentArtifactDevDestinationFilespec(xmlDeploymentArtifact, sourceFilespec);
                RepositoryEnum repositoryEnumType = GetDeploymentArtifactRepositoryType(xmlDeploymentArtifact);

                VixDependencyType dependencyType = (VixDependencyType)Enum.Parse(typeof(VixDependencyType), xmlDeploymentArtifact.Attributes["dependencyType"].Value.Trim());

                DeploymentArtifact deployProject = new DeploymentArtifact(sourceFilespec, destinationFilespec, devDestinationFilespec, dependencyType, repositoryEnumType);
                deploymentArtifacts.Add(deployProject);
            }

            return deploymentArtifacts.ToArray();
        }

        public VixWebApplication[] GetVixWebApplications()
        {
            List<VixWebApplication> vixWebApplications = new List<VixWebApplication>();
            XmlNodeList xmlWebApplications = this.manifest.SelectNodes("Build/WebApplications/WebApplication");
            Debug.Assert(xmlWebApplications != null);
            foreach (XmlNode xmlVixWebApplication in xmlWebApplications)
            {
                string webApplicationOptionTypeAsString = xmlVixWebApplication.Attributes["option"].Value;
                VixWebApplicationOption option = (VixWebApplicationOption)Enum.Parse(typeof(VixWebApplicationOption), webApplicationOptionTypeAsString);

                string useContextFolderAsString = xmlVixWebApplication.Attributes["useContextFile"].Value.Trim();

                VixWebApplication vixWebApplication = new VixWebApplication(option,
                    xmlVixWebApplication.Attributes["path"].Value.Trim(),
                    xmlVixWebApplication.InnerText.Trim(), (useContextFolderAsString == "true" ? true : false));
                vixWebApplications.Add(vixWebApplication);
            }

            return vixWebApplications.ToArray();
        }

        #endregion

        #region private methods
        private string GetMsiVersionNumberFilespec(string repositoryRootDirspec, string attributeName)
        {
            string filespec = null;
            XmlNode xmlBuild = this.manifest.SelectSingleNode("Build/MsiVersion");
            if (xmlBuild != null && xmlBuild.Attributes != null)
            {
                XmlAttribute pathAttribute = xmlBuild.Attributes[attributeName];
                if (pathAttribute != null)
                {
                    filespec = Path.Combine(repositoryRootDirspec, pathAttribute.Value.Trim());
                }
            }
            return filespec;
        }

        private string PerformStringSubstitution(string source)
        {
            Regex regex = new Regex(@"%\w*%");

            if (regex.IsMatch(source))
            {
                MatchCollection matches = regex.Matches(source);
                // reverse iteration so offsets don't get hosed
                for (int i = 0; i < matches.Count; i++)
                {
                    Match match = matches[i];
                    string substitute = GetSubstituteValue(match.Value);
                    source = source.Replace(match.Value, substitute);
                }

            }
            return source;
        }

        private string GetSubstituteValue(string str)
        {
            string substitute = null;
            string substituteTypeAsString = str.Replace("%", "");
            try
            {
                SubstitutionType substitutionType = (SubstitutionType)Enum.Parse(typeof(SubstitutionType), substituteTypeAsString);
                switch (substitutionType)
                {
                    case SubstitutionType.buildversion:
                        substitute = this.FullyQualifiedPatchNumber;
                        break;
                    case SubstitutionType.cvs:
                        substitute = config.CvsProjectRoot;
                        break;
                    case SubstitutionType.tfs:
                        substitute = config.TfsRootDirspec;
                        break;
                    case SubstitutionType.mavenrepo:
                        substitute = config.Maven2RepoDirspec;
                        break;
                    case SubstitutionType.payload:
                        substitute = config.VixPayloadRootDirspec;
                        break;
                    case SubstitutionType.tomcat:
                        substitute = config.ApacheTomcatDirspec;
                        //if (String.IsNullOrEmpty(substitute))
                        //{
                        //    throw new BuildException("Tomcat Root Dirspec not specified in configuration");
                        //}
                        break;
                    case SubstitutionType.vss:
                        substitute = config.VssRootDirspec;
                        break;
                    case SubstitutionType.jre:
                        substitute = Environment.GetEnvironmentVariable("JRE_HOME", EnvironmentVariableTarget.Machine);
                        if (String.IsNullOrEmpty(substitute))
                        {
                            throw new BuildException("JRE_HOME environment variable not defined");
                        }
                        break;
                    case SubstitutionType.vixconfig:
                        if (String.IsNullOrEmpty(config.VixConfigDir) == false)
                        {
                            substitute = config.VixConfigDir;
                        }
                        else
                        {
                            substitute = Environment.GetEnvironmentVariable("vixconfig", EnvironmentVariableTarget.Machine);
                            if (String.IsNullOrEmpty(substitute))
                            {
                                substitute = @"C:\VixConfig";
                            }
                        }
                        break;
                     case SubstitutionType.workspace:
                        substitute = WorkspaceWorkingFolder;
                        break; 
                     default:
                        throw new BuildException("GetSubstituteValue() - unhandled SubstitutionType " + substitutionType.ToString());
                }
                return substitute;
            }
            catch (ArgumentException)
            {
                throw new BuildException(substituteTypeAsString + " is not a valid SubstitutionType");
            }
        }


        private string GetDeploymentArtifactSourceFilespec(XmlNode xmlDeploymentArtifact)
        {
            string sourceFilespec;
            string sourceRelativeFilespec;
            string sourceRelativeDirspec;
            string sourceDirspec;
            StringBuilder sb = new StringBuilder();

            if (xmlDeploymentArtifact.ParentNode.Name == "RepositoryDeploymentArtifacts")
            {
                sb.AppendFormat("Repository[@repositoryType='{0}']", config.BuildRepository.ToString());
                XmlNode xmlRepository = xmlDeploymentArtifact.SelectSingleNode(sb.ToString());
                sb.Length = 0; // reset for for error message construction
                if (xmlRepository == null)
                {
                    sb.AppendLine("Error: DeploymentArtifact element does not have a child Repository element configured for repository type " + config.BuildRepository.ToString());
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                sourceRelativeFilespec = xmlRepository.InnerText.Trim();
                if (string.IsNullOrEmpty(sourceRelativeFilespec))
                {
                    sb.AppendLine("Error: Repository element has an empty source relative filespec specified in inner text of element.");
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                if (xmlRepository.Attributes["source"] == null)
                {
                    sb.AppendLine("Error: Repository element has an missing source attribute.");
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
                sourceRelativeDirspec = xmlRepository.Attributes["source"].Value.Trim();

                if (string.IsNullOrEmpty(sourceRelativeDirspec))
                {
                    sb.AppendLine("Error: Repository element has an empty source attribute.");
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                try
                {
                    sourceDirspec = PerformStringSubstitution(sourceRelativeDirspec);
                }
                catch (BuildException ex)
                {
                    sb.AppendLine("Error: Exception while performing string substitution on Repository element's source attribute: " + sourceRelativeDirspec);
                    sb.AppendLine("Exception is: " + ex.Message);
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
            }
            else
            {
                sourceRelativeFilespec = xmlDeploymentArtifact.InnerText.Trim();
                if (string.IsNullOrEmpty(sourceRelativeFilespec))
                {
                    sb.AppendLine("Error: DeploymentArtifact element has an empty source relative filespec specified in inner text of element.");
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                if (xmlDeploymentArtifact.Attributes["source"] == null)
                {
                    sb.AppendLine("Error: DeploymentArtifact element has an missing source attribute.");
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
                sourceRelativeDirspec = xmlDeploymentArtifact.Attributes["source"].Value.Trim();

                if (string.IsNullOrEmpty(sourceRelativeDirspec))
                {
                    sb.AppendLine("Error: DeploymentArtifact element has an empty source attribute.");
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                try
                {
                    sourceDirspec = PerformStringSubstitution(sourceRelativeDirspec);
                }
                catch (BuildException ex)
                {
                    sb.AppendLine("Error: Exception while performing string substitution on DeploymentArtifact element's source attribute: " + sourceRelativeDirspec);
                    sb.AppendLine("Exception is: " + ex.Message);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
            }
            sourceFilespec = Path.Combine(sourceDirspec, sourceRelativeFilespec);

            return sourceFilespec;
        }

        private string GetDeploymentArtifactDestinationFilespec(XmlNode xmlDeploymentArtifact, string sourceFilespec)
        {
            string destinationFilespec;
            string destAttribValue;
            string destination;
            string sourceFilename = Path.GetFileName(sourceFilespec);
            StringBuilder sb = new StringBuilder();

            if (xmlDeploymentArtifact.ParentNode.Name == "RepositoryDeploymentArtifacts")
            {
                sb.AppendFormat("Repository[@repositoryType='{0}']", config.DeployRepository.ToString());
                XmlNode xmlRepository = xmlDeploymentArtifact.SelectSingleNode(sb.ToString());
                sb.Length = 0; // reset for for error message construction
                if (xmlRepository == null)
                {
                    sb.AppendLine("Error: DeploymentArtifact element does not have a child Repository element configured for repository type " + config.DeployRepository.ToString());
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                if (xmlRepository.Attributes["dest"] == null)
                {
                    sb.AppendLine("Error: Repository element has an missing dest attribute.");
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
                destAttribValue = xmlRepository.Attributes["dest"].Value.Trim();

                if (string.IsNullOrEmpty(destAttribValue))
                {
                    sb.AppendLine("Error: Repository element has an empty dest attribute.");
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                try
                {
                    destination = PerformStringSubstitution(destAttribValue);
                }
                catch (BuildException ex)
                {
                    sb.AppendLine("Error: Exception while performing string substitution on Repository element's dest attribute: " + destAttribValue);
                    sb.AppendLine("Exception is: " + ex.Message);
                    sb.AppendLine("Repository element is: " + xmlRepository.OuterXml);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
                
            }
            else
            {
                if (xmlDeploymentArtifact.Attributes["dest"] == null)
                {
                    sb.AppendLine("Error: DeploymentArtifact element has an missing dest attribute.");
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }

                destAttribValue = xmlDeploymentArtifact.Attributes["dest"].Value.Trim();

                if (string.IsNullOrEmpty(destAttribValue))
                {
                    destination = ""; // a missing dest attribute is allowed - will signal skip artifact for a payload deploy
                }
                else
                {
                    try
                    {
                        destination = PerformStringSubstitution(destAttribValue);
                    }
                    catch (BuildException ex)
                    {
                        sb.AppendLine("Error: Exception while performing string substitution on DeploymentArtifact element's dest attribute: " + destAttribValue);
                        sb.AppendLine("Exception is: " + ex.Message);
                        sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                        throw new BuildException(sb.ToString());
                    }
                }
            }

            if (String.IsNullOrEmpty(destination)) // an empty devDest attribute is allowed - will signal skip artifact for a development deploy
            {
                destinationFilespec = "";
            }
            else if (String.IsNullOrEmpty(Path.GetExtension(destination))) // no extension means its a directory - fragile but unavoidable
            {
                destinationFilespec = Path.Combine(destination, sourceFilename);
            }
            else // its a file
            {
                destinationFilespec = destination;
            }

            return destinationFilespec;
        }

        private string GetDeploymentArtifactDevDestinationFilespec(XmlNode xmlDeploymentArtifact, string sourceFilespec)
        {
            string devDestinationFilespec;
            string sourceFilename = Path.GetFileName(sourceFilespec);
            string destination;
            StringBuilder sb = new StringBuilder();

            if (xmlDeploymentArtifact.Attributes["devDest"] == null)
            {
                destination = ""; // a missing devDest attribute is allowed
            }
            else
            {
                string devDestAttrib = xmlDeploymentArtifact.Attributes["devDest"].Value.Trim();
                try
                {
                    destination = PerformStringSubstitution(devDestAttrib);
                }
                catch (BuildException ex)
                {
                    sb.AppendLine("Error: Exception while performing string substitution on DeploymentArtifact element's devDest attribute: " + devDestAttrib);
                    sb.AppendLine("Exception is: " + ex.Message);
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
            }

            if (String.IsNullOrEmpty(destination)) // an empty devDest attribute is allowed - will signal skip artifact for a development deploy
            {
                devDestinationFilespec   = "";
            }
            else if (String.IsNullOrEmpty(Path.GetExtension(destination))) // no extension means its a directory - fragile but unavoidable
            {
                devDestinationFilespec = Path.Combine(destination, sourceFilename);
            }
            else // its a file
            {
                devDestinationFilespec = destination;
            }

            return devDestinationFilespec;
        }

        private RepositoryEnum GetDeploymentArtifactRepositoryType(XmlNode xmlDeploymentArtifact)
        {
            RepositoryEnum sourceRepositoryType;
            StringBuilder sb = new StringBuilder();

            if (xmlDeploymentArtifact.ParentNode.Name == "RepositoryDeploymentArtifacts")
            {
                sourceRepositoryType = config.BuildRepository;
                sb.AppendFormat("Repository[@repositoryType='{0}']", sourceRepositoryType.ToString());
                XmlNode xmlRepository = xmlDeploymentArtifact.SelectSingleNode(sb.ToString());
                sb.Length = 0; // reset for for error message construction
                if (xmlRepository == null)
                {
                    sb.AppendLine("Error: DeploymentArtifact element does not have a child Repository element configured for repository type " + config.BuildRepository.ToString());
                    sb.AppendLine("DeploymentArtifact element is: " + xmlDeploymentArtifact.OuterXml);
                    throw new BuildException(sb.ToString());
                }
            }
            else
            {
                sourceRepositoryType = RepositoryEnum.none;
            }
            return sourceRepositoryType;
        }


        private string GetProjectRepositoryDirspec(string projectName)
        {
            string dirspec;
            switch (config.BuildRepository)
            {
                case RepositoryEnum.ConcurrentVersionsSystem:
                    dirspec = config.CvsProjectRoot;
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    dirspec = Path.Combine(config.TfsRootDirspec, TfsRelativeBuildProjectPath);
                    break;
                case RepositoryEnum.ClearCase:
                    dirspec = Path.Combine(config.ClearCaseRootDirspec, ClearCaseRelativeBuildProjectPath);
                    break;
                default:
                    throw new BuildConfigurationException("Unknown Repository Type: " + config.BuildRepository.ToString());
            }

            return Path.Combine(dirspec, projectName);
        }

        private string GetProjectWorkingFolderDirspec(string projectName)
        {
            return Path.Combine(WorkspaceWorkingFolder, projectName);
        }

        #endregion
    }

    #region value objects
    /// <summary>
    /// 
    /// </summary>
    public class BuildProject
    {
        public string ProjectName { get; set; }
        public string RepositoryDirspec { get; set; }
        public string WorkingFolderDirspec { get; set; }
        public VixDependencyType DependencyType { get; set; }

        public BuildProject(string projectName, VixDependencyType dependencyType, string repositoryDirspec, string workingFolderDirspec)
        {
            this.ProjectName = projectName;
            this.DependencyType = dependencyType;
            this.RepositoryDirspec = repositoryDirspec;
            this.WorkingFolderDirspec = workingFolderDirspec;
        }
    }

    public class DeployProject
    {
        public string ProjectName { get; set; }
        public bool Recurse { get; set; }
        //public string Destination { get; set; }
        public DeployProjectType DeployProjectType { get; set; }
        public string VssRelativeProjectPath { get; set; }
        public string TfsRelativeProjectPath { get; set; }
        public string ClearCaseRelativeProjectPath { get; set; }
        public DeployProject(string projectName, bool recurse, DeployProjectType deployProjectType, string vssRelativeBuildProjectPath, string tfsRelativeBuildProjectPath, string ccRelativeBuildProjectPath)
        {
            this.ProjectName = projectName;
            this.Recurse = recurse;
            //this.Destination = destination;
            this.DeployProjectType = deployProjectType;
            this.VssRelativeProjectPath = vssRelativeBuildProjectPath;
            this.TfsRelativeProjectPath = tfsRelativeBuildProjectPath;
            this.ClearCaseRelativeProjectPath = ccRelativeBuildProjectPath;
        }

        public string GetProjectRepositoryRelativeDirspec(RepositoryEnum repositoryType)
        {
            string relativePath;
            switch (repositoryType)
            {
                case RepositoryEnum.VisualSourceSafe:
                    relativePath = this.VssRelativeProjectPath;
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    relativePath = this.TfsRelativeProjectPath;
                    break;
                case RepositoryEnum.ClearCase:
                    relativePath = this.ClearCaseRelativeProjectPath;
                    break;
                default:
                    throw new BuildConfigurationException("Unknown Repository Type: " + repositoryType.ToString());
            }

            string repositoryRelativeProjectDirspec = Path.Combine(relativePath, this.ProjectName);
            return repositoryRelativeProjectDirspec;
        }
    }

    public class DeploymentArtifact
    {
        public string SourceFilespec { get; set; }
        public string DestinationFilespec { get; set; }
        public string DevDestinationFilespec { get; set; }
        public VixDependencyType DependencyType { get; set; }
        public RepositoryEnum repositoryType { get; set; }
        public DeploymentArtifact(string sourceFilespec, string destinationFilespec, string devDestinationFilespec, 
                    VixDependencyType dependencyType, RepositoryEnum repositoryType)
        {
            this.SourceFilespec = sourceFilespec;
            this.DestinationFilespec = destinationFilespec;
            this.DevDestinationFilespec = devDestinationFilespec;
            this.DependencyType = dependencyType;
            this.repositoryType = repositoryType;
        }
    }

    /*
    public class RepositoryDeploymentArtifact
    {
        private Dictionary<RepositoryEnum, DeploymentArtifact> deploymentArtifacts = new Dictionary<RepositoryEnum, DeploymentArtifact>();
        public string SourceFilespec(RepositoryEnum repository) { return deploymentArtifacts[repository].SourceFilespec;}
        public string DestinationFilespec(RepositoryEnum repository) { return deploymentArtifacts[repository].DestinationFilespec; }
        public string DevDestinationFilespec(RepositoryEnum repository) { return deploymentArtifacts[repository].DevDestinationFilespec; }
        public VixDependencyType DependencyType(RepositoryEnum repository) { return deploymentArtifacts[repository].DependencyType; }
        public RepositoryDeploymentArtifact(Dictionary<RepositoryEnum, DeploymentArtifact> deploymentArtifacts)
        {
            this.deploymentArtifacts = deploymentArtifacts;
        }
    }
     */

    /// <summary>
    /// Value object for Web Application
    /// </summary>
    public class VixWebApplication
    {
        public bool UseContextFolder { get; set; }
        public string War { get; set;  }
        public string Path { get; set; }
        public VixWebApplicationOption WebApplicationOption { get; set; }

        public string DocBase
        {
            get { return this.Path.StartsWith("/") ? this.Path.Substring(1) : this.Path; }
        }

        public VixWebApplication(VixWebApplicationOption webApplicationOption, string path, string war, bool useContextFolder)
        {
            this.WebApplicationOption = webApplicationOption;
            this.War = war;
            this.Path = path;
            this.UseContextFolder = useContextFolder;
        }

        public bool IsAxis2WebApplication()
        {
            bool isAxis2 = false;
            string ext = System.IO.Path.GetExtension(this.War).ToLower();
            if (ext == ".jar" || ext == ".aar")
            {
                isAxis2 = true;
            }

            return isAxis2;
        }
    }

    #endregion
}
