using System;
using System.Collections.Generic;
using System.Text;
using System.Xml;
using System.IO;
using System.Xml.Serialization;
using System.Diagnostics;
using System.Security.Cryptography;
using System.Security.Cryptography.Xml;

namespace VixBuilderBusiness
{
    public class BuildConfiguration
    {
        #region CVS
        public string CvsExecutable  { get; set; }
        public string CvsProjectRoot { get; set; }
        public string CvsServer  { get; set; }
        public string CvsUsername { get; set; }
        public string CvsPassword  { get; set; }
        public bool DisplayCvsOutput  { get; set; }

        CheckoutModeEnum cvsCheckoutMode = CheckoutModeEnum.Trunk;
        public CheckoutModeEnum CvsCheckoutMode
        {
            get { return cvsCheckoutMode; }
            set { cvsCheckoutMode = value; }
        }

        public bool ValidateCvsInfo()
        {
            bool validated = true;

            if (CvsExecutable == null || CvsExecutable.Trim() == "") validated = false;
            if (CvsProjectRoot == null || CvsProjectRoot.Trim() == "") validated = false;
            if (CvsServer == null || CvsServer.Trim() == "") validated = false;
            if (CvsUsername == null || CvsUsername.Trim() == "") validated = false;
            if (CvsPassword == null || CvsPassword.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region VSS

        public string VssExecutable { get; set; }
        public string VssSharename { get; set; }
        public string VssUsername { get; set; }
        public string VssPassword { get; set; }
        public string VssRootDirspec { get; set; }
        public bool DisplayVssOutput { get; set; }
        public bool SkipWriteableFilesOnGet { get; set; }

        public bool ValidateVssInfo()
        {
            bool validated = true;

            if (VssExecutable == null || VssExecutable.Trim() == "") validated = false;
            if (VssSharename == null || VssSharename.Trim() == "") validated = false;
            if (VssUsername == null || VssUsername.Trim() == "") validated = false;
            if (VssPassword == null || VssPassword.Trim() == "") validated = false;
            if (VssRootDirspec == null || VssRootDirspec.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region TFS

        public string TfsExecutable { get; set; }
        public string TfsServername { get; set; }
        public string TfsWorkspace { get; set; }
        public string TfsUsername { get; set; }
        public string TfsPassword { get; set; }
        public string TfsRootDirspec { get; set; }
        public bool DisplayTfsOutput { get; set; }
        //public bool SkipTfsWriteableFilesOnGet { get; set; }
        public bool GetFromTfsLabel { get; set; }
        public bool ApplyForceOption { get; set; }
        public string TfsWorkspaceWorkingFolder { get; set; }

        public bool ValidateTfsInfo()
        {
            bool validated = true;

            if (TfsExecutable == null || TfsExecutable.Trim() == "") validated = false;
            if (TfsServername == null || TfsServername.Trim() == "") validated = false;
            if (TfsWorkspace == null || TfsWorkspace.Trim() == "") validated = false;
            if (TfsUsername == null || TfsUsername.Trim() == "") validated = false;
            if (TfsPassword == null || TfsPassword.Trim() == "") validated = false;
            if (TfsRootDirspec == null || TfsRootDirspec.Trim() == "") validated = false;
            if (TfsWorkspaceWorkingFolder == null || TfsWorkspaceWorkingFolder.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region ClearCase

        public string ClearCaseExecutable { get; set; }
        public string ClearCaseServername { get; set; }
        public string ClearCaseWorkspace { get; set; }
        public string ClearCaseUsername { get; set; }
        public string ClearCasePassword { get; set; }
        public string ClearCaseRootDirspec { get; set; }
        public bool DisplayClearCaseOutput { get; set; }
        public bool SkipClearCaseWriteableFilesOnGet { get; set; }
        public bool GetFromClearCaseLabel { get; set; }

        public bool ValidateClearCaseInfo()
        {
            bool validated = true;

            if (ClearCaseExecutable == null || ClearCaseExecutable.Trim() == "") validated = false;
            if (ClearCaseServername == null || ClearCaseServername.Trim() == "") validated = false;
            if (ClearCaseWorkspace == null || ClearCaseWorkspace.Trim() == "") validated = false;
            if (ClearCaseUsername == null || ClearCaseUsername.Trim() == "") validated = false;
            if (ClearCasePassword == null || ClearCasePassword.Trim() == "") validated = false;
            if (ClearCaseRootDirspec == null || ClearCaseRootDirspec.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region Build
        public string Maven2Executable  { get; set; }
        public string BuildManifestFilename { get; set; }
        public string Maven2RepoDirspec { get; set; }
        public bool DisplayMavenOutput { get; set; }
        public bool DisableMavenUnitTests { get; set; }
        public bool DeleteBuildProjectsBeforeGet { get; set; }
        public string OverrideJarWarVersioning { get; set; }

        private RepositoryEnum buildRepository = RepositoryEnum.ConcurrentVersionsSystem;
        public RepositoryEnum BuildRepository
        {
            get { return buildRepository; }
            set { buildRepository = value; }
        }

        public bool ValidateBuildInfo()
        {
            bool validated = true;

            if (Maven2Executable == null || Maven2Executable.Trim() == "") validated = false;
            if (BuildManifestFilename == null || BuildManifestFilename.Trim() == "") validated = false;
            if (Maven2RepoDirspec == null || Maven2RepoDirspec.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region Deployment
        /// <summary>
        /// Gets or sets the deployment manifest filespec
        /// </summary>
        public string DeploymentManifestFilename { get; set; }

        public string VixPayloadRootDirspec { get; set; } // the directory where the payload zip file will be assembled
        public string VisualStudioExeFilespec { get; set; } // the Visual Studio 2010 dev enc filespec
        public string VixInstallerRootDirspec { get; set; } // where the installer MSI will be copied to
        public bool DeployPayloadForInstallerDebug { get; set; } // checkbox - will copy the payload zip to debug\bin directory for use with the dev env
        public bool DeployPayloadForNightlyBuild { get; set; } // deploy
        public bool DeletePayloadRootContents { get; set; } // performs a recursive delete on the contents of VixPayloadRootDirspec

        private InstallationTypeEnum installationType = InstallationTypeEnum.VIX;
        public InstallationTypeEnum InstallationType
        {
            get { return installationType; }
            set { installationType = value; }
        }

        private RepositoryEnum deployRepository = RepositoryEnum.VisualSourceSafe;
        public RepositoryEnum DeployRepository
        {
            get { return deployRepository; }
            set { deployRepository = value; }
        }

        public bool ValidateDeploymentInfo()
        {
            bool validated = true;

            if (DeploymentManifestFilename == null || DeploymentManifestFilename.Trim() == "") validated = false;
            if (VixPayloadRootDirspec == null || VixPayloadRootDirspec.Trim() == "") validated = false;
            if (VisualStudioExeFilespec == null || VisualStudioExeFilespec.Trim() == "") validated = false;
            if (VixInstallerRootDirspec == null || VixInstallerRootDirspec.Trim() == "") validated = false;
            return validated;
        }

        #endregion

        #region Development

        public string ApacheTomcatDirspec { get; set; }
        public bool DeployWebApplications { get; set; }
        public string VixConfigDir { get; set; }
        public bool GenerateConfigFiles { get; set; }
        public bool GenerateTomcatConfigFiles { get; set; }

        public bool ValidateDevelopmentInfo()
        {
            bool validated = true;

            if (ApacheTomcatDirspec == null || ApacheTomcatDirspec.Trim() == "") validated = false;
            if (VixConfigDir == null || VixConfigDir.Trim() == "") validated = false;

            return validated;
        }

        #endregion

        #region Public methods

        public void Save(string filespec)
        {
            XmlSerializer serializer = null;
            XmlDocument doc = new XmlDocument();
            doc.PreserveWhitespace = true;
            TripleDESDocumentEncryption cryptotron = null;
            MemoryStream stream = null;

            try
            {
                // Create a serializer for the Sailboat type
                serializer = new XmlSerializer(this.GetType());
                // Create a new writable FileStream, using the path passed
                // as a parameter.
                //stream = new FileStream(filepsec, FileMode.Create, FileAccess.Write);
                stream = new MemoryStream();
                serializer.Serialize(stream, this);
                stream.Position = 0;
                doc.Load(stream);
                cryptotron = new TripleDESDocumentEncryption(doc);
                cryptotron.Encrypt("CvsPassword");
                cryptotron.Encrypt("VssPassword");
                cryptotron.Encrypt("TfsPassword");
                cryptotron.Encrypt("ClearCasePassword");
                doc.Save(filespec);
            }
            finally
            {
                if (stream != null)
                {
                    stream.Close();
                }
            }
        }

        public static BuildConfiguration Open(string filespec)
        {
            XmlSerializer serializer = null;
            BuildConfiguration config = null;

            if (File.Exists(filespec)) // no encryption
            {
                XmlDocument doc = new XmlDocument();
                doc.PreserveWhitespace = true;
                TripleDESDocumentEncryption cryptotron = null;
                MemoryStream stream = null;

                try
                {
                    config = new BuildConfiguration();
                    doc.Load(filespec);
                    cryptotron = new TripleDESDocumentEncryption(doc);
                    cryptotron.Decrypt();
                    stream = new MemoryStream();
                    //doc.Save(Path.Combine(vixConfigDir, "foo.xml"));
                    doc.Save(stream);
                    stream.Position = 0;
                    serializer = new XmlSerializer(config.GetType());
                    config = (BuildConfiguration)serializer.Deserialize(stream);
                }
                finally
                {
                    if (stream != null)
                    {
                        stream.Close();
                    }
                }
            }
            else
            {
                throw new BuildConfigurationException("Build configuration file does not exist: " + filespec);
            }

            return config;
        }

        /// <summary>
        /// Throws a BuildConfigurationException if the configuration information does not contain the required information
        /// to perform the action.
        /// </summary>
        /// <param name="config"></param>
        /// <param name="action"></param>
        public void ValidateBuildConfigurationForBuildAction(BuildActionEnum action)
        {
            switch (action)
            {
                case BuildActionEnum.makeprod:
                    ValidateBuildRepository(action);
                    ValidateDeployRepository(action);
                    if (!ValidateBuildInfo()) throw new BuildConfigurationException("Missing required Build information from the build configuration. Cannot perform action " + action.ToString());
                    if (!ValidateDeploymentInfo()) throw new BuildConfigurationException("Missing required Deployment information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case BuildActionEnum.makedev:
                    ValidateBuildRepository(action);
                    if (!ValidateBuildInfo()) throw new BuildConfigurationException("Missing required Build information from the build configuration. Cannot perform action " + action.ToString());
                    if (!ValidateDevelopmentInfo()) throw new BuildConfigurationException("Missing required Development information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case BuildActionEnum.getBuildProjects:
                    ValidateBuildRepository(action);
                    break;
                case BuildActionEnum.build:
                    ValidateBuildRepository(action);
                    if (!ValidateBuildInfo()) throw new BuildConfigurationException("Missing required Build information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case BuildActionEnum.getDeployProjects:
                    ValidateDeployRepository(action);
                    break;
                case BuildActionEnum.assembleAndStageMsiPayload:
                    if (!ValidateDeploymentInfo()) throw new BuildConfigurationException("Missing required Deployment information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case BuildActionEnum.buildMsi:
                    if (!ValidateDeploymentInfo()) throw new BuildConfigurationException("Missing required Deployment information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case BuildActionEnum.deploytomcat:
                    if (!ValidateDevelopmentInfo()) throw new BuildConfigurationException("Missing required Development information from the build configuration. Cannot perform action " + action.ToString());
                    break;
            }
        }

        private void ValidateBuildRepository(BuildActionEnum action)
        {
            switch (BuildRepository)
            {
                case RepositoryEnum.ConcurrentVersionsSystem:
                    if (!ValidateCvsInfo()) throw new BuildConfigurationException("Missing required CVS information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    if (!ValidateTfsInfo()) throw new BuildConfigurationException("Missing required TFS information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                default:
                    throw new BuildException("Illegal Build Repository specified: " + BuildRepository.ToString());
            }
        }

        private void ValidateDeployRepository(BuildActionEnum action)
        {
            switch (DeployRepository)
            {
                case RepositoryEnum.TeamFoundationServer:
                    if (!ValidateTfsInfo()) throw new BuildConfigurationException("Missing required TFS information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                case RepositoryEnum.VisualSourceSafe:
                    if (!ValidateVssInfo()) throw new BuildConfigurationException("Missing required VSS information from the build configuration. Cannot perform action " + action.ToString());
                    break;
                default:
                    throw new BuildException("Illegal Deploy Repository specified: " + DeployRepository.ToString());
            }
        }

        #endregion

        #region Transient Information

        [XmlIgnore]
        public bool MoveCvsLabelIfExists { get; set; }
        [XmlIgnore]
        public string CvsLabelToApply { get; set; }
        [XmlIgnore]
        public string CvsBranchToApply { get; set; }
        [XmlIgnore]
        public string MsiVersion { get; set; }

        #endregion
    }

    class TripleDESDocumentEncryption
    {
        private XmlDocument xmlDoc;
        private TripleDES algorithm;

        public TripleDESDocumentEncryption(XmlDocument doc)
        {
            byte[] key = { 58, 250, 251, 44, 187, 30, 176, 28, 199, 140, 220, 19, 159, 244, 226, 217, 38, 217, 231, 178, 117, 188, 225, 43 };
            byte[] iv = { 34, 15, 64, 79, 93, 70, 14, 161 };
            if (doc != null)
            {
                xmlDoc = doc;
            }
            else
            {
                throw new ArgumentNullException("Doc");
            }

            algorithm = new TripleDESCryptoServiceProvider();
            algorithm.Key = key;
            algorithm.IV = iv;
        }

        public void Encrypt(string elementName)
        {
            // Find the element by name and create a new
            // XmlElement object.
            XmlElement inputElement = xmlDoc.GetElementsByTagName(elementName)[0] as XmlElement;

            // If the element was not found, we're done.
            if (inputElement == null)
            {
                return; // this is because the XmlSerializer does not create elements with default values
            }

            // Create a new EncryptedXml object.
            EncryptedXml exml = new EncryptedXml(xmlDoc);

            // Encrypt the element using the symmetric key.
            byte[] rgbOutput = exml.EncryptData(inputElement, algorithm, false);

            // Create an EncryptedData object and populate it.
            EncryptedData ed = new EncryptedData();

            // Specify the namespace URI for XML encryption elements.
            ed.Type = EncryptedXml.XmlEncElementUrl;

            // Specify the namespace URI for the TrippleDES algorithm.
            ed.EncryptionMethod = new EncryptionMethod(EncryptedXml.XmlEncTripleDESUrl);

            // Create a CipherData element.
            ed.CipherData = new CipherData();

            // Set the CipherData element to the value of the encrypted XML element.
            ed.CipherData.CipherValue = rgbOutput;

            // Replace the plaintext XML elemnt with an EncryptedData element.
            EncryptedXml.ReplaceElement(inputElement, ed, false);
        }

        public void Decrypt()
        {
            XmlElement encryptedElement = null;
            while ((encryptedElement = xmlDoc.GetElementsByTagName("EncryptedData")[0] as XmlElement) != null)
            {
                // Create an EncryptedData object and populate it.
                EncryptedData ed = new EncryptedData();
                ed.LoadXml(encryptedElement);

                // Create a new EncryptedXml object.
                EncryptedXml exml = new EncryptedXml();

                // Decrypt the element using the symmetric key.
                byte[] rgbOutput = exml.DecryptData(ed, algorithm);

                // Replace the encryptedData element with the plaintext XML elemnt.
                exml.ReplaceData(encryptedElement, rgbOutput);
            }
        }

    }

}
