﻿using System;
using System.Collections.Generic;
using System.Text;
using log4net;
using System.Diagnostics;
using System.IO;

namespace VixBuilderBusiness
{
    public delegate String InfoDelegate(String infoMessage);
    public delegate String OutputDelegate(String outputMessage); // combined display standard output and error from child processes
    public delegate void DoEventsDelegate(); // allows business objects to keep the UI responsive

    abstract internal class AbstractRepository
    {
        public InfoDelegate Info { get; set; } // used to display a line of text in the main form's info text box
        public OutputDelegate Output { get; set; } // used to display combined standard output and error to output text box
        public DoEventsDelegate DoEvents { get; set; } // keep the UI responsive
        public bool CanRun { get; set; }
        protected StringBuilder sbOutput;
        protected BuildConfiguration config;

        public virtual BuildManifest GetBuildManifest()
        {
            return null;
        }

        protected readonly RepositoryEnum repositoryType;

        protected AbstractRepository(BuildConfiguration config, RepositoryEnum repositoryType)
        {
            this.config = config;
            this.repositoryType = repositoryType;
        }

        protected void OutputHandler(object sendingProcess, DataReceivedEventArgs outLine)
        {
            // Collect the sort command output.
            if (!String.IsNullOrEmpty(outLine.Data))
            {
                sbOutput.AppendLine(outLine.Data);
            }
        }

        protected void LogInfo(string message)
        {
            if (Info != null)
            {
                Info(message);
            }
            LogManager.GetLogger(this.GetType().Name).Info(message);
        }

        protected string LogError(string message)
        {
            LogManager.GetLogger(this.GetType().Name).Error(message);
            return message;
        }

        protected void YieldToUserInterface()
        {
            if (DoEvents != null)
            {
                DoEvents();
            }
        }

        protected void DisplayOutput()
        {
            if (Output != null)
            {
                Output(GetOutput());
            }
        }

        protected string GetOutput()
        {
            return sbOutput == null ? "" : sbOutput.ToString();
        }

        protected void ResetStandardOutput()
        {
            sbOutput = new StringBuilder();
        }

        protected void RemoveReadOnlyAttributes(String parentFolder)
        {
            String[] folders = Directory.GetDirectories(parentFolder);
            long folderCount = folders.Length;

            DirectoryInfo parentFolderInfo = new DirectoryInfo(parentFolder);
            foreach (FileInfo fileInfo in parentFolderInfo.GetFiles())
            {
                File.SetAttributes(fileInfo.FullName, FileAttributes.Normal);
            }

            foreach (String folder in folders)
            {
                RemoveReadOnlyAttributes(folder);
            }
        }

        /// <summary>
        /// Copies all.
        /// </summary>
        /// <param name="source">The source.</param>
        /// <param name="target">The target.</param>
        /// <remarks></remarks>
        protected void CopyAll(DirectoryInfo source, DirectoryInfo target)
        {
            // Check if the target directory exists, if not, create it.
            if (Directory.Exists(target.FullName) == false)
            {
                Directory.CreateDirectory(target.FullName);
            }

            // Copy each file into it’s new directory.
            foreach (FileInfo fileInfo in source.GetFiles())
            {
                if (fileInfo.Name.StartsWith("vssver")) // silently skip any source safe marker files
                {
                    continue;
                }

                string targetFilespec = Path.Combine(target.ToString(), fileInfo.Name);
                fileInfo.CopyTo(targetFilespec, true);
                File.SetAttributes(targetFilespec, FileAttributes.Normal);
                LogInfo("\tcopied " + fileInfo.FullName + " to " + targetFilespec);
                if (CanRun == false)
                {
                    return;
                }
            }

            foreach (DirectoryInfo sourceDirectoryInfo in source.GetDirectories())
            {
                DirectoryInfo targetDirectoryInfo = target.CreateSubdirectory(sourceDirectoryInfo.Name);
                CopyAll(sourceDirectoryInfo, targetDirectoryInfo); //recurse
                if (CanRun == false)
                {
                    return;
                }
            }
        }

        protected string GetNextVersionNumberFromFile(string versionNumberFilespec)
        {
            string nextVersionNumber = null;

            try
            {
                if (File.Exists(versionNumberFilespec) == false)
                {
                    throw new BuildException("MSI version number file does not exist: " + versionNumberFilespec);
                }

                string line = null;
                using (StreamReader sr = new StreamReader(versionNumberFilespec))
                {
                    line = sr.ReadLine();
                }

                nextVersionNumber = VisualStudioSetupProjectHelper.GetNextVersion(line);

                using (StreamWriter sw = new StreamWriter(versionNumberFilespec, false))
                {
                    sw.WriteLine(nextVersionNumber);
                }
            }
            catch (Exception ex)
            {
                LogManager.GetLogger(this.GetType().ToString()).Error(ex.Message);
                throw new BuildException(ex.Message);
            }

            return nextVersionNumber;
        }

        //protected DeployProject[] GetFilteredDeployProjectList(List<DeployProjectType> typesToGet)
        //{
        //    List<DeployProject> projects = new List<DeployProject>();
        //    DeployProject[] deployProjects = manifest.GetDeployProjects();
        //    foreach (DeployProject vssProject in deployProjects)
        //    {
        //        if (typesToGet == null || typesToGet.Contains(vssProject.DeployProjectType))
        //        {
        //            projects.Add(vssProject);
        //        }
        //    }
        //    return projects.ToArray();
        //}

    }
}
