using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;
using System.Configuration;
using VixBuilderBusiness;
using System.Threading;

namespace VixBuilder
{
    /// <summary>
    /// This delegate provides the means for business layer code to write a message to the this.textBoxInfo control. It maps to the
    /// Info method.
    /// </summary>
    /// <param name="infoMessage">The message string to be displayed on a new textbox line.</param>
    /// <returns></returns>
    /// 

    public partial class VixBuilderForm : Form
    {
        private const string FORM_TITLE = "VIX Build Utility";
        private const string CANCEL = "Ca&ncel"; // The buttonXrefDirectory text while actions are running
        private BuildConfiguration buildConfiguration = new BuildConfiguration();
        private bool buildConfigurationDirty = false;
        private bool disableEvents = false;
        //private BuildAction[] buildActions = BuildAction.GetBuildActionSteps();
        private BuildAction[] buildActionGroups = BuildAction.GetBuildActionGroups();
        private CheckoutMode[] cvsCheckoutModes = CheckoutMode.GetCheckoutModes();
        private InstallationType[] installationTypes = InstallationType.GetInstallationTypes();
        private string buildConfigurationFilespec = null;
        private System.Configuration.Configuration config = null;
        private string cwd = Directory.GetCurrentDirectory();
        private RepositoryType[] buildRepository = RepositoryType.GetBuildRepositories();
        private RepositoryType[] deployRepository = RepositoryType.GetDeployRepositories();
        private VixBuilderFacade vixBuilderFacade;

        public string BuildConfigurationFilespec
        {
            get
            {
                if (this.buildConfigurationFilespec == null)
                {
                    KeyValueConfigurationElement keyValue = config.AppSettings.Settings["BuildConfigurationFilespec"];
                    if (keyValue != null)
                    {
                        this.buildConfigurationFilespec = keyValue.Value;
                    }
                }
                return buildConfigurationFilespec;
            }
            set
            {
                this.buildConfigurationFilespec = value;
                if (value != null)
                {
                    KeyValueConfigurationElement keyValue = config.AppSettings.Settings["BuildConfigurationFilespec"];
                    if (keyValue != null)
                    {
                        config.AppSettings.Settings.Remove("BuildConfigurationFilespec");
                    }
                    config.AppSettings.Settings.Add("BuildConfigurationFilespec", value);
                    config.Save(ConfigurationSaveMode.Modified);
                    ConfigurationManager.RefreshSection("appSettings"); // Force a reload of a changed section.
                }
            }
        }

        public VixBuilderForm()
        {
            InitializeComponent();
            // rest of initialization is done in Load event
        }

        #region form menu event handlers
        //-------------------------------------------
        // BEGIN REGION
        //-------------------------------------------
        private void newToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (CanProceed())
            {
                New();
            }
        }

        private void openToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (CanProceed())
            {
                Open();
            }
        }

        private void saveToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (this.BuildConfigurationFilespec == null)
            {
                SaveAs();
            }
            else
            {
                Save();
            }
        }

        private void saveAsToolStripMenuItem_Click(object sender, EventArgs e)
        {
            SaveAs();
        }

        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        //-------------------------------------------
        // END REGION
        //-------------------------------------------
        #endregion

        #region form event handlers
        //-------------------------------------------
        // BEGIN REGION
        //-------------------------------------------

        private void VixBuilderForm_Load(object sender, EventArgs e)
        {
            this.config = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);

            // initialize VixBuilderFacade delegates
            //VixBuilderFacade.Info = this.Info;
            //VixBuilderFacade.Output = this.Output;
            //VixBuilderFacade.DoEvents = this.DoEvents;
            // initialize TomcatFacade delegates
            TomcatFacade.Info = this.Info;
            TomcatFacade.DoEvents = this.DoEvents;

            // now that build configuration has been initialized
            this.comboBoxBuildActions.DisplayMember = "Description";
            this.comboBoxBuildActions.ValueMember = "Id";
            //this.SetBuildActionComboBoxDataSource(); // set in open method

            this.comboBoxActionGroup.DataSource = buildActionGroups;
            this.comboBoxActionGroup.DisplayMember = "Description";
            this.comboBoxActionGroup.ValueMember = "Id";

            this.comboBoxCheckoutMode.DataSource = cvsCheckoutModes;
            this.comboBoxCheckoutMode.DisplayMember = "Description";
            this.comboBoxCheckoutMode.ValueMember = "Id";

            this.comboBoxInstallationType.DataSource = installationTypes;
            this.comboBoxInstallationType.DisplayMember = "Description";
            this.comboBoxInstallationType.ValueMember = "Id";

            this.comboBoxBuildRepository.DataSource = buildRepository;
            this.comboBoxBuildRepository.DisplayMember = "Description";
            this.comboBoxBuildRepository.ValueMember = "Id";

            this.comboBoxDeployRepository.DataSource = deployRepository;
            this.comboBoxDeployRepository.DisplayMember = "Description";
            this.comboBoxDeployRepository.ValueMember = "Id";

            try
            {
                this.disableEvents = true;
                this.vixBuilderFormBindingSource.Add(this.buildConfiguration);
            }
            finally
            {
                this.disableEvents = false;
            }

            if (this.BuildConfigurationFilespec != null)
            {
                Open(this.BuildConfigurationFilespec);
            }
            else
            {
                New();
            }
        }

        private void VixBuilderForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (!CanProceed())
            {
                e.Cancel = true;
            }
        }

        private void vixBuilderFormBindingSource_CurrentItemChanged(object sender, EventArgs e)
        {
            if (!this.disableEvents)
            {
                this.vixBuilderFormBindingSource.EndEdit();
                buildConfigurationDirty = true;
                this.SetBuildActionComboBoxDataSource();
            }
        }

        private void comboBoxBuildActions_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (!this.disableEvents)
            {
                try
                {
                    BuildActionEnum action = (BuildActionEnum)this.comboBoxBuildActions.SelectedValue;
                    if (action == BuildActionEnum.labelCvsWorkspace)
                    {
                        DisplayLabelActionAdditionalControls(true); // display move tag option
                    }
                    else if (action == BuildActionEnum.deleteLabelFromCvsWorkspace)
                    {
                        DisplayLabelActionAdditionalControls(false); // dont display move tag option
                    }
                    else if (action == BuildActionEnum.branchCvsWorkspace)
                    {
                        DisplayBranchActionAdditionalControls();
                    }
                    else
                    {
                        HideAllAdditionalControls();
                    }
                }
                catch (Exception) { }
            }
        }


        //private void comboBoxBuildRepository_SelectedIndexChanged(object sender, EventArgs e)
        //{
        //    if (!this.disableEvents)
        //    {
        //        this.SetBuildActionComboBoxDataSource();
        //    }
        //}

        //-------------------------------------------
        // END REGION
        //-------------------------------------------
        #endregion

        #region form button event handlers
        //-------------------------------------------
        // BEGIN REGION
        //-------------------------------------------

        private void buttonCvsExeBrowse_Click(object sender, EventArgs e)
        {
            this.textBoxCvsExeFilespec.Text = SelectFilespec(this.textBoxCvsExeFilespec.Text);
        }

        private void buttonVixPayloadRoot_Click(object sender, EventArgs e)
        {
            this.textBoxVixPayloadRootDirspec.Text = this.SelectDirspec(this.textBoxVixPayloadRootDirspec.Text);
        }

        private void buttonApacheTomcatDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxApacheTomcatDirspec.Text = this.SelectDirspec(this.textBoxApacheTomcatDirspec.Text);
        }

        private void buttonMaven2ExeFilespec_Click(object sender, EventArgs e)
        {
            this.textBoxMaven2ExeFilespec.Text = SelectFilespec(this.textBoxMaven2ExeFilespec.Text);
        }

        private void buttonVssExeFilespec_Click(object sender, EventArgs e)
        {
            this.textBoxVssExecutableFilespec.Text = SelectFilespec(this.textBoxVssExecutableFilespec.Text);
        }

        private void buttonVssRootDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxVssRootDirspec.Text = this.SelectDirspec(this.textBoxVssRootDirspec.Text);
        }

        private void buttonWorkspaceRootDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxCvsRootDirspec.Text = this.SelectDirspec(this.textBoxCvsRootDirspec.Text);
        }

        private void buttonMaven2RepoDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxMaven2RepoDirspec.Text = this.SelectDirspec(this.textBoxMaven2RepoDirspec.Text);
        }

        private void buttonVisualStudio2005Filespec_Click(object sender, EventArgs e)
        {
            this.textBoxVisualStudio2005Filespec.Text = SelectFilespec(this.textBoxVisualStudio2005Filespec.Text);
        }

        private void buttonVixInstallerRootDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxVixInstallerRootDirspec.Text = this.SelectDirspec(this.textBoxVixInstallerRootDirspec.Text);
        }

        private void buttonVixConfigDir_Click(object sender, EventArgs e)
        {
            this.textBoxVixConfigDir.Text = this.SelectDirspec(this.textBoxVixConfigDir.Text);
        }

        private void buttonTfsExeFilespec_Click(object sender, EventArgs e)
        {
            this.textBoxTfsExecutableFilespec.Text = SelectFilespec(this.textBoxTfsExecutableFilespec.Text);
        }

        private void buttonTfsRootDirspec_Click(object sender, EventArgs e)
        {
            this.textBoxTfsRootDirspec.Text = this.SelectDirspec(this.textBoxTfsRootDirspec.Text);
        }

        private void buttonWorkspaceWorkingFolder_Click(object sender, EventArgs e)
        {
            this.textBoxTfsWorkspaceWorkingFolder.Text = this.SelectDirspec(this.textBoxTfsWorkspaceWorkingFolder.Text);
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonExecuteActionGroup_Click(object sender, EventArgs e)
        {
            string buttonExecuteActionGroupText = "";

            if (buttonExecuteActionGroup.Text == CANCEL)
            {
                if (this.ConfirmCancel())
                {
                    vixBuilderFacade.CanRun = false;
                }
            }
            else
            {
                ClearInfo();
                ClearOutput();

                if (CanProceed() == false)
                {
                    Info("Action canceled by user");
                    return;
                }

                //if (this.ValidateConfiguration() == false)
                //{
                //    MessageBox.Show("You must fix any existing build configuration errors before performing this action.", "Build Configuration has errors!",
                //        MessageBoxButtons.OK, MessageBoxIcon.Error);
                //    Info("Build configuration error detected - Action canceled");
                //    return;
                //}

                BuildActionEnum action = (BuildActionEnum)this.comboBoxActionGroup.SelectedValue;

                buttonExecuteActionGroupText = buttonExecuteActionGroup.Text;
                buttonExecuteActionGroup.Text = CANCEL;
                this.buttonExecuteActions.Enabled = false;
                // all exceptions will be caught in here - no try - catch - finally needed
                InitializeVixBuilderFacade();
                vixBuilderFacade.ExecuteBuildAction(action, this.cwd);
                buttonExecuteActionGroup.Text = buttonExecuteActionGroupText;
                this.buttonExecuteActions.Enabled = true;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void buttonExecuteActions_Click(object sender, EventArgs e)
        {
            string buttonExecuteActionsText = "";

            if (buttonExecuteActions.Text == CANCEL)
            {
                if (this.ConfirmCancel())
                {
                    vixBuilderFacade.CanRun = false;
                }
            }
            else
            {
                ClearInfo();
                ClearOutput();

                if (CanProceed() == false)
                {
                    Info("Action canceled by user");
                    return;
                }

                //if (this.ValidateConfiguration() == false)
                //{
                //    MessageBox.Show("You must fix any existing build configuration errors before performing this action.", "Build Configuration has errors!",
                //        MessageBoxButtons.OK, MessageBoxIcon.Error);
                //    Info("Build configuration error detected - Action canceled");
                //    return;
                //}

                BuildActionEnum action = (BuildActionEnum)this.comboBoxBuildActions.SelectedValue;
                if (action == BuildActionEnum.labelCvsWorkspace || action == BuildActionEnum.deleteLabelFromCvsWorkspace)
                {
                    if (this.textBoxCvsLabelName.Text.Trim() == "")
                    {
                        Info("You must provide a CVS label - Action canceled");
                        return;
                    }
                    else // initialize transient build configuration properties
                    {
                        this.buildConfiguration.CvsLabelToApply = this.textBoxCvsLabelName.Text.Trim();
                        this.buildConfiguration.MoveCvsLabelIfExists = this.checkBoxMoveTag.Checked;
                    }
                }

                if (action == BuildActionEnum.branchCvsWorkspace)
                {
                    if (this.textBoxCvsLabelName.Text.Trim() == "")
                    {
                        Info("You must provide a CVS label - Action canceled");
                        return;
                    }
                    else if (this.textBoxCvsBranchName.Text.Trim() == "")
                    {
                        Info("You must provide a CVS branch name - Action canceled");
                        return;
                    }
                    else // initialize transient build configuration properties
                    {
                        this.buildConfiguration.CvsLabelToApply = this.textBoxCvsLabelName.Text.Trim();
                        this.buildConfiguration.MoveCvsLabelIfExists = this.checkBoxMoveTag.Checked;
                        this.buildConfiguration.CvsBranchToApply = this.textBoxCvsBranchName.Text.Trim();
                    }
                }

                buttonExecuteActionGroup.Enabled = false;
                buttonExecuteActionsText = buttonExecuteActions.Text;
                buttonExecuteActions.Text = CANCEL;
                // all exceptions will be caught in here - no try - catch - finally needed
                InitializeVixBuilderFacade();
                vixBuilderFacade.ExecuteBuildAction(action, this.cwd);
                buttonExecuteActions.Text = buttonExecuteActionsText;
                buttonExecuteActionGroup.Enabled = true;
            }
        }


        //-------------------------------------------
        // END REGION
        //-------------------------------------------
        #endregion

        #region private methods
        //-------------------------------------------
        // BEGIN REGION
        //-------------------------------------------

        private void SetBuildActionComboBoxDataSource()
        {
            if (this.buildConfiguration != null)
            {
                switch (this.buildConfiguration.BuildRepository)
                {
                    case RepositoryEnum.ConcurrentVersionsSystem:
                        this.comboBoxBuildActions.DataSource = BuildAction.GetCvsBuildActionSteps();
                        break;
                    default:
                        this.comboBoxBuildActions.DataSource = BuildAction.GetBuildActionSteps();
                        break;
                }
            }
            //else // use most restrictive list
            //{
            //    this.comboBoxBuildActions.DataSource = BuildAction.GetBuildActionSteps();
            //}
        }

        private void InitializeVixBuilderFacade()
        {
            vixBuilderFacade = new VixBuilderFacade(buildConfiguration);
            vixBuilderFacade.Info = this.Info;
            vixBuilderFacade.Output = this.Output;
            vixBuilderFacade.DoEvents = this.DoEvents;
        }

        private void DisplayLabelActionAdditionalControls(bool displayMoveTagOption)
        {
            labelLabelName.Visible = true;
            this.textBoxCvsLabelName.Visible = true;
            if (displayMoveTagOption) checkBoxMoveTag.Visible = true;
            labelBranchName.Visible = false;
            this.textBoxCvsBranchName.Visible = false;
            labelBranchInstructions.Visible = false;
        }


        private void DisplayBranchActionAdditionalControls()
        {
            labelLabelName.Visible = true;
            this.textBoxCvsLabelName.Visible = true;
            checkBoxMoveTag.Visible = true;
            labelBranchName.Visible = true;
            this.textBoxCvsBranchName.Visible = true;
            labelBranchInstructions.Visible = true;
        }

        private void HideAllAdditionalControls()
        {
            labelLabelName.Visible = false;
            this.textBoxCvsLabelName.Visible = false;
            checkBoxMoveTag.Visible = false;
            labelBranchName.Visible = false;
            this.textBoxCvsBranchName.Visible = false;
            labelBranchInstructions.Visible = false;
        }

        private void ClearInfo()
        {
            this.textBoxInfo.Clear();
        }

        private void ClearOutput()
        {
            this.textBoxOutput.Clear();
        }

        /// <summary>
        /// Check the build configuration dirty bit and if true prompt the user to perform a save or save as operation.
        /// </summary>
        /// <returns>true if the calling routine can proceed, false otherwise</returns>
        private bool CanProceed()
        {
            bool canProceed = true;

            this.vixBuilderFormBindingSource.EndEdit();
            if (this.buildConfigurationDirty)
            {
                DialogResult result = MessageBox.Show("Save configuration changes?", "VIX Build Utility", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
                if (result == DialogResult.Cancel)
                {
                    canProceed = false;
                }
                else if (result == DialogResult.Yes)
                {
                    if (ValidateConfiguration())
                    {
                        try
                        {
                            if (this.BuildConfigurationFilespec == null)
                            {
                                canProceed = SaveAs();
                            }
                            else
                            {
                                canProceed = Save();
                            }
                        }
                        catch (Exception ex)
                        {
                            MessageBox.Show(Info(ex.Message), "Error saving configuration", MessageBoxButtons.OK, MessageBoxIcon.Error);
                            canProceed = false;
                        }
                    }
                    else
                    {
                        canProceed = false;
                    }
                }
                else // No
                {
                    canProceed = true;
                }
            }
            return canProceed;
        }

        private void New()
        {
            try
            {
                this.disableEvents = true;
                this.vixBuilderFormBindingSource.Clear();
                this.errorProviderFixed.Clear();
                this.buildConfiguration = new BuildConfiguration();
                this.vixBuilderFormBindingSource.Add(this.buildConfiguration);
                this.buildConfigurationDirty = false;
                this.BuildConfigurationFilespec = null;
                this.SetTitleBar();
                this.SetBuildActionComboBoxDataSource();
            }
            finally
            {
                this.disableEvents = false;
            }
        }

        private void Open()
        {
            DialogResult result = this.openFileDialog.ShowDialog();
            if (result == DialogResult.OK)
            {
                this.BuildConfigurationFilespec = this.openFileDialog.FileName;
                try
                {
                    Open(this.BuildConfigurationFilespec);
                }
                catch (Exception ex)
                {
                    MessageBox.Show(Info(ex.Message), "Error opening configuration", MessageBoxButtons.OK, MessageBoxIcon.Error);
                }
            }
        }

        private void Open(string filespec)
        {
            if (!File.Exists(filespec))
            {
                throw new BuildConfigurationException(filespec + " does not exist");
            }

            try
            {
                this.disableEvents = true;
                this.vixBuilderFormBindingSource.Clear();
                this.errorProviderFixed.Clear();
                this.buildConfiguration = BuildConfiguration.Open(filespec);
                this.vixBuilderFormBindingSource.Add(this.buildConfiguration);
                this.buildConfigurationDirty = false;
                this.BuildConfigurationFilespec = filespec;
                SetTitleBar();
                this.ValidateConfiguration();
                this.SetBuildActionComboBoxDataSource();
            }
            finally
            {
                this.disableEvents = false;
            }
        }

        private bool Save()
        {
            bool ok = false;
            try
            {
                this.vixBuilderFormBindingSource.EndEdit();
                this.errorProviderFixed.Clear();
                ok = this.ValidateConfiguration();
                this.buildConfiguration.Save(this.BuildConfigurationFilespec);
                this.buildConfigurationDirty = false;
            }
            catch (Exception ex)
            {
                MessageBox.Show(Info(ex.Message), "Error saving configuration", MessageBoxButtons.OK, MessageBoxIcon.Error);
                ok = false;
            }
            finally
            {
                SetTitleBar();
            }
            return ok;
        }

        private bool SaveAs()
        {
            bool ok = false;
            try
            {
                this.vixBuilderFormBindingSource.EndEdit();
                this.errorProviderFixed.Clear();
                DialogResult result = this.saveFileDialog.ShowDialog();
                if (result == DialogResult.OK)
                {
                    this.BuildConfigurationFilespec = this.saveFileDialog.FileName;
                    ok = Save();
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(Info(ex.Message), "Error saving configuration as", MessageBoxButtons.OK, MessageBoxIcon.Error);
                ok = false;
            }

            return ok;
        }

        private void SetTitleBar()
        {
            string title = FORM_TITLE;
            if (!String.IsNullOrEmpty(this.BuildConfigurationFilespec))
            {
                title += " " + this.BuildConfigurationFilespec;
            }
            this.Text = title;
        }

        private bool ValidateConfiguration()
        {
            bool validated = true;

            switch (buildConfiguration.BuildRepository)
            {
                case RepositoryEnum.ConcurrentVersionsSystem:
                    validated &= this.ValidateCvs();
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    validated &= this.ValidateTfs();
                    break;
            }

            switch(buildConfiguration.DeployRepository)
            {
                case RepositoryEnum.VisualSourceSafe:
                    validated &= this.ValidateVss();
                    break;
                case RepositoryEnum.TeamFoundationServer:
                    validated &= this.ValidateTfs();
                    break;
            }

            validated &= this.ValidateBuild();

            if (this.IsValidateDeploymentRequired())
            {
                validated &= this.ValidateDeployment();
            }

            if (this.IsValidateDevelopmentRequired())
            {
                validated &= this.ValidateDevelopment();
            }

            return validated;
        }

        /// <summary>
        /// Write a message line to the this.textBoxInfo control. This method is designed to be used from a worker thread.
        /// </summary>
        /// <param name="infoMessage">The message line to be written.</param>
        /// <returns></returns>
        private String Info(String infoMessage)
        {
            if (this.textBoxInfo.InvokeRequired)
            {
                InfoDelegate d = new InfoDelegate(this.Info);
                this.Invoke(d, new object[] { infoMessage });
            }
            else
            {
                try
                {
                    this.textBoxInfo.AppendText(infoMessage + Environment.NewLine);
                }
                catch (ObjectDisposedException)
                {
                    // sometimes happens in the development environment after application shutdown
                }
            }
            Application.DoEvents();
            return infoMessage;
        }

        private String Output(String outputMessage)
        {
            if (this.textBoxOutput.InvokeRequired)
            {
                InfoDelegate d = new InfoDelegate(this.Output);
                this.Invoke(d, new object[] { outputMessage });
            }
            else
            {
                try
                {
                    this.textBoxOutput.AppendText(outputMessage + Environment.NewLine);
                }
                catch (ObjectDisposedException)
                {
                    // sometimes happens in the development environment after application shutdown
                }
            }
            Application.DoEvents();
            return outputMessage;
        }


        private void DoEvents()
        {
            Application.DoEvents();
        }

        /// <summary>
        /// Ask the user if they want to cancel the current task.
        /// </summary>
        /// <returns>Returns true to cancel the current task, false otherwise.</returns>
        private bool ConfirmCancel()
        {
            DialogResult result = MessageBox.Show("Do you really want to cancel?", "Confim Cancel",
                MessageBoxButtons.YesNo, MessageBoxIcon.Question);
            return result == DialogResult.Yes ? true : false;
        }

        private string SelectDirspec(string dirspec)
        {
            DialogResult result = this.folderBrowserDialog.ShowDialog();
            return result == DialogResult.OK ? folderBrowserDialog.SelectedPath : dirspec;
        }

        private string SelectFilespec(string filespec)
        {
            DialogResult result = this.openFileDialog.ShowDialog();
            return result == DialogResult.OK ? openFileDialog.FileName : filespec;
        }

        //-------------------------------------------
        // END REGION
        //-------------------------------------------
        #endregion

        #region private validation methods
        //-------------------------------------------
        // BEGIN REGION
        //-------------------------------------------

        private bool ValidateCvs()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.CvsExecutable))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsExeFilespec, "You must specify the fully qualified CVS executable.");
            }
            else if (File.Exists(this.buildConfiguration.CvsExecutable) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsExeFilespec, "File does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.CvsProjectRoot))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsRootDirspec, "You must specify the root directory for CVS projects.");
            }
            else if (Directory.Exists(this.buildConfiguration.CvsProjectRoot) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsRootDirspec, "Directory does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.CvsServer))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsServer, "You must specify the CVS server name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.CvsUsername))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsUsername, "You must specify your CVS user name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.CvsPassword))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxCvsPassword, "You must specify your CVS password.");
            }

            return validated;
        }

        private bool ValidateVss()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.VssExecutable))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssExecutableFilespec, "You must specify the fully qualified VSS executable.");
            }
            else if (File.Exists(this.buildConfiguration.VssExecutable) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssExecutableFilespec, "File does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VssSharename))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssSharename, "You must specify the VSS share name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VssUsername))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssUsername, "You must specify your VSS user name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VssPassword))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssPassword, "You must specify your VSS password.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VssRootDirspec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssRootDirspec, "You must specify the fully qualified VSS Root working directory.");
            }
            else if (Directory.Exists(this.buildConfiguration.VssRootDirspec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVssRootDirspec, "Directory does not exist.");
            }

            return validated;
        }

        private bool ValidateTfs()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsExecutable))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsExecutableFilespec, "You must specify the fully qualified TFS executable.");
            }
            else if (File.Exists(this.buildConfiguration.TfsExecutable) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsExecutableFilespec, "File does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsServername))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsServerName, "You must specify the TFS server name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsWorkspace))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsWorkspace, "You must specify the TFS workspace name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsUsername))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsUsername, "You must specify your TFS user name.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsPassword))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsPassword, "You must specify your TFS password.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsRootDirspec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsRootDirspec, "You must specify the fully qualified VSS Root working directory.");
            }
            else if (Directory.Exists(this.buildConfiguration.TfsRootDirspec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsRootDirspec, "Directory does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.TfsWorkspaceWorkingFolder))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsWorkspaceWorkingFolder, "You must specify the fully qualified VISA workspace directory.");
            }
            else if (Directory.Exists(this.buildConfiguration.TfsWorkspaceWorkingFolder) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxTfsWorkspaceWorkingFolder, "Directory does not exist.");
            }

            return validated;
        }

        private bool ValidateBuild()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.BuildManifestFilename))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxBuildManifestFilename, "You must specify the build manifest file name. This file must be located in the CVS ImagingExchange project.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.Maven2Executable))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxMaven2ExeFilespec, "You must specify the fully qualified Maven 2 executable.");
            }
            else if (File.Exists(this.buildConfiguration.Maven2Executable) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxMaven2ExeFilespec, "File does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.Maven2RepoDirspec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxMaven2RepoDirspec, "You must specify the fully qualified Maven 2 repository.");
            }
            else if (Directory.Exists(this.buildConfiguration.Maven2RepoDirspec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxMaven2RepoDirspec, "Directory does not exist.");
            }

            return validated;
        }

        /// <summary>
        /// If all deployment fields blank then skip validation.
        /// </summary>
        /// <returns></returns>
        private bool IsValidateDeploymentRequired()
        {
            bool validationRequired = false;

            //if (this.textBoxVixPayloadRootDirspec.Text.Trim() != "" || this.textBoxVisualStudio2005Filespec.Text.Trim() != "" || this.textBoxVixInstallerRootDirspec.Text.Trim() != ""
            //    || this.textBoxDeployManifestFilename.Text.Trim() != "")
            if (String.IsNullOrEmpty(this.buildConfiguration.VixPayloadRootDirspec) == false || String.IsNullOrEmpty(this.buildConfiguration.VisualStudioExeFilespec) == false ||
                String.IsNullOrEmpty(this.buildConfiguration.VixInstallerRootDirspec) == false || String.IsNullOrEmpty(this.buildConfiguration.DeploymentManifestFilename) == false)
            {
                validationRequired = true;
            }

            return validationRequired;
        }

        private bool ValidateDeployment()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.VixPayloadRootDirspec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVixPayloadRootDirspec, "You must specify the fully qualified directory where the VIX installer payload will be assembled.");
            }
            else if (Directory.Exists(this.buildConfiguration.VixPayloadRootDirspec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVixPayloadRootDirspec, "Directory does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VisualStudioExeFilespec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVisualStudio2005Filespec, "You must specify the fully qualified Visual Studio 2005 executable.");
            }
            else if (File.Exists(this.buildConfiguration.VisualStudioExeFilespec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVisualStudio2005Filespec, "File does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VixInstallerRootDirspec))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVixInstallerRootDirspec, "You must specify the fully qualified directory where the VIX installer msi file will be copied.");
            }
            else if (Directory.Exists(this.buildConfiguration.VixInstallerRootDirspec) == false)
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxVixInstallerRootDirspec, "Directory does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.DeploymentManifestFilename))
            {
                validated = false;
                errorProviderFixed.SetError(this.textBoxDeployManifestFilename, "You must specify the VIX installer deployment manifest file name. This file must be located in the CVS ImagingExchange project.");
            }

            return validated;
        }

        /// <summary>
        /// If all development fields blank then skip validation.
        /// </summary>
        /// <returns></returns>
        private bool IsValidateDevelopmentRequired()
        {
            bool validationRequired = false;

            //if (this.textBoxApacheTomcatDirspec.Text.Trim() != "" || this.textBoxVixConfigDir.Text.Trim() != "" || this.checkBoxDeployWebApplications.Checked ||
            //    this.checkBoxGenerateConfigFiles.Checked || this.checkBoxGenerateTomcatConfigFiles.Checked)
            if (String.IsNullOrEmpty(this.buildConfiguration.ApacheTomcatDirspec) == false || String.IsNullOrEmpty(this.buildConfiguration.VixConfigDir) == false ||
                this.buildConfiguration.DeployWebApplications == true || this.buildConfiguration.GenerateConfigFiles == true || this.buildConfiguration.GenerateTomcatConfigFiles == true)
            {
                validationRequired = true;
            }

            return validationRequired;
        }

        private bool ValidateDevelopment()
        {
            bool validated = true;

            if (String.IsNullOrEmpty(this.buildConfiguration.ApacheTomcatDirspec) == false && Directory.Exists(this.buildConfiguration.ApacheTomcatDirspec) == false)
            {
                validated &= false;
                errorProviderFixed.SetError(this.textBoxApacheTomcatDirspec, "Directory does not exist.");
            }

            if (String.IsNullOrEmpty(this.buildConfiguration.VixConfigDir) == false && Directory.Exists(this.buildConfiguration.VixConfigDir) == false)
            {
                validated &= false;
                errorProviderFixed.SetError(this.textBoxVixConfigDir, "Directory does not exist.");
            }

            return validated;
        }

        //-------------------------------------------
        // END REGION
        //-------------------------------------------
        #endregion

        #region testbench

        private void buttonTestCreateDirectory_Click(object sender, EventArgs e)
        {
            try
            {
                if (!Directory.Exists(this.textBoxTestDirectory.Text))
                {
                    Directory.CreateDirectory(this.textBoxTestDirectory.Text);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private void buttonNextVersionTest_Click(object sender, EventArgs e)
        {
            try
            {
                this.textBoxNextVersion.Text = vixBuilderFacade.GetNextVersion(this.textBoxVersion.Text);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }
        #endregion

    }
}