/**
 * 
 * Property of ISI Group, LLC
 * Date Created: Aug 25, 2014
 * Developer: Julian Werfel
 */
package gov.va.med.imaging.study.web.rest.translator;

import gov.va.med.SERIALIZATION_FORMAT;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.exchange.business.Image;
import gov.va.med.imaging.exchange.business.Series;
import gov.va.med.imaging.exchange.business.Study;
import gov.va.med.imaging.exchange.business.StudyFilter;
import gov.va.med.imaging.exchange.enums.ImageQuality;
import gov.va.med.imaging.exchange.enums.PatientSensitivityLevel;
import gov.va.med.imaging.exchange.enums.VistaImageType;
import gov.va.med.imaging.study.web.configuration.ViewerStudyFacadeConfiguration;
import gov.va.med.imaging.study.web.rest.types.ViewerStudyFilterType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudyImageType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudyImagesType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudySeriesType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudySeriesesType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudyStudiesType;
import gov.va.med.imaging.study.web.rest.types.ViewerStudyStudyType;
import gov.va.med.imaging.translator.AbstractClinicalTranslator;

import java.util.List;

/**
 * @author Julian
 *
 */
public class ViewerStudyWebTranslator
extends AbstractClinicalTranslator
{
	
	public static ViewerStudyStudiesType translateStudies(List<Study> studies)
	throws MethodException
	{
		if(studies == null)
			return null;
		
		ViewerStudyStudyType result[] = new ViewerStudyStudyType[studies.size()];
		for(int i = 0; i < studies.size(); i++)
		{
			result[i] = translate(studies.get(i));
		}
		
		return new ViewerStudyStudiesType(result);
	}
	
	public static StudyFilter translate(ViewerStudyFilterType filter)
	{
		StudyFilter result = new StudyFilter();
		result.setMaximumAllowedLevel(PatientSensitivityLevel.DISPLAY_WARNING_REQUIRE_OK);
		if(filter != null)
		{
			result.setOrigin(filter.getFilterOrigin());
			result.setFromDate(filter.getFromDate());
			result.setToDate(filter.getToDate());
			result.setStudy_class(filter.getFilterClass());
			result.setStudy_event(filter.getFilterEvent());
			result.setStudy_package(filter.getFilterPackage());
			result.setStudy_specialty(filter.getFilterSpecialty());
			result.setStudy_type(filter.getFilterType());
		}
		
		return result;		
	}
	
	private static ViewerStudyStudyType translate(Study study)
	throws MethodException
	{
		ViewerStudyStudyType result = new ViewerStudyStudyType();
		if(study == null)
			return result;
		
		result.setDescription(study.getDescription());
		
		result.setEvent(study.getEvent());
		result.setImageCount(study.getImageCount());
		result.setImagePackage(study.getImagePackage());
		result.setImageType(study.getImageType());
		result.setNoteTitle(study.getNoteTitle());
		result.setOrigin(study.getOrigin());
		result.setPatientId(study.getPatientId());
		result.setPatientName(study.getPatientName());
		result.setProcedureDescription(study.getProcedure());
		result.setSiteNumber(study.getSiteNumber());
		result.setSiteName(study.getSiteName());
		result.setSiteAbbreviation(study.getSiteAbbr());
		result.setSpecialtyDescription(study.getSpecialty());
		result.setProcedureDate(study.getProcedureDate());
		
		result.setStudyPackage(study.getImagePackage());
		result.setStudyClass(study.getStudyClass() == null ? "" : study.getStudyClass());
		result.setStudyType(study.getImageType());
		result.setCaptureDate(study.getCaptureDate());
		result.setCapturedBy(study.getCaptureBy());	
		result.setGroupIen(study.getGroupIen());
		result.setAlternateExamNumber(study.getAlternateExamNumber());
		
		result.setDocumentDate(study.getDocumentDate());
		result.setSensitive(study.isSensitive());
		// return null for the UID instead of the empty string to be consistent with the WSDL - DKB
		if (study.getStudyUid() != null && study.getStudyUid().trim().length() > 0)
			result.setDicomUid(study.getStudyUid());
		else
			result.setDicomUid(null);
		
		//result.setStudyId(study.getStudyUrn().toString(SERIALIZATION_FORMAT.CDTP));
		// JMW 1/3/13 - using VFTP format so any additional parameters are URL encoded properly
		result.setStudyId(study.getStudyUrn().toString(SERIALIZATION_FORMAT.VFTP));
		
		if(study.getSeries() != null)
		{
			ViewerStudySeriesType [] seriesType = new ViewerStudySeriesType[study.getSeriesCount()];
			int i = 0;
			for(Series ser : study)
			{
				seriesType[i] = translate(ser);
				i++;
			}			
			result.setSeries(new ViewerStudySeriesesType(seriesType));
		}
		
		if(study.getFirstImage() == null)
			throw new MethodException("first image is null");
		
		ViewerStudyImageType firstImage = translate(study.getFirstImage());
		result.setFirstImage(firstImage);
		result.setFirstImageId(firstImage.getImageId());
		
		if(study.getModalities() != null)
			result.setStudyModalities(study.getModalities().toArray(new String [study.getModalities().size()]));
		
		
		return result;
	}
	
	private static ViewerStudySeriesType translate(Series series)
	{
		ViewerStudySeriesType result = new ViewerStudySeriesType();
		result.setModality(series.getModality());
		//result.setSeriesIen(series.getSeriesIen());
		result.setSeriesNumber(series.getSeriesNumber());
		result.setSeriesUid(series.getSeriesUid());
		
		ViewerStudyImageType [] images = new ViewerStudyImageType[series.getImageCount()];
		int i = 0;
		for(Image image : series)
		{
			images[i] = translate(image);
			i++;
		}
		
		result.setImages(new ViewerStudyImagesType(images));
		return result;
	}
	
	private static ViewerStudyImageType translate(Image image)
	{
		ViewerStudyImageType result = new ViewerStudyImageType();
		
		if(image == null)
			return result;
		
		String imageId = image.getImageUrn().toString(SERIALIZATION_FORMAT.RAW);
		
		result.setImageId( imageId );		
		
		// Exchange fields
		// return null for the UID instead of the empty string to be consistent with the WSDL - DKB
		if (image.getImageUid()!= null && image.getImageUid().trim().length() > 0)
		{
			result.setDicomUid(image.getImageUid().trim());
		}
		
		
		if (image.getImageNumber() != null && image.getImageNumber().trim().length() > 0)
		{
			try
			{
				Integer imageNumber = new Integer(image.getImageNumber());
				result.setImageNumber(imageNumber);
			}
			catch (NumberFormatException ex)
			{
				result.setImageNumber(null);
			}
		}
		else
		{
			result.setImageNumber(null);
		}
		
		result.setDicomImageNumber(image.getDicomImageNumberForDisplay());
		result.setDicomSequenceNumber(image.getDicomSequenceNumberForDisplay());
		
		VistaImageType vistaImageType = VistaImageType.valueOfImageType(image.getImgType());
		result.setImageType(vistaImageType == null ? "" : vistaImageType.name());
		result.setThumbnailImageUri(getThumbnailImageUri(image, imageId));
		result.setReferenceImageUri(getReferenceImageFullUri(image, imageId));
		result.setDiagnosticImageUri(getDiagnosticImageUri(image, imageId));
		result.setImageModality(image.getImageModality());		
		
		result.setCaptureDate(image.getCaptureDate());
		result.setDocumentDate(image.getDocumentDate());
		result.setSensitive(image.isSensitive());
		result.setDescription(image.getDescription());
		result.setAlternateExamNumber(image.getAlternateExamNumber());
		
		return result;
	}
	
	private static String getThumbnailImageUri(Image image, String imageUrn)
	{
		if((image.getAbsFilename() != null) && (image.getAbsFilename().startsWith("-1")))
		{
			return "";
		}
		else
		{
			return "imageURN=" + imageUrn + "&imageQuality=20&contentType=" + getContentType(image, ImageQuality.THUMBNAIL);
		}
	}
	
	private  static String getReferenceImageFullUri(Image image, String imageUrn)
	{
		boolean isRadImage = isRadImage(image);
		if((image.getFullFilename() != null) && (image.getFullFilename().startsWith("-1")))
		{
			return "";
		}
		else
		{
			// in this interface, if a rad image then there is a reference quality, if not rad then no ref quality - just diagnostic
			if(isRadImage)
			{
				int imageQuality = ImageQuality.REFERENCE.getCanonical();
				return "imageURN=" + imageUrn + "&imageQuality=" + imageQuality + "&contentType=" + getContentType(image, 
						ImageQuality.REFERENCE) + "&contentTypeWithSubType=" + getContentType(image, ImageQuality.REFERENCE);
			}
			else
			{
				return "";
			}
		}
	}
	
	
	
	private static String getDiagnosticImageUri(Image image, String imageUrn)
	{
		boolean isRadImage = isRadImage(image);		
		if((image.getBigFilename() != null) && (image.getBigFilename().startsWith("-1")))
		{
			return "";
		}
		else
		{
			if(isRadImage)
			{
				return "imageURN=" + imageUrn + "&imageQuality=90&contentType=" + 
					getContentType(image, ImageQuality.DIAGNOSTIC) + "&contentTypeWithSubType=" + getContentType(image, ImageQuality.DIAGNOSTIC);
			}
			else
			{
				return "imageURN=" + imageUrn + "&imageQuality=" + ImageQuality.DIAGNOSTICUNCOMPRESSED.getCanonical() + "&contentType=" + getContentType(image, 
						ImageQuality.REFERENCE);
			}
		}		
	}
	
	private static String getContentType(Image image, ImageQuality imageQuality)
	{
		return  getContentType(image, imageQuality, ViewerStudyFacadeConfiguration.getConfiguration());
	}

}
