/**
 * ImageMixMetadataImpl.java
 *
 * for MIX metadata service (a CVIX only service) the client interface is placed here
 * and implemented locally
 * 
 * @author DNS
 */
package gov.va.med.imaging.mix.proxy.v1;

import java.util.HashMap;
// import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Date;
import java.text.SimpleDateFormat;

import javax.ws.rs.core.MediaType;

// import java.text.ParseException;
import org.apache.commons.httpclient.methods.GetMethod;

import gov.va.med.imaging.core.interfaces.exceptions.ConnectionException;
import gov.va.med.imaging.core.interfaces.exceptions.MethodException;
import gov.va.med.imaging.mix.rest.endpoints.MixDiagnosticReportRestUri;
import gov.va.med.imaging.mix.rest.endpoints.MixImagingStudyRestUri;
import gov.va.med.imaging.mix.rest.proxy.MixRestGetClient;
import gov.va.med.imaging.mix.rest.proxy.AbstractMixRestImageProxy;
import gov.va.med.imaging.mix.translator.MixJSONConverter;
import gov.va.med.imaging.url.mix.configuration.MIXConfiguration;
import gov.va.med.imaging.proxy.services.ProxyServiceType;
import gov.va.med.imaging.proxy.services.ProxyServices;
// import gov.va.med.SERIALIZATION_FORMAT;
// import gov.va.med.imaging.exchange.enums.StudyLoadLevel;
// import gov.va.med.imaging.exchange.translation.exceptions.TranslationException;
import gov.va.med.imaging.mix.webservices.fhir.types.v1.FilterType;
import gov.va.med.imaging.mix.webservices.fhir.types.v1.RequestorType;
// import gov.va.med.imaging.mix.webservices.fhir.types.v1.RequestorTypePurposeOfUse;
import gov.va.med.imaging.mix.webservices.fhir.types.v1.ReportStudyListResponseType;
import gov.va.med.imaging.mix.webservices.fhir.types.v1.ReportType;
import gov.va.med.imaging.mix.webservices.fhir.types.v1.StudyType;
import gov.va.med.imaging.mix.webservices.fhir.v1.ImageMetadata;
import gov.va.med.imaging.mix.webservices.fhir.exceptions.*;


public class ImageMixMetadataImpl
extends AbstractMixRestImageProxy
implements ImageMetadata
{
	private static String defaultFromDate = "1900-01-01";
				
	private String mixNode;
	private RequestorType requestor;
	private FilterType filter;
	private String patientId;
	private Boolean fullStudylist;
	private String transactionId;

	private ReportStudyListResponseType reportStudyListResponse=null;
	
	public ImageMixMetadataImpl (ProxyServices proxyServices, MIXConfiguration mixConfiguration) {
		super(proxyServices, mixConfiguration);
	}

	@Override
	protected ProxyServiceType getProxyServiceType() {
		return ProxyServiceType.metadata;
	}
	
	@Override
	protected String getDataSourceVersion() {
		return "1";
	}
	
	@Override
	protected String getRestServicePath() {
		// Note, mix service path is the same for all MIX services
		return MixDiagnosticReportRestUri.mixServicePath;
	}

	@Override
	protected void addOptionalGetInstanceHeaders(GetMethod getMethod) {
		// no Image access here, nothing to do...
	}

	@Override
	protected ProxyServiceType getInstanceRequestProxyServiceType() {
		return ProxyServiceType.image;
	}

	@Override
	protected ProxyServiceType getTextFileRequestProxyServiceType() {
		// unused method in MIX world
		return ProxyServiceType.text;
	}
	
	/**
	 * Make MIX client calls to collect Study Graphs (including reports) for given patient within date range; with limited timeout
	 * @param datasource the target MIX server (DAS, etc.) gateway: 200 - default
	 * @param reqtor requesting entity (Name, SSN, site)
	 * @param filtr from/to date, study id
	 * @param patId ICN
	 * @param transactId for transaction logging
	 * @return ReportStudyListResponseType have status, error info and study graphs (including report)
	 * @throws MIXMetadataException
	 */
	// TODO: limit timeout to 30-45 second!?
	public ReportStudyListResponseType getPatientReportStudyList(String datasource, RequestorType reqtor,
			FilterType filtr, String patId, Boolean fullTree, String transactId, String requestedSite) // requestedSite is ignored, it is 200, the datasource
			throws MIXMetadataException
	{
		if ((datasource==null) || (datasource.isEmpty()) ||
			(filtr==null) ||
			(transactId == null) ||(transactId.isEmpty()) ||
			(patId == null) || (patId.isEmpty())) {
			throw new MIXMetadataException("getPatientReportStudyList: Invalid or null input parameter(s)!");
		}
		this.mixNode = datasource;
		this.requestor = reqtor;
		this.filter = filtr;
		this.patientId = patId;
		this.fullStudylist = fullTree;
		this.transactionId = transactId;
		// requestedSite is ignored, it is the datasource (200)
		
		try { 
			// get the Patient's study shallow list with reports (JSON-> DiagnosticReport) in local variable
			getPatientReportsAndShallowStudyList();
		} 
		catch (MIXDiagnosticReportException mdre) {
			// ***
			throw new MIXMetadataException("getPatientReportStudyList: Failed to get reports with shallow study list!");
		}
		
		// exit on empty list or if no full tree needed
		if ((reportStudyListResponse==null) || (reportStudyListResponse.getStudies().length == 0) || !fullTree) {
			return reportStudyListResponse;
		}

		reportStudyListResponse.setPartialResponse(true);

		// collect study graphs for each study (JSON-> ImagingSudy-s of DiagnosticReport)
		// *** decide to give up after x failures?!
		Integer failedStudyCount = 0;
		for (StudyType study : reportStudyListResponse.getStudies()) {
			try {
				StudyType studyType = getPatientFullStudyList(study.getDicomUid());
				if (studyType==null) {
					failedStudyCount++;
				} else {
					// make sure the report there is kept!
					studyType.setReportContent(study.getReportContent());
					study = studyType;
				}
			}
			catch (MIXImagingStudyException mdre) {
				// *** fill errors array
				failedStudyCount++;
			}
		}
		if (failedStudyCount == 0)
			reportStudyListResponse.setPartialResponse(false);
	
		return reportStudyListResponse;
	}
	
	// fills reports and shallow study list only
	private void getPatientReportsAndShallowStudyList()
			throws MIXDiagnosticReportException
	{
		// ConnectionException & MethodException
		getLogger().info("getPatientReportsAndShallowStudyList, Transaction [" + transactionId + "] initiated, patient '" + patientId + "'."); // + routingToken.toRoutingTokenString() + "'.");
		setDataSourceMethodAndVersion("getPatientReportsAndShallowStudyList");

		String fromDate = defaultFromDate;
		if ((filter != null) && (filter.getFromDate() != null) && !filter.getFromDate().isEmpty())
			fromDate = filter.getFromDate(); // make sure of yyyy-MM-dd format

		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd", Locale.US);
		Date d = new Date();
		String toDate = sdf.format(d);
		if ((filter != null) && (filter.getToDate() != null) && !filter.getToDate().isEmpty())
			toDate = filter.getToDate(); // make sure of yyyy-MM-dd format
		
		// compose URL to client
		Map<String, String> urlParameterKeyValues = new HashMap<String, String>();
		// urlParameterKeyValues.put("{routingToken}", routingToken.toRoutingTokenString());
		urlParameterKeyValues.put("{patientIcn}", patientId);
		urlParameterKeyValues.put("{fromDate}", fromDate);
		urlParameterKeyValues.put("{toDate}", toDate);
		String url="";
		try {
			url = getWebResourceUrl(MixDiagnosticReportRestUri.reportStudyListPath, urlParameterKeyValues); 
		}
		catch (ConnectionException ce) {
			throw new MIXDiagnosticReportException("getPatientReportsAndShallowStudyList: Failed to connect to/compose URL source!" + ce.getMessage());
		}
		
		// request to Client and translate (json) response
		MixRestGetClient getClient = new MixRestGetClient(url, MediaType.APPLICATION_JSON_TYPE, mixConfiguration);
		String jsonResponse="";
		try{
			jsonResponse = getClient.executeRequest(String.class);
		}
		catch (ConnectionException ce) {
			throw new MIXDiagnosticReportException("getPatientReportsAndShallowStudyList: Failed to connect to Client!");
		}
		catch (MethodException me) {
			throw new MIXDiagnosticReportException("getPatientReportsAndShallowStudyList: Error getting Client response!" + me.getMessage());
		}
		getLogger().info("getPatientReportsAndShallowStudyList, Transaction [" + transactionId + "] returned [" + (reportStudyListResponse == null ? "null" : "not null") + "] reportStudyListResponse.");
		ReportStudyListResponseType result = MixJSONConverter.ConvertDiagnosticReportToJava(jsonResponse);
		getLogger().info("getPatientReportsAndShallowStudyList, Transaction [" + transactionId + "] returned response of [" + (result == null ? "null" : "not null") + "] reportStudyListResponse business object.");
		reportStudyListResponse = result;
	}


	// fills study graph for given study
	private StudyType getPatientFullStudyList(String studyUID)
			throws MIXImagingStudyException
	{
		// ConnectionException & MethodException
		getLogger().info("getPatientFullStudyList, Transaction [" + transactionId + "] initiated, study '" + studyUID + "'."); // + routingToken.toRoutingTokenString() + "'.");
		setDataSourceMethodAndVersion("getPatientFullStudyList");

		// compose URL to client
		Map<String, String> urlParameterKeyValues = new HashMap<String, String>();
		// urlParameterKeyValues.put("{routingToken}", routingToken.toRoutingTokenString());
		urlParameterKeyValues.put("{studyUid}", studyUID);
		String url="";
		try {
			url = getWebResourceUrl(MixImagingStudyRestUri.studyListPath, urlParameterKeyValues); 
		}
		catch (ConnectionException ce) {
			throw new MIXImagingStudyException("getPatientFullStudyList: Failed to connect to/compose URL source!" + ce.getMessage());
		}
		
		// request to Client and translate (json) response
		MixRestGetClient getClient = new MixRestGetClient(url, MediaType.APPLICATION_JSON_TYPE, mixConfiguration);
		String jsonResponse="";
		try{
			jsonResponse = getClient.executeRequest(String.class);
		}
		catch (ConnectionException ce) {
			throw new MIXImagingStudyException("getPatientFullStudyList: Failed to connect to Client!");
		}
		catch (MethodException me) {
			throw new MIXImagingStudyException("getPatientFullStudyList: Error getting Client response!" + me.getMessage());
		}
		getLogger().info("getPatientFullStudyList, Transaction [" + transactionId + "] returned [" + (reportStudyListResponse == null ? "null" : "not null") + "] reportStudyListResponse.");
		StudyType result = MixJSONConverter.ConvertImagingStudyToJava(jsonResponse);
		getLogger().info("getPatientFullStudyList, Transaction [" + transactionId + "] returned response of [" + (result == null ? "null" : "not null") + "] reportStudyListResponse business object.");
        return result;
	}
	
	/**
	 * Make MIX client call(s) to collect one Study Report with limited timeout
	 * @param datasource the target MIX server (DAS, etc.) gateway: 200 - default
	 * @param reqtor requesting entity (Name, SSN, site)
	 * @param patientId ICN
	 * @param transactionId for transaction logging
	 * @param studyId opaque, unique study identifier (preferably Study UID)
	 * @return ReportType including the report or null if not found
	 * @throws MIXMetadataException
	 */
	// TODO: limit timeout to 30-45 second!?
	public 	ReportType getPatientReport(String datasource, RequestorType reqtor, String patId,  String studyId, String transactId) // DICOM Study UID expected, not studyUrn.toString(SERIALIZATION_FORMAT.NATIVE)!
			throws MIXMetadataException {
		ReportType reportType=null;
		if ((datasource==null) || (datasource.isEmpty()) ||
				(transactId == null) ||(transactId.isEmpty()) ||
				(studyId == null) ||(studyId.isEmpty()) ||
				(patId == null) || (patId.isEmpty())) {
				throw new MIXMetadataException("getPatientReport: Invalid or null input parameter(s)!");
		}
		this.mixNode = datasource;
		this.requestor = reqtor;
		this.patientId = patId;
		this.transactionId = transactId;

		if (reportStudyListResponse==null) {
			// request entire report/shallow study list for patient (filter is null)
			try {
				reportStudyListResponse = getPatientReportStudyList(datasource, reqtor, null, patId, false, transactId, mixNode);
			} 
			catch (MIXMetadataException mmde) {
				throw new MIXMetadataException("getPatientReport: Failed getting Reports/Shallow study list.");
			}
		}

		// get report from local reportStudyListResponse structure using studyId
		for (StudyType study : reportStudyListResponse.getStudies()) {
			if (study.getDicomUid().equals(studyId) || study.getStudyId().equals(studyId)) {
				reportType = new ReportType();
				reportType.setRadiologyReport(study.getReportContent());
				reportType.setPatientId(patId);
				reportType.setStudyId(studyId);
				reportType.setSiteNumber(mixNode); // 200
				reportType.setSiteName("DOD");
				reportType.setSiteAbbreviation("DOD");
				reportType.setProcedureDate(study.getProcedureDate()); // *** check/fix format translate!
			}
		}
		return reportType;
	}

}
